<?php
/**
 * Vacation_Driver_ldap:: implements the Vacation_Driver API for
 * LDAP-compliant mail servers (such as Exim).
 *
 * Parameters:
 * (required)
 * - host     - hostname of the LDAP server
 * - port     - port number of the LDAP service
 * - basedn   - base DN of the user directory
 * - uid      - attribute to use for uid
 * - vacation - attribute to use for storing the vacation message
 * - active   - attribute which determines if the vacation message is active
 * (optional)
 * - userdn  - another way of specifying the user DN (instead of constructing
 *             it from uid+basedn).
 * - version - Protocol version for the LDAP server (PHP defaults to version 2.
 *             OpenLDAP >= 2.1.4 uses version 3, and so must be set explicitly).
 *
 * $Horde: vacation/lib/Driver/ldap.php,v 1.17.2.14 2009/01/06 15:28:08 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @author  Jan Schneider <jan@horde.org>
 * @package Vacation
 */
class Vacation_Driver_ldap extends Vacation_Driver {

    /**
     * Pointer to the ldap connection.
     */
    var $_ds;

    /**
     * Sets up vacation notices for a user.
     *
     * @param string $password  The password for the user.
     * @param string $message   The text of the vacation notice.
     * @param string $subject   The subject of the vacation notice.
     * @param string $from      The From: address of the vacation notice.
     * @param string $alias     The alias email address.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        // Make sure the configuration file is correct.
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Get the user's DN.
        if (isset($this->_params[$this->_realm]['userdn'])) {
            $userdn = $this->_params[$this->_realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn();
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $res = $this->_connect($userdn, $password);
        if (is_a($res, 'PEAR_Error')) {
            $this->_disconnect();
            if ($res->getMessage() == _("Could not bind to ldap server")) {
                return PEAR::raiseError(_("Incorrect Password"));
            }
            return $res;
        }

        // Prepare the message. \n->\n\n and UTF-8 encode.
        $full_message = $this->_buildMessage($message, $subject, $from);
        $full_message = str_replace("\r\n", "\\n", $full_message);
        $full_message = String::convertCharset($full_message, NLS::getCharset(), 'UTF-8');

        // Change the user's vacation.
        $newDetails[$this->_params[$this->_realm]['vacation']] = $full_message;
        $newDetails[$this->_params[$this->_realm]['active']] = explode(
            '|', $this->_params[$this->_realm]['enabled']);
        $res = ldap_mod_replace($this->_ds, $userdn, $newDetails);
        if (!$res) {
            $res = PEAR::raiseError(ldap_error($this->_ds));
            $this->_disconnect();
            return $res;
        }

        $res = $this->_setVacationAlias($userdn);

        // Disconnect.
        $this->_disconnect();

        return $res;
    }

    /**
     * Sets or creates a vacation mail alias.
     *
     * Some MTA/LDAP/Vacation implementations require an extra mail alias
     * (ex. user@example.org -> user@example.org, user@autoreply.example.org)
     *
     * You should override this method in your extended LDAP driver class, if
     * you need this feature.
     *
     * @param string $userdn  The LDAP DN for the current user.
     */
    function _setVacationAlias($userdn)
    {
        return true;
    }

    /**
     * Deactivates the vacation notice.
     *
     * This does not delete the vacation message, just marks it as disabled.
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        // Make sure the configuration file is correct.
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Get the user's dn.
        if (isset($this->_params[$this->_realm]['userdn'])) {
            $userdn = $this->_params[$this->_realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn();
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $result = $this->_connect($userdn, $password);
        if (is_a($result, 'PEAR_Error')) {
            $this->_disconnect();
            if ($result->getMessage() == _("Could not bind to ldap server")) {
                return PEAR::raiseError(_("Incorrect Password"));
            }
            return $result;
        }

        // Set the vacation message to inactive.
        $newDetails[$this->_params[$this->_realm]['active']] =
            $this->_params[$this->_realm]['disabled'];
        $result = ldap_mod_replace($this->_ds, $userdn, $newDetails);
        if (!$result) {
            return PEAR::raiseError(ldap_error($this->_ds));
        }

        // Delete the unnecessary vacation alias (if present).
        $result = $this->_unsetVacationAlias($userdn);

        // Disconnect.
        $this->_disconnect();

        return $result;
    }

    /**
     * Unsets or removes a vacation mail alias.
     *
     * @see _setVacationAlias()
     *
     * @param string $userdn  The LDAP DN for the current user.
     */
    function _unsetVacationAlias($userdn)
    {
        return true;
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return array  Vacation details or PEAR_Error on failure.
     */
    function _getUserDetails($password)
    {
        // Make sure the configuration file is correct.
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Get the user's DN.
        if (isset($this->_params[$this->_realm]['userdn'])) {
            $userdn = $this->_params[$this->_realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn();
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $result = $this->_connect($userdn, $password);
        if (is_a($result, 'PEAR_Error')) {
            $this->_disconnect();
            if ($result->getMessage() == _("Could not bind to ldap server")) {
                return PEAR::raiseError(_("Incorrect Password"));
            }
            return $result;
        }

        // Retrieve vacation information.
        $vacation = $this->_getVacation($userdn);

        // Prepare the message.
        $vacation['message'] = String::convertCharset($vacation['message'], 'UTF-8');

        // Parse message.
        $vacation = array_merge($vacation, $this->_parseMessage($vacation['message']));

        return $vacation;
    }

    function _getVacation($userdn)
    {
        $filter = $this->_params[$this->_realm]['uid'] . '=' . $this->_user;
        $searchAttrs = array($this->_params[$this->_realm]['vacation'],
                             $this->_params[$this->_realm]['active']);

        /* Log the query at a DEBUG log level. */
        Horde::logMessage(sprintf('LDAP query by Vacation_Driver_ldap::_getVacation(): root = "%s"; filter = "%s"; attributes = "%s"; timelimit = %d',
                                  $userdn, $filter, implode(', ', $searchAttrs),
                                  $this->_params[$this->_realm]['timeout']),
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);

        // Get the vacation message and the vacation status.
        $sr = ldap_search($this->_ds, $userdn, $filter, $searchAttrs, 0, 0,
                          $this->_params[$this->_realm]['timeout']);

        $entry = ldap_first_entry($this->_ds, $sr);
        if (!$entry) {
            return array('vacation' => false);
        }

        $retAttrs = ldap_get_attributes($this->_ds, $entry);
        if (!$retAttrs) {
            return array('vacation' => false);
        }

        // Set default values.
        $vacationInfo['message']  = '';
        $vacationInfo['vacation'] = $this->_params[$this->_realm]['disabled'];

        // Are there any returned attributes / values?
        $messageAttr = $this->_params[$this->_realm]['vacation'];
        if (isset($retAttrs[$messageAttr])) {
            $vacationInfo['message'] = $retAttrs[$messageAttr][0];
        }

        $vacationAttr = $this->_params[$this->_realm]['active'];
        if (isset($retAttrs[$vacationAttr])) {
            unset($retAttrs[$vacationAttr]['count']);
            $vacationInfo['vacation'] = implode('|', $retAttrs[$vacationAttr]);
        }

        return $vacationInfo;
    }

    /**
     * Check if the realm has a specific configuration.  If not, try to fall
     * back on the default configuration.  If still not a valid configuration
     * then exit with an error.
     */
    function _checkConfig()
    {
        // If no realm passed in, or no host config for the realm passed in,
        // then we fall back to the default realm
        if (empty($this->_params[$this->_realm]['server'])) {
            $this->_realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['host']) ||
            empty($this->_params[$this->_realm]['port']) ) {
            return PEAR::raiseError(_("The module is not properly configured!"));
        }
    }

    /**
     * Lookup and return the user's dn.
     *
     * @return string  The LDAP dn for the user.
     */
    function _lookupdn()
    {
        // Bind as guest.
        $this->_connect();

        // Construct search.
        $search = $this->_params[$this->_realm]['uid'] . '=' . $this->_user;
        if (!empty($this->_params[$this->_realm]['realm'])) {
            $search .= '@' . $this->_params[$this->_realm]['realm'];
        }

        /* Log the query at a DEBUG log level. */
        Horde::logMessage(sprintf('LDAP query by Vacation_Driver_ldap::_lookupdn(): root = "%s"; filter = "%s"; timelimit = %d',
                                  $this->_params[$this->_realm]['basedn'],
                                  $search, $this->_params[$this->_realm]['timeout']),
                          __FILE__, __LINE__, PEAR_LOG_DEBUG);

        // Get userdn.
        $result = @ldap_search($this->_ds, $this->_params[$this->_realm]['basedn'],
                               $search, array(), 0, 0,
                               $this->_params[$this->_realm]['timeout']);
        if (!$result ||
            !($entry = ldap_first_entry($this->_ds, $result))) {
            $this->_disconnect();
            return PEAR::raiseError(_("User not found."));
        }
        $userdn = ldap_get_dn($this->_ds, $entry);

        // Disconnect.
        $this->_disconnect();

        return $userdn;
    }

    /**
     * Connects to the LDAP server and binds as the guest user or as the
     * optional userdn.
     *
     * @param string $userdn    The DN to use when binding non-anonymously.
     * @param string $password  The password for $userdn.
     *
     * @return boolean  True on success, PEAR_Error otherwise.
     */
    function _connect($userdn = null, $password = null)
    {
        $this->_ds = ldap_connect($this->_params[$this->_realm]['host'],
                                  $this->_params[$this->_realm]['port']);
        if (!$this->_ds) {
            return PEAR::raiseError(_("Could not connect to ldap server"));
        }
        if (isset($this->_params[$this->_realm]['version'])) {
            ldap_set_option($this->_ds, LDAP_OPT_PROTOCOL_VERSION,
                            $this->_params[$this->_realm]['version']);
        }

        if (!empty($this->_params[$this->_realm]['binddn'])) {
            $result = @ldap_bind($this->_ds, $this->_params[$this->_realm]['binddn'],
                                 $this->_params[$this->_realm]['bindpw']);
        } elseif (!is_null($userdn)) {
            $result = @ldap_bind($this->_ds, $userdn, $password);
        } else {
            $result = @ldap_bind($this->_ds);
        }

        if (!$result) {
            return PEAR::raiseError(_("Could not bind to ldap server"));
        }

        return true;
    }

    /**
     * Close the ldap connection.
     */
    function _disconnect()
    {
        @ldap_close($this->_ds);
    }

}
