
#include "config.h"
#include "logger.h"
#include "convertpluginloader.h"
#include "replaygainpluginloader.h"
#include "ripperpluginloader.h"
#include "options.h"

#include <qdir.h>
#include <qfile.h>

#include <klocale.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kmimetype.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kio/netaccess.h>

// NOTE currently only the possibilities of a file format are respected,
//       the possibilities of the current encoder are ignored
// - seems to be fixed

FormatItem::FormatItem()
{
    // reset everything
    mime_types.clear();
    size = 0;
    compressionType = FormatInfo::lossy;
    compressionLevel = 0;
    internalReplayGain = false;
    encoders.clear();
    decoders.clear();
    replaygains.clear();
    //repairers.clear();
    encoder = 0;
    decoder = 0;
    replaygain = 0;
    //repairer = 0;
}

FormatItem::~FormatItem()
{}


Config::Config( Logger* _logger )
{
    logger = _logger;

    currentRipper = 0; // this seems to be unnecessary
}

Config::~Config()
{}

void Config::read()
{
    loadPlugins();
    readProfiles();

    KConfig *conf = kapp->config();
    QStringList listDefaults;
    int intDefault;
    bool boolDefault;
    QString ripper;
    QString encoder;
    QString decoder;
    QString replaygain;
    int rank;

    conf->setGroup( "General" ) ;
    data.app.configVersion = conf->readNumEntry( "configVersion", 0 );

/*    if( configVersion == 0 ) {
        conf->setGroup( "Interface" );
        data.general.updateDelay = conf->readNumEntry( "pauseMS", 500 );
        data.general.startTab = conf->readNumEntry( "startTab", 0 );
        data.general.showToolBar = conf->readBoolEntry( "showToolBar", false );

        conf->setGroup( "Backends" );
        listDefaults.clear();
        QString datadir = locateLocal( "data", "soundkonverter/bin/" );
        datadir.remove( datadir.length() - 1, 1 );
        listDefaults.append( datadir );
        listDefaults.append( QDir::homeDirPath() + "/bin" );
        listDefaults.append( "/usr/local/bin" );
        listDefaults.append( "/usr/bin" );
        data.environment.directories = conf->readListEntry( "directories", listDefaults );
        listDefaults.clear();
        data.environment.foundPrograms = conf->readListEntry( "programs", listDefaults );

        conf->setGroup( "Misc" );
        //configVersion = conf->readNumEntry( "configVersion", 0 );
        data.general.askForNewOptions = conf->readBoolEntry( "askForNewOptions", true );

        return;
    }*/

    conf->setGroup( "General" ) ;
    //configVersion = conf->readNumEntry( "configVersion", 0 );
    data.general.startTab = conf->readNumEntry( "startTab", 0 );
    data.general.lastTab = conf->readNumEntry( "lastTab", 0 );
    data.general.defaultProfile = conf->readEntry( "defaultProfile", i18n("Last used") );
    data.general.defaultFormat = conf->readEntry( "defaultFormat", i18n("Last used") );
//     data.general.defaultOutputDirectory = conf->readEntry( "defaultOutputDirectory", QDir::homeDirPath() + "/soundKonverter/%b/%d - %n - %a - %t" );
    data.general.specifyOutputDirectory = conf->readEntry( "specifyOutputDirectory", QDir::homeDirPath() + "/soundKonverter" );
    data.general.metaDataOutputDirectory = conf->readEntry( "metaDataOutputDirectory", QDir::homeDirPath() + "/soundKonverter/%b/%d - %n - %a - %t" );
    data.general.copyStructureOutputDirectory = conf->readEntry( "copyStructureOutputDirectory", QDir::homeDirPath() + "/soundKonverter" );
    data.general.useVFATNames = conf->readBoolEntry( "useVFATNames", true );
    data.general.priority = conf->readNumEntry( "priority", 10 );
    data.general.numFiles = conf->readNumEntry( "numFiles", 3 );
    data.general.updateDelay = conf->readNumEntry( "updateDelay", 500 );
    data.general.askForNewOptions = conf->readBoolEntry( "askForNewOptions", true );
    data.general.showToolBar = conf->readBoolEntry( "showToolBar", false );

    conf->setGroup( "Plugins" );
    data.plugins.checkForUpdates = conf->readBoolEntry( "checkForUpdates", false );

    conf->setGroup( "Environment" );
    listDefaults.clear();
    QString datadir = locateLocal( "data", "soundkonverter/bin/" );
    datadir.remove( datadir.length() - 1, 1 );
    listDefaults.append( datadir );
    listDefaults.append( QDir::homeDirPath() + "/bin" );
    listDefaults.append( "/usr/local/bin" );
    listDefaults.append( "/usr/bin" );
    data.environment.directories = conf->readListEntry( "directories", listDefaults );
    listDefaults.clear();
    data.environment.foundPrograms = conf->readListEntry( "foundPrograms", listDefaults );

    conf->setGroup( "Ripper" );
    ripper = conf->readEntry( "ripper", "kio_audiocd" );

    if( ripper == "kio_audiocd" ) rank = 10000;
    else rank = 60;
    currentRipper = 0; // this is a valid ripper (kio_audiocd)
    for( QValueList<RipperPlugin*>::Iterator b = rippers.begin(); b != rippers.end(); ++b ) {
        binaries[ (*b)->rip.bin ] = "";
        if( (*b)->rip.rank > rank ) {
            rank = (*b)->rip.rank;
            currentRipper = (*b);
        }
        if( (*b)->rip.bin == ripper ) {
            rank = 10000; // should be high enougth to overbid the other plugins
            currentRipper = (*b);
        }
    }

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it )
    {
        for( QValueList<ConvertPlugin*>::Iterator b = (*it).encoders.begin(); b != (*it).encoders.end(); ++b ) {
            binaries[ (*b)->enc.bin ] = "";
        }

        for( QValueList<ConvertPlugin*>::Iterator b = (*it).decoders.begin(); b != (*it).decoders.end(); ++b ) {
            binaries[ (*b)->dec.bin ] = "";
        }

        for( QValueList<ReplayGainPlugin*>::Iterator b = (*it).replaygains.begin(); b != (*it).replaygains.end(); ++b ) {
            binaries[ (*b)->replaygain.bin ] = "";
        }

        for( QMap<QString, QString>::Iterator b = binaries.begin(); b != binaries.end(); ++b ) {
            for( QStringList::Iterator c = data.environment.directories.begin(); c != data.environment.directories.end(); ++c )
            {
                if( b.data() == "" && QFile::exists((*c) + "/" + b.key()) ) {
                    b.data() = (*c) + "/" + b.key();
                }
            }
        }
        QStringList foundPrograms;
        for( QMap<QString, QString>::Iterator b = binaries.begin(); b != binaries.end(); ++b ) {
            if( b.data() != "" ) {
                foundPrograms.append( b.data() );
            }
        }
        if( foundPrograms != data.environment.foundPrograms ) {
            backendsChanged = true;
        }
        else {
            backendsChanged = false;
        }
    }

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it )
    {
        conf->setGroup( (*it).mime_types.first() );
        encoder = conf->readEntry( "encoder" );
        decoder = conf->readEntry( "decoder" );
        replaygain = conf->readEntry( "replaygain" );

        (*it).encoder = 0;
        (*it).decoder = 0;
        (*it).replaygain = 0;

        rank = 0;
        for( QValueList<ConvertPlugin*>::Iterator b = (*it).encoders.begin(); b != (*it).encoders.end(); ++b ) {
            if( (*b)->enc.rank > rank && binaries[(*b)->enc.bin] != "" ) {
                rank = (*b)->enc.rank;
                (*it).encoder = (*b);
            }
            if( (*b)->enc.bin == encoder && binaries[(*b)->enc.bin] != "" ) {
                rank = 10000; // should be high enougth to overbid the other plugins
                (*it).encoder = (*b);
            }
        }

        rank = 0;
        for( QValueList<ConvertPlugin*>::Iterator b = (*it).decoders.begin(); b != (*it).decoders.end(); ++b ) {
            if( (*b)->dec.rank > rank && binaries[(*b)->dec.bin] != "" ) {
                rank = (*b)->dec.rank;
                (*it).decoder = (*b);
            }
            if( (*b)->dec.bin == decoder && binaries[(*b)->dec.bin] != "" ) {
                rank = 10000; // should be high enougth to overbid the other plugins
                (*it).decoder = (*b);
            }
        }

        rank = 0;
        for( QValueList<ReplayGainPlugin*>::Iterator b = (*it).replaygains.begin(); b != (*it).replaygains.end(); ++b ) {
            if( (*b)->replaygain.rank > rank && binaries[(*b)->replaygain.bin] != "" ) {
                rank = (*b)->replaygain.rank;
                (*it).replaygain = (*b);
            }
            if( (*b)->replaygain.bin == replaygain && binaries[(*b)->replaygain.bin] != "" ) {
                rank = 10000; // should be high enougth to overbid the other plugins
                (*it).replaygain = (*b);
            }
        }

        if( (*it).encoder != 0 && (*it).encoder->enc.strength.enabled ) {
            if( (*it).encoder->enc.strength.range_max >= (*it).encoder->enc.strength.range_min )
                intDefault = (*it).encoder->enc.strength.default_value / (*it).encoder->enc.strength.step;
            else
                intDefault = ( (*it).encoder->enc.strength.range_min - (*it).encoder->enc.strength.default_value ) / (*it).encoder->enc.strength.step;
        }
        else {
            intDefault = 0;
        }
        (*it).compressionLevel = conf->readNumEntry( "compressionLevel", intDefault );

        if( (*it).encoder != 0 && (*it).encoder->enc.replaygain.enabled ) {
            if( (*it).replaygain != 0 && (*it).replaygain->replaygain.rank > (*it).encoder->enc.replaygain.rank ) {
                boolDefault = false;
            }
            else {
                boolDefault = true;
            }
        }
        else {
            boolDefault = false;
        }
        (*it).internalReplayGain = conf->readBoolEntry( "internalReplayGain", boolDefault );
    }
}

void Config::write()
{
    writeProfiles();
    writeServiceMenu();
    writeAmarokScript();

    KConfig *conf = kapp->config();

    conf->setGroup( "Ripper" );
    conf->writeEntry( "ripper", (currentRipper)?currentRipper->rip.bin:"kio_audiocd" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it )
    {
        conf->setGroup( (*it).mime_types.first() );
        if( (*it).encoder ) conf->writeEntry( "encoder", (*it).encoder->enc.bin );
        if( (*it).decoder ) conf->writeEntry( "decoder", (*it).decoder->dec.bin );
        if( (*it).replaygain ) conf->writeEntry( "replaygain", (*it).replaygain->replaygain.bin );
        conf->writeEntry( "compressionLevel", (*it).compressionLevel );
        conf->writeEntry( "internalReplayGain", (*it).internalReplayGain );
    }

    conf->setGroup( "General" ) ;
    conf->writeEntry( "configVersion", 300 );
    conf->writeEntry( "startTab", data.general.startTab );
    conf->writeEntry( "lastTab", data.general.lastTab );
    conf->writeEntry( "defaultProfile", data.general.defaultProfile );
    conf->writeEntry( "defaultFormat", data.general.defaultFormat );
//     conf->writeEntry( "defaultOutputDirectory", data.general.defaultOutputDirectory );
    conf->writeEntry( "specifyOutputDirectory", data.general.specifyOutputDirectory );
    conf->writeEntry( "metaDataOutputDirectory", data.general.metaDataOutputDirectory );
    conf->writeEntry( "copyStructureOutputDirectory", data.general.copyStructureOutputDirectory );
    conf->writeEntry( "useVFATNames", data.general.useVFATNames );
    conf->writeEntry( "priority", data.general.priority );
    conf->writeEntry( "numFiles", data.general.numFiles );
    conf->writeEntry( "updateDelay", data.general.updateDelay );
    conf->writeEntry( "askForNewOptions", data.general.askForNewOptions );
    conf->writeEntry( "showToolBar", data.general.showToolBar );

    conf->setGroup( "Plugins" );
    conf->writeEntry( "checkForUpdates", data.plugins.checkForUpdates );

    conf->setGroup( "Environment" );
    conf->writeEntry( "directories", data.environment.directories );
    data.environment.foundPrograms.clear();
    for( QMap<QString, QString>::Iterator b = binaries.begin(); b != binaries.end(); ++b ) {
        if( b.data() != "" ) {
            data.environment.foundPrograms.append( b.data() );
        }
    }
    conf->writeEntry( "foundPrograms", data.environment.foundPrograms );

    conf->sync();

    emit configChanged();
}

void Config::readProfiles()
{
    int version;
    QString name;
    ConversionOptions options;

    QDomDocument domTree;
    QFile opmlFile( locateLocal("data","soundkonverter/profiles.xml") );
    if( !opmlFile.open( IO_ReadOnly ) ) return;
    if( !domTree.setContent( &opmlFile ) ) return;
    opmlFile.close();

    QDomElement root = domTree.documentElement();
    if( root.attribute("type") != "profiles" ) return;
    QDomNode node, sub1Node, sub2Node;
    node = root.firstChild();

    while( !node.isNull() ) {
        if( node.isElement() && node.nodeName() == "info" ) {

            version = node.toElement().attribute("version").toInt();

        }
        else if( node.isElement() && node.nodeName() == "profile" ) {

            name = node.toElement().attribute("name");

            sub1Node = node.toElement().firstChild();
            while( !sub1Node.isNull() ) {
                if( sub1Node.isElement() && sub1Node.nodeName() == "encodingOptions" ) {
// TODO clean up
                    options.encodingOptions.sFormat = sub1Node.toElement().attribute("sFormat");
                    options.encodingOptions.sQualityMode = sub1Node.toElement().attribute("sQualityMode");
                    options.encodingOptions.iQuality = sub1Node.toElement().attribute("iQuality").toInt();
                    options.encodingOptions.sBitrateMode = sub1Node.toElement().attribute("sBitrateMode");
                    options.encodingOptions.bBitrateRange = sub1Node.toElement().attribute("bBitrateRange").toInt();
                    options.encodingOptions.iMinBitrate = sub1Node.toElement().attribute("iMinBitrate").toInt();
                    options.encodingOptions.iMaxBitrate = sub1Node.toElement().attribute("iMaxBitrate").toInt();

                    sub2Node = sub1Node.toElement().firstChild();
                    while( !sub2Node.isNull() ) {
                        if( sub2Node.isElement() && sub2Node.nodeName() == "samplingRate" ) {

                            options.encodingOptions.samplingRate.bEnabled = sub2Node.toElement().attribute("bEnabled").toInt();
                            options.encodingOptions.samplingRate.iSamplingRate = sub2Node.toElement().attribute("iSamplingRate").toInt();

                        }
                        else if( sub2Node.isElement() && sub2Node.nodeName() == "channels" ) {

                            options.encodingOptions.channels.bEnabled = sub2Node.toElement().attribute("bEnabled").toInt();
                            options.encodingOptions.channels.sChannels = sub2Node.toElement().attribute("sChannels");

                        }
                        else if( sub2Node.isElement() && sub2Node.nodeName() == "replaygain" ) {

                            options.encodingOptions.replaygain.bEnabled = sub2Node.toElement().attribute("bEnabled").toInt();

                        }
                        sub2Node = sub2Node.nextSibling();
                    }

                    options.encodingOptions.sInOutFiles =  sub1Node.toElement().attribute("sInOutFiles");

                }
                else if( sub1Node.isElement() && sub1Node.nodeName() == "outputOptions" ) {

                    options.outputOptions.mode = (OutputDirectory::Mode)sub1Node.toElement().attribute("mode").toInt();
                    options.outputOptions.directory = sub1Node.toElement().attribute("directory");

                }
                sub1Node = sub1Node.nextSibling();
            }
            ProfileData profileData;
            profileData.name = name;
            profileData.options = options;
            profiles += profileData;
        }
        node = node.nextSibling();
    }
}

void Config::writeProfiles()
{
    QDomDocument domTree;
    QDomElement root = domTree.createElement( "soundkonverter" );
    root.setAttribute( "type", "profiles" );
    domTree.appendChild( root );

    QDomElement info = domTree.createElement( "info" );
    info.setAttribute( "version", "300" );
    root.appendChild( info );

    for( QValueList<ProfileData>::Iterator it = profiles.begin(); it != profiles.end(); ++it ) {
        QDomElement profile = domTree.createElement( "profile" );
        profile.setAttribute( "name", (*it).name );

            QDomElement encodingOptions = domTree.createElement( "encodingOptions" );

                encodingOptions.setAttribute( "sFormat", (*it).options.encodingOptions.sFormat );
                encodingOptions.setAttribute( "sQualityMode", (*it).options.encodingOptions.sQualityMode );
                encodingOptions.setAttribute( "iQuality", (*it).options.encodingOptions.iQuality );
                encodingOptions.setAttribute( "sBitrateMode", (*it).options.encodingOptions.sBitrateMode );
                encodingOptions.setAttribute( "bBitrateRange", (*it).options.encodingOptions.bBitrateRange );
                encodingOptions.setAttribute( "iMinBitrate", (*it).options.encodingOptions.iMinBitrate );
                encodingOptions.setAttribute( "iMaxBitrate", (*it).options.encodingOptions.iMaxBitrate );

                QDomElement samplingRate = domTree.createElement( "samplingRate" );

                    samplingRate.setAttribute( "bEnabled", (*it).options.encodingOptions.samplingRate.bEnabled );
                    samplingRate.setAttribute( "iSamplingRate", (*it).options.encodingOptions.samplingRate.iSamplingRate );

                encodingOptions.appendChild( samplingRate );

                QDomElement channels = domTree.createElement( "channels" );

                    channels.setAttribute( "bEnabled", (*it).options.encodingOptions.channels.bEnabled );
                    channels.setAttribute( "sChannels", (*it).options.encodingOptions.channels.sChannels );

                encodingOptions.appendChild( channels );

                QDomElement replaygain = domTree.createElement( "replaygain" );

                    replaygain.setAttribute( "bEnabled", (*it).options.encodingOptions.replaygain.bEnabled );

                encodingOptions.appendChild( replaygain );

                encodingOptions.setAttribute( "sInOutFiles", (*it).options.encodingOptions.sInOutFiles );

            profile.appendChild( encodingOptions );

            QDomElement outputOptions = domTree.createElement( "outputOptions" );

                outputOptions.setAttribute( "mode", int((*it).options.outputOptions.mode) );
                outputOptions.setAttribute( "directory", (*it).options.outputOptions.directory );

            profile.appendChild( outputOptions );

        root.appendChild( profile );
    }

    QFile opmlFile( locateLocal("data","soundkonverter/profiles.xml") );
    if( !opmlFile.open( IO_WriteOnly ) ) return;

    QTextStream ts( &opmlFile );
    ts << domTree.toString();

    opmlFile.close();
}

void Config::writeServiceMenu()
{
    int num;
    QString content;
    QFile file;

    num = 0;
    content = "";
    content += "[Desktop Entry]\n";
    content += "ServiceTypes=";

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
//         if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
//             if( !(*it).encoder->enc.lossy.enabled && !(*it).encoder->enc.lossless.enabled && !(*it).encoder->enc.hybrid.enabled ) continue;
//             for( QStringList::Iterator b = (*it).mime_types.begin(); b != (*it).mime_types.end(); ++b ) {
//                 content += (*b) + ",";
//                 num++;
//             }
//         }
        if( (*it).decoder != 0 && binaries[(*it).decoder->dec.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
            for( QStringList::Iterator b = (*it).mime_types.begin(); b != (*it).mime_types.end(); ++b ) {
                content += (*b) + ",";
                num++;
            }
        }
    }

    if( num != 0 )
    {
        content += "audio/x-wav";

        content += "\n";
        content += "Icon=soundkonverter\n";
        content += "Actions=ConvertWithSoundkonverter\n\n";

        content += "[Desktop Action ConvertWithSoundkonverter]\n";
        content += "Name="+i18n("Convert with soundKonverter ...")+"\n";
        content += "Icon=soundkonverter\n";
        content += "Exec=soundkonverter %F\n";

        file.setName( locateLocal("data","konqueror/servicemenus/")+"convert_with_soundkonverter.desktop" );
        if ( file.open( IO_WriteOnly ) ) {
            QTextStream stream( &file );
            stream << content;
            file.close();
        }
    }

    num = 0;
    content = "";
    content += "[Desktop Entry]\n";
    content += "ServiceTypes=";

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).replaygain != 0 && binaries[(*it).replaygain->replaygain.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
            for( QStringList::Iterator b = (*it).mime_types.begin(); b != (*it).mime_types.end(); ++b ) {
                content += (*b) + ",";
                num++;
            }
        }
    }

    if( num != 0 )
    {
        content = content.left(content.length()-1);

        content += "\n";
        content += "Icon=soundkonverter_replaygain\n";
        content += "Actions=AddReplayGainWithSoundkonverter\n\n";

        content += "[Desktop Action AddReplayGainWithSoundkonverter]\n";
        content += "Name="+i18n("Add Replay Gain with soundKonverter ...")+"\n";
        content += "Icon=soundkonverter_replaygain\n";
        content += "Exec=soundkonverter --replaygain %F\n";

        file.setName( locateLocal("data","konqueror/servicemenus/")+"add_replaygain_with_soundkonverter.desktop" );
        if( file.open(IO_WriteOnly) ) {
            QTextStream st( &file );
            st << content;
            file.close();
        }
    }
}

void Config::writeAmarokScript()
{
    int num, num1, num2;
    QString content, content1, content2;
    QFile file;

    num = 0;
    content = "";
    content += "name = soundKonverter\n";
    content += "type = transcode\n\n";
    content += "[Transcode]\n";
    content += "target_formats = ";

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
            if( !(*it).encoder->enc.lossy.enabled && !(*it).encoder->enc.lossless.enabled && !(*it).encoder->enc.hybrid.enabled ) continue;
            for( QStringList::Iterator b = (*it).extensions.begin(); b != (*it).extensions.end(); ++b ) {
                if( content.find(" "+(*b).lower()+" ") == -1 ) content += (*b).lower() + " ";
                num++;
            }
        }
    }

    if( num != 0 ) // TODO what is, if there are only decoders?
    {
        content += "wav";
        content += "\n";

        file.setName( locateLocal("data","amarok/scripts/soundKonverter/")+"soundKonverter.spec" );
        if( file.open(IO_WriteOnly) ) {
            QTextStream st( &file );
            st << content;
            file.close();
        }
    }

    num1 = num2 = 0;
    content = content1 = content2 = "";

    // NOTE duplicate entries won't be shown to the user at any time, so they aren't filtered
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
            if( (*it).encoder->enc.lossless.enabled ) {
                for( QStringList::Iterator b = (*it).extensions.begin(); b != (*it).extensions.end(); ++b ) {
                    if( content1.find(","+(*b).lower()+",") == -1 ) content1 += (*b).lower() + ","; // NOTE the first entry will be shown twice
                    num1++;
                }
            }
            if( (*it).encoder->enc.hybrid.enabled ) {
                for( QStringList::Iterator b = (*it).extensions.begin(); b != (*it).extensions.end(); ++b ) {
                    if( content2.find(","+(*b).lower()+",") == -1 ) content2 += (*b).lower() + ","; // NOTE the first entry will be shown twice
                    num2++;
                }
            }
        }
    }

    if( num1 != 0 )
    {
        content1 = content1.left(content1.length()-1);
        content += "Lossless";
        content += "\n";
        content += content1;
        content += "\n";
    }

    if( num2 != 0 )
    {
        content2 = content2.left(content2.length()-1);
        content += "Hybrid";
        content += "\n";
        content += content2;
        content += "\n";
    }

    if( num1 != 0 || num2 != 0 )
    {
        file.setName( locateLocal("data","amarok/scripts/soundKonverter/")+"formats" );
        if( file.open(IO_WriteOnly) ) {
            QTextStream st( &file );
            st << content;
            file.close();
        }
    }

    KStandardDirs* stdDirs = new KStandardDirs();
    KIO::NetAccess::file_copy( stdDirs->findResource("data","soundkonverter/amarokscript/soundKonverter.rb"), locateLocal("data","amarok/scripts/soundKonverter/soundKonverter.rb"), 0755, true );
    KIO::NetAccess::file_copy( stdDirs->findResource("data","soundkonverter/amarokscript/README"), locateLocal("data","amarok/scripts/soundKonverter/README"), -1, true );
    delete stdDirs;
}

void Config::loadFileInfos()
{}

void Config::loadPlugins()
{
    // NOTE can be speeded up

    //kdDebug() << "entering: `" << "Config::loadPlugins()" << "'" << endl;
    logger->log( 1000, "entering: `Config::loadPlugins()'" );
    QStringList list;
    QDir dir;
    QString correction_file_mime_type;

    /** a map that holds the _identifier_ and the _version_ number of all plugins */
    QMap<QString, PluginMapData> pluginMap;

    int version;
    QString identifier;
    int i;

    ConvertPluginLoader* convertPluginLoader = new ConvertPluginLoader();
    ReplayGainPluginLoader* replaygainPluginLoader = new ReplayGainPluginLoader();
    RipperPluginLoader* ripperPluginLoader = new RipperPluginLoader();

    KStandardDirs stddir;
    QStringList directories = stddir.findDirs( "data", "soundkonverter/plugins/" );
    for( QStringList::Iterator a = directories.begin(); a != directories.end(); ++a )
    {
        //kdDebug() << " searching directory: `" << *a << "'" << endl;
        logger->log( 1000, " searching directory: `" + *a + "'" );
        dir.setPath( *a );
        list = dir.entryList( "*", QDir::All, QDir::Name );
        for( QStringList::Iterator b = list.begin(); b != list.end(); ++b )
        {
            if( *b != "." && *b != ".." && (*b).right(19) == ".soundkonverter.xml" )
            {
                //kdDebug() << "  found file: `" << *b << "'" << endl;
                logger->log( 1000, "  found file: `" + *b + "'" );
                if( convertPluginLoader->verifyFile( QString(*a).append(*b) ) != -1 )
                {
                    version = convertPluginLoader->verifyFile( QString(*a).append(*b) );
                    identifier = *b;
                    identifier.remove( identifier.length() - 19, 19 );
                    i = identifier.findRev( "-" );
                    identifier.remove( i, identifier.length() - i );
//                     i = identifier.find( "." );
//                     identifier.remove( 0, i + 1 );
                    if( !pluginMap.contains(identifier) ) {
                        PluginMapData data;
                        data.version = 0;
                        pluginMap.insert( identifier, data );
                    }
                    if( pluginMap[identifier].version < version )
                    {
                        pluginMap[identifier].version = version;
                        pluginMap[identifier].filename = QString(*a).append(*b);
                        pluginMap[identifier].type = "converter";
                        logger->log( 1000, "   updating version for: `" + identifier + "'" );
                    }
                }
                else if( replaygainPluginLoader->verifyFile( QString(*a).append(*b) ) != -1 )
                {
                    version = replaygainPluginLoader->verifyFile( QString(*a).append(*b) );
                    identifier = *b;
                    identifier.remove( identifier.length() - 19, 19 );
                    i = identifier.findRev( "-" );
                    identifier.remove( i, identifier.length() - i );
//                     i = identifier.find( "." );
//                     identifier.remove( 0, i + 1 );
                    if( !pluginMap.contains(identifier) ) {
                        PluginMapData data;
                        data.version = 0;
                        pluginMap.insert( identifier, data );
                    }
                    if( pluginMap[identifier].version < version )
                    {
                        pluginMap[identifier].version = version;
                        pluginMap[identifier].filename = QString(*a).append(*b);
                        pluginMap[identifier].type = "replaygain";
                        logger->log( 1000, "   updating version for: `" + identifier + "'" );
                    }
                }
                else if( ripperPluginLoader->verifyFile( QString(*a).append(*b) ) != -1 )
                {
                    version = ripperPluginLoader->verifyFile( QString(*a).append(*b) );
                    identifier = *b;
                    identifier.remove( identifier.length() - 19, 19 );
                    i = identifier.findRev( "-" );
                    identifier.remove( i, identifier.length() - i );
//                     i = identifier.find( "." );
//                     identifier.remove( 0, i + 1 );
                    if( !pluginMap.contains(identifier) ) {
                        PluginMapData data;
                        data.version = 0;
                        pluginMap.insert( identifier, data );
                    }
                    if( pluginMap[identifier].version < version )
                    {
                        pluginMap[identifier].version = version;
                        pluginMap[identifier].filename = QString(*a).append(*b);
                        pluginMap[identifier].type = "ripper";
                        logger->log( 1000, "   updating version for: `" + identifier + "'" );
                    }
                }
                else
                {
                    //kdDebug() << "  file is invalid: `" << *b << "'" << endl;
                    logger->log( 1000, "  file is invalid: `" + *b + "'" );
                }
            }
        }
    }

    for( QMap<QString, PluginMapData>::Iterator a = pluginMap.begin(); a != pluginMap.end(); ++a ) {
        if( a.data().type == "converter" )
        {
            ConvertPlugin* plugin = convertPluginLoader->loadFile( a.data().filename );
            if( plugin->info.version != -1 )
            {
                if( plugin->enc.hybrid.enabled )
                {
                    correction_file_mime_type = plugin->enc.hybrid.correction_file_mime_type;
                }
                if( plugin->enc.enabled )
                {
                    for( QStringList::Iterator b = plugin->enc.mime_types.begin(); b != plugin->enc.mime_types.end(); ++b )
                    {
                        //kdDebug() << "   registering encoder for: `" << *b << "'" << endl;
                        logger->log( 1000, "   registering encoder for: `" + *b + "'" );
                        registerFormatFeatures( *b, plugin, 0, 0, correction_file_mime_type );
                    }
                }
                if( plugin->dec.enabled )
                {
                    for( QStringList::Iterator b = plugin->dec.mime_types.begin(); b != plugin->dec.mime_types.end(); ++b )
                    {
                        //kdDebug() << "   registering decoder for: `" << *b << "'" << endl;
                        logger->log( 1000, "   registering decoder for: `" + *b + "'" );
                        registerFormatFeatures( *b, 0, plugin, 0, correction_file_mime_type );
                    }
                }
                converters.append( plugin );
                //kdDebug() << "  plugin sucessfully loaded: `" << a.key() << "'" << endl;
                logger->log( 1000, "  plugin sucessfully loaded: `" + a.key() + "'" );
            }
            else
            {
                //kdError() << "  failed to load Plugin: `" << a.key() << "'" << endl;
                logger->log( 1000, "  failed to load Plugin: `" + a.key() + "'" );
            }
        }
        else if( a.data().type == "replaygain" )
        {
            ReplayGainPlugin* plugin = replaygainPluginLoader->loadFile( a.data().filename );
            if( plugin->info.version != -1 )
            {
                for( QStringList::Iterator b = plugin->replaygain.mime_types.begin(); b != plugin->replaygain.mime_types.end(); ++b )
                {
                    //kdDebug() << "   registering replaygain for: `" << *b << "'" << endl;
                    logger->log( 1000, "   registering replaygain for: `" + *b + "'" );
                    registerFormatFeatures( *b, 0, 0, plugin );
                }
                replaygains.append( plugin );
                //kdDebug() << "  plugin sucessfully loaded: `" << a.key() << "'" << endl;
                logger->log( 1000, "  plugin sucessfully loaded: `" + a.key() + "'" );
            }
            else
            {
                //kdError() << "  failed to load Plugin: `" << a.key() << "'" << endl;
                logger->log( 1000, "  failed to load Plugin: `" + a.key() + "'" );
            }
        }
        else if( a.data().type == "ripper" )
        {
            RipperPlugin* plugin = ripperPluginLoader->loadFile( a.data().filename );
            if( plugin->info.version != -1 )
            {
                rippers.append( plugin );
                //kdDebug() << "  plugin sucessfully loaded: `" << a.key() << "'" << endl;
                logger->log( 1000, "  plugin sucessfully loaded: `" + a.key() + "'" );
            }
            else
            {
                //kdError() << "  failed to load Plugin: `" << a.key() << "'" << endl;
                logger->log( 1000, "  failed to load Plugin: `" + a.key() + "'" );
            }
        }
    }

    delete convertPluginLoader;
    delete replaygainPluginLoader;
    delete ripperPluginLoader;

    // NOTE in order to make a plugin working, there must be a format_infos file.

    // TODO remove duplicate extensions

    FormatInfoLoader* formatInfoLoader = new FormatInfoLoader();

    //kdDebug() << " trying to open: `" << "soundkonverter/format_infos/"+language+"/" << "'" << endl;
//     logger->log( 1000, " trying to open: `soundkonverter/format_infos/" + language + "/'" );
//     directories = stddir.findDirs( "data", "soundkonverter/format_infos/"+language+"/" );
//     if( directories.isEmpty() ) directories = stddir.findDirs( "data", "soundkonverter/format_infos/en/" );
    directories = stddir.findDirs( "data", "soundkonverter/format_infos/" );
    for( QStringList::Iterator a = directories.begin(); a != directories.end(); ++a )
    {
        //kdDebug() << " searching directory: `" << *a << "'" << endl;
        logger->log( 1000, " searching directory: `" + *a + "'" );
        dir.setPath( *a );
        list = dir.entryList( "*", QDir::All, QDir::Name );
        for( QStringList::Iterator b = list.begin(); b != list.end(); ++b )
        {
            if( *b != "." && *b != ".." && (*b).right(4) == ".xml" )
            {
                //kdDebug() << "  found file: `" << *b << "'" << endl;
                logger->log( 1000, "  found file: `" + *b + "'" );
                if( formatInfoLoader->verifyFile( QString(*a).append(*b) ) )
                {
                    FormatInfo* formatInfo = formatInfoLoader->loadFile( QString(*a).append(*b) );

                    for( QValueList<FormatItem>::Iterator c = formats.begin(); c != formats.end(); ++c ) {
                        for( QStringList::Iterator d = formatInfo->mime_types.begin(); d != formatInfo->mime_types.end(); ++d ) {
                            if( (*c).mime_types.findIndex(*d) != -1 ) {
                                (*c).description = "<p>" + formatInfo->description + "</p>";
                                (*c).description.replace("\\n","</p>\n<p>");
                                for( QStringList::Iterator d = formatInfo->urls.begin(); d != formatInfo->urls.end(); ++d ) {
                                    (*c).description += "\n<p><a href=\"" + (*d) + "\">" + (*d) + "</a></p>";
                                }
                                (*c).compressionType = formatInfo->compressionType;
                                (*c).size = formatInfo->size;
                                // add extensions for mime types
                                logger->log( 1000, "   found mime type: `" + *d + "'" );
                                QStringList extensions = KMimeType::mimeType( *d )->patterns();
                                for( QStringList::Iterator c = extensions.begin(); c != extensions.end(); ++c ) {
                                    (*c).remove( 0, 2 );
                                    logger->log( 1000, "    adding extension: `" + *c + "'" );
                                }
                                (*c).extensions += extensions;
                                // add extensions for correction file mime types
                                for( QStringList::Iterator d = (*c).correction_file_mime_types.begin(); d != (*c).correction_file_mime_types.end(); ++d ) {
                                    logger->log( 1000, "   found correction mime type: `" + *d + "'" );
                                    QStringList extensions = KMimeType::mimeType( *d )->patterns();
                                    for( QStringList::Iterator e = extensions.begin(); e != extensions.end(); ++e ) {
                                        (*e).remove( 0, 2 );
                                        logger->log( 1000, "    adding correction extension: `" + *e + "'" );
                                        (*c).correction_file_extensions += (*e);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    delete formatInfoLoader;
}

void Config::registerFormatFeatures( const QString &mime_type,
                                     ConvertPlugin* encoder, ConvertPlugin* decoder,
                                     ReplayGainPlugin* replaygain/*, RepairPlugin* repairer*/,
                                     const QString &correction_file_mime_type )
{
    //kdDebug() << "    entering: `" << "Config::registerFormatFeatures( ... )" << "'" << endl;
    logger->log( 1000, "    entering: `Config::registerFormatFeatures( ... )'" );

    // iterate through all file formats and search for an existing one
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(mime_type) != -1 ) { // we found an existing entry for our file format
            //kdDebug() << "     found an existing entry: `" << mime_type << "'" << endl;
            logger->log( 1000, "     found an existing entry: `" + mime_type + "'" );

            if( encoder != 0 ) (*it).encoders.append( encoder );
//             if( (*it).encoder == 0 && encoder != 0 ) (*it).encoder = encoder;
//             if( (*it).encoder != 0 && binaries[(*it).encoder.enc.bin] == "" )
            if( decoder != 0 ) (*it).decoders.append( decoder );
//             if( (*it).decoder == 0 && decoder != 0 ) (*it).decoder = decoder;
            if( replaygain != 0 ) (*it).replaygains.append( replaygain );
//             if( (*it).replaygain == 0 && replaygain != 0 ) (*it).replaygain = replaygain;
            //if( repairer ) (*it).repairer.append( repairer );*/
            if( !correction_file_mime_type.isEmpty() ) (*it).correction_file_mime_types.append( correction_file_mime_type );

            // everything done, we can return!
            return;
        }
    }
    //kdDebug() << "     creating a new entry: `" << mime_type << "'" << endl;
    logger->log( 1000, "     creating a new entry: `" + mime_type + "'" );

    // well it seems, we haven't found an entry. create a new!
    QValueList<FormatItem>::Iterator newItem = formats.append( FormatItem() );
    (*newItem).mime_types = mime_type;
    if( encoder != 0 ) (*newItem).encoders.append( encoder );
//     if( encoder != 0 ) (*newItem).encoder = encoder;
    if( decoder != 0 ) (*newItem).decoders.append( decoder );
//     if( decoder != 0 ) (*newItem).decoder = decoder;
    if( replaygain != 0 ) (*newItem).replaygains.append( replaygain );
//     if( replaygain != 0 ) (*newItem).replaygain = replaygain;
    //if( repairer ) (*newItem).repairer.append( repairer );
    if( !correction_file_mime_type.isEmpty() ) (*newItem).correction_file_mime_types.append( correction_file_mime_type );

}

/*void Config::registerFileFormat( const QString &mime_type, const QString &option, const QString &value )
{
    // iterate through all file formats and search for an existing one
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(mime_type) != -1 ) { // we found an existing entry for our file format
            if( option == "mime_type" ) (*it).mime_types.append( value );
            //else if( option == "description" ) // TODO implement a qmap ?!
            else if( option == "size" ) (*it).size = value.toInt();
            else if( option == "compression_type" ) {
                if( value == "lossy" ) (*it).compressionType = FormatInfo::lossy;
                else if( value == "lossless" ) (*it).compressionType = FormatInfo::lossless;
                else (*it).compressionType = FormatInfo::hybrid;
            }

            // everything done, we can return!
            return;
        }
    }

    // well it seems, we haven't found an entry. create a new!
    QValueList<FormatItem>::Iterator newItem = formats.append( FormatItem() );
    if( option == "mime_type" ) (*newItem).mime_types.append( value );
    //else if( option == "description" ) // TODO implement a qmap ?!
    else if( option == "size" ) (*newItem).size = value.toInt();
    else if( option == "compression_type" ) {
        if( value == "lossy" ) (*newItem).compressionType = FormatInfo::lossy;
        else if( value == "lossless" ) (*newItem).compressionType = FormatInfo::lossless;
        else (*newItem).compressionType = FormatInfo::hybrid;
    }
}*/

ConvertPlugin* Config::encoderForFormat( const QString &format )
{
    // iterate through all file formats and search for our format in mime_types and extensions
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return (*it).encoder;
        }
    }

    return 0;
}

ConvertPlugin* Config::decoderForFormat( const QString &format )
{
    // iterate through all file formats and search for our format in mime_types and extensions
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return (*it).decoder;
        }
    }

    return 0;
}

ReplayGainPlugin* Config::replaygainForFormat( const QString &format )
{
    // iterate through all file formats and search for our format in mime_types and extensions
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return (*it).replaygain;
        }
    }

    return 0;
}

FormatItem* Config::getFormatItem( const QString &format )
{
    // iterate through all file formats and search for our format in mime_types and extensions
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return &(*it);
        }
    }

    return 0;
}

// NOTE speed up the following 3 functions (cache the data) (use the extensions variable in FormatItem)
// NOTE seems to be called too often ???
QStringList Config::allFormats()
{
    QStringList list;

    //kdDebug() << "entering: `" << "Config::allFormats()" << "'" << endl;
    logger->log( 1000, "entering: `Config::allFormats()'" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        //kdDebug() << " mime type: `" << (*it).mime_types.first() << "'" << endl;
        logger->log( 1000, " mime type: `" + (*it).mime_types.first() + "'" );
        if( (*it).mime_types.first() != "application/octet-stream" ) {
            QString extension = KMimeType::mimeType( (*it).mime_types.first() )->patterns().first();
            extension.remove( 0, 2 );
            //kdDebug() << "  extension: `" << extension << "'" << endl;
            logger->log( 1000, "  extension: `" + extension + "'" );
            if( !extension.isEmpty() ) {
                list.append( extension );
                //kdDebug() << "   (added)" << endl;
                logger->log( 1000, "   (added)" );
            }
        }
    }

    return list;
}

QStringList Config::allEncodableFormats()
{
    QStringList list;

    //kdDebug() << "entering: `" << "Config::allEncodableFormats()" << "'" << endl;
    logger->log( 1000, "entering: `Config::allEncodableFormats()'" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        //kdDebug() << " mime type: `" << (*it).mime_types.first() << "'" << endl;
        logger->log( 1000, " mime type: `" + (*it).mime_types.first() + "'" );
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).mime_types.first() != "application/octet-stream" ) {
            if( !(*it).encoder->enc.lossy.enabled && !(*it).encoder->enc.lossless.enabled && !(*it).encoder->enc.hybrid.enabled ) continue;
            QString extension = KMimeType::mimeType( (*it).mime_types.first() )->patterns().first();
            extension.remove( 0, 2 );
            //kdDebug() << "  extension: `" << extension << "'" << endl;
            logger->log( 1000, "  extension: `" + extension + "'" );
            if( !extension.isEmpty() ) {
                list.append( extension );
                //kdDebug() << "   (added)" << endl;
                logger->log( 1000, "   (added)" );
            }
        }
    }

    return list;
}

QStringList Config::allLossyEncodableFormats()
{
    QStringList list;

    //kdDebug() << "entering: `" << "Config::allLossyEncodableFormats()" << "'" << endl;
    logger->log( 1000, "entering: `Config::allLossyEncodableFormats()'" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        //kdDebug() << " mime type: `" << (*it).mime_types.first() << "'" << endl;
        logger->log( 1000, " mime type: `" + (*it).mime_types.first() + "'" );
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).encoder->enc.lossy.enabled && (*it).compressionType & FormatInfo::lossy && (*it).mime_types.first() != "application/octet-stream" ) {
            QString extension = KMimeType::mimeType( (*it).mime_types.first() )->patterns().first();
            extension.remove( 0, 2 );
            //kdDebug() << "  extension: `" << extension << "'" << endl;
            logger->log( 1000, "  extension: `" + extension + "'" );
            if( !extension.isEmpty() ) {
                list.append( extension );
                //kdDebug() << "   (added)" << endl;
                logger->log( 1000, "   (added)" );
            }
        }
    }

    return list;
}

QStringList Config::allLosslessEncodableFormats()
{
    QStringList list;

    //kdDebug() << "entering: `" << "Config::allLosslessEncodableFormats()" << "'" << endl;
    logger->log( 1000, "entering: `Config::allLosslessEncodableFormats()'" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        //kdDebug() << " mime type: `" << (*it).mime_types.first() << "'" << endl;
        logger->log( 1000, " mime type: `" + (*it).mime_types.first() + "'" );
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).encoder->enc.lossless.enabled && (*it).compressionType & FormatInfo::lossless && (*it).mime_types.first() != "application/octet-stream" ) {
            QString extension = KMimeType::mimeType( (*it).mime_types.first() )->patterns().first();
            extension.remove( 0, 2 );
            //kdDebug() << "  extension: `" << extension << "'" << endl;
            logger->log( 1000, "  extension: `" + extension + "'" );
            if( !extension.isEmpty() ) {
                list.append( extension );
                //kdDebug() << "   (added)" << endl;
                logger->log( 1000, "   (added)" );
            }
        }
    }

    return list;
}

QStringList Config::allHybridEncodableFormats()
{
    QStringList list;

    //kdDebug() << "entering: `" << "Config::allHybridEncodableFormats()" << "'" << endl;
    logger->log( 1000, "entering: `Config::allHybridEncodableFormats()'" );

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        //kdDebug() << " mime type: `" << (*it).mime_types.first() << "'" << endl;
        logger->log( 1000, " mime type: `" + (*it).mime_types.first() + "'" );
        if( (*it).encoder != 0 && binaries[(*it).encoder->enc.bin] != "" && (*it).encoder->enc.hybrid.enabled && (*it).compressionType & FormatInfo::hybrid && (*it).mime_types.first() != "application/octet-stream" ) {
            QString extension = KMimeType::mimeType( (*it).mime_types.first() )->patterns().first();
            extension.remove( 0, 2 );
            //kdDebug() << "  extension: `" << extension << "'" << endl;
            logger->log( 1000, "  extension: `" + extension + "'" );
            if( !extension.isEmpty() ) {
                list.append( extension );
                //kdDebug() << "   (added)" << endl;
                logger->log( 1000, "   (added)" );
            }
        }
    }

    return list;
}

QString Config::getCorrectionExtension( const QString &format )
{
    // iterate through all file formats and search for our format
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return (*it).correction_file_extensions.first();
        }
    }

    return "";
}

QString Config::getFormatDescription( const QString &format ) // NOTE could be removed
{
    // iterate through all file formats and search for our format
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).mime_types.findIndex(format) != -1 || (*it).extensions.findIndex(format) != -1 ) { // we found it
            return (*it).description;
        }
    }

    return "";
}

void Config::addProfile( const QString &name, const ConversionOptions& profile )
{
    ProfileData profileData;
    profileData.name = name;
    profileData.options = profile;
    profiles += profileData;
    if( name != i18n("Last used") && name != "Last used" ) writeProfiles(); // will only be saved at app exit, when saving everything anyway
    emit configChanged();
}

void Config::removeProfile( const QString &name )
{
    for( QValueList<ProfileData>::Iterator it = profiles.begin(); it != profiles.end(); ++it ) {
        if( (*it).name == name ) {
            profiles.remove( it );
            if( name != i18n("Last used") && name != "Last used" ) writeProfiles(); // will only be saved at app exit, when saving everything anyway
            return;
        }
    }
    emit configChanged();
}

ConversionOptions Config::getProfile( const QString &name )
{
    for( QValueList<ProfileData>::Iterator it = profiles.begin(); it != profiles.end(); ++it ) {
        if( /*(*it).name != i18n("Last used") &&*/ (*it).name == name ) {
            return (*it).options;
        }
    }

    ConversionOptions options;
    options.encodingOptions.sFormat = "";
    return options;
}

QString Config::getProfileName( const ConversionOptions& options )
{
    if( options.encodingOptions.sQualityMode == i18n("Lossless") ) {
        return i18n("Lossless");
    }

    if( options.encodingOptions.sQualityMode == i18n("Hybrid") ) {
        return i18n("Hybrid");
    }

    for( QValueList<ProfileData>::Iterator it = profiles.begin(); it != profiles.end(); ++it ) {
        if( (*it).name != i18n("Last used") && (*it).options.nearlyEqual( options ) ) {
            return (*it).name;
        }
    }

    ConvertPlugin* plugin = encoderForFormat( options.encodingOptions.sFormat );
    if( plugin != 0 ) {
        if( plugin->enc.lossy.quality.enabled ) {
            if( options.encodingOptions.sQualityMode == i18n("Quality") && options.encodingOptions.iQuality == 20 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == true && options.encodingOptions.channels.sChannels == i18n("Mono") &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very low");
            }
            if( options.encodingOptions.sQualityMode == i18n("Quality") && options.encodingOptions.iQuality == 30 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Low");
            }
            if( options.encodingOptions.sQualityMode == i18n("Quality") && options.encodingOptions.iQuality == 40 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Medium");
            }
            if( options.encodingOptions.sQualityMode == i18n("Quality") && options.encodingOptions.iQuality == 50 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("High");
            }
            if( options.encodingOptions.sQualityMode == i18n("Quality") && options.encodingOptions.iQuality == 60 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very high");
            }
        }
        if( plugin->enc.lossy.bitrate.abr.enabled || plugin->enc.lossy.bitrate.cbr.enabled ) {
            if( options.encodingOptions.sQualityMode == i18n("Bitrate") && options.encodingOptions.iQuality == 64 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == true && options.encodingOptions.channels.sChannels == i18n("Mono") &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very low");
            }
            if( options.encodingOptions.sQualityMode == i18n("Bitrate") && options.encodingOptions.iQuality == 96 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Low");
            }
            if( options.encodingOptions.sQualityMode == i18n("Bitrate") && options.encodingOptions.iQuality == 192 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Medium");
            }
            if( options.encodingOptions.sQualityMode == i18n("Bitrate") && options.encodingOptions.iQuality == 240 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("High");
            }
            if( options.encodingOptions.sQualityMode == i18n("Bitrate") && options.encodingOptions.iQuality == 320 &&
              options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very high");
            }
        }
        /*{
            if( options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == true && options.encodingOptions.channels.sChannels == i18n("Mono") &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very low");
            }
            if( options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == true && options.encodingOptions.samplingRate.iSamplingRate == 22050 &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Low");
            }
            if( options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Medium");
            }
            if( options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("High");
            }
            if( options.encodingOptions.bBitrateRange == false &&
              options.encodingOptions.samplingRate.bEnabled == false &&
              options.encodingOptions.channels.bEnabled == false &&
              options.encodingOptions.sInOutFiles == binaries[plugin->enc.bin] + " " + plugin->enc.in_out_files ) {
                return i18n("Very high");
            }
        }*/
    }
    return i18n("User defined");
}

QStringList Config::getAllProfiles()
{
    QStringList list;

    for( QValueList<ProfileData>::Iterator it = profiles.begin(); it != profiles.end(); ++it ) {
        /*if( (*it).name != i18n("Last used") )*/ list += (*it).name;
    }

    return list;
}

bool Config::acceptFile( const QString& mimeType )
{
    if( mimeType == "audio/x-wav" ) return true;

    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).decoder != 0 && (*it).mime_types.findIndex(mimeType) != -1 ) {
            return true;
        }
    }

    return false;
}

bool Config::acceptReplayGainFile( const QString& mimeType )
{
    for( QValueList<FormatItem>::Iterator it = formats.begin(); it != formats.end(); ++it ) {
        if( (*it).replaygain != 0 && (*it).mime_types.findIndex(mimeType) != -1 ) {
            return true;
        }
    }

    return false;
}

QString Config::fileFilter( bool wav )
{
    QString filter1;
    if( wav ) filter1 += "*.wav";
    QString filter2;
    if( wav ) filter2 += "\n*.wav *.WAV|wav " + i18n("files") + " (*.wav *.WAV)";
    QString temp;

    for( QValueList<FormatItem>::Iterator a = formats.begin(); a != formats.end(); ++a ) {
        if( (*a).decoder != 0 ) {
            temp = "";
            for( QStringList::Iterator b = (*a).extensions.begin(); b != (*a).extensions.end(); ++b ) {
                filter1 += " *." + (*b);
                temp += " *." + (*b);
            }
            //temp.stripWhiteSpace(); // NOTE doesn't work
            if( temp != "" ) {
                temp.remove( 0, 1 );
                filter2 += "\n" + temp + "|" + (*a).extensions.first() + " " + i18n("files") + " (" + temp + ")";
            }
        }
    }

    filter1.stripWhiteSpace();
    filter2.stripWhiteSpace();

    return filter1 + "|" + i18n("all supported formats") + "\n*.*|" + i18n("all formats") + filter2;
}

QString Config::replayGainFilter()
{}

