
#include "outputdirectory.h"
#include "filelist.h"
#include "conversionoptions.h"
#include "tagengine.h"
#include "config.h"

#include <qlayout.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qlabel.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <ktoolbarbutton.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <kuser.h>
#include <kmountpoint.h>


OutputDirectory::OutputDirectory( Config* _config, QWidget* parent, const char* name )
    : QWidget( parent, name )
{
    config = _config;

    // create an icon loader object for loading icons
    KIconLoader *iconLoader = new KIconLoader();

    QGridLayout *grid = new QGridLayout( this, 1, 1, 0, 3, "grid" );

    QHBoxLayout *box = new QHBoxLayout( );
    grid->addLayout( box, 0, 0 );

    QLabel *lOutput = new QLabel( i18n("Output")+":", this, "lOutput" );
    box->addWidget(lOutput);

    cMode = new KComboBox( this );
//     cMode->insertItem( i18n("Default output directory") );
    cMode->insertItem( i18n("By meta data") );
    cMode->insertItem( i18n("Source directory") );
    cMode->insertItem( i18n("Specify output directory") );
    cMode->insertItem( i18n("Copy directory structure") );
    //QToolTip::add( cMode, i18n("Output all converted files into...") );
    box->addWidget( cMode );
    connect( cMode, SIGNAL(activated(int)),
               this, SLOT(modeChangedSlot(int))
             );
    modeJustChanged = false;
    /*pModeInfo = new KToolBarButton( "messagebox_info", 1010, this, "pModeInfo" );
    QToolTip::add( pModeInfo, i18n("Information about the output mode") );
    box->addWidget( pModeInfo );
    connect( pModeInfo, SIGNAL(clicked()),
               this, SLOT(modeInfo())
             );*/
    pClear = new KToolBarButton( "locationbar_erase", 1001, this, "pClear" );
    QToolTip::add( pClear, i18n("Clear the directory input field") );
    box->addWidget( pClear );
    lDir = new KLineEdit( this, "lDir" );
    box->addWidget( lDir );
    connect( lDir, SIGNAL(textChanged(const QString&)),
               this, SLOT(directoryChangedSlot(const QString&))
             );
    connect( pClear, SIGNAL(clicked()),
               lDir, SLOT(setFocus())
             );
    connect( pClear, SIGNAL(clicked()),
               lDir, SLOT(clear())
             );
    /*pDirInfo = new KToolBarButton( "messagebox_info", 1011, this, "pDirInfo" );
    QToolTip::add( pDirInfo, i18n("Information about the wildcards") );
    box->addWidget( pDirInfo );
    connect( pDirInfo, SIGNAL(clicked()),
               this, SLOT(dirInfo())
             );*/
    pDirSelect = new KToolBarButton( "folder", 1012, this, "pDirSelect" );
    QToolTip::add( pDirSelect, i18n("Choose an output directory") );
    box->addWidget( pDirSelect );
    connect( pDirSelect, SIGNAL(clicked()),
               this, SLOT(selectDir())
             );
    pDirGoto = new KToolBarButton( "konqueror", 1013, this, "pDirGoto" );
    QToolTip::add( pDirGoto, i18n("Open Konqueror with the output directory") );
    box->addWidget( pDirGoto );
    connect( pDirGoto, SIGNAL(clicked()),
               this, SLOT(gotoDir())
             );

    // delete the icon loader object
    delete iconLoader;

    modeChangedSlot( MetaData ); // TODO implement proper
                                // save the current directory always on text change
}

OutputDirectory::~OutputDirectory()
{}

void OutputDirectory::disable()
{
    cMode->setEnabled( false );
    lDir->setEnabled( false );
    pClear->setEnabled( false );
    pDirSelect->setEnabled( false );
}

void OutputDirectory::enable()
{
    cMode->setEnabled( true );
    modeChangedSlot( cMode->currentItem() );
}

OutputDirectory::Mode OutputDirectory::mode()
{
    return (Mode)cMode->currentItem();
}

void OutputDirectory::setMode( OutputDirectory::Mode mode )
{
    cMode->setCurrentItem( (int)mode );
    modeChangedSlot( (int)mode );
}

QString OutputDirectory::directory()
{
    if( /*(Mode)cMode->currentItem() != Default && */(Mode)cMode->currentItem() != Source ) return lDir->text();
    else return "";
}

void OutputDirectory::setDirectory( const QString& directory )
{
    if( /*(Mode)cMode->currentItem() != Default && */(Mode)cMode->currentItem() != Source ) lDir->setText( directory );
}

QString OutputDirectory::calcPath( FileListItem* fileListItem, Config* config, QString extension )
{
    // TODO replace '//' by '/' ???
    // FIXME test fvat names
    QString path;
    if( extension.isEmpty() ) extension = fileListItem->options.encodingOptions.sFormat;

    // FIXME file name
    QString fileName;
    if( fileListItem->track == -1 ) fileName = fileListItem->fileName;
    else if( fileListItem->tags != 0 ) fileName =  QString().sprintf("%02i",fileListItem->tags->track) + " - " + fileListItem->tags->title + "." + extension;
    else fileName = "track" + QString::number(fileListItem->track) + "." + extension; // NOTE shouldn't be possible

    // if the user wants to change the output directory/file name per file!
    if( !fileListItem->options.outputFilePathName.isEmpty() ) {
//         path = uniqueFileName( changeExtension(fileListItem->options.outputFilePathName,extension) );
        path = changeExtension(fileListItem->options.outputFilePathName,extension);
        if( config->data.general.useVFATNames ) path = vfatPath( path );
        return path;
    }

    if( fileListItem->options.outputOptions.mode == Specify ) {
//         path = uniqueFileName( changeExtension(fileListItem->options.outputOptions.directory+"/"+fileName,extension) );
        path = changeExtension(fileListItem->options.outputOptions.directory+"/"+fileName,extension);
        if( config->data.general.useVFATNames ) path = vfatPath( path );
        return path;
    }
    else if( /*fileListItem->options.outputOptions.mode == Default || */fileListItem->options.outputOptions.mode == MetaData ) {
        /*if( fileListItem->options.outputOptions.mode == Default ) path = config->data.general.defaultOutputDirectory;
        else */path = fileListItem->options.outputOptions.directory;

        if( path.right(1) == "/" ) path += "%f";
        else if( path.findRev(QRegExp("%[aAbBcCdDfFgGnNpPtTyY]{1,1}")) < path.findRev("/") ) path += "/%f";

        path.replace( "%a", "$replace_by_artist$" );
        path.replace( "%b", "$replace_by_album$" );
        path.replace( "%c", "$replace_by_comment$" );
        path.replace( "%d", "$replace_by_disc$" );
        path.replace( "%g", "$replace_by_genre$" );
        path.replace( "%n", "$replace_by_track$" );
        path.replace( "%p", "$replace_by_composer$" );
        path.replace( "%t", "$replace_by_title$" );
        path.replace( "%y", "$replace_by_year$" );
        path.replace( "%f", "$replace_by_filename$" );

        QString artist = ( fileListItem->tags == 0 || fileListItem->tags->artist.isEmpty() ) ? i18n("Unknown Artist") : fileListItem->tags->artist;
///        artist.replace("/","\\");
        path.replace( "$replace_by_artist$", KURL::encode_string(artist).replace("/","%2f") );

        QString album = ( fileListItem->tags == 0 || fileListItem->tags->album.isEmpty() ) ? i18n("Unknown Album") : fileListItem->tags->album;
///        album.replace("/","\\");
        path.replace( "$replace_by_album$", KURL::encode_string(album).replace("/","%2f") );

        QString comment = ( fileListItem->tags == 0 || fileListItem->tags->comment.isEmpty() ) ? i18n("No Comment") : fileListItem->tags->comment;
///        comment.replace("/","\\");
        path.replace( "$replace_by_comment$", KURL::encode_string(comment).replace("/","%2f") );

        QString disc = ( fileListItem->tags == 0 ) ? "0" : QString().sprintf("%i",fileListItem->tags->disc);
        path.replace( "$replace_by_disc$", disc );

        QString genre = ( fileListItem->tags == 0 || fileListItem->tags->genre.isEmpty() ) ? i18n("Unknown Genre") : fileListItem->tags->genre;
///        genre.replace("/","\\");
        path.replace( "$replace_by_genre$", KURL::encode_string(genre).replace("/","%2f") );

        QString track = ( fileListItem->tags == 0 ) ? "00" : QString().sprintf("%02i",fileListItem->tags->track);
        path.replace( "$replace_by_track$", track );

        QString composer = ( fileListItem->tags == 0 || fileListItem->tags->composer.isEmpty() ) ? i18n("Unknown Composer") : fileListItem->tags->composer;
///        composer.replace("/","\\");
        path.replace( "$replace_by_composer$", KURL::encode_string(composer).replace("/","%2f") );

        QString title = ( fileListItem->tags == 0 || fileListItem->tags->title.isEmpty() ) ? i18n("Unknown Title") : fileListItem->tags->title;
///        title.replace("/","\\");
        path.replace( "$replace_by_title$", KURL::encode_string(title).replace("/","%2f") );

        QString year = ( fileListItem->tags == 0 ) ? "0000" : QString().sprintf("%04i",fileListItem->tags->year);
        path.replace( "$replace_by_year$", year );

        QString filename = fileName.left( fileName.findRev(".") );
///        filename.replace("/","\\");
        path.replace( "$replace_by_filename$", filename );

//         path = uniqueFileName( path + "." + extension );
        path = path + "." + extension;
        if( config->data.general.useVFATNames ) path = vfatPath( path );
        return path;
    }
    else if( fileListItem->options.outputOptions.mode == CopyStructure ) { // TODO is that correct ???
        QString basePath = fileListItem->options.outputOptions.directory;
        QString originalPath = fileListItem->options.filePathName;
        QString cutted;
        while( basePath.length() > 0 ) {
            if( fileListItem->options.filePathName.find(basePath) == 0 ) {
                originalPath.replace( basePath, "" );
                return uniqueFileName( changeExtension(basePath+cutted+originalPath,extension) );
            }
            else {
                cutted = basePath.right( basePath.length() - basePath.findRev("/") ) + cutted;
                basePath = basePath.left( basePath.findRev("/") );
            }
        }
//         path = uniqueFileName( changeExtension(fileListItem->options.outputOptions.directory+"/"+fileListItem->options.filePathName,extension) );
        path = changeExtension(fileListItem->options.outputOptions.directory+"/"+fileListItem->options.filePathName,extension);
        if( config->data.general.useVFATNames ) path = vfatPath( path );
        return path;
    }
    else {
//         path = uniqueFileName( changeExtension(fileListItem->options.filePathName,extension) );
        path = changeExtension(fileListItem->options.filePathName,extension);
        if( config->data.general.useVFATNames ) path = vfatPath( path );
        return path;
    }
}

QString OutputDirectory::changeExtension( const QString& filename, const QString& extension )
{
    return filename.left( filename.findRev(".") + 1 ) + extension;
}

QString OutputDirectory::uniqueFileName( const QString& filename )
{
    QString filePathName;

    if( filename.left( 1 ) == "/" ) {
        filePathName = filename;
    }
    else if( filename.left( 7 ) == "file://" ) {
        filePathName = filename;
        filePathName.remove( 0, 7 );
    }
    else if( filename.left( 13 ) == "system:/home/" ) {
        filePathName = filename;
        filePathName.remove( 0, 13 );
        filePathName = QDir::homeDirPath() + "/" + filePathName;
    }
    else if( filename.left( 14 ) == "system:/users/" || filename.left( 6 ) == "home:/" ) {
        int length = ( filename.left(6) == "home:/" ) ? 6 : 14;
        QString username = filename;
        username.remove( 0, length );
        username = username.left( username.find("/") );
        filePathName = filename;
        filePathName.remove( 0, length + username.length() );
        KUser user( username );
        filePathName = user.homeDir() + filePathName;
    }
    else if( filename.left( 14 ) == "system:/media/" || filename.left( 7 ) == "media:/" ) {
        int length = ( filename.left(7) == "media:/" ) ? 7 : 14;
        QString device = filename;
        device.remove( 0, length );
        device = "/dev/" + device.left( device.find( "/" ) );

        KMountPoint::List mountPoints = KMountPoint::possibleMountPoints();

        for( KMountPoint::List::ConstIterator jt = mountPoints.begin(); jt != mountPoints.end(); ++jt )
        {
            const KSharedPtr<KMountPoint> mp = *jt;
            if( mp->mountedFrom() == device )
            {
                filePathName = ( mp->mountPoint() == "/" ) ? mp->mountPoint() : mp->mountPoint() + "/";
                filePathName += filename.right( filename.length() - device.length() - length + 4 );
            }
        }
    }
    else {
        filePathName = filename;
    }

    QFileInfo fileInfo( KURL::decode_string(filePathName) );

    // generate a unique file name
    while( fileInfo.exists() ) {
        fileInfo.setFile( fileInfo.filePath().left( fileInfo.filePath().findRev(".")+1 ) + i18n("new") + fileInfo.filePath().right( fileInfo.filePath().length() - fileInfo.filePath().findRev(".") ) );
    }

    return KURL::encode_string( fileInfo.filePath() ); // was: .replace( "//", "/" )
}

QString OutputDirectory::makePath( const QString& path )
{
    QFileInfo fileInfo( path );

    QStringList dirs = QStringList::split( "/", fileInfo.dirPath() );
    QString mkDir;
    QDir dir;
    for( QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it ) {
        mkDir += "/" + KURL::decode_string(*it).replace("/","%2f");
        dir.setPath( mkDir );
        if( !dir.exists() ) dir.mkdir( mkDir );
    }

    return path;
}

// copyright            : (C) 2002 by Mark Kretschmann
// email                : markey@web.de
// modified             : 2008 Daniel Faust <hessijames@gmail.com>
QString OutputDirectory::vfatPath( const QString& path )
{
    QString s = KURL::decode_string(path.right( path.length() - path.findRev("/") - 1 ));
    QString p = KURL::decode_string(path.left( path.findRev("/") + 1 ));

    for( uint i = 0; i < s.length(); i++ )
    {
        QChar c = s.ref( i );
        if( c < QChar(0x20)
                || c=='*' || c=='?' || c=='<' || c=='>'
                || c=='|' || c=='"' || c==':' || c=='/'
                || c=='\\' )
            c = '_';
        s.ref( i ) = c;
    }

    uint len = s.length();
    if( len == 3 || (len > 3 && s[3] == '.') )
    {
        QString l = s.left(3).lower();
        if( l=="aux" || l=="con" || l=="nul" || l=="prn" )
            s = "_" + s;
    }
    else if( len == 4 || (len > 4 && s[4] == '.') )
    {
        QString l = s.left(3).lower();
        QString d = s.mid(3,1);
        if( (l=="com" || l=="lpt") &&
                (d=="0" || d=="1" || d=="2" || d=="3" || d=="4" ||
                    d=="5" || d=="6" || d=="7" || d=="8" || d=="9") )
            s = "_" + s;
    }

    while( s.startsWith( "." ) )
        s = s.mid(1);

    while( s.endsWith( "." ) )
        s = s.left( s.length()-1 );

    s = s.left(255);
    len = s.length();
    if( s[len-1] == ' ' )
        s[len-1] = '_';

    return QString( p + s ).replace("%2f","_");
//     return p + s;
}

void OutputDirectory::selectDir()
{
    QString startDir = lDir->text();
    int i = startDir.find( QRegExp("%[aAbBcCdDfFgGnNpPtTyY]{1,1}") );
    if( i != -1 ) {
        i = startDir.findRev( "/", i );
        startDir = startDir.left( i );
    }

    QString directory = KFileDialog::getExistingDirectory( startDir, this, i18n("Choose an output directory") );
    if( !directory.isEmpty() ) {
        QString dir = lDir->text();
        i = dir.find( QRegExp("%[aAbBcCdDfFgGnNpPtTyY]{1,1}") );
        if( i != -1 && (Mode)cMode->currentItem() == MetaData ) {
            i = dir.findRev( "/", i );
            lDir->setText( directory + dir.mid(i) );
            emit directoryChanged( directory + dir.mid(i) );
        }
        else {
            lDir->setText( directory );
            emit directoryChanged( directory );
        }
    }
}

void OutputDirectory::gotoDir()
{
    QString startDir = lDir->originalText();
    int i = startDir.find( QRegExp("%[aAbBcCdDfFgGnNpPtTyY]{1,1}") );
    if( i != -1 ) {
        i = startDir.findRev( "/", i );
        startDir = startDir.left( i );
    }

    kfm.clearArguments();
    kfm << "kfmclient";
    kfm << "openURL";
    kfm << startDir;
    kfm.start( KProcess::DontCare );
}

void OutputDirectory::modeChangedSlot( int mode )
{
    modeJustChanged = true;

    /*if( (Mode)mode == Default ) {
        pClear->setEnabled( false );
        lDir->setText( config->data.general.defaultOutputDirectory );
        lDir->setEnabled( true );
        lDir->setReadOnly( true );
        lDir->setEnableSqueezedText( true );
        pDirSelect->setEnabled( false );
        // TODO hide pDirSelect and show pDirEdit
        pDirGoto->setEnabled( true );
        //pDirInfo->hide();
        QToolTip::remove( cMode );
        QToolTip::add( cMode, i18n("Output all converted files into the soundKonverter default output directory") );
        QToolTip::remove( lDir );
    }
    else */
    if( (Mode)mode == MetaData ) {
        pClear->setEnabled( true );
        if( config->data.general.metaDataOutputDirectory.isEmpty() ) config->data.general.metaDataOutputDirectory = QDir::homeDirPath() + "/soundKonverter/%b/%d - %n - %a - %t";
        lDir->setText( config->data.general.metaDataOutputDirectory );
        lDir->setEnabled( true );
        lDir->setReadOnly( false );
//         lDir->setEnableSqueezedText( false );
        pDirSelect->setEnabled( true );
        pDirGoto->setEnabled( true );
        //pDirInfo->show();
        QToolTip::remove( cMode );
        QToolTip::add( cMode, i18n("Name all converted files according to the specified pattern") );
        QToolTip::remove( lDir );
        QToolTip::add( lDir, i18n("<p>The following strings are wildcards, that will be replaced by the information in the meta data:</p><p>%a - Artist<br>%b - Album<br>%c - Comment<br>%d - Disc number<br>%g - Genre<br>%n - Track number<br>%p - Composer<br>%t - Title<br>%y - Year<br>%f - Original file name<p>") );
    }
    else if( (Mode)mode == Source ) {
        pClear->setEnabled( false );
        lDir->setText( "" );
        lDir->setEnabled( false );
        lDir->setReadOnly( false );
//         lDir->setEnableSqueezedText( false );
        pDirSelect->setEnabled( false );
        pDirGoto->setEnabled( false );
        //pDirInfo->hide();
        QToolTip::remove( cMode );
        QToolTip::add( cMode, i18n("Output all converted files into the same directory as the original files") );
        QToolTip::remove( lDir );
    }
    else if( (Mode)mode == Specify ) {
        pClear->setEnabled( true );
        if( config->data.general.specifyOutputDirectory.isEmpty() ) config->data.general.specifyOutputDirectory = QDir::homeDirPath() + "/soundKonverter";
        lDir->setText( config->data.general.specifyOutputDirectory );
        lDir->setEnabled( true );
        lDir->setReadOnly( false );
//         lDir->setEnableSqueezedText( false );
        pDirSelect->setEnabled( true );
        pDirGoto->setEnabled( true );
        //pDirInfo->hide();
        QToolTip::remove( cMode );
        QToolTip::add( cMode, i18n("Output all converted files into the specified output directory") );
        QToolTip::remove( lDir );
    }
    else if( (Mode)mode == CopyStructure ) {
        pClear->setEnabled( true );
        if( config->data.general.copyStructureOutputDirectory.isEmpty() ) config->data.general.copyStructureOutputDirectory = QDir::homeDirPath() + "/soundKonverter";
        lDir->setText( config->data.general.copyStructureOutputDirectory );
        lDir->setEnabled( true );
        lDir->setReadOnly( false );
//         lDir->setEnableSqueezedText( false );
        pDirSelect->setEnabled( true );
        pDirGoto->setEnabled( true );
        //pDirInfo->hide();
        QToolTip::remove( cMode );
        QToolTip::add( cMode, i18n("Copy the whole directory structure for all converted files") );
        QToolTip::remove( lDir );
    }

    emit modeChanged( (Mode)mode );

    modeJustChanged = false;
}

void OutputDirectory::directoryChangedSlot( const QString& directory )
{
    if( modeJustChanged ) {
        modeJustChanged = false;
        return;
    }

    Mode mode = (Mode)cMode->currentItem();

    if( mode == MetaData ) {
        config->data.general.metaDataOutputDirectory = directory;
    }
    else if( mode == Specify ) {
        config->data.general.specifyOutputDirectory = directory;
    }
    else if( mode == CopyStructure ) {
        config->data.general.copyStructureOutputDirectory = directory;
    }

    emit directoryChanged( directory );
}

/*void OutputDirectory::modeInfo()
{
    int mode = cMode->currentItem();
    QString sModeString = cMode->currentText();

    if( (Mode)mode == Default ) {
        KMessageBox::information( this,
            i18n("This will output each file into the soundKonverter default directory."),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
    else if( (Mode)mode == Source ) {
        KMessageBox::information( this,
            i18n("This will output each file into the same directory as the original file."),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
    else if( (Mode)mode == Specify ) {
        KMessageBox::information( this,
            i18n("This will output each file into the directory specified in the editbox behind."),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
    else if( (Mode)mode == MetaData ) {
        KMessageBox::information( this,
            i18n("This will output each file into a directory, which is created based on the metadata in the audio files. Select a directory, where the new directories should be created."),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
    else if( (Mode)mode == CopyStructure ) {
        KMessageBox::information( this,
            i18n("This will output each file into a directory, which is created based on the name of the original directory. So you can copy a whole directory structure, in one you have the original files, in the other the converted."),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
    else {
        KMessageBox::error( this,
            i18n("This mode (%s) doesn't exist.", sModeString),
            QString(i18n("Mode")+": ").append(sModeString) );
    }
}*/

/*void OutputDirectory::dirInfo()
{
    KMessageBox::information( this,
        i18n("<p>The following strings are space holders, that will be replaced by the information in the metatags.</p><p>%a - Artist<br>%b - Album<br>%c - Comment<br>%d - Disc number<br>%g - Genre<br>%n - Track number<br>%p - Composer<br>%t - Title<br>%y - Year<br>%f - Original file name<p>"),
        QString(i18n("Legend")) );
}*/

