/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*************************************************
 * atmosphere.c : define how background is cleared
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 *************************************************/

#include <math.h>

#include "p3_base.h"
#include "math3d.h"
#include "coordsys.h"
#include "material.h"
#include "renderer.h"
#include "util.h"
#include "atmosphere.h"

extern P3_renderer* renderer;
extern GLfloat black[4];


void P3_clear_screen (GLfloat* color) {
  if (renderer->c_camera->option & P3_CAMERA_PARTIAL) {
    int* size = renderer->c_camera->size;
    glDisable (GL_LIGHTING);
    glDisable (GL_FOG);
    glDisable (GL_TEXTURE_2D);
    glDisable (GL_CULL_FACE);
    glDepthMask (GL_FALSE);
    if (color == NULL) {
      glColor4fv (black);
    } else {
      glColor4fv (color);
    }
    glLoadIdentity ();
    glMatrixMode (GL_PROJECTION);
    glPushMatrix ();
    glLoadIdentity ();
    glOrtho (0.0, (GLfloat) size[2], (GLfloat) size[3], 0.0, -1.0, 1.0);
    glBegin (GL_QUADS);
    glVertex2i (0, 0);
    glVertex2i (size[2], 0);
    glVertex2i (size[2], size[3]);
    glVertex2i (0, size[3]);
    glEnd ();
    glMatrixMode (GL_PROJECTION);
    glPopMatrix ();
    glMatrixMode (GL_MODELVIEW);
    glEnable (GL_CULL_FACE);
    glEnable (GL_TEXTURE_2D);
    glEnable (GL_FOG);
    glEnable (GL_LIGHTING);
    glDepthMask (GL_TRUE);
    glClear (GL_DEPTH_BUFFER_BIT);
  } else {
    if (color == NULL) {
      glClearColor (0.0, 0.0, 0.0, 1.0);
    } else {
      glClearColor (color[0], color[1], color[2], color[3]);
    }
    glClear (GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
  }
}

GLfloat P3_fog_factor_at (P3_atmosphere* atm, GLfloat p[3]) {
  GLfloat z = sqrt(p[0] * p[0] + p[1] * p[1] + p[2] * p[2]);
  if (atm->fog_type == GL_LINEAR) {
    return (GLfloat) (1.0 - (atm->fog_end - z) / (atm->fog_end - atm->fog_start));
  } else if (atm->fog_type == GL_EXP) {
    return (GLfloat) (1.0 - exp(atm->fog_density * z));
  } else if (atm->fog_type == GL_EXP2) {
    return (GLfloat) (1.0 - exp(atm->fog_density * atm->fog_density * z * z));
  }
  return 0.0;
}

P3_atmosphere* P3_atmosphere_new (P3_atmosphere* atm) {
  if (atm == NULL) {
    atm = (P3_atmosphere*) malloc (sizeof (P3_atmosphere));
  }
  atm->option = 0;
  atm->ambient[0] = 0.5;
  atm->ambient[1] = 0.5;
  atm->ambient[2] = 0.5;
  atm->ambient[3] = 1.0;
  atm->bg_color[0] = 0.0;
  atm->bg_color[1] = 0.0;
  atm->bg_color[2] = 0.0;
  atm->bg_color[3] = 1.0;
  atm->fog_color[0] = 0.0;
  atm->fog_color[1] = 0.0;
  atm->fog_color[2] = 0.0;
  atm->fog_color[3] = 1.0;
  atm->fog_type = GL_LINEAR;
  atm->fog_start = 10.0;
  atm->fog_end = 100.0;
  atm->fog_density = 1.0;
  atm->cloud = NULL;
  atm->skybox = NULL;
  return atm;
}

static void P3_atmosphere_draw_skyplane (P3_atmosphere* atm) {
  GLfloat plane[4];
  GLfloat face1[12];
  GLfloat y; GLfloat h;
  GLfloat f; GLfloat g;
  GLfloat* ptr;
  GLfloat* face2;
  GLfloat* face3;
  int nb2;
  int nb3;
  int i;

  glLoadMatrixf (renderer->c_camera->render_matrix);

  /* draw gradient */
  P3_material_activate (NULL);
  ptr = renderer->c_camera->frustum->points;
  for (i = 0; i < 12; i++) {
    face1[i] = ptr[12 + i] * 0.5;
  }

  ptr = P3_coordsys_get_root_matrix ((P3_coordsys*) renderer->c_camera);
  P3_point_by_matrix (face1,     ptr);
  P3_point_by_matrix (face1 + 3, ptr);
  P3_point_by_matrix (face1 + 6, ptr);
  P3_point_by_matrix (face1 + 9, ptr);

  y = renderer->c_camera->back * 0.5;
  h = renderer->r_frustum->position[1];
  plane[0] = 0.0;
  plane[1] = - 1.0;
  plane[2] = 0.0;
  plane[3] = h + y;
  P3_face_intersect_plane (face1, 4, plane, &face2, &nb2);
  if (nb2 > 0) {
    glColor4fv (atm->sky_color);
    glBegin (GL_POLYGON);
    for (i = 0; i < nb2; i++) {
      glVertex3fv (face2 + i * 3);
    }
    glEnd ();
  }
  free (face2);

  plane[1] = 1.0;
  plane[3] = - plane[3];
  P3_face_intersect_plane (face1, 4, plane, &face3, &nb3);
  plane[1] = - 1.0;
  plane[3] = renderer->r_frustum->position[1];
  P3_face_intersect_plane (face3, nb3, plane, &face2, &nb2);
  free (face3);
  if (nb2 > 0) {
    glBegin (GL_POLYGON);
    for (i = 0; i < nb2 * 3; i += 3) {
      f = (face2[i + 1] - h) / y;
      g = 1.0 - f;
      glColor4f (atm->sky_color[0] * f + atm->bg_color[0] * g, 
                 atm->sky_color[1] * f + atm->bg_color[1] * g, 
                 atm->sky_color[2] * f + atm->bg_color[2] * g, 
                 atm->sky_color[3] * f + atm->bg_color[3] * g);
      glVertex3fv (face2 + i);
    }
    glEnd ();
  }
  free (face2);

  /* draw clouds */
  if (atm->cloud != NULL) {
    plane[1] = renderer->r_frustum->position[1] + 5.0;
    h = renderer->c_camera->back * 0.7;
    glEnable (GL_BLEND);
    P3_material_activate (atm->cloud);

    ptr = renderer->r_frustum->position;
    plane[0] = ptr[0] * 0.01;
    plane[2] = ptr[2] * 0.01;
    y = h * 0.1;

    ptr = renderer->r_frustum->position;
    glTranslatef (ptr[0], 0.0, ptr[2]);

    glBegin (GL_TRIANGLE_FAN);

    glTexCoord2f (plane[0], plane[2]);
    glVertex3f (0.0, plane[1], 0.0);

    ptr = atm->cloud->diffuse;
    glColor4f (ptr[0], ptr[1], ptr[2], 0.0);

    glTexCoord2f (plane[0] - y, plane[2] - y);
    glVertex3f (- h, plane[1], - h);
    glTexCoord2f (plane[0] + y, plane[2] - y);
    glVertex3f (h, plane[1], - h);
    glTexCoord2f (plane[0] + y, plane[2] + y);
    glVertex3f (h, plane[1], h);
    glTexCoord2f (plane[0] - y, plane[2] + y);
    glVertex3f (- h, plane[1], h);
    glTexCoord2f (plane[0] - y, plane[2] - y);
    glVertex3f (- h, plane[1], - h);

    glEnd ();
    glDisable (GL_BLEND);
  }
}

static void P3_atmosphere_draw_skybox (P3_atmosphere* atm) {
  GLfloat* ptr;

#define SKYBOX_DISTANCE 10.0

  ptr = renderer->r_frustum->position;
  glLoadMatrixf (renderer->c_camera->render_matrix);
  glTranslatef (ptr[0], ptr[1], ptr[2]);
  if (atm->option & P3_ATMOSPHERE_SKYBOX_ALPHA) { glEnable (GL_BLEND); }
  /* skybox material order is: front, right, back, left, bottom, top */
  /* font */
  P3_material_activate (atm->skybox[0]);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glEnd ();
  /* right */
  P3_material_activate (atm->skybox[1]);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glEnd ();
  /* back */
  P3_material_activate (atm->skybox[2]);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glEnd ();
  /* left */
  P3_material_activate (atm->skybox[3]);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glEnd ();
  /* bottom */
  P3_material_activate (atm->skybox[4]);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, SKYBOX_DISTANCE);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (-SKYBOX_DISTANCE, -SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
  glEnd ();
  if (atm->option & P3_ATMOSPHERE_SKYBOX_6) {
    /* top */
    P3_material_activate (atm->skybox[5]);
    glBegin (GL_QUADS);
    glTexCoord2f (0.0, 0.0);
    glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
    glTexCoord2f (1.0, 0.0);
    glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, -SKYBOX_DISTANCE);
    glTexCoord2f (1.0, 1.0);
    glVertex3f (SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
    glTexCoord2f (0.0, 1.0);
    glVertex3f (-SKYBOX_DISTANCE, SKYBOX_DISTANCE, SKYBOX_DISTANCE);
    glEnd ();
  }
  glDisable (GL_BLEND);

#undef SKYBOX_DISTANCE

}

void P3_atmosphere_draw_bg (P3_atmosphere* atm) {
  glDisable (GL_LIGHTING);
  glDisable (GL_FOG);
  glDisable (GL_DEPTH_TEST);
  glDepthMask (GL_FALSE);
  glDisable (GL_CULL_FACE);
  if (atm->option & P3_ATMOSPHERE_SKYPLANE) {
    P3_atmosphere_draw_skyplane (atm);
  }
  if (atm->option & (P3_ATMOSPHERE_SKYBOX_5 | P3_ATMOSPHERE_SKYBOX_6)) {
    P3_atmosphere_draw_skybox (atm);
  }
  glEnable (GL_LIGHTING);
  glEnable (GL_FOG);
  glEnable (GL_DEPTH_TEST);
  glDepthMask (GL_TRUE);
  glEnable (GL_CULL_FACE);
}

void P3_atmosphere_clear (P3_atmosphere* atm) {
  /* clear the screen. only called with root atmosphere, by the renderer */
  if (!(atm->option & P3_ATMOSPHERE_DONT_CLEAR)) {
    P3_clear_screen (atm->bg_color);
    P3_atmosphere_draw_bg (atm);
  }
}

void P3_atmosphere_render (P3_atmosphere* atm) {
  /* apply fog and ambient */
  glLightModelfv (GL_LIGHT_MODEL_AMBIENT, atm->ambient);
  if (atm->option & P3_ATMOSPHERE_FOG) { 
    glFogf (GL_FOG_MODE, atm->fog_type);
    glFogf (GL_FOG_START, atm->fog_start);
    glFogf (GL_FOG_END, atm->fog_end);
    glFogf (GL_FOG_DENSITY, atm->fog_density);
    glFogfv (GL_FOG_COLOR, atm->fog_color);
    glEnable (GL_FOG);
  } else {
    glDisable (GL_FOG);
  }
}

void P3_atmosphere_get_data (P3_atmosphere* a, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, a->option);
  P3_chunk_save_int (chunk, a->fog_type);
  P3_chunk_save_float (chunk, a->fog_start);
  P3_chunk_save_float (chunk, a->fog_end);
  P3_chunk_save_float (chunk, a->fog_density);
  P3_chunk_save (chunk, a->bg_color,  4 * sizeof (GLfloat));
  P3_chunk_save (chunk, a->fog_color, 4 * sizeof (GLfloat));
  P3_chunk_save (chunk, a->ambient,   4 * sizeof (GLfloat));
  if (a->option & P3_ATMOSPHERE_SKYPLANE) {
    P3_chunk_save (chunk, a->sky_color, 4 * sizeof (GLfloat));
  }
}

void P3_atmosphere_set_data (P3_atmosphere* a, P3_chunk* chunk) {
  a->option      = P3_chunk_load_int (chunk);
  a->fog_type    = P3_chunk_load_int (chunk);
  a->fog_start   = P3_chunk_load_float (chunk);
  a->fog_end     = P3_chunk_load_float (chunk);
  a->fog_density = P3_chunk_load_float (chunk);
  P3_chunk_load (chunk, a->bg_color,  4 * sizeof (GLfloat));
  P3_chunk_load (chunk, a->fog_color, 4 * sizeof (GLfloat));
  P3_chunk_load (chunk, a->ambient,   4 * sizeof (GLfloat));
  if (a->option & P3_ATMOSPHERE_SKYPLANE) {
    P3_chunk_load (chunk, a->sky_color, 4 * sizeof (GLfloat));
  }
}

