/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************************
 * engine.c : general engine functions and global variables
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************************/

#ifdef USE_GLX
#include <X11/Xlib.h>
#include <X11/extensions/Xext.h>
#include <X11/extensions/xf86vmode.h>
#include <GL/glx.h>
#endif /* USE_GLX */

#ifdef USE_SDL
#include <SDL/SDL.h>
#endif /* USE_SDL */

#include "p3_base.h"
#include "util.h"
#include "material.h"
#include "renderer.h"
#include "gladd.h"
#include "fx.h"
//#include "bsp.h"
#include "cal3d4p3.h"
#include "engine.h"


/* - GLOBAL VARIABLES - */

/* OpenGL extension function */

#if !(GL_VERSION_1_3)
void (*glMultiTexCoord2f)     (GLenum, GLfloat, GLfloat) = 0;
void (*glMultiTexCoord2fv)    (GLenum, GLfloat*) = 0;
void (*glActiveTexture)       (GLenum) = 0;
void (*glClientActiveTexture) (GLenum) = 0;
#endif /* ! GL_VERSION_1_3 */

/*
void (*glBindBuffer)    (GLenum, GLuint) = 0;
void (*glBufferData)    (GLenum, GLuint, void*, GLenum) = 0;
void (*glDeleteBuffers) (GLsizei, GLuint*) = 0;
void (*glGenBuffers)    (GLsizei, GLuint*) = 0;
*/

/* general engine variables : */

#ifdef USE_SDL
SDL_Surface* screen = NULL;
int nb_joysticks = 0;
SDL_Joystick** joysticks = NULL;
#endif /* USE_SDL */

#ifdef USE_GLX
Display* display;
Window window;
GLXContext context = NULL;
int old_screen_w;
int old_screen_h;
#endif /* USE_GLX */

int          screen_w = 0;
int          screen_h = 0;
int          engine_option = P3_USE_MIPMAP | P3_SHADOWS;// | P3_USE_ALPHA_BSP;
int          quality = P3_QUALITY_HIGH;
P3_renderer* renderer = NULL;
P3_list*     land_tri_recycler;
//P3_bsp*      bsp;
//P3_list*     bsp_node_recycler;
P3_list* chunks;

int         maxtextures = 0;
int         maxtexturesize = 0;
int         maxclipplanes = 0;
//int         nbclipplanes  = 0;    /* the number of enabled clip planes */
int         maxlights = 0;   /* maximum number of light activated at a time (OpenGL limit) */
P3_light**  lights_gl = NULL;   
/* contain pointer to the current lights activated for OpenGL
 * this number of activated lights is limited by OpenGL */
int*        lights_gl_activated = NULL;
/* flag for each OpenGL lights to know if it is activated */
P3_material* current_material = NULL;

float delta_time = 0.0;

/* this list is never dealloc'd, allways usable. for use in functions where
 * you need a list but don't want to malloc one :) */
GLfloat  user_matrix[19];

GLfloat white[4]        = { 1.0, 1.0, 1.0, 1.0 };
GLfloat black[4]        = { 0.0, 0.0, 0.0, 1.0 };
GLfloat transparency[4] = { 1.0, 1.0, 1.0, 0.0 };

/* NULL material packs */
int        NULL_nb_packs = 0;
P3_xpack** NULL_packs = NULL;

P3_material* light_shader = NULL;
int shadow_display_list = -1;

/* fx */
P3_list* fx_color_faders = NULL;
// TO DO ?
//P3_list* fx_colors = NULL;


/* - FUNCTIONS - */

void P3_error (char* fmt, ...) {
  va_list ap;
  va_start (ap, fmt);
  fprintf (stderr, "ERROR : ");
  vfprintf (stderr, fmt, ap);
  fprintf (stderr, "\n");
  va_end (ap);
}

#ifdef USE_GLX

static int P3_compare_mode (XF86VidModeModeInfo** a, XF86VidModeModeInfo** b) {
  if ((*a)->hdisplay > (*b)->hdisplay) { return -1; } else { return 1; }
}

void P3_set_mode (int width, int height) {
  XF86VidModeModeLine mode;
  XF86VidModeModeInfo** modes;
  int i;
  int nmodes;
  /* copied from SDL */
  if (XF86VidModeGetModeLine (display, DefaultScreen (display), &i, &mode) &&
      XF86VidModeGetAllModeLines (display, DefaultScreen (display), &nmodes, &modes)) {
    qsort (modes, nmodes, sizeof (XF86VidModeModeInfo*), (int (*)(const void *, const void *)) P3_compare_mode);
    for (i = nmodes - 1; i > 0 ; i--) {
      if ((modes[i]->hdisplay >= width) && (modes[i]->vdisplay >= height)) {
        break;
      }
    }
    if ((modes[i]->hdisplay != mode.hdisplay) || (modes[i]->vdisplay != mode.vdisplay)) {
      old_screen_w = mode.hdisplay;
      old_screen_h = mode.vdisplay;
      XF86VidModeSwitchToMode (display, DefaultScreen (display), modes[i]);
      screen_w = modes[i]->hdisplay;
      screen_h = modes[i]->vdisplay;
    }
    XFree(modes);
  }
}

void P3_set_video (int width, int height, int fullscreen, int resizable) {
  XVisualInfo* visual_info;
  int attrib_list[] = { GLX_RGBA, GLX_DOUBLEBUFFER, None };
  screen_w = width;
  screen_h = height;
  if (context == NULL) {
    visual_info = glXChooseVisual (display, 0, attrib_list);
    context = glXCreateContext (display, visual_info, NULL, True);
    if (context == NULL) {
      P3_error ("unable to create GLX context");
    }
    if (glXMakeCurrent (display, window, context) == False) {
      P3_error ("unable to make GLX context current");
    }
    XFree (visual_info);
  }

  if (fullscreen == P3_TRUE) {
    P3_set_mode (width, height);
    engine_option |= P3_FULLSCREEN;
    XMoveResizeWindow (display, window, 0, 0, width, height);
    XRaiseWindow (display, window);
  } else {
    if (engine_option & P3_FULLSCREEN) { 
      engine_option -= P3_FULLSCREEN; 
      if (old_screen_w != screen_w || old_screen_h != screen_h) {
        P3_set_mode (old_screen_w, old_screen_h);
      }
    }
  }
  XSync (display, False);
}

int P3_X_error (Display* d, XErrorEvent* e) {
  char str[64];
  XGetErrorText (display, e->error_code, str, 64);
  P3_error ("X error : %s", str);
  return 0;
}

void P3_init_video (char* title, int width, int height, int fullscreen, int resizable) {
  XSetErrorHandler ((XErrorHandler) P3_X_error);

  display = XOpenDisplay (NULL);
  window = XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0, width, height, 0, 0, 0);
  if (window == 0) {
    P3_error ("unable to create X window");
  }
  /* possible masks are listed in X11/X.h */
  XSelectInput (display, window, ResizeRedirectMask | ExposureMask | KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  if (title != NULL) {
    XTextProperty titleprop;
		XStringListToTextProperty ((char**) &title, 1, &titleprop);
		XSetWMName (display, window, &titleprop);
		XFree (titleprop.value);
  }

  XMapWindow (display, window);

  P3_set_video (width, height, fullscreen, resizable);

  P3_init_GL ();
}

#endif /* USE_GLX */

#ifdef USE_GLUT

void P3_init_video (char* title, int width, int height, int fullscreen, int resizable) {
  glutInit (NULL, NULL);
  glutInitDisplayMode ();
  glutInitWindowSize (width, height);
  glutSetWindowTitle (title);
}

#endif /* USE_GLUT */

#ifdef USE_SDL
void P3_set_video (int width, int height, int fullscreen, int resizable) {
  int stencil;
  int bits_per_pixel;
  Uint32 flags;
  SDL_VideoInfo* info = NULL;
  screen_w = width;
  screen_h = height;
  /* Information about the current video settings. */
  info = (SDL_VideoInfo*) SDL_GetVideoInfo ();
  if (!info) {
    P3_error ("Video query failed : %s", SDL_GetError());
    exit (P3_FALSE);
  }
  /* On X11, VidMode can't change
   * resolution, so this is probably being overly
   * safe. Under Win32, ChangeDisplaySettings
   * can change the bpp.
   */
  bits_per_pixel = info->vfmt->BitsPerPixel;
  flags = SDL_OPENGL | SDL_GL_DOUBLEBUFFER | SDL_HWPALETTE;
  if (fullscreen == P3_FALSE) { 
    engine_option &= ~P3_FULLSCREEN;
  } else { 
    engine_option |= P3_FULLSCREEN;
    flags |= SDL_FULLSCREEN; 
  }
  if (resizable == P3_TRUE) flags |= SDL_RESIZABLE;
  if (info->hw_available) 
    flags |= SDL_HWSURFACE;
  else
    flags |= SDL_SWSURFACE;
//  SDL_GL_SetAttribute (SDL_GL_RED_SIZE, 5);
//  SDL_GL_SetAttribute (SDL_GL_GREEN_SIZE, 5);
//  SDL_GL_SetAttribute (SDL_GL_BLUE_SIZE, 5);
//  SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE, 16);
  stencil = 16;
  while (stencil > 1) {
    SDL_GL_SetAttribute (SDL_GL_STENCIL_SIZE, stencil);
    /* Set the video mode */
    screen = SDL_SetVideoMode (width, height, bits_per_pixel, flags);
    if (screen == NULL) stencil = stencil >> 1;
    else break;
  }
  if (screen == NULL) {
    SDL_GL_SetAttribute (SDL_GL_STENCIL_SIZE, 0);
    screen = SDL_SetVideoMode (width, height, bits_per_pixel, flags);
    if (screen == NULL) {
      P3_error ("Video mode set failed : %s", SDL_GetError());
      exit (P3_FALSE);
    }
    fprintf (stdout, "Failed to set stencil buffer, shadows will be disabled\n");
    engine_option &= ~P3_SHADOWS;
  } else {
    fprintf (stdout, "Using %i bits stencil buffer\n", stencil);
  }
  glViewport (0, 0, (GLsizei) screen_w, (GLsizei) screen_h);
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho (0.0, (GLfloat) screen_w, (GLfloat) screen_h, 0.0, -1.0, 1.0);
  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();
}

void P3_init_joysticks (void) {
  int i;
  nb_joysticks = SDL_NumJoysticks ();
  if (nb_joysticks > 0) {
    joysticks = (SDL_Joystick**) malloc (nb_joysticks * sizeof (SDL_Joystick*));
    SDL_JoystickEventState (SDL_ENABLE);
    for (i = 0; i < nb_joysticks; i++) joysticks[i] = SDL_JoystickOpen (i);
  }
}

void P3_init_video (char* title, int width, int height, int fullscreen, int resizable) {
  /* initialize SDL */
  if (SDL_Init (SDL_INIT_VIDEO | SDL_INIT_JOYSTICK  | SDL_INIT_NOPARACHUTE) == -1) {
    P3_error ("Could not initialize SDL : %s.", SDL_GetError());
    exit (P3_FALSE);
  }
  P3_set_video (width, height, fullscreen, resizable);
  if (title != NULL) {
    SDL_WM_SetCaption (title, NULL);
  }
  P3_init_GL ();
  /* ok that has nothing to do here */
  P3_init_joysticks ();
}

#endif /* USE_SDL */

void P3_dump_info (void) {
  printf ("\n");
  printf ("Using OpenGL %s\n", glGetString (GL_VERSION));
  printf ("  - renderer : %s\n", glGetString (GL_RENDERER));
  printf ("  - vendor : %s\n", glGetString (GL_VENDOR));
  printf ("  - maximum number of lights : %i\n", maxlights);
  printf ("  - maximum number of clip planes : %i\n", maxclipplanes);
  printf ("  - maximum number of texture units : %i\n", maxtextures);
  printf ("  - maximum texture size : %i pixels\n", maxtexturesize);
//  printf ("  - extensions : %s\n", glGetString (GL_EXTENSIONS));
  printf ("\n");
}

void P3_init_GL (void) {
  int i;
  
  glGetIntegerv (GL_MAX_LIGHTS, &maxlights);
  lights_gl = (P3_light**) malloc (maxlights * sizeof (P3_light*));
  lights_gl_activated = (int*) malloc (maxlights * sizeof (int));
  for (i = 0; i < maxlights; i++) {
    lights_gl[i] = NULL;
    lights_gl_activated[i] = P3_FALSE;
  }

  glGetIntegerv (GL_MAX_CLIP_PLANES, &maxclipplanes);

#if !(GL_VERSION_1_3)
  glGetIntegerv (GL_MAX_TEXTURE_UNITS_ARB, &maxtextures);
#else /* GL_VERSION_1_3 */
  glGetIntegerv (GL_MAX_TEXTURE_UNITS, &maxtextures);
#endif /* GL_VERSION_1_3 */
  glGetIntegerv (GL_MAX_TEXTURE_SIZE, &maxtexturesize);

  glClearDepth (1.0);
  glDepthMask (GL_FALSE);
  glDisable (GL_DEPTH_TEST);

  glDepthFunc (GL_LESS);

  glEnable (GL_TEXTURE_2D);
  glDisable (GL_COLOR_MATERIAL);
  glColorMaterial (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
  glEnable (GL_COLOR_MATERIAL);

  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, black);
  glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
  glDisable (GL_LIGHTING);

  glDisable (GL_NORMALIZE);
    
  glDisable (GL_BLEND);
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
//  glAlphaFunc (GL_GREATER, 0.5);
// TO DO
  glDisable (GL_ALPHA_TEST);
//  glEnable (GL_ALPHA_TEST);
  glAlphaFunc (GL_NOTEQUAL, 0.0);
//  glAlphaFunc (GL_GEQUAL, P3_LIGHT_NULL_ATTENUATION);

  glEnable (GL_CULL_FACE);
  glCullFace (GL_BACK);
  glFrontFace (GL_CCW);
    
  glPolygonMode (GL_FRONT_AND_BACK, GL_FILL); 
  glEnable (GL_POINT_SMOOTH);
  glDisable (GL_LINE_SMOOTH);
  glDisable (GL_POLYGON_SMOOTH);
  glShadeModel (GL_SMOOTH);

//  glEnable (GL_DITHER);
  glDisable (GL_DITHER);

  glPixelStorei (GL_UNPACK_ALIGNMENT, 1);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

  P3_set_quality (quality);

  shadow_display_list = glGenLists (1);

#ifdef USE_GLX

// TO DO see SDL or GLUT source

//	this->gl_data->glXGetProcAddress =
//		(void *(*)(const GLubyte *)) dlsym(handle, "glXGetProcAddressARB");

#endif /* USE_GLX */

#ifdef USE_SDL
#if !(GL_VERSION_1_3)
  glActiveTexture       = SDL_GL_GetProcAddress ("glActiveTextureARB");
  glClientActiveTexture = SDL_GL_GetProcAddress ("glClientActiveTextureARB");
  glMultiTexCoord2f     = SDL_GL_GetProcAddress ("glMultiTexCoord2fARB");
  glMultiTexCoord2fv    = SDL_GL_GetProcAddress ("glMultiTexCoord2fvARB");
#endif /* ! GL_VERSION_1_3 */
#endif /* USE_SDL */

/*
  glBindBuffer    = SDL_GL_GetProcAddress ("glBindBufferARB");
  glGenBuffers    = SDL_GL_GetProcAddress ("glGenBuffersARB");
  glBufferData    = SDL_GL_GetProcAddress ("glBufferDataARB");
  glDeleteBuffers = SDL_GL_GetProcAddress ("glDeleteBuffersARB");
  if (glBufferData == 0 || glDeleteBuffers == 0 || glGenBuffers == 0) {
    glBindBuffer = 0;
  }
*/

  engine_option = engine_option | P3_GL_INITED;
  P3_dump_info ();
}

void P3_set_quality (int q) {
  quality = q;
  switch (quality) {
  case P3_QUALITY_LOW:
    glHint (GL_FOG_HINT, GL_FASTEST);
    glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
    glHint (GL_POINT_SMOOTH_HINT, GL_FASTEST);
    glHint (GL_LINE_SMOOTH_HINT, GL_FASTEST);
    glHint (GL_POLYGON_SMOOTH_HINT, GL_FASTEST);
    break;
  case P3_QUALITY_MEDIUM:
    glHint (GL_FOG_HINT, GL_DONT_CARE);
    glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_DONT_CARE);
    glHint (GL_POINT_SMOOTH_HINT, GL_DONT_CARE);
    glHint (GL_LINE_SMOOTH_HINT, GL_DONT_CARE);
    glHint (GL_POLYGON_SMOOTH_HINT, GL_DONT_CARE);
    break;
  case P3_QUALITY_HIGH:
    glHint (GL_FOG_HINT, GL_NICEST);
    glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
    glHint (GL_POINT_SMOOTH_HINT, GL_NICEST);
    glHint (GL_LINE_SMOOTH_HINT, GL_NICEST);
    glHint (GL_POLYGON_SMOOTH_HINT, GL_NICEST);
    break;
  }
}

void P3_toggle_wireframe_mode (void) {
  if (engine_option & P3_WIREFRAME) { 
    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL); 
    engine_option &= ~P3_WIREFRAME;
  } else {
    glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); 
    engine_option |= P3_WIREFRAME;
  }
}

void P3_base_init (void) {
  srand (1);
//  user_list = P3_list_new (8);
  /* initialize renderer */
  renderer = P3_renderer_new ();
  land_tri_recycler = P3_list_new (20);
  chunks = P3_list_new (2);
//  bsp = P3_bsp_new ();
//  bsp_node_recycler = P3_list_new (12);
#ifdef HAVE_CAL3D
  P3_cal3d_init ();
#endif /* HAVE_CAL3D */
}

void P3_base_quit (void) {
  int i;

  /* lights */
  free (lights_gl);
  free (lights_gl_activated);
//  P3_list_dealloc (user_list);
  /* renderer */
  if (renderer != NULL) { P3_renderer_dealloc (renderer); }
  P3_list_dealloc (land_tri_recycler);
  free (NULL_packs);
  /* fx */
  if (engine_option & P3_FX_INITED) P3_fx_quit ();
  /* chunks */
  for (i = 0; i < chunks->nb; i++) {
    P3_chunk_dealloc ((P3_chunk*) P3_list_get (chunks, i));
  }
  P3_list_dealloc (chunks);

#ifdef HAVE_CAL3D
  P3_cal3d_quit ();
#endif /* HAVE_CAL3D */

  if (shadow_display_list != -1) glDeleteLists (shadow_display_list, 1);


  /* bsp */
/*
  P3_bsp_dealloc (bsp);
  for (i = 0; i < bsp_node_recycler->nb; i++) {
    P3_bsp_node_dealloc (P3_list_get (bsp_node_recycler, i));
  }
  P3_list_dealloc (bsp_node_recycler);
*/

#ifdef USE_GLX
  if (engine_option & P3_FULLSCREEN) { 
    engine_option -= P3_FULLSCREEN; 
    if (old_screen_w != screen_w || old_screen_h != screen_h) {
      P3_set_mode (old_screen_w, old_screen_h);
    }
  }
  if (context != NULL) {
    glXMakeCurrent (display, None, NULL);
    glXDestroyContext (display, context);
    XDestroyWindow (display, window);
    XCloseDisplay (display);
  }
#endif /* USE_GLX */

#ifdef USE_SDL
  for (i = 0; i < nb_joysticks; i++) SDL_JoystickClose (joysticks[i]);
  SDL_Quit ();
#endif /* USE_SDL */

  free (joysticks);
  engine_option &= ~P3_GL_INITED;
}

