/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*******************************************
 * image.c : load standard image file format
 * Copyright (C) 2002 Bertrand 'blam' LAMY
 *******************************************/

#include <SDL/SDL.h>

#ifdef HAVE_LIBPNG
#include <png.h>
#endif /* HAVE_LIBPNG */

#include "p3_base.h"
#include "util.h"
#include "material.h"
#include "image.h"


void P3_image_check (P3_image* img) {
  int desired_w;
  int desired_h;
  int i;
  if (img == NULL) return;
  i = 0;
  while (1) {
    desired_w = (1 << i);
    if (desired_w >= img->width) { break; }
    i++;
  }
  i = 0;
  while (1) {
    desired_h = (1 << i);
    if (desired_h >= img->height) { break; }
    i++;
  }
  if (img->width != desired_w || img->height != desired_h) {
    P3_error ("WARNING Image dimensions are not power of 2 (%i x %i)", img->width, img->height);
  }
}

/* - internal prototypes - */
#ifdef HAVE_LIBPNG
GLubyte* P3_load_PNG (int* width, int* height, int* type, FILE* fp);
#endif /* HAVE_LIBPNG */
GLubyte* P3_load_TGA (int* width, int* height, int* type, FILE* fp);


/* - load images - */

P3_image* P3_image_new (P3_image* img) {
  if (img == NULL) { img = (P3_image*) malloc (sizeof (P3_image)); }
  img->nb_color = 0;
  img->width    = 0;
  img->height   = 0;
  img->pixels   = NULL;
  return img;
}

GLubyte* P3_load_image (int* width, int* height, int* type, char* filename) {
  GLubyte* pixels;
  char* ext;
  char buffer[8];
  FILE* file;
  /* open file */
  file = fopen (filename, "rb");
  if (file == NULL) {
    P3_error ("can't open file : %s", filename);
    return NULL;
  }
  /* try to use extension to find image format */
  pixels = NULL;
  ext = P3_filename_extension (filename);
  if (ext != NULL) {
#ifdef HAVE_LIBPNG
    if (strcmp (ext, "png") == 0) { pixels = P3_load_PNG (width, height, type, file); }
#endif /* HAVE_LIBPNG */
    if (strcmp (ext, "tga") == 0) { pixels = P3_load_TGA (width, height, type, file); }
  }
  if (pixels == NULL) { 
    /* try to use magic number to find image format */
    fread (buffer, 1, 8, file);
#ifdef HAVE_LIBPNG
    if (png_sig_cmp (buffer, 0, 8)) {
      /* that's a png */
      rewind (file);
      pixels = P3_load_PNG (width, height, type, file); 
    }
#endif /* HAVE_LIBPNG */
  }
  /* close the file */
  fclose (file);
  if (pixels == NULL) { P3_error ("in loading file %s or unknown image format", filename); }
  return pixels;
}

P3_image* P3_image_load (P3_image* img, char* filename) {
  if (img == NULL) { img = (P3_image*) malloc (sizeof (P3_image)); }
  img->pixels = P3_load_image (&(img->width), &(img->height), &(img->nb_color), filename);
  return img;
}

/* - load PNG image - */

/* this code is stolen and adapted from the example.c code of the libpng */
#ifdef HAVE_LIBPNG
GLubyte* P3_load_PNG (int* width, int* height, int* type, FILE* fp) {
  png_structp png_ptr;
  png_infop info_ptr;
  png_bytep* row_pointers;
  int i; int j;
  int nb;
  png_color* palette;
  int nbcolor = 0;
  int usepalette = P3_FALSE;
  png_color color;
  GLubyte* pixels;
  /* create png struct */
  png_ptr = png_create_read_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (png_ptr == NULL) {
    fclose (fp);
    P3_error ("can't open png file");
    return NULL;
  }
  info_ptr = png_create_info_struct (png_ptr);
  if (info_ptr == NULL) {
    fclose (fp);
    png_destroy_read_struct (&png_ptr, (png_infopp) NULL, (png_infopp) NULL);
    P3_error ("can't read info in png file");
    return NULL;
  }
  /* Set error handling if you are using the setjmp/longjmp method */
  if (setjmp (png_jmpbuf (png_ptr))) {
    /* Free all of the memory associated with the png_ptr and info_ptr */
    png_destroy_read_struct (&png_ptr, &info_ptr, (png_infopp) NULL);
    fclose (fp);
    /* If we get here, we had a problem reading the file */
    P3_error ("can't read png file");
    return NULL;
  }
  /* I/O initialization */
  png_init_io (png_ptr, fp);
  /* read */
  png_read_png (png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);
  /* create the texture from image data */
  *width  = png_get_image_width  (png_ptr, info_ptr);
  *height = png_get_image_height (png_ptr, info_ptr);
  *type   = png_get_color_type   (png_ptr, info_ptr);
  if (*type == PNG_COLOR_TYPE_RGB_ALPHA) {
//    *type = GL_RGBA;
//    *type = 4;
    nb = 4;
//  } else if (*type == PNG_COLOR_TYPE_GRAY) {
//    int bit_depth;
//    bit_depth = png_get_bit_depth (png_ptr, info_ptr);
// TO DO
  } else {
    if (*type != PNG_COLOR_TYPE_RGB && *type != PNG_COLOR_TYPE_PALETTE) {
// TO DO support more
      P3_error ("unsupported color type in png file");
    }
//    *type = GL_RGB;
//    *type = 3;
    nb = 3;
  }
  *type = nb;
  pixels = (GLubyte*) malloc (*width * *height * nb * sizeof (GLubyte));
  /* array of pointers to the pixel data for each row */
  row_pointers = png_get_rows (png_ptr, info_ptr);
  if (png_get_color_type (png_ptr, info_ptr) == PNG_COLOR_TYPE_PALETTE) {
    png_get_PLTE (png_ptr, info_ptr, &palette, &nbcolor);
    usepalette = P3_TRUE;
    nb = 1;
  }
  /* grab pixels */
  for (i = 0; i < *height; i++) {
    for (j = 0; j < *width * nb; j++) {
      if (usepalette == P3_FALSE) {
        *(pixels + i * *width * nb + j) = (GLubyte) *(*(row_pointers + i) + j);
      } else {
        color = *(palette + *(*(row_pointers + i) + j));
        *(pixels + i * *width * 3 + j * 3)     = color.red;
        *(pixels + i * *width * 3 + j * 3 + 1) = color.green;
        *(pixels + i * *width * 3 + j * 3 + 2) = color.blue;
      }
    }
  }
  /* clean up and free memory allocated */
  png_destroy_read_struct (&png_ptr, &info_ptr, (png_infopp) NULL);
  return pixels;
}
#endif /* HAVE_LIBPNG */


/* - load TGA image - */

GLubyte* P3_finalize_TGA (GLubyte* pixels, int width, int height, int* type) {
  int i;
  GLubyte* pixs;
  GLubyte* p;
  /* test for alpha */
//  *type = GL_RGB;
  *type = 3;
  for (i = 0; i < width * height * 4; i += 4) {
    if (pixels[i + 3] < 255) {
//      *type = GL_RGBA; 
      *type = 4;
      return pixels;
    }
  }
  /* finally that was 24 bits image coded on 32 -> must convert */
  pixs = malloc (width * height * 3 * sizeof (GLubyte));
  p = pixs;
  for (i = 0; i < width * height * 4; i += 4) {
    *(p++) = *(pixels + i);
    *(p++) = *(pixels + i + 1);
    *(p++) = *(pixels + i + 2);
  }
  free (pixels);
  return pixs;
}

/* this function is adapted from the Quake II engine */
GLubyte* P3_load_TGA (int* rwidth, int* rheight, int* rtype, FILE* file) {
  unsigned char id_length;
  unsigned char colormap_type;
  unsigned char image_type;
  unsigned short colormap_index;
  unsigned short colormap_length;
  unsigned char colormap_size;
  unsigned short x_origin;
  unsigned short y_origin;
  unsigned short width;
  unsigned short height;
  unsigned char pixel_size;
  unsigned char attributes;
  int nb = 0;
  int i; int j; int k; 
  unsigned char colors[4];
  GLubyte* rpixels = NULL;
  GLubyte* pixels;
  /* read header */
  fread (&id_length, sizeof(unsigned char), 1, file);
  fread (&colormap_type, sizeof(unsigned char), 1, file);
  fread (&image_type, sizeof(unsigned char), 1, file);
  fread (&colormap_index, sizeof(unsigned short), 1, file);
  fread (&colormap_length, sizeof(unsigned short), 1, file);
  fread (&colormap_size, sizeof(unsigned char), 1, file);
  fread (&x_origin, sizeof(unsigned short), 1, file);
  fread (&y_origin, sizeof(unsigned short), 1, file);
  fread (&width, sizeof(unsigned short), 1, file);
  fread (&height, sizeof(unsigned short), 1, file);
  fread (&pixel_size, sizeof(unsigned char), 1, file);
  fread (&attributes, sizeof(unsigned char), 1, file);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap16 (colormap_index);
  SDL_Swap16 (colormap_length);
  SDL_Swap16 (x_origin);
  SDL_Swap16 (y_origin);
  SDL_Swap16 (width);
  SDL_Swap16 (height);
#endif
// TO DO support more...
  if (image_type != 2 && image_type != 10 && image_type != 3) {
    P3_error ("load targa type %i : only type 2, 3, and 10 targa RGB images are supported", image_type);
    return NULL;
  }
  if (colormap_type != 0 || (pixel_size != 32 && pixel_size != 24 && pixel_size != 8)) {
    P3_error ("load targa : only 8, 24 and 32 bit image with no colormap are supported (%i)", pixel_size);
    return NULL;
  }
  /* initialize material */
  *rwidth  = (int) width;
  *rheight = (int) height;
  if (pixel_size == 24) { 
//    *rtype = GL_RGB; 
    *rtype = 3;
    nb = 3;
  }
  if (pixel_size == 32) {
    nb = 4;
  }
  if (id_length != 0) {
    /* skip TARGA image comment */
    fseek (file, (long) id_length, SEEK_CUR);
  }
  /* get pixels */
  if (image_type == 2) {
    rpixels = (GLubyte*) malloc (*rwidth * *rheight * nb * sizeof (GLubyte));
    /* uncompressed, RGB images */
    for (j = 0; j < *rheight; j++) {
      pixels = rpixels + j * *rwidth * nb;
      for (i = 0; i < *rwidth; i++) {
        fread (colors, sizeof (unsigned char), nb, file);
        *(pixels++) = (GLubyte) colors[2];
        *(pixels++) = (GLubyte) colors[1];
        *(pixels++) = (GLubyte) colors[0];
        if (nb == 4) { *(pixels++) = (GLubyte) colors[3]; }
      }
    }
  }
  if (image_type == 3) {
    /* grayscale image */
    rpixels = (GLubyte*) malloc (*rwidth * *rheight * sizeof (GLubyte));
    for (j = 0; j < *rheight; j++) {
      pixels = rpixels + j * *rwidth;
      for (i = 0; i < *rwidth; i++) {
        fread (colors, sizeof (unsigned char), 1, file);
        *(pixels++) = (GLubyte) colors[0];
      }
    }
//    *rtype = GL_LUMINANCE;
    *rtype = 1;
    return rpixels;
  }
  if (image_type == 10) {
    unsigned char packet_header;
    unsigned char packet_size;
    rpixels = (GLubyte*) malloc (*rwidth * *rheight * nb * sizeof (GLubyte));
    /* runlength encoded RGB images */
    for (j = 0; j < *rheight; j++) {
      pixels = rpixels + j * *rwidth * nb;
      for (i = 0; i < *rwidth; ) {
        fread (&packet_header, sizeof (unsigned char), 1, file);
        packet_size = 1 + (packet_header & 0x7f);
        if (packet_header & 0x80) {
          /* run-length packet */
          fread (colors, sizeof (unsigned char), nb, file);
          for (k = 0;k < packet_size; k++) {
            *(pixels++) = (GLubyte) colors[2];
            *(pixels++) = (GLubyte) colors[1];
            *(pixels++) = (GLubyte) colors[0];
            if (nb == 4) { *(pixels++) = (GLubyte) colors[3]; }
            i++;
            if (i >= *rwidth) {
              i = 0;
              j++;
              if (j >= *rheight) { 
                if (nb == 4) { rpixels = P3_finalize_TGA (rpixels, *rwidth, *rheight, rtype); }
                return rpixels;
              }
              pixels = rpixels + j * *rwidth * nb;
            }
          }
        } else {
          /* non run-length packet */
          for (k= 0 ; k < packet_size; k++) {
            fread (colors, sizeof(unsigned char), nb, file);
            *(pixels++) = (GLubyte) colors[2];
            *(pixels++) = (GLubyte) colors[1];
            *(pixels++) = (GLubyte) colors[0];
            if (nb == 4) { *(pixels++) = (GLubyte) colors[3]; }
            i++;
            if (i >= *rwidth) {
              /* pixel packet run spans across rows */
              i = 0;
              j++;
              if (j >= *rheight) { 
                if (nb == 4) { rpixels = P3_finalize_TGA (rpixels, *rwidth, *rheight, rtype); }
                return rpixels;
              }
              pixels = rpixels + j * *rwidth * nb;
            }
          }
        }
      }
    }
  }
/*
  if(image_type == 11) {
    // grayscale image with runlengths
  }
*/
  if (nb == 4) { rpixels = P3_finalize_TGA (rpixels, *rwidth, *rheight, rtype); }
  return rpixels;
}


/* - save images - */

void P3_save_RGB_TGA (GLubyte* pixels, int width, int height, FILE* file) {
  unsigned char  uc;
  unsigned short us;
  int i; int j;
  GLubyte* ptr;
  /* write header */
  uc = 0; /* id_length */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  uc = 0; /* colormap type */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  uc = 2; /* image type */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  us = 0; /* colormap_index */
  fwrite (&us, sizeof(unsigned short), 1, file);
  us = 0; /* colormap_length */
  fwrite (&us, sizeof(unsigned short), 1, file);
  uc = 0; /* colormap_size */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  us = 0; /* x_origin */
  fwrite (&us, sizeof(unsigned short), 1, file);
  us = 0; /* y_origin */
  fwrite (&us, sizeof(unsigned short), 1, file);
  fwrite (&width, sizeof(unsigned short), 1, file);
  fwrite (&height, sizeof(unsigned short), 1, file);
  uc = 24; /* pixel size */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  uc = 0; /* attributes */
  fwrite (&uc, sizeof(unsigned char), 1, file);
  /* uncompressed, RGB images */
  ptr = pixels;
  for (j = 0; j < height; j++) {
    for (i = 0; i < width; i++) {
      fwrite (ptr + 2, sizeof (unsigned char), 1, file);
      fwrite (ptr + 1, sizeof (unsigned char), 1, file);
      fwrite (ptr,     sizeof (unsigned char), 1, file);
      ptr += 3;
    }
  }
}
