/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * xmesh_build.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "coordsys.h"
#include "world.h"
#include "light.h"
#include "material.h"
#include "mesh.h"

#ifdef COMPILE_FOR_PYTHON
#include "python/face_python.h"
#endif /* COMPILE_FOR_PYTHON */

extern GLfloat white[4];
extern GLfloat black[4];


/*====================+
 | XMESH CONSTRUCTION |
 +====================*/

void P3_xmesh_initialize (P3_xmesh* mesh) {
  mesh->option = 0;
  mesh->nb_materials = 0;
  mesh->materials = NULL;
  mesh->nb_vertices = 0;
  mesh->vertex_options   = NULL;
  mesh->vertex_coords    = NULL;
  mesh->vertex_normals   = NULL;
  mesh->vertex_texcoords = NULL;
  mesh->vertex_diffuses  = NULL;
  mesh->vertex_emissives = NULL;
  mesh->vertex_warfogs   = NULL;
  mesh->nb_coords = 0;
  mesh->coords = NULL;
  mesh->nb_vnormals = 0;
  mesh->vnormals = NULL;
  mesh->nb_colors = 0;
  mesh->colors = NULL;
  mesh->nb_values = 0;
  mesh->values = NULL;
  mesh->faces_size = 0;
  mesh->faces = NULL;
}

void P3_xmesh_free_data (P3_xmesh* mesh) {
  int i;
  for (i = 0; i < mesh->nb_materials; i++) P3_material_decref (mesh->materials[i]);
  free (mesh->materials);
  free (mesh->vertex_options);
  free (mesh->vertex_coords);
  free (mesh->vertex_normals);
  free (mesh->vertex_texcoords);
  free (mesh->vertex_diffuses);
  free (mesh->vertex_emissives);
  free (mesh->vertex_warfogs);
  free (mesh->coords);
  free (mesh->vnormals);
  free (mesh->colors);
  free (mesh->values);
  free (mesh->faces);
}

int P3_xmesh_face_size (P3_xmesh* mesh, P3_xface* face) {
  int s = sizeof (int) + sizeof (GLfloat*) + sizeof (P3_xpack*) + 3 * sizeof (int);
  if (face->option & P3_FACE_QUAD) { 
    if (mesh->option & P3_MESH_NEIGHBORS) {
      s += sizeof (int) + 4 * sizeof (P3_xface*); 
    } else {
      s += sizeof (int); 
    }
  } else if (face->option & P3_FACE_TRIANGLE && mesh->option & P3_MESH_NEIGHBORS) {
    s += 3 * sizeof (P3_xface*); 
  }
  return s;
}

static int P3_xface_vertices_number (P3_xface* face) {
  if (face->option & P3_FACE_TRIANGLE) {
    return 3;
  } else if (face->option & P3_FACE_QUAD) {
    return 4;
  }
  return 0;
}

void P3_xmesh_register_material (P3_xmesh* mesh, P3_material* material) {
  int i;
  for (i = 0; i < mesh->nb_materials; i++) {
    if (mesh->materials[i] == material) return;
  }
  mesh->materials = (P3_material**) realloc (mesh->materials, (mesh->nb_materials + 1) * sizeof (P3_material*));
  mesh->materials[mesh->nb_materials] = material;
  if (material != NULL) P3_material_incref (material);
  (mesh->nb_materials)++;
}

static GLfloat* P3_xmesh_register_coord (P3_xmesh* mesh, GLfloat* coord) {
  GLfloat* old;
  int i;
  for (i = 0; i < mesh->nb_coords; i++) {
    if (P3_float_array_compare (coord, mesh->coords + i * 3, 3) == P3_TRUE) return mesh->coords + i * 3;
  }
  i = mesh->nb_coords;
  (mesh->nb_coords)++;
  old = mesh->coords;
  mesh->coords = (GLfloat*) realloc (mesh->coords, mesh->nb_coords * 3 * sizeof (GLfloat));
  memcpy (mesh->coords + i * 3, coord, 3 * sizeof (GLfloat));
  if (mesh->coords != old) {
    int j;
    for (j = 0; j < mesh->nb_vertices; j++) {
      if (mesh->vertex_coords[j] != NULL) mesh->vertex_coords[j] = mesh->coords + (mesh->vertex_coords[j] - old);
    }
  }
  return mesh->coords + i * 3;
}

static GLfloat* P3_xmesh_register_vnormal (P3_xmesh* mesh) {
  GLfloat* old;
  GLfloat* ptr;
  old = mesh->vnormals;
  mesh->vnormals = realloc (mesh->vnormals, (mesh->nb_vnormals + 1) * 3 * sizeof (GLfloat));
  ptr = mesh->vnormals + 3 * mesh->nb_vnormals;
  ptr[0] = 0.0f;
  ptr[1] = 0.0f;
  ptr[2] = 0.0f;
  (mesh->nb_vnormals)++;
  if (mesh->vnormals != old) {
    int j;
    for (j = 0; j < mesh->nb_vertices; j++) {
      if (mesh->vertex_normals[j] != NULL) mesh->vertex_normals[j] = mesh->vnormals + (mesh->vertex_normals[j] - old);
    }
  }
  return ptr;
}

int P3_xmesh_register_value (P3_xmesh* mesh, GLfloat* value, int nb) {
  GLfloat* old;
  int r;
  for (r = 0; r <= mesh->nb_values - nb; r++) {
    if (P3_float_array_compare (value, mesh->values + r, nb) == P3_TRUE) return r;
  }
  r = mesh->nb_values;
  mesh->nb_values += nb;
  old = mesh->values;
  mesh->values = (GLfloat*) realloc (mesh->values, mesh->nb_values * sizeof (GLfloat));
  memcpy (mesh->values + r, value, nb * sizeof (GLfloat));
  if (mesh->values != old) {
    P3_xface* face;
    void* max;
    int j;
    if (mesh->option & P3_MESH_TEXCOORDS) {
      for (j = 0; j < mesh->nb_vertices; j++) {
        if (mesh->vertex_texcoords[j] != NULL) mesh->vertex_texcoords[j] = mesh->values + (mesh->vertex_texcoords[j] - old);
      }
    }
    face = mesh->faces;
    max = mesh->faces + mesh->faces_size;
    while ((void*) face < max) {
      if (face->normal != NULL) face->normal = mesh->values + (face->normal - old);
      face = ((void*) face) + P3_xmesh_face_size (mesh, face);
    }
  }
  return r;
}

GLfloat* P3_xmesh_register_color (P3_xmesh* mesh, GLfloat color[4]) {
  GLfloat* ptr;
  int i;
  ptr = mesh->colors;
  for (i = 0; i < mesh->nb_colors; i++) {
    if (fabs (color[0] - ptr[0]) < P3_EPSILON &&
        fabs (color[1] - ptr[1]) < P3_EPSILON &&
        fabs (color[2] - ptr[2]) < P3_EPSILON &&
        fabs (color[3] - ptr[3]) < P3_EPSILON) return ptr;
    ptr += 4;
  }
  i = mesh->nb_colors * 4;
  (mesh->nb_colors)++;
  ptr = mesh->colors;
  mesh->colors = (GLfloat*) realloc (mesh->colors, mesh->nb_colors * 4 * sizeof (GLfloat));
  memcpy (mesh->colors + i, color, 4 * sizeof (GLfloat));
  if (mesh->colors != ptr) {
    int j;
    if (mesh->option & P3_MESH_DIFFUSES) {
      for (j = 0; j < mesh->nb_vertices; j++) {
        if (mesh->vertex_diffuses[j] != NULL) mesh->vertex_diffuses[j] = mesh->colors + (mesh->vertex_diffuses[j] - ptr);
      }
    }
    if (mesh->option & P3_MESH_EMISSIVES) {
      for (j = 0; j < mesh->nb_vertices; j++) {
        if (mesh->vertex_emissives[j] != NULL) mesh->vertex_emissives[j] = mesh->colors + (mesh->vertex_emissives[j] - ptr);
      }
    }
    if (mesh->option & P3_MESH_WARFOGS) {
      GLfloat* fmax = ptr + (mesh->nb_colors - 1) * 4;
      for (j = 0; j < mesh->nb_vertices; j++) {
        if (mesh->vertex_warfogs[j] >= ptr && mesh->vertex_warfogs[j] < fmax) mesh->vertex_warfogs[j] = mesh->colors + (mesh->vertex_warfogs[j] - ptr);
      }
    }
  }
  return mesh->colors + i;
}

static int P3_xmesh_register_vertex (P3_xmesh* mesh, GLfloat* coord, GLfloat* normal, GLfloat* texcoord, GLfloat* diffuse, GLfloat* emissive) {
  int i;
  for (i = 0; i < mesh->nb_vertices; i++) {
    if (mesh->vertex_coords[i] == coord &&
        (!(mesh->option & P3_MESH_VERTEX_NORMALS) || mesh->vertex_normals[i]   == normal)   &&
        (!(mesh->option & P3_MESH_TEXCOORDS)      || mesh->vertex_texcoords[i] == texcoord) &&
        (!(mesh->option & P3_MESH_DIFFUSES)       || mesh->vertex_diffuses[i]  == diffuse)  &&
        (!(mesh->option & P3_MESH_EMISSIVES)      || mesh->vertex_emissives[i] == emissive))
      return i;
  }
  i = mesh->nb_vertices;
  (mesh->nb_vertices)++;
  mesh->vertex_coords = (GLfloat**) realloc (mesh->vertex_coords, mesh->nb_vertices * sizeof (GLfloat*));
  mesh->vertex_coords[i] = coord;
  if (mesh->option & P3_MESH_VERTEX_NORMALS) {
    mesh->vertex_normals = (GLfloat**) realloc (mesh->vertex_normals, mesh->nb_vertices * sizeof (GLfloat*));
    mesh->vertex_normals[i] = normal;
  }
  if (mesh->option & P3_MESH_TEXCOORDS) {
    mesh->vertex_texcoords = (GLfloat**) realloc (mesh->vertex_texcoords, mesh->nb_vertices * sizeof (GLfloat*));
    mesh->vertex_texcoords[i] = texcoord;
  }
  if (mesh->option & P3_MESH_DIFFUSES) {
    mesh->vertex_diffuses = (GLfloat**) realloc (mesh->vertex_diffuses, mesh->nb_vertices * sizeof (GLfloat*));
    mesh->vertex_diffuses[i] = diffuse;
  }
  if (mesh->option & P3_MESH_EMISSIVES) {
    mesh->vertex_emissives = (GLfloat**) realloc (mesh->vertex_emissives, mesh->nb_vertices * sizeof (GLfloat*));
    mesh->vertex_emissives[i] = emissive;
  }
  return i;
}

static int P3_xmesh_add_vertex (P3_xmesh* mesh, P3_vertex* vertex, P3_face* face) {
  P3_coordsys* csys;
  GLfloat* coord;
  GLfloat* matrix;
  GLfloat value[4];
  int texcoord = 0;
  int diffuse  = 0;
  int emissive = 0;
  /* coord */
  P3_vertex_get_coord (vertex, value);
  csys = P3_vertex_get_coordsys (vertex);
  matrix = P3_coordsys_get_root_matrix (csys);
  P3_point_by_matrix (value, matrix);
  coord = P3_xmesh_register_coord (mesh, value);
  /* texcoord */
  if (mesh->option & P3_MESH_TEXCOORDS) {
    P3_vertex_get_texcoord (vertex, value);
    texcoord = P3_xmesh_register_value (mesh, value, 2);
  }
  /* diffuse color */
  if (mesh->option & P3_MESH_DIFFUSES) {
    P3_vertex_get_diffuse (vertex, value);
    matrix = P3_xmesh_register_color (mesh, value);
    diffuse = matrix - mesh->colors;
  }
  /* emissive color */
  if (mesh->option & P3_MESH_EMISSIVES) {
    P3_vertex_get_emissive (vertex, value);
    matrix = P3_xmesh_register_color (mesh, value);
    emissive = matrix - mesh->colors;
  }
  /* vertex */
  return P3_xmesh_register_vertex (mesh, coord, NULL, mesh->values + texcoord, mesh->colors + diffuse, mesh->colors + emissive);
}

static void P3_xmesh_add_face (P3_xmesh* mesh, P3_face* face, P3_chunk* faces) {
  P3_material* material;
  int vertices[4];
  GLfloat n[4];
  GLfloat* p;
  int nb_vertices;
  int option = 0;
  int m, m2;
  /* compute data and then register all in 1 pass cause registering data
   * can have side effect on the faces chunk */
  nb_vertices = P3_face_get_vertices_number (face);
  if (nb_vertices == 3) {
    option |= P3_FACE_TRIANGLE;
  } else if (nb_vertices == 4) {
    option |= P3_FACE_QUAD;
  } else {
    P3_error ("XMesh doesn't support face with %i vertices", nb_vertices);
    return;
  }
  if (!(mesh->option & P3_MESH_SHADOW_CAST) && P3_face_is_double_sided (face)) option |= P3_FACE_DOUBLE_SIDED;
  if (!P3_face_is_solid        (face)) option |= P3_FACE_NON_SOLID;
  if ( P3_face_is_alpha        (face)) option |= P3_FACE_ALPHA;
  if ( P3_face_is_smoothlit    (face)) option |= P3_FACE_SMOOTHLIT;
  vertices[0] = P3_xmesh_add_vertex (mesh, P3_face_get_vertex (face, 0), face);
  vertices[1] = P3_xmesh_add_vertex (mesh, P3_face_get_vertex (face, 1), face);
  vertices[2] = P3_xmesh_add_vertex (mesh, P3_face_get_vertex (face, 2), face);
  if (option & P3_FACE_QUAD) vertices[3] = P3_xmesh_add_vertex (mesh, P3_face_get_vertex (face, 3), face);
  P3_face_normal (n, mesh->vertex_coords[vertices[0]], mesh->vertex_coords[vertices[1]], mesh->vertex_coords[vertices[2]]);
  P3_vector_normalize (n);
  if (mesh->option & P3_MESH_PLANE_EQUATION) {
    p = mesh->vertex_coords[vertices[0]];
    n[3] = -(p[0] * n[0] + p[1] * n[1] + p[2] * n[2]);
    m = P3_xmesh_register_value (mesh, n, 4);
  } else {
    m = P3_xmesh_register_value (mesh, n, 3);
  }
  if (mesh->option & P3_MESH_SHADOW_CAST && P3_face_is_double_sided (face)) {
    n[0] = - n[0];
    n[1] = - n[1];
    n[2] = - n[2];
    if (mesh->option & P3_MESH_PLANE_EQUATION) {
      n[3] = - n[3];
      m2 = P3_xmesh_register_value (mesh, n, 4);
    } else {
      m2 = P3_xmesh_register_value (mesh, n, 3);
    }
  }
  material = P3_face_get_material (face);
  if (material != NULL) P3_xmesh_register_material (mesh, material);
  /* option */
  P3_chunk_add_int (faces, option);
  /* pack */
  P3_chunk_add_ptr (faces, P3_xpack_get (option, material));
  /* normal */
  P3_chunk_add_ptr (faces, mesh->values + m);
  /* vertices */
  P3_chunk_add_int (faces, vertices[0]);
  P3_chunk_add_int (faces, vertices[1]);
  P3_chunk_add_int (faces, vertices[2]);
  if (option & P3_FACE_QUAD) P3_chunk_add_int (faces, vertices[3]);
  /* neighbors */
  if (mesh->option & P3_MESH_NEIGHBORS) {
    P3_chunk_add_ptr (faces, NULL);
    P3_chunk_add_ptr (faces, NULL);
    P3_chunk_add_ptr (faces, NULL);
    if (option & P3_FACE_QUAD) P3_chunk_add_ptr (faces, NULL);
  }
  if (mesh->option & P3_MESH_SHADOW_CAST && P3_face_is_double_sided (face)) {
    /* if double sided add also the other side face 
     * shadow_cast doesn't like double sided faces...
     */
    P3_chunk_add_int (faces, option | P3_FACE_NO_VERTEX_NORMAL_INFLUENCE);
    P3_chunk_add_ptr (faces, P3_xpack_get (option, material));
    P3_chunk_add_ptr (faces, mesh->values + m2);
    P3_chunk_add_int (faces, vertices[2]);
    P3_chunk_add_int (faces, vertices[1]);
    P3_chunk_add_int (faces, vertices[0]);
    if (option & P3_FACE_QUAD) P3_chunk_add_int (faces, vertices[3]);
    if (mesh->option & P3_MESH_NEIGHBORS) {
      P3_chunk_add_ptr (faces, NULL);
      P3_chunk_add_ptr (faces, NULL);
      P3_chunk_add_ptr (faces, NULL);
      if (option & P3_FACE_QUAD) P3_chunk_add_ptr (faces, NULL);
    }
  }
  mesh->faces = faces->content;
  mesh->faces_size = faces->nb;
}

static int P3_xmesh_face_is_neighbor_by (P3_xmesh* mesh, P3_xface* f1, P3_xface* f2, int vertice) {
  /* vertice is the index vertice in the f1 vertices array 
   * face f1 and f2 must share this vertice coord
   */
  GLfloat* c1[4];
  GLfloat* c2[4];
  int nbv1 = P3_xface_vertices_number (f1);
  int nbv2 = P3_xface_vertices_number (f2);
  int i;
  int a1, a2, b1, b2;
  c1[0] = mesh->vertex_coords[f1->v1];
  c1[1] = mesh->vertex_coords[f1->v2];
  c1[2] = mesh->vertex_coords[f1->v3];
  if (f1->option & P3_FACE_QUAD) c1[3] = mesh->vertex_coords[f1->v4];
  c2[0] = mesh->vertex_coords[f2->v1];
  c2[1] = mesh->vertex_coords[f2->v2];
  c2[2] = mesh->vertex_coords[f2->v3];
  if (f1->option & P3_FACE_QUAD) c2[3] = mesh->vertex_coords[f2->v4];
  /* find the same coord in f2 */
  for (i = 0; i < nbv2; i++) {
    if (c2[i] == c1[vertice]) {
      a1 = vertice + 1;
      if (a1 >= nbv1) { a1 = 0; }
      b1 = vertice - 1;
      if (b1 < 0) { b1 = nbv1 - 1; }
      a2 = i + 1;
      if (a2 >= nbv2) { a2 = 0; }
      b2 = i - 1;
      if (b2 < 0) { b2 = nbv2 - 1; }
      if (c1[a1] == c2[a2] || c1[a1] == c2[b2] || c1[b1] == c2[a2] || c1[b1] == c2[b2]) {
        return P3_TRUE;
      } else {
        return P3_FALSE;
      }
    }
  }
  return P3_FALSE;
}

static void P3_xmesh_face_compute_vertex_normal (P3_xmesh* mesh, P3_xface* face, int vertex_index) {
  GLfloat* ptr;
  int* vertices;
  int v0, v, v2;
  GLfloat u1[3];
  GLfloat u2[3];
  GLfloat w;
  int nb_vertices = P3_xface_vertices_number (face);
  vertices = &(face->v1);
  v = vertices[vertex_index];
  if (vertex_index == 0)
    v0 = vertices[nb_vertices - 1];
  else
    v0 = vertices[vertex_index - 1];
  if (vertex_index < nb_vertices - 1)
    v2 = vertices[vertex_index + 1];
  else
    v2 = vertices[0];
  P3_vector_from_points (u1, mesh->vertex_coords[v2], mesh->vertex_coords[v]);
  P3_vector_from_points (u2, mesh->vertex_coords[v0], mesh->vertex_coords[v]);
  w = P3_vector_angle (u1, u2);
  if (mesh->option & P3_MESH_VERTEX_NORMALS)
    ptr = mesh->vertex_normals[v];
  else
    ptr = mesh->vnormals + (mesh->vertex_coords[v] - mesh->coords);
  ptr[0] += (w * face->normal[0]);
  ptr[1] += (w * face->normal[1]);
  ptr[2] += (w * face->normal[2]);
}

static void P3_xmesh_compute_single_vertex_normals (P3_xmesh* mesh) {
  GLfloat* cmax = NULL;
  GLfloat* ptr;
  P3_xface* face;
  void* max;
  /* find each smoothlit vertex */
  /* only faces know if a vertex is smoothlit or not */
  face = mesh->faces;
  max = mesh->faces + mesh->faces_size;
  while ((void*) face < max) {
    if (face->option & P3_FACE_SMOOTHLIT) {
      ptr = mesh->vertex_coords[face->v1];
      if (ptr > cmax) cmax = ptr;
      ptr = mesh->vertex_coords[face->v2];
      if (ptr > cmax) cmax = ptr;
      ptr = mesh->vertex_coords[face->v3];
      if (ptr > cmax) cmax = ptr;
      if (face->option & P3_FACE_QUAD) {
        ptr = mesh->vertex_coords[face->v4];
        if (ptr > cmax) cmax = ptr;
      }
    }
    face = ((void*) face) + P3_xmesh_face_size (mesh, face);
  }
  mesh->option &= ~P3_MESH_VERTEX_NORMALS;
  if (cmax != NULL) {
    mesh->nb_vnormals = (cmax - mesh->coords) / 3 + 1;
    mesh->vnormals = (GLfloat*) calloc (mesh->nb_vnormals * 3, sizeof (GLfloat));
  }
}

static void P3_xmesh_compute_multiple_vertex_normals (P3_xmesh* mesh, GLfloat angle) {
  P3_list* faces;
  P3_list* face_ok;
  P3_chunk* indices;
  P3_chunk* index_ok;
  GLfloat* coord;
  GLfloat* normal;
  GLfloat* n;
  GLfloat* texcoord = NULL;
  GLfloat* diffuse  = NULL;
  GLfloat* emissive = NULL;
  P3_xface* face;
  P3_xface* f;
  int index;
  int i, j, k, q;
  int nb_placed;
  /* suppose all coords as a normal */
  mesh->option |= P3_MESH_VERTEX_NORMALS;
  mesh->vertex_normals = (GLfloat**) malloc (mesh->nb_vertices * sizeof (GLfloat*));
  for (i = 0; i < mesh->nb_vertices; i++) mesh->vertex_normals[i] = NULL;
  faces    = P3_list_new (8);
  indices  = P3_chunk_new ();
  index_ok = P3_chunk_new ();
  face_ok  = P3_list_new (4);
  /* for each vertex coord */
  for (i = 0; i < mesh->nb_coords; i++) {
    coord = mesh->coords + i * 3;
    /* find each face that has a vertex with this coord */
    faces->nb = 0;
    indices->nb = 0;
    face = mesh->faces;
    while ((void*) face < mesh->faces + mesh->faces_size) {
      if (face->option & P3_FACE_SMOOTHLIT) {
        if (mesh->vertex_coords[face->v1] == coord) {
          P3_list_add (faces, face);
          P3_chunk_add_int (indices, 0);
        } else if (mesh->vertex_coords[face->v2] == coord) {
          P3_list_add (faces, face);
          P3_chunk_add_int (indices, 1);
        } else if (mesh->vertex_coords[face->v3] == coord) {
          P3_list_add (faces, face);
          P3_chunk_add_int (indices, 2);
        } else if (face->option & P3_FACE_QUAD && mesh->vertex_coords[face->v4] == coord) {
          P3_list_add (faces, face);
          P3_chunk_add_int (indices, 3);
        }
      }
      face = ((void*) face) + P3_xmesh_face_size (mesh, face);
    }
    if (faces->nb == 0) {
      /* no smoothlit face with this coord found */
      continue;
    }
    face_ok->nb = 0;
    index_ok->nb = 0;
    while (1) {
      nb_placed = 0;
      j = 0;
      while (j < face_ok->nb) {
        face = P3_list_get (face_ok, j);
        for (k = 0; k < faces->nb; k++) {
          f = P3_list_get (faces, k);
          if (f == NULL) { continue; }
          index = ((int*) indices->content)[k];
          if (P3_xmesh_face_is_neighbor_by (mesh, f, face, index) == P3_TRUE) {
            if (P3_vector_angle (face->normal, f->normal) <= angle) {
              /* the 2 faces share the same vertex normal */
              normal = mesh->vertex_normals[(&(face->v1))[((int*) index_ok->content)[j]]];
              q = (&(f->v1))[index];
              n = mesh->vertex_normals[q];
              if (n == NULL) {
                mesh->vertex_normals[q] = normal;
              } else if (n != normal) {
                /* we must duplicate the vertex to add a different vertex normal */
                /* but maybe in fact it is the same vertex as neighbor face... => use register vertex */
                if (mesh->option & P3_MESH_TEXCOORDS) 
                  texcoord = mesh->vertex_texcoords[q];
                if (mesh->option & P3_MESH_DIFFUSES) 
                  diffuse = mesh->vertex_diffuses[q];
                if (mesh->option & P3_MESH_EMISSIVES) 
                  emissive = mesh->vertex_emissives[q];
                (&(f->v1))[index] = P3_xmesh_register_vertex (mesh, coord, normal, texcoord, diffuse, emissive);
              }
              P3_list_add (face_ok, f);
              P3_chunk_add_int (index_ok, index);
              faces->content[k] = NULL;
              nb_placed++;
              break;
            }
          }
        }
        if (nb_placed > 0) break;
        j++;
      }
      if (nb_placed == 0) {
        /* create a new vertex normal for the first encountered face */
        for (j = 0; j < faces->nb; j++) {
          face = (P3_xface*) P3_list_get (faces, j);
          if (face != NULL) break;
        }
        if (face == NULL) break;
        index = ((int*) indices->content)[j];
        q = (&(face->v1))[index];
        normal = P3_xmesh_register_vnormal (mesh);
        if (mesh->vertex_normals[q] == NULL) {
          mesh->vertex_normals[q] = normal;
        } else {
          /* duplicate the vertex */
          if (mesh->option & P3_MESH_TEXCOORDS) 
            texcoord = mesh->vertex_texcoords[q];
          if (mesh->option & P3_MESH_DIFFUSES) 
            diffuse = mesh->vertex_diffuses[q];
          if (mesh->option & P3_MESH_EMISSIVES) 
            emissive = mesh->vertex_emissives[q];
          (&(face->v1))[index] = P3_xmesh_register_vertex (mesh, coord, normal, texcoord, diffuse, emissive);
        }
        P3_list_add (face_ok, face);
        P3_chunk_add_int (index_ok, index);
        faces->content[j] = NULL;
      }
    }
  }
  P3_list_dealloc (faces);
  P3_chunk_dealloc (indices);
  P3_chunk_dealloc (index_ok);
  P3_list_dealloc (face_ok);
}

static void P3_xmesh_compute_vertex_normals (P3_xmesh* mesh, GLfloat angle) {
  P3_xface* face;
  int i;
  if (angle >= P3_pi) {
    P3_xmesh_compute_single_vertex_normals (mesh);
  } else {
    P3_xmesh_compute_multiple_vertex_normals (mesh, angle);
  }
  /* really compute vertex normals */
  face = mesh->faces;
  while ((void*) face < mesh->faces + mesh->faces_size) {
    if (face->option & P3_FACE_SMOOTHLIT && !(face->option & P3_FACE_NO_VERTEX_NORMAL_INFLUENCE)) {
      P3_xmesh_face_compute_vertex_normal (mesh, face, 0);
      P3_xmesh_face_compute_vertex_normal (mesh, face, 1);
      P3_xmesh_face_compute_vertex_normal (mesh, face, 2);
      if (face->option & P3_FACE_QUAD) {
        P3_xmesh_face_compute_vertex_normal (mesh, face, 3);
      }
    }
    face = ((void*) face) + P3_xmesh_face_size (mesh, face);
  }
  /* normalize vertex normals */
  for (i = 0; i < mesh->nb_vnormals; i++) P3_vector_normalize (mesh->vnormals + i * 3);
}

static int P3_xmesh_face_set_neighbor (P3_xface* face, P3_xface** face_neighbor, P3_xface* neighbor) {
  if (*face_neighbor == NULL) {
    *face_neighbor = neighbor;
    return 1;
  } else {
    GLfloat a, b;
    a = P3_vector_dot_product (face->normal, neighbor->normal);
    b = P3_vector_dot_product (face->normal, (*face_neighbor)->normal);
    if (a > b) {
      *face_neighbor = neighbor;
      return 1;
    }
  }
  return 0;
}

static void P3_xmesh_face_set_neighborhood (P3_xmesh* mesh, P3_xface* f1, P3_xface* f2) {
  int* v1 = &(f1->v1);
  int* v2 = &(f2->v1);
  int b2, a1, a2;
  P3_xface** n1;
  P3_xface** n2;
  int nbv1 = P3_xface_vertices_number (f1);
  int nbv2 = P3_xface_vertices_number (f2);
  int i; int j;
  if (f1->option & P3_FACE_QUAD)
    n1 = (P3_xface**) (((void*) f1) + sizeof (int) + 4 * sizeof (int) + sizeof (P3_xpack*) + sizeof (GLfloat*));
  else
    n1 = (P3_xface**) (((void*) f1) + sizeof (int) + 3 * sizeof (int) + sizeof (P3_xpack*) + sizeof (GLfloat*));
  if (f2->option & P3_FACE_QUAD)
    n2 = (P3_xface**) (((void*) f2) + sizeof (int) + 4 * sizeof (int) + sizeof (P3_xpack*) + sizeof (GLfloat*));
  else
    n2 = (P3_xface**) (((void*) f2) + sizeof (int) + 3 * sizeof (int) + sizeof (P3_xpack*) + sizeof (GLfloat*));
  for (i = 0; i < nbv1; i++) {
    for (j = 0; j < nbv2; j++) {
      if (mesh->vertex_coords[v1[i]] == mesh->vertex_coords[v2[j]]) {
        if (i < nbv1 - 1) a1 = v1[i + 1]; else a1 = v1[0];
        if (j == 0) b2 = v2[nbv2 - 1]; else b2 = v2[j - 1];
        if (j < nbv2 - 1) a2 = v2[j + 1]; else a2 = v2[0];
        if (mesh->vertex_coords[a1] == mesh->vertex_coords[a2]) {
          if (P3_xmesh_face_set_neighbor (f1, n1 + i, f2) == 0) return;
          P3_xmesh_face_set_neighbor (f2, n2 + j, f1);
          /* face can't normally have 2 times the same neighbor */
          return;
        }
        if (mesh->vertex_coords[a1] == mesh->vertex_coords[b2]) {
          if (P3_xmesh_face_set_neighbor (f1, n1 + i, f2) == 0) return;
          if (j == 0) { 
            P3_xmesh_face_set_neighbor (f2, n2 + nbv2 - 1, f1);
          } else {
            P3_xmesh_face_set_neighbor (f2, n2 + j - 1, f1);
          }
          /* face can't normally have 2 times the same neighbor */
          return;
        }
      }
    }
  }
}

void P3_xmesh_face_neighbors (P3_xmesh* mesh) {
  P3_list* face_list;
  P3_xface* face;
  int i;
  int j;

//printf ("Compute face neighborhood...\n");

  /* create a face list for faster lookup */
  face_list = P3_list_new (32);
  face = (P3_xface*) mesh->faces;
  while ((void*) face < mesh->faces + mesh->faces_size) {
    P3_list_add (face_list, face);
    face = ((void*) face) + P3_xmesh_face_size (mesh, face);
  }
  /* take each face and find its neighbors */
  for (i = 0; i < face_list->nb; i++) {
    face = (P3_xface*) P3_list_get (face_list, i);
    for (j = i + 1; j < face_list->nb; j++) {
      P3_xmesh_face_set_neighborhood (mesh, face, (P3_xface*) P3_list_get (face_list, j));
    }
  }
  P3_list_dealloc (face_list);
}

static P3_xface* P3_xmesh_find_face (P3_xmesh* mesh, P3_face* face) {
  P3_material* material = P3_face_get_material (face);
  P3_xface* f;
  P3_xface* max;
  int vertices[4];
  int nb, i;
  nb = P3_face_get_vertices_number (face);
  if (nb != 3 && nb != 4) return NULL;
  for (i = 0; i < nb; i++) 
    vertices[i] = P3_xmesh_add_vertex (mesh, P3_face_get_vertex (face, i), face);
  f = (P3_xface*) mesh->faces;
  max = (P3_xface*) (mesh->faces + mesh->faces_size);
  while (f < max) {
    if (f->pack->material == material &&
        ((nb == 3 && f->option & P3_FACE_TRIANGLE) || (nb == 4 && f->option & P3_FACE_QUAD)) &&
        f->v1 == vertices[0] && f->v2 == vertices[1] && f->v3 == vertices[2] && 
        (f->option & P3_FACE_TRIANGLE || (f->option & P3_FACE_QUAD && f->v4 == vertices[3]))) {
      return f;
    }
    f = (P3_xface*) (((void*) f) + P3_xmesh_face_size (mesh, f));
  }
  return NULL;
}

void P3_xmesh_face_force_neighbor (P3_xmesh* mesh, P3_face* face, P3_face* neighbor) {
  P3_xface* f = P3_xmesh_find_face (mesh, face);
  P3_xface* n = P3_xmesh_find_face (mesh, neighbor);
  P3_xmesh_face_set_neighborhood (mesh, f, n);
}

void P3_xmesh_to_faces (P3_xmesh* mesh, P3_world* world) {
  P3_xface* face;
  P3_vertex* pv;
  P3_face* pf;
  GLfloat* ptr;
  int nbv, v;
  int i;
  face = (P3_xface*) mesh->faces;
  while ((void*) face < mesh->faces + mesh->faces_size) {
    nbv = P3_xface_vertices_number (face);
    pf = P3_face_new (world, face->pack->material, 0);
    if (face->option & P3_FACE_SMOOTHLIT)    P3_face_set_smoothlit    (pf, 1);
    if (face->option & P3_FACE_DOUBLE_SIDED) P3_face_set_double_sided (pf, 1);
    if (face->option & P3_FACE_NON_SOLID)    P3_face_set_solid        (pf, 0);
    for (i = 0; i < nbv; i++) {
      v = (&(face->v1))[i];
      ptr = mesh->vertex_coords[v];
      pv = P3_vertex_new (world, ptr[0], ptr[1], ptr[2]);
      if (mesh->option & P3_MESH_TEXCOORDS) P3_vertex_set_texcoord (pv, mesh->vertex_texcoords[v]);
      if (mesh->option & P3_MESH_DIFFUSES)  P3_vertex_set_diffuse  (pv, mesh->vertex_diffuses[v]);
      if (mesh->option & P3_MESH_EMISSIVES) P3_vertex_set_emissive (pv, mesh->vertex_emissives[v]);
      P3_face_add_vertex (pf, pv);
    }
    face = ((void*) face) + P3_xmesh_face_size (mesh, face);
  }
}

P3_xmesh* P3_xmesh_from_faces (P3_xmesh* mesh, P3_list* given_faces, GLfloat angle, int mesh_option) {
  P3_chunk* faces;
  P3_face* face;
  P3_vertex* vertex;
  GLfloat values[4];
  int i, k, n;

printf ("Building Mesh %p from %i faces...\n", mesh, given_faces->nb);

  mesh->option |= mesh_option;
  faces = P3_chunk_new ();
  /* part of mesh option */
  for (i = 0; i < given_faces->nb; i++) {
    face = (P3_face*) P3_list_get (given_faces, i);
    n = P3_face_get_vertices_number (face);
    for (k = 0; k < n; k++) {
      vertex = P3_face_get_vertex (face, k);
      P3_vertex_get_texcoord (vertex, values);
      if (P3_float_array_compare (values, black, 2) == P3_FALSE) mesh->option |= P3_MESH_TEXCOORDS;
      P3_vertex_get_diffuse (vertex, values);
      if (P3_float_array_compare (values, white, 4) == P3_FALSE) mesh->option |= P3_MESH_DIFFUSES;
      P3_vertex_get_emissive (vertex, values);
      if (P3_float_array_compare (values, black, 4) == P3_FALSE) mesh->option |= P3_MESH_EMISSIVES;
    }
  }
  if (mesh->option & P3_MESH_TEXCOORDS) {
    for (i = 0; i < given_faces->nb; i++) {
      face = (P3_face*) P3_list_get (given_faces, i);
      if (P3_face_get_material (face) != NULL) {
        i = -1;
        break;
      }
    }
    if (i != -1) mesh->option &= ~P3_MESH_TEXCOORDS;
  }
  k = 0;
  /* smoothlit vertex first (memory optimization) */
  for (i = 0; i < given_faces->nb; i++) {
    face = (P3_face*) P3_list_get (given_faces, i);
    if (P3_face_is_smoothlit (face) == P3_TRUE) {
      P3_xmesh_add_face (mesh, face, faces);
      given_faces->content[i] = NULL;
      k = 1;
    }
  }
  for (i = 0; i < given_faces->nb; i++) {
    face = (P3_face*) P3_list_get (given_faces, i);
    if (face != NULL) P3_xmesh_add_face (mesh, face, faces);
  }
  mesh->faces = realloc (mesh->faces, mesh->faces_size);
  free (faces);

  /* compute vertex normals */
  if (k == 1) P3_xmesh_compute_vertex_normals (mesh, angle);

  /* find face neighbors */
  if (mesh->option & P3_MESH_NEIGHBORS) P3_xmesh_face_neighbors (mesh);

// TO DO ?
//  P3_cmesh_compute_dimension (mesh);

printf ("  [DONE]\n");

  return mesh;
}

P3_xmesh* P3_xmesh_from_world (P3_xmesh* mesh, P3_world* world, GLfloat angle, int mesh_option) {
  P3_list* faces;
  faces = P3_list_new (256);
  /* take all faces of the world */
  P3_world_extract (world, P3_ID_FACE, faces);
  /* create the mesh */
  mesh = P3_xmesh_from_faces (mesh, faces, angle, mesh_option);
  P3_list_dealloc (faces);
  return mesh;
}





