/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/********************************************************
 * p3_base.h :
 *
 * define base structure.
 * READ THIS FILE FIRST IF YOU WANT TO KNOW HOW P3 WORKS!
 *
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 ********************************************************/

// TO DO double sided is dead

// TO DO delete cmesh
// TO DO stock frustum ?
// TO DO stock the plane constant (d) with the face normal ?
// TO DO keyword static
// TO DO cal3d volume/shape saving
// TO DO another possibility for rendering process of land: put the frustum in the renderer data chunk
//       and do both visibiilty computation and drawing at the rendering step

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <float.h>
#include <GL/gl.h>

/*===================*
 * - ENGINE DESIGN - *
 *===================*
 * This is a 2 steps rendering process:
 * 1) batch: traverse the Worlds hierarchy, compute matrices and visibility, 
 *     find renderable objects and add them into the Renderer.
 * 2) render: let's the Renderer do its job!
 */

/*========================*
 * general engine options *
 *========================*/

#define P3_GL_INITED             (1<<0)
#define P3_USE_MIPMAP            (1<<1)
#define P3_FULLSCREEN            (1<<2)
#define P3_WIREFRAME             (1<<3)
#define P3_FX_INITED             (1<<4)
#define P3_SHADOWS               (1<<5)
// TO DO
//#define P3_USE_EMISSIVE  (1<<3)
//#define P3_USE_ALPHA_BSP (1<<4)

#define P3_QUALITY_LOW    0
#define P3_QUALITY_MEDIUM 1
#define P3_QUALITY_HIGH   2

/*===========*
 * CONSTANTS *
 *===========*/

#define P3_FALSE  0
#define P3_TRUE   1

#define P3_pi          3.1415927
#define P3_2_pi        6.2831853  /* pi * 2 */
#define P3_pi_2        1.5707963  /* pi / 2 */
#define P3_pi_180      0.0174533  /* pi / 180 */
#define P3_pi_180_inv  57.216848  /* 180 / pi */
#define P3_sqrt_2      1.4142136  /* sqrt(2) */
#define P3_sqrt_2_2    0.7071068  /* sqrt(2) / 2 */
#define P3_EPSILON     0.001      /* values under this value are considered as 0.0 */
#define P3_UPSILON     0.00001
#define P3_INFINITY    FLT_MAX


/*=======*
 * UTILS *
 *=======*/

typedef struct {
  void** content;
  int    nb;
  int    max;
} P3_list;

typedef struct {
  void* content;
  int nb;
  int max;
} P3_chunk;

typedef struct {
  /* next is before data to allow data to have variable size (not always a void*...) */
  int next;
  void* data;
} P3_chain;

typedef struct {
  /* P3_double_chain is a P3_chain */
  int next;
  void* data;
  int link;
} P3_double_chain;


/*=================*
 * 3D CLASS OBJECT *
 *=================*/

/* object system 
 * some macros to easy wrap for any language (in fact this is not wrapper but real extension to any language...)
 *  - currently Python supported
 *  - can be done for C to have a pure C 3D engine also. is it really usefull ? contact me if you are interested
 *  - why not port it to Perl ? waiting for Parrot...
 *
 * P3_SYSTEM_OBJECT_HEADER must be put for all structs that are not stucked to C
 */

#ifdef COMPILE_FOR_PYTHON
#include <Python.h>
#define P3_SYSTEM_OBJECT_HEADER \
  PyObject_HEAD
#define P3_children PyObject*
#endif /* COMPILE_FOR_PYTHON */


/* class id (P3_class->id): used to know what is the type of a P3_any_object */
#define P3_ID_VOLUME           1
#define P3_ID_WORLD            2
#define P3_ID_CAMERA           3
#define P3_ID_LIGHT            4
#define P3_ID_PORTAL           5
#define P3_ID_FACE             6
#define P3_ID_WATERCUBE        7
#define P3_ID_MESH             8
#define P3_ID_MESH_LOD         9
#define P3_ID_MESH_FXINSTANCE 10
#define P3_ID_MORPH           11
#define P3_ID_MORPH_MESH      12
#define P3_ID_LAND            13
#define P3_ID_SPRITE          14
#define P3_ID_CYLINDER        15
#define P3_ID_BONUS           16
#define P3_ID_PARTICLES       17
#define P3_ID_INSTANCE        18 /* an external coordsys object (not coded in C) */
#define P3_ID_SHAPE           19 /* an external shape    object (not coded in C) */
#define P3_ID_CAL3D_SHAPE     20
#define P3_ID_CAL3D_SUBSHAPE  21
#define P3_ID_CAL3D_VOLUME    22

/* some structs defined later */
typedef struct _p3_coordsys      P3_coordsys;
typedef struct _p3_instance      P3_instance;
typedef struct _p3_raypickable   P3_raypickable;
typedef struct _p3_world         P3_world;
typedef struct _p3_camera        P3_camera;
typedef struct _p3_node          P3_node;
typedef struct _p3_portal        P3_portal;
typedef struct _p3_light         P3_light;
typedef struct _p3_xpack         P3_xpack;


/* option for raypick functions */
#define P3_RAYPICK_CULL_FACE  (1<<0)
#define P3_RAYPICK_HALF_LINE  (1<<1)
/* returned by math3d raypick functions */
#define P3_RAYPICK_DIRECT   1
#define P3_RAYPICK_INDIRECT 2

typedef struct {
  P3_list*  raypicked;
  P3_chunk* raypick_data;
  GLfloat   root_data[7];
  int option; 
  GLfloat result;
  GLfloat normal[3];
  void*   ret_csys;
} P3_raypick_data;


/* Init func is called when a shape is added in a P3_world or a P3_volume and return stuff stocked on
 * the parent (visibility stuff)
 */
/* Batch and Render functions corresponds to the 3 steps of the rendering process. */
typedef void  (*batch_func)     (void*, P3_instance*);
typedef void  (*render_func)    (void*, P3_instance*);
typedef int   (*shadow_func)    (void*, P3_instance*, P3_light*);
typedef void  (*raypick_func)   (void*, P3_raypick_data*, P3_raypickable*);
typedef int   (*raypick_b_func) (void*, P3_raypick_data*, P3_raypickable*);
//typedef void  (*raypick_func)   (void*, GLfloat*, int, GLfloat*, GLfloat*, void**, P3_raypickable*);
//typedef int   (*raypick_b_func) (void*, GLfloat*, int, P3_raypickable*);


/* Functions that create a New object must always be like that:
 *   object_type* new_func (object_type* obj, ...args...);
 * If obj is NULL, returns a new object_type else initializes obj.
 */

typedef struct {
  int             id;
  batch_func      batch;
  render_func     render;
  shadow_func     shadow;
  raypick_func    raypick;
  raypick_b_func  raypick_b;
} P3_class;

/* There are 2 kind of objects : Shapes and Coordsys objects. The main difference is that Coordsys objects
 * can be added into a world as child whereas Shapes can be added into worlds or volumes as shape.
 *
 * Adding a shape into a world or setting a Coordsys object as a shape may segfault !!!
 *
 *  |--------------|-------|-----------|
 *  | object       | shape | coordsys  |
 *  |--------------|-------|-----------|
 *  | P3_world     |       |     X     |
 *  | P3_volume    |       |     X     |
 *  | P3_model     |   X   |           |
 *  | P3_land      |   X   |           |
 *  | P3_morph     |       |     X     |
 *  | P3_portal    |       |     X     |
 *  | P3_light     |       |     X     |
 *  | P3_sprite    |       |     X     |
 *  | P3_cylinder  |       |     X     |
 *  | P3_particles |       |     X     |
 *  | P3_bonus     |   X   |           |
 *  | P3_watercube |       |     X     |
 *  |--------------|-------|-----------|
 */

/* Headers definitions: only macros for convenience ! All headers are not mixable, and they
 * must be added in a certain order.
 */

#define P3_OBJECT_HEADER \
  P3_class* class; \
  int option

/* WARNING the 5 first option flag are reserved for a struct with P3_OBJECT_HEADER
 * that is added in the 3D hierarchy (option (1<<0) to (1<<4))
 */
/* Shared option flags */
#define P3_OBJECT_HIDDEN       (1<<0) /* if set, object won't be rendered at all (set by user) */
#define P3_COORDSYS_LEFTHANDED (1<<3)
#define P3_OBJECT_NON_SOLID    (1<<4)

#define P3_CHILD_HEADER \
  P3_coordsys* parent

#define P3_COORDSYS_HEADER \
  GLfloat m[19]; \
  GLfloat mroot[19]; \
  GLfloat mrooti[19]; \
  int validity

/* Coordsys header:
 *  m      is the coordsys matrix (value 16, 17, 18, and 19 are the x,y,z scale factor of the matrix)
 *  mroot  is the root parent matrix
 *  mrooti is the root parent inverted matrix
 */

#define P3_RENDER_MATRIX_HEADER \
  GLfloat render_matrix[19]; \
  int frustum_data

#define P3_RAYPICK_HEADER \
  int raypick_data

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
} P3_any_object;

struct _p3_instance {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
};

struct _p3_raypickable {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  P3_RAYPICK_HEADER;
};

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
} P3_child;


/*======*
 * FONT *
 *======*/

#define P3_TEXT_ALIGN_LEFT   0
#define P3_TEXT_ALIGN_CENTER 1

#define P3_FONT_RASTER  1
#define P3_FONT_TEXTURE 2

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int font_type;
} P3_font;


/*=============*
 * RASTER FONT * text with only 1 color, each character has the same width
 *=============*/

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int font_type;
  int offset;
  int call_list;
  int first_char;
  int last_char;
  int nb_chars;
  GLubyte* chars;
  int stored_width;  /* width of a character in multiple of 8 */
  int width;  /* size of a character in pixels */
  int height;
} P3_font_raster;


/*==============*
 * TEXTURE FONT * multiple colors, variable width for each character, scalable
 *==============*/

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int font_type;
  int first_char;
  int last_char;
  int nb_chars;
  int space;  /* size in pixels of ' ' */
  int width;
  int height;
  int type;   /* GL_RGB or GL_RGBA or GL_LUMINANCE */
  int char_height;
  GLfloat char_verti;    /* height of all characters in value for glTexCoord2f */
  GLuint id;  /* for glBindTexture */
  GLubyte* pixels;
  int*      char_size;   /* width of each character in pixels */
  int*      char_line;   /* vertical start of each character (in number of line) */
  GLfloat*  char_start;  /* horizontal start of each character (range from 0.0 to 1.0 = value for glTexCoord2f) */
} P3_font_texture;


/*==========*
 * MATERIAL *
 *==========*/

#define P3_MATERIAL_SEPARATE_SPECULAR (1<<1)
#define P3_MATERIAL_ADDITIVE_BLENDING (1<<2)
#define P3_MATERIAL_ALPHA             (1<<3)
#define P3_MATERIAL_MASK              (1<<4)
#define P3_MATERIAL_CLAMP             (1<<5)
#define P3_MATERIAL_DIFFUSE           (1<<6)
#define P3_MATERIAL_SPECULAR          (1<<7)
#define P3_MATERIAL_EMISSIVE          (1<<8)
#define P3_MATERIAL_IMAGE             (1<<9)

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int nb_color;
  int width;
  int height;
  GLubyte* pixels;
} P3_image;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_image* image;
  int option;
  GLuint id;  /* for glBindTexture */
  GLfloat shininess;
  GLfloat* diffuse;
  GLfloat* specular;
  GLfloat* emissive;
  /* xpacks are inside the material structure to be sure that when the material
   * is dealloced, the xpacks are dealloced too. it also fastens xpack look up.
   */

// TO DO chain rather than list ?

  int nb_packs;
  P3_xpack** packs;
} P3_material;


/*==========*
 * COORDSYS *
 *==========*/

/* option (1<<0) is P3_OBJECT_HIDDEN  */

#define P3_COORDSYS_INVALID                0
#define P3_COORDSYS_ROOT_VALID         (1<<0)
#define P3_COORDSYS_INVERSE_ROOT_VALID (1<<1)

struct _p3_coordsys {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
};

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_coordsys* parent;
  GLfloat coord[3];
} P3_point;

typedef P3_point P3_vector;

/*========*
 * VOLUME *
 *========*/

/* option (1<<0) is P3_OBJECT_HIDDEN  */
/* option (1<<3) is P3_COORDSYS_LEFTHANDED */

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  P3_RAYPICK_HEADER;
  P3_any_object* shape;
  char* visibility;
} P3_volume;


/*============*
 * ATMOSPHERE *
 *============*/

#define P3_ATMOSPHERE_SKYPLANE      (1<<2)
#define P3_ATMOSPHERE_FOG           (1<<3)
#define P3_ATMOSPHERE_DONT_CLEAR    (1<<4)
#define P3_ATMOSPHERE_SKYBOX_5      (1<<5)
#define P3_ATMOSPHERE_SKYBOX_6      (1<<6)
#define P3_ATMOSPHERE_SKYBOX_ALPHA  (1<<7)
#define P3_ATMOSPHERE_HAS_CHILD     (1<<8)

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int     option;
  GLfloat bg_color[4];
  int     fog_type;  /* GL_LINEAR, GL_EXP or GL_EXP2 */
  GLfloat fog_start;
  GLfloat fog_end;
  GLfloat fog_density;
  GLfloat fog_color[4];
  GLfloat ambient[4];
  /* only used if P3_ATMOSPHERE_SKYPLANE is defined */
  GLfloat sky_color[4];
  P3_material* cloud;
  /* only used if P3_ATMOSPHERE_SKYBOX is defined */
  P3_material** skybox;
} P3_atmosphere;


/*=========*
 * FRUSTUM *
 *=========*/

typedef struct {
  GLfloat position[3];  /* camera position (x,y,z) */
  GLfloat points[24];   /* points : (x,y,z) * 8 */
  GLfloat planes[24];   /* planes equation : (a,b,c,d) * 6 */
} P3_frustum;


/*=======*
 * WORLD *
 *=======*/

/* option (1<<0) is P3_OBJECT_HIDDEN  */
/* option (1<<3) is P3_COORDSYS_LEFTHANDED */
/* option (1<<4) is P3_OBJECT_NON_SOLID */
#define P3_WORLD_BATCHED       (1<<6)
#define P3_WORLD_PORTAL_LINKED (1<<7)

struct _p3_world {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  P3_RAYPICK_HEADER;
  P3_any_object* shape; /* world is a volume */
  char* visibility;
  P3_children children; /* standard rendering pathway */
  P3_atmosphere* atmosphere;
//  P3_frustum* frustum;
};


/*========*
 * CAMERA *
 *========*/

/* option (1<<3) is P3_COORDSYS_LEFTHANDED */
#define P3_CAMERA_PARTIAL (1<<5)
#define P3_CAMERA_ORTHO   (1<<6)

struct _p3_camera {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;  /* the modelview matrix */
  P3_frustum* frustum;
  P3_any_object* to_render;  /* object that the camera render */
  GLfloat fov;
  GLfloat front;
  GLfloat back;
  int size[4];  /* screen-viewport (x, y, width, height) */
};


/*==========*
 * RENDERER *
 *==========*/

#define P3_RENDERER_STATE_OPAQUE      0
#define P3_RENDERER_STATE_SECONDPASS  1
#define P3_RENDERER_STATE_ALPHA       2
#define P3_RENDERER_STATE_SPECIAL     3

typedef struct {
  P3_list* lights;
  P3_atmosphere* atmosphere;
  P3_portal* portal;
} P3_context;

typedef struct {
  P3_any_object* obj;
  P3_instance*   ins;
  P3_context*    ctx;
  int data;
} P3_renderer_obj;

typedef struct {
  /* renderer state */
  int state;
  P3_any_object* root_object;
  P3_camera*   c_camera;    /* current camera */


// TO DO useless now I think
  P3_frustum*  c_frustum;   /* current frustum */
  P3_coordsys* c_frustum_coordsys;


  P3_frustum*  r_frustum;   /* root frustum */
  P3_chunk* frustums;
  P3_instance* c_instance;
  /* contexts */
  P3_context*  c_context;
  P3_context** contexts;
  int nb_contexts;
  int max_contexts;
  /* list of collected objects to render */
  P3_chunk* opaque;
  P3_chunk* secondpass;
  P3_chunk* alpha;
  P3_chunk* specials;  /* objects that are rendered after the shadows (= not shadowed) */
  P3_list* top_lights; /* contain top level activated lights */
  P3_list* all_lights;
  P3_atmosphere* r_atmosphere; /* root atmosphere (the one to clear the screen with) */
  P3_list* worlds_made;  /* list of world whose context has been made (used by portals to determine if
                          * a world must be batched or not */
  P3_list* portals;  /* a list of encountered portals to clear_part the atmosphere before any other rendering 
                      * and to draw fog at the end */
  P3_list* watercubes;  /* watercube must be rendered at last cause there is a hack with the depth buffer */
  P3_chunk* data;
  P3_chunk* faces;
  /* mesh renderer */
  int pack_start;
  int face_start;
  int last_opaque;
  int last_secondpass;
  int last_alpha;
  GLfloat** colors;
} P3_renderer;


/*=======*
 * LIGHT *
 *=======*/

/* light option flags */
/* option (1<<0) is P3_OBJECT_HIDDEN */
/* option (1<<3) is P3_COORDSYS_LEFTHANDED */
//#define P3_LIGHT_STATIC       (1<<5)
//#define P3_LIGHT_PURE_STATIC  (1<<6)
#define P3_LIGHT_TOP_LEVEL      (1<<7)
#define P3_LIGHT_DIRECTIONAL    (1<<8)
#define P3_LIGHT_NO_SHADOW      (1<<9)
#define P3_LIGHT_SHADOW_COLOR  (1<<10)

/* attenuation below this value is considered as NULL and light is turned off
 * for objects far enought to have an attenuation value below this value
 */
#define P3_LIGHT_NULL_ATTENUATION 0.05

struct _p3_light {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  int id;
  GLfloat w; /* 4th value of the light position */
  GLfloat radius;
  GLfloat angle;      /* spot angle */
  GLfloat exponent;   /* spot exponent */
// TO DO pointer and not array ? only for specular ?
  GLfloat colors[12]; /* ambient + diffuse + specular color */
  GLfloat linear;     /* linear attenuation */
  GLfloat constant;   /* constant attenuation */
  GLfloat quadratic;  /* quadratic attenuation */
  GLfloat data[3];    /* used by cell-shading and shadow */
  GLfloat* shadow_color;
};


/*========*
 * PORTAL *
 *========*/

/* option (1<<0) is P3_OBJECT_HIDDEN */
/* option (1<<3) is P3_COORDSYS_LEFTHANDED */
#define P3_PORTAL_USE_4_CLIP_PLANES    (1<<5)
#define P3_PORTAL_USE_5_CLIP_PLANES    (1<<6)
#define P3_PORTAL_INFINITE             (1<<7)
#define P3_PORTAL_BOUND_ATMOSPHERE     (1<<8)
//#define P3_PORTAL_TELEPORTER           (1<<9)

/* PORTAL OPTIONS :
// * - teleporter : the elements that are behind the portal are not rendered (protect Z-buffer)
 * - bound atmosphere : if world and world beyond doesn't have the same atmosphere and you want
 *    to see beyond atmosphere through the portal
 * - infinite : the portal is an infinite plane (for now, doesn't works with bound atmosphere)
 * - use 4 clip planes : the part of world beyond that is out of the portal rectangle is not rendered
 * - use 5 clip planes : id and the part of world beyond that is in front of the portal is not rendered
 */

struct _p3_portal {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
//  GLfloat* transform;  
  /* triple 4x4 matrix used to go from parent to beyond world if the 2 worlds are not 
   * well positioned one for the other = P3_PORTAL_NON_COHERENT
   * (transform) is applied to the drawer current matrix (the GL modelview when rendering)
   * (transform + 16) is applied to the camera when the camera go through the portal
   * (transform + 32) is applied to the camera frustum */
  P3_world* beyond;
  void* beyond_name;
  double* equation;  /* for clip planes */
  int nbpoints;
  GLfloat* coords;
  /* when portal is teleporter or bounds atmosphere we need to draw a quad representing the
   * portal. this is the points coordinates and the number of points (special rendering to
   * allow us to draw below the camera front plane) */
  P3_context* context;
};


/*====*
 * FX *
 *====*/

typedef struct {
  GLfloat** vertex_warfogs;
  GLfloat** vertex_colors;
  char*     vertex_options;
  P3_chunk* chunk;
} P3_fx_transition_data;

typedef struct _p3_fx             P3_fx;
typedef struct _p3_fx_color_fader P3_fx_color_fader;
typedef GLfloat* (*register_color_func) (void* obj, GLfloat color[4]);

struct _p3_fx {
  GLfloat* color;
  GLfloat alpha;
  float duration;
  void* obj;
  GLfloat** vertex_warfogs;
  GLfloat** vertex_colors;
  char*     vertex_options;
  P3_list* list;
  P3_fx_transition_data* data;
  void (*apply) (P3_fx* fx, int vertex_index);
  register_color_func register_color;
};

struct _p3_fx_color_fader {
  GLfloat* from;
  GLfloat* to;
  GLfloat color[4];
  float percent;
  float time_factor;
  void (*callback) (P3_fx_color_fader* fader, P3_fx_transition_data* data);
  P3_fx_transition_data* callback_data;
};


/*======*
 * MESH *
 *======*/

#define P3_VERTEX_ALPHA          (1<<1)
#define P3_VERTEX_MADE           (1<<3)  /* temporaly used in FX */
#define P3_VERTEX_INVISIBLE      (1<<4)
#define P3_VERTEX_FX_TRANSITION  (1<<5)

#define P3_FACE_TRIANGLE      (1<<0)
#define P3_FACE_QUAD          (1<<1)
#define P3_FACE_NON_SOLID     (1<<2)
#define P3_FACE_HIDDEN        (1<<3)
#define P3_FACE_ALPHA         (1<<4)
#define P3_FACE_DOUBLE_SIDED  (1<<5)
#define P3_FACE_SMOOTHLIT     (1<<6)
#define P3_FACE_FRONT         (1<<7)
#define P3_FACE_BACK          (1<<8)
#define P3_FACE_NO_VERTEX_NORMAL_INFLUENCE  (1<<9)

#define P3_PACK_SECONDPASS            (1<<2)
#define P3_PACK_SPECIAL               (1<<3)


typedef struct {
  int option;
  P3_xpack* pack;
  GLfloat* normal;
  int v1;
  int v2;
  int v3;
  int v4;  /* optional */
  /* Optional fields follow:
   *   P3_xface* neighbor1;
   *   P3_xface* neighbor2;
   *   P3_xface* neighbor3;
   *   P3_xface* neighbor4;
   */
} P3_xface;

#define P3_PACK_OPTIONS \
  (P3_FACE_TRIANGLE | P3_FACE_QUAD | P3_FACE_ALPHA | P3_FACE_DOUBLE_SIDED)

#define P3_DISPLAY_LIST_OPTIONS \
  (P3_FACE_TRIANGLE | P3_FACE_QUAD | P3_FACE_ALPHA | P3_FACE_DOUBLE_SIDED)


struct _p3_xpack {
  int option;
  P3_material* material;
  int data;
  P3_xpack* secondpass;
  P3_xpack* alpha;
};

typedef struct _p3_xnode P3_xnode;

struct _p3_xnode {
  int nb_faces;
  void** faces;
  GLfloat sphere[4];
  int nb_child;
  P3_xnode** child;
};

#define P3_XMESH_HEADER \
  int nb_materials; \
  P3_material** materials; \
  int nb_vertices; \
  char*     vertex_options; \
  GLfloat** vertex_coords; \
  GLfloat** vertex_normals; \
  GLfloat** vertex_texcoords; \
  GLfloat** vertex_diffuses; \
  GLfloat** vertex_emissives; \
  GLfloat** vertex_warfogs; \
  int nb_coords; \
  GLfloat* coords; \
  int nb_vnormals; \
  GLfloat* vnormals; \
  int nb_colors; \
  GLfloat* colors; \
  int nb_values; \
  GLfloat* values; \
  int faces_size; \
  void* faces

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_XMESH_HEADER;
} P3_xmesh;

/* Mesh render option */
/* BE CAREFULL: if you change these values, you must also
 * change some value in soya.soya3d
 */
#define P3_MESH_DIFFUSES         (1<<5)
#define P3_MESH_EMISSIVES        (1<<6)
#define P3_MESH_WARFOGS          (1<<7)
#define P3_MESH_TEXCOORDS        (1<<8)
#define P3_MESH_VERTEX_NORMALS   (1<<9)
#define P3_MESH_VERTEX_OPTIONS  (1<<10)
#define P3_MESH_CELL_SHADING    (1<<11)
#define P3_MESH_NEVER_LIT       (1<<12)
#define P3_MESH_NORMALIZE       (1<<13)
#define P3_MESH_PLANE_EQUATION  (1<<14)
#define P3_MESH_NEIGHBORS       (1<<15)
#define P3_MESH_INITED          (1<<16)
#define P3_MESH_TREE            (1<<17)
#define P3_MESH_DISPLAY_LISTS   (1<<18)
#define P3_MESH_FACE_LIST       (1<<19)
#define P3_MESH_HAS_SPHERE      (1<<20)
#define P3_MESH_SHADOW_CAST     (1<<21)

/* MESH OPTIONAL STRUCTURE (XTRA) 

XTRA 1 : tree         (P3_xnode)
         display list (P3_mesh_display_list)
         cell shading (P3_mesh_cell_shading)

XTRA 2 : face list    (P3_mesh_face_list)
         sphere       (GLfloat[4])

*/

typedef struct {
  int option;
  P3_material* material;
  int dlist;
} P3_display_list;

typedef struct {
  int nb_list_opaque;
//  int nb_list_secondpass;
  int nb_list_alpha;
  P3_display_list* dlists;
} P3_mesh_display_list;

typedef struct {
  GLfloat sphere[4];
  int nb_faces;
  P3_xface** faces;
} P3_mesh_face_list;

typedef struct {
  P3_material* shader;
  GLfloat line_color[4];
  GLfloat line_width_factor;
} P3_mesh_cell_shading;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_XMESH_HEADER;
  void* xtra1;
  void* xtra2;
} P3_mesh;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  int nb_meshes;
  P3_mesh** meshes;
  GLfloat LOD_factor;
} P3_mesh_LOD;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_mesh* mesh;
  char*     vertex_options;
  GLfloat** vertex_warfogs;
} P3_mesh_fxinstance;


/*========*
 * SPRITE *
 *========*/

#define P3_SPRITE_ALPHA            (1<<7)
#define P3_SPRITE_COLORED          (1<<9)
#define P3_SPRITE_NEVER_LIT       (1<<11)
#define P3_SPRITE_RECEIVE_SHADOW  (1<<12)

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  GLfloat position[3];
  P3_material* material;
  GLfloat color[4];
  GLfloat w;
  GLfloat h;
} P3_sprite;

typedef struct {
  /* P3_cylinder is a P3_sprite */
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  GLfloat position[3];
  P3_material* material;
  GLfloat color[4];
  GLfloat w;
  GLfloat h;
  GLfloat direction[3];
} P3_cylinder;

#define P3_BONUS_BATCHED  (1<<14)

typedef struct {
  /* a rotating 2D square + a sprite halo */
  /* P3_bonus is a shape */
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_material* material;
  P3_material* halo;
  GLfloat color[4];  /* color is only used for the halo */
  GLfloat angle;
} P3_bonus;


/*===========*
 * PARTICLES *
 *===========*/

/* option (1<<0)  is P3_OBJECT_HIDDEN  */
/* option (1<<7)  is P3_R_ALPHA */
/* option (1<<9)  is P3_R_COLORED */
/* option (1<<11) is P3_R_NEVER_LIT */
#define P3_PARTICLES_COLOREDS    (1<<14)
#define P3_PARTICLES_MULTI_SIZE  (1<<15)
#define P3_PARTICLES_CYLINDER    (1<<16)

typedef struct _p3_particles P3_particles;

struct _p3_particles {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
// TO DO render matrix is useless ?
  P3_RENDER_MATRIX_HEADER;
  P3_material* material;
  P3_coordsys* coordsys;
  int nb;
  int max_nb;
  int particle_size;   /* range from 11 to 20 GLfloat */
  GLfloat* particles;  /* life, max_life, position, speed, acceleration, [color], [size], [direction] 
                        * life, max_life, x/y/z, u/v/w, a/b/c, [r/g/b/a], [w/h], [m/n/o] */
  float delta_time;
  /* fading color */
  int nb_colors;
  GLfloat* fading_colors;
  /* size gain */
  int nb_sizes;
  GLfloat* sizes;
  /* generator */
  int (*generator) (P3_particles*, int);
};


/*==================*
 * LAND (HEIGHTMAP) *
 *==================*/

typedef struct _p3_land_patch P3_land_patch;
typedef struct _p3_land_tri   P3_land_tri;

#define P3_LAND_VERTEX_HIDDEN          (1<<0)
#define P3_LAND_VERTEX_ALPHA           P3_VERTEX_ALPHA
//#define P3_LAND_VERTEX_ALPHA           (1<<1)
#define P3_LAND_VERTEX_NON_SOLID       (1<<2)
#define P3_LAND_VERTEX_FORCE_PRESENCE  (1<<3)
#define P3_LAND_VERTEX_INVISIBLE       P3_VERTEX_INVISIBLE
#define P3_LAND_VERTEX_FX_TRANSITION   P3_VERTEX_FX_TRANSITION
//#define P3_LAND_VERTEX_INVISIBLE       (1<<4)
//#define P3_LAND_VERTEX_FX_TRANSITION   (1<<5)

typedef struct {
  GLfloat texcoord[2];
  GLfloat normal[3];
  GLfloat coord[3];
  P3_xpack* pack;
} P3_land_vertex;

#define P3_LAND_INITED            (1<<2)
#define P3_LAND_NON_SQUARRED      (1<<2) /* only used for saving, for compatibility with old version of soya */
#define P3_LAND_REAL_LOD_RAYPICK  (1<<3)
/* option (1<<6) was static lit */
#define P3_LAND_VERTEX_OPTIONS    (1<<7)
#define P3_LAND_COLORED           (1<<8)
#define P3_LAND_WARFOG            (1<<9) /* warfog implies vertex options */

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  /* LAND OPTIONS
   * (1<<2) P3_LAND_INITED
   * (1<<3) P3_LAND_RAYPICK_REAL_LOD
   * (1<<4) P3_OBJECT_NON_SOLID
   * (1<<6) P3_LAND_COLORED
   * (1<<7) P3_LAND_HAS_VERTEX_OPTIONS
   * (1<<8) P3_LAND_WARFOG
   */
  int nb_materials;       /* the number of materials used by the heightmap */
  P3_material** materials;
  P3_land_vertex* vertices;
  char* vertex_options;
  /* (1<<0) P3_LAND_FACE_HIDDEN
   * (1<<2) P3_LAND_FACE_NON_SOLID
   * (1<<3) P3_LAND_VERTEX_FORCE_PRESENCE
   */
  /* optional vertex fields */
  GLfloat** vertex_colors;
  GLfloat** vertex_warfogs;
  GLfloat* normals;       /* full LOD triangles normals */
  int nb_colors;
  GLfloat* colors;        /* vertices colors */
  int nb_vertex_width;    /* size_width and size_depth must be (2^n) + 1 or I don't know what happen (crash?) */
  int nb_vertex_depth;
  int patch_size;
  int max_level;
  GLfloat texture_factor; /* a factor that multiply the texture coordinates */
  GLfloat scale_factor;   /* a factor to decide when the triangle must split (higher value means more accuracy) */
  GLfloat split_factor;
  int nb_patchs;
  int nb_patch_width;
  int nb_patch_depth;
  P3_land_patch* patchs;
} P3_land;

struct _p3_land_patch {
  GLfloat sphere[4];
  char level;      /* precision level for rendering */
  P3_land_tri* tri_top;
  P3_land_tri* tri_left;
  P3_land_tri* tri_right;
  P3_land_tri* tri_bottom;
};

struct _p3_land_tri {
  char level;
  GLfloat normal[3];
  GLfloat sphere[4];
  /* 3 vertices that determine the triangle. turn right to left (CCW) */
  P3_land_vertex* v1;     /* apex vertex */
  P3_land_vertex* v2;     /* right vertex for the apex */
  P3_land_vertex* v3;     /* left vertex for the apex */
  P3_land_tri* parent;
  P3_land_tri* left_child;
  P3_land_tri* right_child;
  P3_land_tri* left_neighbor;
  P3_land_tri* right_neighbor;
  P3_land_tri* base_neighbor;
  P3_land_patch* patch;
};


/*===========*
 * WATERCUBE *
 *===========*/

#define P3_WATERCUBE_POINT_ENABLE          (1<<0)
#define P3_WATERCUBE_POINT_ALWAYS_ENABLED  (1<<1)
#define P3_WATERCUBE_POINT_ENABLED         (P3_WATERCUBE_POINT_ENABLE | P3_WATERCUBE_POINT_ALWAYS_ENABLED)

typedef struct {
  GLfloat* points;
  int nb_points;
  int* seq_sizes;
  int nb_seqs;
} P3_watercube_underwater;

typedef struct {
  GLfloat texcoord[2];
  GLfloat normal[3];
  GLfloat coord[3];
  int option;
  /* WATERCUBE POINT OPTIONS
   * (1<<0) P3_WATERCUBE_POINT_ENABLE
   * (1<<1) P3_WATERCUBE_POINT_ALWAYS_ENABLED
   */
} P3_watercube_point;

typedef struct {
  GLfloat position[2];
  GLfloat height; /* maximum height of the wave */
  GLfloat radius; /* distance from wave center where is the maximum height */
  GLfloat decay;  /* amount of height decrease when the distance from maximum height radius increase */
  GLfloat wavelength;
  GLfloat speed;        /* increase radius */
  GLfloat height_decay; /* decrease height */
  GLfloat max_radius;
} P3_watercube_wave;

#define P3_WATERCUBE_INITED  (1<<5)

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  /* WATERCUBE OPTIONS
   * (1<<2) P3_WATERCUBE_INITED
   */
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  GLfloat size[3];
  P3_material* material;
  GLfloat texture_factor;
  GLfloat color[4];
  GLfloat speed;
  int nb_points_width;
  int nb_points_depth;
  P3_watercube_point* points;
  int patch_size;
  int nb_patches_width;
  int nb_patches_depth;
  int* patch_levels;
  int max_camera_level;
  float lod_linear;
  float wave_lod_linear;
  int nb_phases;
  GLfloat* phases;
  GLfloat wave_height;
  /* data for rendering */
  P3_context* context;
  int pbox[4];
  GLfloat fbox[6];
  GLfloat camera_position[2];
  GLfloat camera_z_axis[2];
  GLfloat ubox[6];
  GLfloat front_points[12];
//  float delta_time;
  int nb_waves;
  P3_watercube_wave* waves;
} P3_watercube;


/*===========*
 * ANIMATION *
 *===========*/

typedef struct {
  float time;
  GLfloat quaternion[4];
  GLfloat m[19];
} P3_state_coordsys;

/*
typedef struct {
//  P3_SYSTEM_OBJECT_HEADER
  GLfloat position[3];
} P3_state_vertex;
*/

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  float cyclic_lap;
  float time_min;
  float time_max;
  int nb_states;
  P3_state_coordsys* states;
} P3_anim_coordsys;


/*=======*
 * MORPH *
 *=======*/

#if 0 // NEW TO DO NOW

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
} P3_morph_bone;

typedef struct {
  GLfloat coord[3];
  GLfloat normal[3];
  int nb_bones;
  P3_morph_bone* bones;
  GLfloat* weights;
} P3_morph_vertex;

typedef struct {
  int option;
  P3_xpack* pack;
  GLfloat normal[3];
  int v1;
  int v2;
  int v3;
  P3_xface* neighbor1;
  P3_xface* neighbor2;
  P3_xface* neighbor3;
} P3_morph_face;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  int nb_materials;
  P3_material** materials;
  int nb_vertices;
  char*     vertex_options;
  GLfloat** vertex_data;
  GLfloat** vertex_texcoords;
  GLfloat** vertex_diffuses;
  GLfloat** vertex_emissives;
  GLfloat** vertex_warfogs;
  int nb_vertex_data;
  P3_morph_vertex* vdata;
  int nb_colors;
  GLfloat* colors;
  int nb_values;
  GLfloat* values;
  int faces_size;
  void* faces;
} P3_morph_data;

#endif











// !!! DEPRECATED !!!

/* P3_morph* f_visibility */
#define P3_MORPH_VISIBLE   0
#define P3_MORPH_INVISIBLE 1 /* set by animation */
#define P3_MORPH_HIDDEN    2 /* set by user (superior priority) */

/* option for both morph vertex and face */
#define P3_MORPH_NOT_COMPUTED                    (1<<0)
#define P3_MORPH_HAS_NORMAL                      (1<<1)
#define P3_MORPH_MORPHING_NORMAL                 (1<<2)
#define P3_MORPH_SMOOTHLIT                       (1<<3)
#define P3_MORPH_HAS_VERTEX_WITH_MORPHING_NORMAL (1<<4)

typedef struct {
  int option;
  int coord;
  int texcoord;
  int color;
  int world;
} P3_morph_vertex;

typedef struct {
  int  option;
  int  normal;
  int  world;
  int  nb_v;
  int* v;
// TO DO default visibility ? in option ?
} P3_morph_face;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER  /* only for convenience (not tracked by GC) */
  P3_OBJECT_HEADER;
  int material;
  int  nb_f;
  int* f;
} P3_morph_mesh;

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  int nb_v;
  P3_morph_vertex* v;
  int   nb_v_coords;
  GLfloat* v_coords;
  int   nb_v_texcoords;
  GLfloat* v_texcoords;
  int   nb_v_normals;
  GLfloat* v_normals;
  int   nb_v_colors;
  GLfloat* v_colors;
//  GLfloat* v_c_coords;   /* computed vertices coords */
//  GLfloat* v_c_normals;  /* computed vertices normals */
  int nb_f;
  P3_morph_face* f;
  GLfloat* f_normals;    /* normals of faces (1 for each face, not shared) */
//  GLfloat* f_c_normals;  /* computed faces normals */
  int nb_m;
  P3_morph_mesh* meshes;
  int nb_materials;   /* mesh material */
  P3_material** materials;
  int  nb_w;          /* number of worlds */
  P3_children w;
  int option; /* only alpha option above */
//  int  major_option; // highest option of the morph meshes (= the one used for array)
//  GLenum array_format; // GL_T2P3_C4P3_N3P3_V3F or GL_C4P3_N3P3_V3F or GL_T2P3_N3P3_V3F or GL_N3P3_V3F (or GL_V3F ?)
// TO DO ?
//  int nb_anims;
//  P3_anim* anims;
  GLfloat bsphere[4];
} P3_morph_data;

/* option (1<<0) is P3_OBJECT_HIDDEN  */
/* option (1<<3) is P3_COORDSYS_LEFTHANDED */
/* option (1<<4) is P3_OBJECT_NON_SOLID */
/* option (1<<5) is P3_OBJECT_RAYPICKED */
#define P3_MORPH_COMPUTED         (1<<6)
#define P3_MORPH_RAYPICK_COMPUTED (1<<7)

typedef struct {
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
  P3_RAYPICK_HEADER;
  /* P3_morph is the volume of a P3_morph_data and is quite a specific volume... 
   * but P3_morph looks also like a world...
   */
  P3_morph_data* data;
  P3_children    worlds;
//  P3_frustum*    frustum;
//  P3_coordsys**  worlds;
  char*          f_visibility;
//  GLfloat*       f_normals; // only normals of morphing faces
//  GLfloat*       vertices;  // transformed array vertices
//  float time;
  P3_material** materials;
  GLfloat* v_c_coords;   /* computed vertices coords */
  GLfloat* v_c_normals;  /* computed vertices normals */
  GLfloat* f_c_normals;  /* computed faces normals */
} P3_morph;


/*===============*
 * CAL3D OBJECTS *
 *===============*/

#ifdef HAVE_CAL3D

#include "cal3d_wrapper.h" //#include <cal3d/cal3d_wrapper.h> // Buggous file ?

#define P3_CAL3D_ALPHA         (1<<5)
#define P3_CAL3D_CELL_SHADING  (1<<6)
#define P3_CAL3D_SHADOW_CAST   (1<<7)
#define P3_CAL3D_NEIGHBORS     (1<<8)
#define P3_CAL3D_INITED        (1<<9)

typedef struct _p3_cal3d_shape    P3_cal3d_shape;
typedef struct _p3_cal3d_volume   P3_cal3d_volume;

typedef struct {
  int mesh;
  int submesh;
  int first_vertex;
  int nb_vertices;
  int first_face;
  int nb_faces;
  P3_material* material;
} P3_cal3d_submesh;

struct _p3_cal3d_shape {
  /* data part of the cal3d object */
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  struct CalCoreModel* core_model;
  int nb_materials;
  P3_material** materials;
  int nb_submeshes;
  P3_cal3d_submesh* submeshes;
  int nb_faces;
  int nb_vertices;
  int* faces;
  int* face_neighbors;
  void* xtra;
};

struct _p3_cal3d_volume {
  /* instance part of the cal3d object */
  P3_SYSTEM_OBJECT_HEADER
  P3_OBJECT_HEADER;
  P3_CHILD_HEADER;
  P3_COORDSYS_HEADER;
  P3_RENDER_MATRIX_HEADER;
//  P3_RAYPICK_HEADER;
  
  P3_cal3d_shape* shape;
  struct CalModel* model;
  
//  char* vertex_options;

  char* submeshes;
//  int nb_submeshes;
//  P3_cal3d_submesh** submeshes;
//  int nb_faces;
  GLfloat* face_normals;
//  int nb_vertices;
  GLfloat* vertex_coords;
  int nb_attached_states;
  int* attached_states;
  float delta_time;
  P3_children attached_coordsys;
// why is that the volume that has the sphere and not the shape ??
  GLfloat sphere[4];
};

#endif /* HAVE_CAL3D */


/*===================*
 * GENERIC FUNCTIONS *
 *===================*/

void P3_error (char* fmt, ...);

#ifdef COMPILE_FOR_PYTHON
#include "python/p3_python.h"
#endif /* COMPILE_FOR_PYTHON */

#if !(GL_VERSION_1_3)
extern void (*glMultiTexCoord2f)  (GLenum, GLfloat, GLfloat);
extern void (*glMultiTexCoord2fv) (GLenum, GLfloat*);
extern void (*glActiveTexture)    (GLenum);
#define GL_TEXTURE0 GL_TEXTURE0_ARB
#define GL_TEXTURE1 GL_TEXTURE1_ARB
#define GL_TEXTURE2 GL_TEXTURE2_ARB
#endif /* ! GL_VERSION_1_3 */


