/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*======+
 | LAND |
 +======*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Land_Init (P3_land* a, PyObject* args, PyObject* kwds) {
  if (args == Py_None || PySequence_Size (args) == 0) {
    P3_land_new ((P3_land*) a, 0, 0);
  } else if (PySequence_Size (args) == 1) {
    P3_land_new ((P3_land*) a, PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)));
  } else {
    P3_land_new ((P3_land*) a, PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)));
  }
  return 0;
}

static void PyP3Land_Dealloc (P3_land* a) {
  int i;
  PyObject_GC_UnTrack ((PyObject*) a);
  for (i = 0; i < a->nb_materials; i++) {
    Py_XDECREF (a->materials[i]);
  }
  P3_land_dealloc (a);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3Land_Traverse (P3_land* a, visitproc visit, void* arg) {
  int err;
  int i;
  for (i = 0; i < a->nb_materials; i++) {
    if (a->materials[i] != NULL) {
      err = visit ((PyObject*) a->materials[i], arg);
      if (err) { return err; }
    }
  }
  return 0;
}

static int PyP3Land_Clear (P3_land* a) {
  int i;
  for (i = 0; i < a->nb_materials; i++) {
    Py_XDECREF (a->materials[i]);
    a->materials[i] = NULL;
  }
  return 0;
}

static PyObject* PyP3Land_GetState (P3_land* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  PyObject* list;
  P3_land_get_data (a, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  list = PyP3XMesh_GetMaterials ((P3_xmesh*) a);
  PyTuple_SET_ITEM (tuple, 1, list);
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Land_SetState (P3_land* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* o;
  PyP3XMesh_SetMaterials ((P3_xmesh*) a, PySequence_Fast_GET_ITEM (args, 1));
  o = PySequence_Fast_GET_ITEM (args, 0);
  chunk->content = PyString_AS_STRING (o);
  P3_land_set_data (a, chunk);
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

PyObject* PyP3Land_FromImage (P3_land* a, PyObject* arg) {
  if ((PyObject*) arg == Py_None) { arg = NULL; }
  P3_land_from_image (a, (P3_image*) arg);
  Py_INCREF (Py_None);
  return Py_None;
}

PyObject* PyP3Land_GetHeight (P3_land* a, PyObject* args) {
  return PyFloat_FromDouble ((double) P3_land_get_height (a, (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1))));
}

PyObject* PyP3Land_GetHeightAt (P3_land* a, PyObject* args) {
  return PyFloat_FromDouble ((double) P3_land_get_height_at (a, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)), NULL));
}

static PyObject* PyP3Land_GetTrueHeightAndNormal (P3_land* self, PyObject* args) {
  GLfloat* n;
  PyObject* tuple;
  P3_vector* v;
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble (P3_land_get_height_at (self, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)), &n)));
  v = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  v->parent = NULL;
  memcpy (v->coord, n, 3 * sizeof (GLfloat));
  PyTuple_SET_ITEM (tuple, 1, (PyObject*) v);
  return tuple;
}

PyObject* PyP3Land_SetHeight (P3_land* a, PyObject* args) {
  P3_land_set_height (a, (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

PyObject* PyP3Land_SetMaterialLayer (P3_land* a, PyObject* args) {
  P3_material* arg1 = (P3_material*) PySequence_Fast_GET_ITEM (args, 0);
  if ((PyObject*) arg1 == Py_None) { arg1 = NULL; }
  P3_land_set_material_layer (a, (P3_material*) arg1, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

PyObject* PyP3Land_SetMaterialLayerAngle (P3_land* a, PyObject* args) {
  P3_material* arg1 = (P3_material*) PySequence_Fast_GET_ITEM (args, 0);
  if ((PyObject*) arg1 == Py_None) { arg1 = NULL; }
  P3_land_set_material_layer_angle (a, (P3_material*) arg1, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 3)), (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 4)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_MultiplyHeight (P3_land* self, PyObject* arg) {
  int i; int j;
  GLfloat factor;
  factor = (GLfloat) PyFloat_AS_DOUBLE (arg);
  for (j = 0; j < self->nb_vertex_depth; j++) {
    for (i = 0; i < self->nb_vertex_width; i++) {
      P3_land_get_vertex (self, i, j)->coord[1] *= factor;
    }
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_AddHeight (P3_land* self, PyObject* arg) {
  int i; int j;
  GLfloat factor;
  factor = (GLfloat) PyFloat_AS_DOUBLE (arg);
  for (j = 0; j < self->nb_vertex_depth; j++) {
    for (i = 0; i < self->nb_vertex_width; i++) {
      P3_land_get_vertex (self, i, j)->coord[1] += factor;
    }
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_GetMaterial (P3_land* self, PyObject* args) {
  int x; int y;
  P3_material* m;
  x = PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0));
  y = PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1));
  m = P3_land_get_vertex (self, x, y)->pack->material;
  Py_INCREF ((PyObject*) m);
  return (PyObject*) m;
}

static PyObject* PyP3Land_SetMaterial (P3_land* a, PyObject* args) {
  int x; int y; 
  P3_xpack* pack;
  PyObject* mat;
  mat = PySequence_Fast_GET_ITEM (args, 2);
  if (mat == Py_None) { mat = NULL; }
  P3_land_add_material (a, (P3_material*) mat);
  pack = P3_xpack_get (P3_FACE_TRIANGLE, (P3_material*) mat);
  x = PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0));
  y = PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1));
  if (x >= 0 || x < a->nb_vertex_width || y >= 0 || y < a->nb_vertex_depth) {
    P3_land_get_vertex (a, x, y)->pack = pack;
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_RemoveColors (P3_land* a) {
  free (a->colors);
  free (a->vertex_colors);
  free (a->vertex_warfogs);
  a->nb_colors = 0;
  a->colors = NULL;
  a->vertex_colors = NULL;
  a->vertex_warfogs = NULL;
  a->option &= ~(P3_LAND_COLORED | P3_LAND_WARFOG);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_ComputeShadowColor (P3_land* a, PyObject* args) {
  GLfloat color[4];
  PY_TUPLE_FLOAT_TO_ARRAY_4 (color, PySequence_Fast_GET_ITEM (args, 1));
  P3_land_compute_shadow_color (a, (P3_light*) PySequence_Fast_GET_ITEM (args, 0), color);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_GetVertexOption (P3_land* a, PyObject* args) {
  P3_land_check_vertex_options (a);
  return PyInt_FromLong ((long) a->vertex_options[(int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)) + (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)) * a->nb_vertex_width]);
}

static PyObject* PyP3Land_SetVertexOption (P3_land* a, PyObject* args) {
  P3_land_check_vertex_options (a);
  a->vertex_options[(int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)) + (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)) * a->nb_vertex_width] = (char) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 2));
  Py_INCREF (Py_None);
  return Py_None;
}

/*
static PyObject* PyP3Land_AddVertexOption (P3_land* a, PyObject* args) {
  int n = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)) + (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)) * a->nb_vertex_width;
  P3_land_check_vertex_options (a);
  a->vertex_options[n] |= PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 2));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_RemVertexOption (P3_land* a, PyObject* args) {
  int opt = PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 2));
  int n = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)) + (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)) * a->nb_vertex_width;
  P3_land_check_vertex_options (a);
  a->vertex_options[n] &= ~opt;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_AddAllVertexOption (P3_land* a, PyObject* arg) {
  P3_land_check_vertex_options (a);
  P3_land_add_vertex_option_all (a, (int) PyInt_AS_LONG (arg));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_RemAllVertexOption (P3_land* a, PyObject* arg) {
  P3_land_check_vertex_options (a);
  P3_land_rem_vertex_option_all (a, (int) PyInt_AS_LONG (arg));
  Py_INCREF (Py_None);
  return Py_None;
}
*/

static PyObject* PyP3Land_SetVertexColor (P3_land* a, PyObject* args) {
  GLfloat* color_ptr;
  GLfloat color[4];
  int n = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)) + (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)) * a->nb_vertex_width;
  PY_TUPLE_FLOAT_TO_ARRAY_4 (color, PySequence_Fast_GET_ITEM (args, 2));
  color_ptr = P3_land_check_color (a, color);
  a->vertex_colors[n] = color_ptr;
  if (1.0 - color[3] > P3_EPSILON) 
    a->vertex_options[n] |= P3_LAND_VERTEX_ALPHA;
  else if (a->option & P3_LAND_VERTEX_OPTIONS)
    a->vertex_options[n] &= ~P3_LAND_VERTEX_ALPHA;
  if (color[3] < P3_EPSILON) 
    a->vertex_options[n] |= P3_LAND_VERTEX_HIDDEN;
  else if (a->option & P3_LAND_VERTEX_OPTIONS)
    a->vertex_options[n] &= ~P3_LAND_VERTEX_HIDDEN;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_InitFx (P3_land* a) {
  P3_land_check_warfog (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_VertexFx (P3_land* a, PyObject* args) {
  /* args are: 
   *   fx selection (sphere or cylinderY or None)
   *   fx params (alpha or color or None)
   *   [fx duration] (optional, if not given no transition)
   */
  PyObject* select = PySequence_Fast_GET_ITEM (args, 0);
  P3_fx fx;
  GLfloat sphere[4];
  fx.obj = a;
  fx.register_color = (register_color_func) P3_land_register_color;
  fx.vertex_warfogs = a->vertex_warfogs;
  fx.vertex_colors  = a->vertex_colors;
  fx.vertex_options = a->vertex_options;
  PyP3Fx_GetFx (&fx, args);
  if (select == Py_None) {
    P3_land_fx_all (a, &fx);
  } else if (PySequence_Size (select) == 3) {
    PY_TUPLE_FLOAT_TO_ARRAY_3 (sphere, select);
    P3_land_fx_in_cylinderY (a, &fx, sphere);
  } else {
    PY_TUPLE_FLOAT_TO_ARRAY_4 (sphere, select);
    P3_land_fx_in_sphere(a, &fx, sphere);
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Land_ReInit (P3_land* a) {
  if (a->option & P3_LAND_INITED) { a->option -= P3_LAND_INITED; }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3Land_Methods[] = {
  { "_getstate",                    (PyCFunction) PyP3Land_GetState,                 METH_NOARGS },
  { "_setstate",                    (PyCFunction) PyP3Land_SetState,                 METH_O },
  { "from_image",                   (PyCFunction) PyP3Land_FromImage,                METH_O },
  { "get_height",                   (PyCFunction) PyP3Land_GetHeight,                METH_VARARGS },
  { "set_height",                   (PyCFunction) PyP3Land_SetHeight,                METH_VARARGS },
  { "get_true_height",              (PyCFunction) PyP3Land_GetHeightAt,              METH_VARARGS },
  { "get_true_height_normal",       (PyCFunction) PyP3Land_GetTrueHeightAndNormal,   METH_VARARGS },
  { "set_material_layer",           (PyCFunction) PyP3Land_SetMaterialLayer,         METH_VARARGS },
  { "set_material_layer_angle",     (PyCFunction) PyP3Land_SetMaterialLayerAngle,    METH_VARARGS },
  { "multiply_height",              (PyCFunction) PyP3Land_MultiplyHeight,           METH_O },
  { "add_height",                   (PyCFunction) PyP3Land_AddHeight,                METH_O },
  { "get_material",                 (PyCFunction) PyP3Land_GetMaterial,              METH_VARARGS },
  { "set_material",                 (PyCFunction) PyP3Land_SetMaterial,              METH_VARARGS },
  { "set_vertex_color",             (PyCFunction) PyP3Land_SetVertexColor,           METH_VARARGS },
  { "remove_color",                 (PyCFunction) PyP3Land_RemoveColors,             METH_NOARGS },
  { "compute_shadow_color",         (PyCFunction) PyP3Land_ComputeShadowColor,       METH_VARARGS },
  { "set_vertex_option",            (PyCFunction) PyP3Land_SetVertexOption,          METH_VARARGS },
  { "get_vertex_option",            (PyCFunction) PyP3Land_GetVertexOption,          METH_VARARGS },
//  { "add_vertex_option",            (PyCFunction) PyP3Land_AddVertexOption,          METH_VARARGS },
//  { "rem_vertex_option",            (PyCFunction) PyP3Land_RemVertexOption,          METH_VARARGS },
//  { "add_all_vertexs_option",       (PyCFunction) PyP3Land_AddAllVertexOption,       METH_O },
//  { "rem_all_vertexs_option",       (PyCFunction) PyP3Land_RemAllVertexOption,       METH_O },
  { "init_fx",                      (PyCFunction) PyP3Land_InitFx,                   METH_NOARGS },
  { "vertex_fx",                    (PyCFunction) PyP3Land_VertexFx,                 METH_VARARGS },
  { "reinit",                       (PyCFunction) PyP3Land_ReInit,                   METH_NOARGS },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_FLOAT     (Land, P3_land*, SplitFactor, split_factor)
PY_GET_SET_ON_FLOAT_ADD (Land, P3_land*, ScaleFactor, scale_factor, if (a->option & P3_LAND_INITED) { a->option -= P3_LAND_INITED; })
PY_GET_SET_ON_FLOAT_ADD (Land, P3_land*, TextureFactor, texture_factor, P3_land_compute_coords (a);)
PY_GET_SET_ON_INT_ADD   (Land, P3_land*, PatchSize, patch_size, if (a->option & P3_LAND_INITED) { a->option -= P3_LAND_INITED; })
PY_GET_SET_ON_OPTION    (Land, P3_land*, RaypickWithLOD, P3_LAND_REAL_LOD_RAYPICK)

static PyObject* PyP3Land_GetWidth (P3_land* a, void* context) {
  return PyInt_FromLong((long) a->nb_vertex_width);
}

static PyObject* PyP3Land_GetDepth (P3_land* a, void* context) {
  return PyInt_FromLong((long) a->nb_vertex_depth);
}

static PyObject* PyP3Land_GetHasVertexOptions (P3_land* a, void* context) {
  if (a->option & P3_LAND_VERTEX_OPTIONS) {
    return PyInt_FromLong (1);
  } else {
    return PyInt_FromLong (0);
  }
}

static int PyP3Land_SetHasVertexOptions (P3_land* a, PyObject* value, void* context) {
  free (a->vertex_options);
  if (PyObject_IsTrue (value) == 1) {
    a->option |= P3_LAND_VERTEX_OPTIONS;
    a->vertex_options = (char*) malloc (a->nb_vertex_width * a->nb_vertex_depth * sizeof (char));
  } else {
    a->option &= ~P3_LAND_VERTEX_OPTIONS;
    a->vertex_options = NULL;
  }
  return 0;
}

static PyGetSetDef PyP3Land_GetSets[] = {
  { "width",              (getter) PyP3Land_GetWidth,            (setter) 0,                            NULL },
  { "depth",              (getter) PyP3Land_GetDepth,            (setter) 0,                            NULL },
  { "split_factor",       (getter) PyP3Land_GetSplitFactor,      (setter) PyP3Land_SetSplitFactor,      NULL },
  { "raypick_with_LOD",   (getter) PyP3Land_GetRaypickWithLOD,   (setter) PyP3Land_SetRaypickWithLOD,   NULL },
  { "scale_factor",       (getter) PyP3Land_GetScaleFactor,      (setter) PyP3Land_SetScaleFactor,      NULL },
  { "texture_factor",     (getter) PyP3Land_GetTextureFactor,    (setter) PyP3Land_SetTextureFactor,    NULL },
  { "patch_size",         (getter) PyP3Land_GetPatchSize,        (setter) PyP3Land_SetPatchSize,        NULL },
  { "has_vertex_options", (getter) PyP3Land_GetHasVertexOptions, (setter) PyP3Land_SetHasVertexOptions, NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Land_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Land",
  sizeof(P3_land),
  0,
  (destructor) PyP3Land_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Land_Traverse,/* tp_traverse */
  (inquiry) PyP3Land_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Land_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Land_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Land_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

