# Soya 3D
# Copyright (C) 2001-2003 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""soya.idler

This module provide a main loop for Soya, with FPS regulation. It is a simplified
version of the Py2Play idler ; if you want to use Py2Play in addition to Soya,
you should rather use the Py2Play idler.

The following global module attributes are noticeable:

 - ROUND_DURATION: The duration of a round. Round is the idler's time unit. The idler calls
   successively begin_round(), advance_time() (possibly several times) and end_round(); it
   is granted that ALL rounds correspond to a period of duration ROUND_DURATION (though
   the different period may not be regularly spread over time).
   Default is 0.030.

 - MIN_FRAME_DURATION: minimum duration for a frame. This attribute can be used to limit
   the maximum FPS to save CPU time; e.g. FPS higher than 30-40 is usually useless.
   Default is 0.025, which limits FPS to 40 in theory and to about 33 in practice
   (I don't know why there is a difference between theory and practice !).

 - IDLER: a reference to the last created idler."""

import time
import soya

ROUND_DURATION     = 0.030
MIN_FRAME_DURATION = 0.020

IDLER = None
STOPPED = 0
PLAYING = 1

class Idler:
  """Idler

A main loop with FPS regulation.

Interesting attributes:

 - next_round_tasks: a list of callable (taking no arg) that will be called once, just
   after the beginning of the next round.

 - scene: the scene associated to this idler."""
  
  def __init__(self, scene = None):
    """Idler(scene) -> Idler

Creates a new idler for scene SCENE."""
    self.next_round_tasks = []
    self.fps   = 0.0
    self.state = STOPPED
    self.scene = scene
    
    global IDLER
    IDLER = self
    
  def start(self):
    """Idler.start()

Starts idling with a new thread."""
    import thread
    thread.start_new_thread(self.idle, ())
    
  def stop (self):
    """Idler.stop()

Stops idling. The stopping may not occur immediately, but at the end of the next iteration."""
    self.state = STOPPED
    
  def idle(self):
    """Idler.idle()

Starts idling with the current thread. This method doesn't finish, until you call Idler.stop()."""
    self.state = PLAYING
    self.time = last_fps_computation_time = time.time()
    self.time_since_last_round = 0.0

    self.begin_round()
    
    nb_frame = 0
    while self.state >= PLAYING:
      nb_frame = 0
      while (self.state >= PLAYING) and (nb_frame < 80):
        nb_frame = nb_frame + 1
        
        while 1: # Sleep until at least MIN_FRAME_DURATION second has passed since the last frame
          current = time.time()
          delta = current - self.time
          
          if delta > MIN_FRAME_DURATION: break
          time.sleep(MIN_FRAME_DURATION - delta)
          
        self.time = current
        
        while self.time_since_last_round + delta > ROUND_DURATION: # Start a new frame
          spent_time = ROUND_DURATION - self.time_since_last_round
          
          self.advance_time(spent_time / ROUND_DURATION) # Complete the previous round
          self.end_round()                               # Ends the previous round
          
          self.begin_round()                             # Prepare the following round
          
          if self.next_round_tasks:
            for task in self.next_round_tasks: task()
            self.next_round_tasks = []
            
          delta = delta - spent_time
          self.time_since_last_round = 0
          
        self.advance_time(delta / ROUND_DURATION) # start the current round
        self.time_since_last_round = self.time_since_last_round + delta
        
        self.render()
        
      current = time.time()
      self.fps = nb_frame / (current - last_fps_computation_time)
      last_fps_computation_time = current
      
  def begin_round(self):
    """Idler.begin_round()

Called by Idler.idle when a new round begins; default implementation delegates to Idler.scene.begin_round."""
    if self.scene: self.scene.begin_round()
    
  def end_round(self):
    """Idler.end_round()

Called by Idler.idle when a round is finished; default implementation delegates to Idler.scene.end_round."""
    if self.scene: self.scene.end_round()
    
  def advance_time(self, proportion):
    """Idler.advance_time()

Called by Idler.idle when a piece of a round has occured; default implementation delegates to Idler.scene.advance_time.
PROPORTION is the proportion of the current round's time that has passed (1.0 for an entire round)."""
    if self.scene:
      soya.advance_time(proportion) # for C coded stuff
      self.scene.advance_time(proportion)
    
  def render(self):
    """Idler.render()

Called by Idler.idle when rendering is needed; default implementation calls soya.render."""
    soya.render()
    
    
