# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""soya.ray

Provide a Ray -- a line that draw a fading trace behind it when it moves.
Rays are usually used for sword or reactor.

The line starts at the ray position and ends at "endpoint" (by default,
0.0, 0.0, -1.0 in the ray coordinate system). Move, rotate or scale the
ray if you want a different position or size.

The color of the ray can be changed, as well as the length of the faded trace.

See tutorial lesson 106."""

import soya, _soya, soya.soya3d as soya3d
from soya.math3d import *


class Ray(soya._CObj, soya3d.GraphicElement, _soya._Coordsys):
  def __init__(self, parent = None, material = None, length = 10):
    _soya._Coordsys.__init__(self)
    
    self.name     = ""
    self.material = material
    self.length   = length
    self.old_pos  = []
    
    self.endpoint = Point(self, 0.0, 0.0, -1.0)
    
    if parent: parent.add(self)
    
  def __getstate__(self):
    state = soya._CObj.__getstate__(self)
    del state["old_pos"]
    return state
  def __setstate__(self, state):
    soya._CObj.__setstate__(self, state)
    self.old_pos  = []
    
  def zap(self):
    self.old_pos *= 0
    
  def batch(self): return (3, self.get_root())#, self.material)
  
  def render(self):
    root = self.get_root()
    
    #self.old_pos.insert(0, (self % root, self.endpoint % root))
    self.old_pos.insert(0, (root.transform_point(self.x, self.y, self.z, self.parent),
                            root.transform_point(self.endpoint.x, self.endpoint.y, self.endpoint.z, self.endpoint.parent)))
    
    if len(self.old_pos) > self.length:
      del self.old_pos[-1]
      
    #if self.material: self.material.activate()
    #else: _soya.glDisable(_soya.GL_TEXTURE_2D)
    _soya.activate_material(self.material)
    _soya.glDisable(_soya.GL_LIGHTING)
    _soya.glDisable(_soya.GL_CULL_FACE)
    
    _soya.glBegin(0x0008) #_soya.GL_QUAD_STRIP
    
    color = (self.material and list(self.material.diffuse)) or [0.05, 0.5, 1.0, 1.0]
    if self.material and self.material.tex_filename:
      for start, end in self.old_pos:
        _soya.glColor4f(*color)
        color[3] -= 1.0 / (self.length - 1)
        
        _soya.glTexCoord2f(0.0, color[3])
        _soya.glVertex3f(*end)
        
        _soya.glTexCoord2f(1.0, color[3])
        _soya.glVertex3f(*start)
        
    else:
      for start, end in self.old_pos:
        _soya.glColor4f(*color)
        color[3] -= 1.0 / (self.length - 1)
        
        _soya.glVertex3f(*end)
        _soya.glVertex3f(*start)
        
    _soya.glEnd()
    _soya.glEnable(_soya.GL_LIGHTING)
    _soya.glEnable(_soya.GL_CULL_FACE)
    #if self.material: self.material.inactivate()
    #else: _soya.glEnable(_soya.GL_TEXTURE_2D)
    

