# Soya 3D tutorial
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Lesson 006: Event management : a keyboard-controlled cube

# This lesson allows you to move the cube of lesson 001 !
# Use the keyboard cursor keys to move the cube.

import sys, soya, soya.soya3d as soya3d, soya.model as model, soya.cube as cube, soya.math3d as math3d

soya.init()

scene = soya3d.World()

material = model.Material()

cube = cube.Cube(scene, material)
cube.rotate_vertical(30.0)

light = soya3d.Light(scene)
light.set_xyz(0.0, 0.2, 1.0)

camera = soya3d.Camera(scene)
camera.z = 5.0
soya.set_root_widget(camera)

# This function will be called in our main loop to check if the user has
# pressed a key.

def manage_event():
  global cube
  
  # soya.process_event() returns all the events that have occurred since the
  # previous call. We scan all the event.
  
  for event in soya.process_event():
    
    # Each event is a tuple; the first item indicates the nature of the event.
    # The following
    
    # If the event is a keydown, ends the demo.
    # Events are tupples; the first item indicates the nature of the event and
    # is one of the KEYDOWN, KEYUP, MOUSEBUTTONDOWN,
    # MOUSEBUTTONUP, MOUSEMOTION constant.
    
    if event[0] == soya.KEYDOWN: # Keyboard
      
      # For keyboard event, the second element of the tupple is the key code
      # (a K_* constant), and the third is the state (combition of the
      # KMOD_SHIFT, KMOD_ALT, KMOD_CTRL, KMOD_META, flags).
      
      if   event[1] == soya.K_LEFT:
        
        # Rotate the cube on the left.
        # 2 different type of rotating functions are available: turn_* and rotate_*,
        # where * is either "lateral" (rotation around the Y axis), "vertical"
        # (rotation around the X axis) or "incline" (rotation around the Z axis).
        # For turn_*, the X, Y, Z axis are the axis of the object you are rotating.
        # For rotate_*, the X, Y, Z axis are the axis of the parent  of the object
        # you are rotating.

        # Here, turn_* is what we need! You can try to replace turn_ by rotate_
        # to see the difference (replace also the other turn_ 5 lines below).
        cube.turn_lateral(15.0)
        
      elif event[1] == soya.K_RIGHT:
        
        # Rotate the cube on the right.
        cube.turn_lateral(-15.0)
        
      elif event[1] == soya.K_UP:
        
        # Move the cube forward.
        # First we create a Vector object in the cube's coordinates system,
        # with coordinates 0, 0, -1. The -Z direction we use here is always
        # the "front" direction in Soya.
        
        translation = math3d.Vector(cube, 0.0, 0.0, -1.0)
        
        # Apply the translation. The "add_vector" method means translation for
        # Point or absolute position object (like here), and vectorial addition
        # for Vector.
        # 
        # Soya automatically handles the coordinates system conversion, since
        # translation is defined in the cube coordinates system, and cube's
        # position is defined in the scene coordinates system.
        # 
        # One may force coordinates system conversion by the use of the
        # "convert_to(coord_syst)" method, which applies the conversion IN
        # PLACE, or with the "%" operator, which returns a Point or a Vector
        # in the given coordinate system, possibly the same.
        # 
        #     translation.convert_to(scene)
        #     translation = translation % scene
        
        cube.add_vector(translation)
        
      elif event[1] == soya.K_DOWN:
        
        # Move the cube backward.
        
        translation = math3d.Vector(cube, 0.0, 0.0, 1.0)
        
        # The "+=" operator is a shortcut for "add_vector"
        
        cube += translation
        
      elif event[1] == soya.K_q:
        
        # Quit.
        
        sys.exit()


import time

while 1:
  soya.render()
  
  scene.begin_round()
  scene.advance_time()
  scene.end_round()
  
  time.sleep(0.1)
  
  # Treats the events.
  
  manage_event()
