# Soya 3D tutorial
# Copyright (C) 2001-2002 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ----------------------
# Lesson 107: Atmosphere
# ----------------------

# Atmosphere is an object that gather environmental information about a
# World, such as Ambient light, Fog...
# The root atmosphere (ie the first one encountered when traversing the
# Worlds hierarchy) is also responsible for clearing the screen (with a
# Background color or with a Skyplane)

import sys, os, os.path, time

import soya
import soya.model
import soya.soya3d
import soya.cube

soya.init()

data_dir = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data")

soya.model.Image   .PATH = os.path.join(data_dir, "images")
soya.model.Material.PATH = os.path.join(data_dir, "materials")
soya.soya3d.World  .PATH = os.path.join(data_dir, "worlds")
soya.model.Shape   .PATH = os.path.join(data_dir, "shapes")

# Create a world
world = soya.soya3d.World()

# Add a light
light = soya.soya3d.Light(world)
light.set_xyz(0.0, 10.0, 0.0)


# Atmosphere creation
# -------------------

atm1 = soya.soya3d.Atmosphere()

# Set Ambient light
atm1.ambient = (0.3, 0.3, 0.3, 1.0)

# Set Background color
atm1.bg_color  = (0.2, 0.1, 0.4, 1.0)

# Fog attributes
atm1.fog_color   = (0.0, 0.0, 1.0, 1.0)
# Possible fog types are:
#  - 0 = linear      (GL_LINEAR)
#  - 1 = exponential (GL_EXP)
#  - 2 = exponential (GL_EXP2)
atm1.fog_type    = 0
atm1.fog_start   = 0.0
atm1.fog_end     = 50.0
atm1.fog_density = 0.0
# This line enable Fog. Set atm1.fog to 0 to disable the fog
atm1.fog         = 1

# As this Atmosphere will be the root one, let's attribute it a Skyplane
atm1.skyplane  = 1
# This line enable Skyplane. Set atm1.skyplane = 0 to disable it
atm1.sky_color = (1.0, 1.0, 0.5, 1.0)
# The Skyplane will consist into a gradient from atm1.sky_color (up) to
# atm1.bg_color (down). You can also add clouds with a Material
atm1.cloud = soya.model.Material()
atm1.cloud.tex_filename = "cloud.tga"

# Attribute our new Atmosphere to the World
world.atmosphere = atm1

# Add a cube
material = soya.model.Material()
material.tex_filename = "block2.tga"
cube = soya.cube.Cube(world, material)
world.set_shape(cube.shapify())

# Create another World containing another cube but with another Atmosphere
world2 = soya.soya3d.World(world)
atm2 = soya.soya3d.Atmosphere()
atm2.ambient = (0.5, 0.5, 0.0, 1.0)
atm2.fog_color   = (0.0, 1.0, 0.0, 1.0)
atm2.fog_type    = 0
atm2.fog_start   = 0.0
atm2.fog_end     = 50.0
atm2.fog_density = 0.0
atm2.fog         = 1
world2.atmosphere = atm2
world2.set_shape(cube.shapify())
world2.set_xyz(2.0, 0.0, -2.0)


# Add a camera and a loop to render

camera = soya.soya3d.Camera(world)
camera.set_xyz(5.0, -2.0, 5.0)
camera.look_at(soya.soya3d.Point(world, 0.0, 0.0, 0.0))
soya.set_root_widget(camera)

while(1):
  time.sleep(0.1)
  soya.render()


# IMPORTANT NOTICE
# ----------------
# In the editor, if you add a world with an Atmosphere,
# you won't see the Atmosphere Background color or Skyplane
# cause all objects are added into a parent World that have
# a black atmosphere (this is to set a default Ambient light
# in the editor). If you want to see you Atmosphere as a
# root one, take the editor console and type:
# > root.parent.parent.Atmosphere = None
