# Soya 3D tutorial
# Copyright (C) 2003 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ----------------------------------------
# Lesson 117: Advanced Landscape functions
# ----------------------------------------

# This lesson shows some advanced landscape feature:
#   - landscape colored shadows
#   - landscape vertex fx to perform a 'fog of war' effect
#   - getting the height of the land instead of using raypicking (faster but less powerfull)

import os, os.path, sys, time

import soya.model
import soya.soya3d
import soya.land


soya.init()

soya.model.Image.PATH = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "images")

# Create a world
world = soya.soya3d.World()

# Add a light
light = soya.soya3d.Light(world)
light.ambient = (0.4, 0.4, 0.4, 1.0)
light.set_xyz(0.0, 15.0, 0.0)

# Create a land
land = soya.land.Land(0)
land.from_image(soya.model.Image("map1.tga"))
land.multiply_height(8.0)
# Texture the land
material1 = soya.model.Material()
material1.tex_filename  = "block2.tga"
material2 = soya.model.Material()
material2.tex_filename  = "metal1.tga"
land.set_material_layer(material1, 0.0, 6.0)
land.set_material_layer(material2, 6.0, 8.0)

world.set_shape(land)


# Land colors
# -----------
#   You can set the color of each vertex of the Land like this:
#
#     land.set_vertex_color(X, Z, COLOR)
#
#   X: position of the vertex along the X axis of the Land (integer)
#   Z: position of the vertex along the Z axis of the Land (integer)
#   COLOR: 4-elements tuple (RED, GREEN, BLUE, ALPHA)
#
#   If you want to drop all the colors of your Land use:
#
#     land.remove_color()



# Land colored shadows
# --------------------
#   This function will use raypicking to determine for each vertex of the
#   Land if it is shadowed or not. If the vertex is shadowed it will have
#   the given SHADOW_COLOR, else it will keep its color (or will have a
#   white color if the vertex had no color yet).
#
#     land.compute_shadow_color(LIGHT, SHADOW_COLOR)
#
#   LIGHT: the light that will cast static shadows on the map
#   SHADOW_COLOR: the color of the shadowed regions (RED, GREEN, BLUE, ALPHA)

land.compute_shadow_color(light, (0.6, 0.6, 0.6, 1.0))


# This function will set the given to color to the vertices that are on
# the borders of the Land
land.fade_borders((1.0, 1.0, 1.0, 0.0))


# Setting land visibility
# -----------------------
#   Each vertex of the land can have an option flag. Here are basic functions
#   to deal with vertex option:
#
#     land.get_vertex_option(X, Z)
#     land.set_vertex_option(X, Z, OPTION)
#
#   X and Z must be integer, OPTION is a flag of the following components:
#     1 : hidden vertex
#     4 : non solid vertex
#     8 : force vertex presence (independantly of LOD level)
#
#   If a face has all its vertices hidden, it won't be rendered at all.
#   If a face has all its vertices non solid, the face will be non solid (ie
#     it will no be taken in account during raypicking)
#   A vertex with forced presence will be present whatever the LOD level is
#     (the triangles of the Land will stay splitted to preserve the presence
#     of that vertex)
#   You must not use other values (specially value 2 and 16 are reserved).


# Add a zone on the map that is invisible, not solid and always present (= a hole!)
# 1 | 4 | 8 = 13
land.set_vertex_option(10, 10, 13)
land.set_vertex_option(11, 10, 13)
land.set_vertex_option(10, 11, 13)
land.set_vertex_option(11, 11, 13)
land.set_vertex_option(12, 10, 13)
land.set_vertex_option(12, 11, 13)
land.set_vertex_option(10, 12, 13)
land.set_vertex_option(11, 12, 13)
land.set_vertex_option(12, 12, 13)

# Make the Land split triangles to be sure that the points that we have said to be
# always present will be created
land.reinit()


# Land vertex fx
# --------------
#   First before using the vertex fx we must initialize the fx for the Land.
#   After fx initialization the Land vertex colors is undefined.
land.init_fx()

#   Then we can use this powerfull function:
#
#      land.vertex_fx(SELECTION, COLOR, TRANSITION_DURATION)
#
#   This function can dynamicly change the vertices colors of the Land.
#   SELECTION: this is the way to determine which vertices will have their color changed.
#     It can be:
#     - None: color will be applied on all the vertices of the Land
#     - 3-elements tuple representing a cylinder along Y axis: (CENTER_X, CENTER_Z, RADIUS)
#         the center coordinates must be given in the Land coordsys
#     - 4-elements tuple representing a sphere: (CENTER_X, CENTER_Y, CENTER_Z, RADIUS)
#         the center coordinates must be given in the Land coordsys
#   COLOR: this is the color the selected vertices will finally have. It can be:
#     - 4-elements tuple: (RED, GREEN, BLUE, ALPHA)
#     - a float: only alpha component, the RED, GREEN, BLUE component won't be changed
#     - None: restore vertex color (the color set by set_vertex_color or compute_shadow_color
#         else white)
#   TRANSITION_DURATION: optional (if not given, no transition). This is the time of the
#     smoothed transition between the color.


# Initialize all the Land color to black
land.vertex_fx(None, (0.0, 0.0, 0.0, 0.0))


# Add a camera
camera = soya.soya3d.Camera(world)
camera.set_xyz (0.0, 16.0, 0.0)
camera.turn_vertical(-90.0)
soya.set_root_widget(camera)

# Add a laser to see the hole in the land
import soya.laser
laser = soya.laser.Laser(world)
laser.reflect = 1
laser.set_xyz (0.0, 8.0, 0.0)
laser.turn_vertical(-75.0)


# A loop to render

while(1):

# Reveal the part of the land that is around the camera
#  land.vertex_fx((camera.x, camera.z, 5.0), 1.0, 10.0)
#  land.vertex_fx((camera.x, camera.z, 5.0), (1.0, 1.0, 1.0, 1.0), 10.0)
  land.vertex_fx((camera.x, camera.z, 5.0), None, 10.0)

  soya.advance_time(1.0)
  soya.render()

  time.sleep(0.1)
  
# Use Up, Down, Left, Right arrows to move the camera

  for event in soya.process_event():
    if event[0] == soya.KEYDOWN:
      if event[1] == soya.K_UP:
        camera.translate(0.0, 0.0, -0.5)
        laser.translate(0.0, 0.0, -0.5)
      elif event[1] == soya.K_DOWN:
        camera.translate(0.0, 0.0, 0.5)
        laser.translate(0.0, 0.0, 0.5)
      elif event[1] == soya.K_LEFT:
        camera.translate(-0.5, 0.0, 0.0)
        laser.translate(-0.5, 0.0, 0.0)
      elif event[1] == soya.K_RIGHT:
        camera.translate(0.5, 0.0, 0.0)
        laser.translate(0.5, 0.0, 0.0)
      elif event[1] == soya.K_q:
        sys.exit()

# Landscape get_true_height
# -------------------------
#
# height = get_true_height(x, z)
#   This function will return the height associated with the position x and z.
#   x and z must be Float and given in the land coordsys. The height is interpolated
#   (this is different from the get_height function).
#
# (height, normal) = get_true_height_normal(x, z)
#   Same as above but return a tuple with the height and the point normal associated
#   with the position (x, z). Notice that normal is a 3 floats tuple, not a Vector
#   cause the land is a shape that doesn't have a parent and so doesn't know its coordsys.

      camera.y = land.get_true_height(camera.x, camera.z) + 16.0
      laser.y = camera.y - 8.0


# Conclusion
# ----------
#  Now that you know all of this you can use this 'fog of war' effect to create a wargame
#  or real time strategy game (like Warcraft III... ;). Anybody interested?
    


