# Soya 3D
# Copyright (C) 2001-2004 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# XXX disable GC
#import gc; gc.disable()

import sys, os, os.path, weakref,cPickle as  pickle

from soya._soya import *
from soya._soya import _CObj

#from soya import _soya

_soya = sys.modules["soya._soya"]

#from soya import opengl
#from soya import sdlconst

# For file compatibility
sys.modules["soyapyrex"] = _soya

def set_root_widget(widget):
  global root_widget
  root_widget = widget
  _soya.set_root_widget(widget)
  

def coalesce_motion_event(events):
  """coalesce_motion_event(events) -> sequence

Prunes from EVENTS all mouse motion events except the last one.
This is usefull since only the last one is usually releavant (Though
be carrefull if you use the relative mouse coordinate !).

EVENTS should be a list of events, as returned by soya.process_event().
The returned list has the same structure."""
  import soya.sdlconst
  
  events = list(events)
  events.reverse()
  
  def keep_last(event):
    if event[0] == soya.sdlconst.MOUSEMOTION:
      if keep_last.last_found: return 0
      else: keep_last.last_found = 1
    return 1
  keep_last.last_found = 0
  
  events = filter(keep_last, events)
  events.reverse()
  return events


DATADIR = os.path.join(os.path.dirname(__file__), "data")
if not os.path.exists(DATADIR):
  DATADIR = "/usr/share/soya"
  if not os.path.exists(DATADIR):
    DATADIR = "/usr/local/share/soya"
    if not os.path.exists(DATADIR):
      DATADIR = "/usr/share/python-soya"
      if not os.path.exists(DATADIR):
        DATADIR = os.path.join(os.getcwd(), "data")
        if not os.path.exists(DATADIR):
          import warnings
          warnings.warn("Soya's data directory cannot be found !")
          
        
path = []

_SAVING = None # The object currently being saved. XXX not thread-safe, hackish

def _loader(klass, filename): return klass.load(filename)
def _getter(klass, filename): return klass.get (filename)

class SavedInAPath(object):
  """SavedInAPath

Base class for all objects that can be saved in a path, such as Material,
World,..."""
  DIRNAME = ""
  
  def _check_export(klass, exported_filename, filename, *source_dirnames):
    """_check_export(FILENAME, *(SOURCE_DIRNAME, SOURCE_FILENAME)) -> (SOURCE_DIRNAME, SOURCE_FULL_FILENAME, FULL_FILENAME)

Search soya.path for a file FILENAME.data in the self.DIRNAME directory,
and check if the file needs to be re-exporter from any of the SOURCE_DIRNAMES directory.

Returned tuple :
FULL_FILENAME is the complete filename of the object.
If SOURCE_DIRNAME is None, the exported object is up-to-date.
If SOURCE_DIRNAME is one of SOURCE_DIRNAMES, the source object of this directory have been
modified more recently that the exported one, and SOURCE_FULL_FILENAME is the complete
filename of the source that needs to be re-exported."""
    filename        = filename.replace("/", os.sep)
    #source_filename = filename[:filename.index("@")]
    for p in path:
      file        = os.path.join(p, klass.DIRNAME , exported_filename)
      
      if   os.path.exists(file):
        for source_dirname, source_filename in source_dirnames:
          source_file = os.path.join(p, source_dirname, source_filename)
          if os.path.exists(source_file) and (os.path.getmtime(file) < os.path.getmtime(source_file)):
            print "* Soya * Converting %s to %s..." % (source_file, klass.__name__)
            return source_dirname, source_file, file
        return None, None, file
      else:
        for source_dirname, source_filename in source_dirnames:
          source_file = os.path.join(p, source_dirname, source_filename)
          if os.path.exists(source_file):
            print "* Soya * Converting %s to %s..." % (source_file, klass.__name__)
            return source_dirname, source_file, file
          
    raise ValueError("No %s or %s named %s" % (klass.__name__, source_dirname, filename))
  _check_export = classmethod(_check_export)
  
  def get(klass, filename):
    """SavedInAPath.get(filename)

Gets the object of this class with the given FILENAME attribute.
The object is loaded from the path if it is not already loaded.
If it is already loaded, the SAME object is returned.
Raise ValueError if the file is not found in soya.path."""
    return klass._alls.get(filename) or klass._alls.setdefault(filename, klass.load(filename))
  get = classmethod(get)
  
  def load(klass, filename):
    """SavedInAPath.get(filename)

Loads the object of this class with the given FILENAME attribute.
Contrary to get, load ALWAYS returns a new object.
Raise ValueError if the file is not found in soya.path."""
    filename = filename.replace("/", os.sep)
    for p in path:
      file = os.path.join(p, klass.DIRNAME, filename + ".data")
      if os.path.exists(file):
        obj = klass._alls[filename] = pickle.loads(open(file, "rb").read())
        return obj
    raise ValueError("No %s named %s" % (klass.__name__, filename))
  load = classmethod(load)
  
  def save(self, filename = None):
    """SavedInAPath.save(filename = None)

Saves this object. If no FILENAME is given, the object is saved in the path,
using its filename attribute. If FILENAME is given, it is saved at this
location."""
    global _SAVING
    _SAVING = self # Hack !!
    #pickle.dump(self, open(filename or os.path.join(path[0], self.DIRNAME, self.filename.replace("/", os.sep)) + ".data", "wb"), 1)
    # Avoid destroying the file if the serialization causes an error.
    data = pickle.dumps(self, 1)
    open(filename or os.path.join(path[0], self.DIRNAME, self.filename.replace("/", os.sep)) + ".data", "wb").write(data)
    _SAVING = None
    
  def __reduce__(self):
    if (not _SAVING is self) and self._filename: # self is saved in another file, save filename only
      if isinstance(self, CoordSyst): return (_loader, (self.__class__, self.filename)) # not shareable
      else:                           return (_getter, (self.__class__, self.filename)) # can be shared
    return _CObj.__reduce__(self)
  
  def get_filename(self): return self._filename
  def set_filename(self, filename):
    if self._filename:
      try: del self._alls[self.filename]
      except KeyError: pass
    if filename: self._alls[filename] = self
    self._filename = filename
  filename = property(get_filename, set_filename)
  
  def availables(klass):
    """SavedInAPath.availables() -> list

Returns the list of the filename all the objects available in the current path."""
    import dircache
    filenames = dict(klass._alls)
    for p in path:
      for filename in dircache.listdir(os.path.join(p, klass.DIRNAME)):
        if filename.endswith(".data"): filenames[filename[:-5]] = 1
    filenames = filenames.keys()
    filenames.sort()
    return filenames
  availables = classmethod(availables)
  
  def __setstate__(self, state):
    super(SavedInAPath, self).__setstate__(state)

# We MUST extends all Pyrex classes in Python,
# at least for weakref-ing their instance.

class Image(SavedInAPath, _soya._Image):
  """A Soya image, suitable for e.g. texturing.

Attributes are:

 - pixels : the raw image data (e.g. in a form suitable for PIL).

 - width.

 - height.

 - nb_color: the number of color channels (1 => monochrome, 3 => RGB, 4 => RGBA).
"""
  DIRNAME = "images"
  _alls = weakref.WeakValueDictionary()
  def __reduce__(self): return _CObj.__reduce__(self)
  
  def load(klass, filename):
    filename = filename.replace("/", os.sep)
    for p in path:
      file = os.path.join(p, klass.DIRNAME, filename)
      if os.path.exists(file): return open_image(file)
    raise ValueError("No %s named %s" % (klass.__name__, filename))
  load = classmethod(load)
  
  def save(klass, filename = None): raise NotImplementedError("Soya cannot save image.")
  
  def availables(klass):
    """SavedInAPath.availables() -> list

Returns the list of the filename all the objects available in the current path."""
    import dircache
    filenames = dict(klass._alls)
    for p in path:
      for filename in dircache.listdir(os.path.join(p, klass.DIRNAME)):
        filenames[filename] = 1
    filenames = filenames.keys()
    filenames.sort()
    return filenames
  availables = classmethod(availables)

  
class Material(SavedInAPath, _soya._Material):
  """Material

A material regroups all the surface attributes, like colors, shininess and
texture. You should NEVER use None as a material, use soya._DEFAULT_MATERIAL instead.

Attributes are:

- diffuse: the diffuse color.

- specular: the specular color; used for shiny part of the surface.

- emissive: the emissive color; this color is applied even in the dark.

- separate_specular: set it to true to enable separate specular; this usually
  results in a more shiny specular effect.

- shininess: the shininess ranges from 0.0 to 128.0; 0.0 is the most metallic / shiny
  and 128.0 is the most plastic.

- texture: the texture (a soya.Image object, or None if no texture).

- clamp: set it to true if you don't want to repeat the texture when the texture
  coordinates are out of the range 0 - 1.0.

- additive_blending: set it to true for additive blending. For semi-transparent surfaces
  (=alpha blended) only. Usefull for special effect (Ray, Sprite,...)."""

  DIRNAME = "materials"
  _alls = weakref.WeakValueDictionary()
  
  def load(klass, filename):
    need_export, image_file, file = klass._check_export(filename + ".data", filename, (Image.DIRNAME, filename + ".png"), (Image.DIRNAME, filename + ".jpeg"))
    if need_export:
      image = Image.get(os.path.basename(image_file))
      if os.path.exists(file): material = pickle.loads(open(file, "rb").read())
      else:
        material = Material()
        material._filename = filename
      material.texture = image
      try: material.save()
      except:
        sys.excepthook(*sys.exc_info())
        print "* Soya * WARNING : can't save material %s!" % filename
      return material
    else:
      return pickle.loads(open(file, "rb").read())
  load = classmethod(load)
  
class Shape(SavedInAPath, _soya._Shape):
  DIRNAME = "shapes"
  _alls = weakref.WeakValueDictionary()
  
  def load(klass, filename):
    need_export, world_file, file = klass._check_export(filename + ".data", filename, (World.DIRNAME, filename + ".data"), ("blender", filename.split("@")[0] + ".blend"), ("obj", filename + ".obj"), ("obj", filename + ".mtl"), ("3ds", filename + ".3ds"))
    if need_export:
      shape = World.get(filename).shapify()
      shape._filename = filename
      try: shape.save()
      except:
        sys.excepthook(*sys.exc_info())
        print "* Soya * WARNING : can't save compiled shape %s!" % filename
      return shape
    else:
      return pickle.loads(open(file, "rb").read())
  load = classmethod(load)
  
  def availables(klass): return World.availables()
  availables = classmethod(availables)
  
  
class SimpleShape(Shape, _soya._SimpleShape):
  pass
  
class TreeShape(Shape, _soya._TreeShape):
  pass

class CellShadingShape(Shape, _soya._CellShadingShape):
  pass


class Point(_soya._Point):
  pass

class Vector(_soya._Vector):
  pass

class Vertex(_soya._Vertex):
  pass

  
class Volume(_soya._Volume):
  pass

def do_cmd(cmd):
  print "* Soya * Running '%s'..." % cmd
  os.system(cmd)
  
class World(SavedInAPath, _soya._World, Volume):
  DIRNAME = "worlds"
  _alls = weakref.WeakValueDictionary()
  
  def load(klass, filename):
    need_export, source_file, file = klass._check_export(filename + ".data", filename, ("blender", filename.split("@")[0] + ".blend"), ("obj", filename + ".obj"), ("obj", filename + ".mtl"), ("3ds", filename + ".3ds"))
    if need_export:
      if   need_export == "blender":
        animation, time = "", 0
        extra = ""
        
        if "@" in filename:
          args = filename[filename.index("@") + 1:].split(",")
          if args[0]: animation, time = args[0].split(":")
          del args[0]
          for arg in args:
            print arg, args
            extra += "MATERIAL_%s=%s" % tuple(arg.split(":"))
            
        do_cmd("blender %s -P %s --blender2soya FILENAME=%s ANIMATION=%s ANIMATION_TIME=%s %s" % (
          source_file,
          os.path.join(os.path.dirname(__file__), "blender2soya.py"),
          filename,
          animation,
          time,
          extra,
          ))
        
      elif need_export == "obj":
        import soya.objmtl2soya
        world = soya.objmtl2soya.loadObj(os.path.splitext(source_file)[0] + ".obj")
        world.filename = filename
        world.save()
        return world
      
      elif need_export == "3ds":
        import soya._3DS2soya
        world = soya._3DS2soya.load_3ds(os.path.splitext(source_file)[0] + ".3ds")
        world.filename = filename
        world.save()
        return world
      
    return pickle.loads(open(file, "rb").read())
  load = classmethod(load)
  

class Light(_soya._Light):
  pass

class Camera(_soya._Camera):
  pass

class Face(_soya._Face):
  pass

class Atmosphere(_soya._Atmosphere):
  pass

class NoBackgroundAtmosphere(_soya._NoBackgroundAtmosphere, Atmosphere):
  pass

class SkyAtmosphere(_soya._SkyAtmosphere, Atmosphere):
  pass

class Sprite(_soya._Sprite):
  pass

class CylinderSprite(_soya._CylinderSprite, Sprite):
  pass

class Particles(_soya._Particles):
  pass

class Bonus(_soya._Bonus):
  pass

class Portal(_soya._Portal):
  # Implement in Python due to the lambda
  def pass_through(self, coordsyst):
    """Portal.pass_though(self, coordsyst)

Makes COORDSYST pass through the portal. If needed (=if coordsyst.parent is self.parent),
it removes COORDSYST from its current parent and add it in the new one,
at the right location.
If coordsyst is a camera, it change the 'to_render' attribute too.

The passing through does NOT occur immediately, but after the beginning of the round
(this is usually what you need, in order to avoid that moving COORDSYST from a parent
to another makes it plays twice)."""
    def do_it():
      if isinstance(coordsyst, Camera) and coordsyst.to_render:
        coordsyst.to_render = self.beyond
      if coordsyst.parent is self.parent:
        self.beyond.add(coordsyst)
    IDLER.next_round_tasks.append(do_it)
    
    
class Land(_soya._Land):
  pass

class WaterCube(_soya._WaterCube):
  pass

class TravelingCamera(_soya._TravelingCamera):
  pass

class ThirdPersonTraveling(_soya._ThirdPersonTraveling):
  pass

class FixTraveling(_soya._FixTraveling):
  pass

class Cal3dVolume(_soya._Cal3dVolume):
  pass

class Cal3dShape(Shape, _soya._Cal3dShape):
  def load(klass, filename):
    need_export, source_file, file = klass._check_export(os.path.join(filename, filename + ".cfg"), filename, ("blender", filename + ".blend"))
    if need_export:
      if need_export == "blender":
        do_cmd("blender %s -P %s --blender2cal3d FILENAME=%s EXPORT_FOR_SOYA=1 XML=0" % (
          source_file,
          os.path.join(os.path.dirname(__file__), "blender2cal3d.py"),
          os.path.join(os.path.dirname(source_file), "..", "shapes", filename, filename + ".cfg"),
          ))
      
    return parse_cal3d_cfg_file(file)
  load = classmethod(load)

_soya.Image            = Image
_soya.Material         = Material
_soya.Shape            = Shape
_soya.SimpleShape      = SimpleShape
_soya.TreeShape        = TreeShape
_soya.CellShadingShape = CellShadingShape
_soya.Point            = Point
_soya.Vector           = Vector
_soya.Camera           = Camera
_soya.Light            = Light
_soya.Volume           = Volume
_soya.World            = World
_soya.Cal3dVolume      = Cal3dVolume
_soya.Cal3dShape       = Cal3dShape
_soya.Face             = Face
_soya.Atmosphere       = Atmosphere
_soya.Portal           = Portal
_soya.Land             = Land
_soya.WaterCube        = WaterCube
_soya.Particles        = Particles

DEFAULT_MATERIAL = Material()
DEFAULT_MATERIAL.filename  = "__DEFAULT_MATERIAL__"
DEFAULT_MATERIAL.shininess = 128.0
_soya._set_default_material(DEFAULT_MATERIAL)

PARTICLE_DEFAULT_MATERIAL = pickle.load(open(os.path.join(DATADIR, "particle_default.data"), "rb"))
PARTICLE_DEFAULT_MATERIAL.filename = "__PARTICLE_DEFAULT_MATERIAL__"
# PARTICLE_DEFAULT_MATERIAL = Material()
# PARTICLE_DEFAULT_MATERIAL.additive_blending = 1
# PARTICLE_DEFAULT_MATERIAL.texture = open_image(os.path.join(DATADIR, "fx.png"))
# PARTICLE_DEFAULT_MATERIAL.filename = "__PARTICLE_DEFAULT_MATERIAL__"
# PARTICLE_DEFAULT_MATERIAL.diffuse = (1.0, 1.0, 1.0, 1.0)
# PARTICLE_DEFAULT_MATERIAL.save("/home/jiba/src/soya/data/particle_default.data")
_soya._set_particle_default_material(PARTICLE_DEFAULT_MATERIAL)

SHADER_DEFAULT_MATERIAL = pickle.load(open(os.path.join(DATADIR, "shader_default.data"), "rb"))
SHADER_DEFAULT_MATERIAL.filename = "__SHADER_DEFAULT_MATERIAL__"
#SHADER_DEFAULT_MATERIAL = Material()
#SHADER_DEFAULT_MATERIAL.texture = open_image(os.path.join(DATADIR, "shader.png"))
#SHADER_DEFAULT_MATERIAL.filename = "__SHADER_DEFAULT_MATERIAL__"
#SHADER_DEFAULT_MATERIAL.diffuse = (1.0, 1.0, 1.0, 1.0)
#SHADER_DEFAULT_MATERIAL.save("/home/jiba/src/soya/data/shader_default.data")
_soya._set_shader_default_material(SHADER_DEFAULT_MATERIAL)


inited = 0
