-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
function GetBinaryOperatorType
  (Name  : SPSymbols.SPSymbol;
   Left  : Symbol;
   Right : Symbol)
  return  Symbol
is
   Result : Symbol;

   --------------------------------------------------------------------------------

   function GetSymmetricOperatorType (Left, Right : Symbol) return Symbol
   --# global in Dict;
   is
      Result : Symbol;
   begin

      if TypeIsUnknown (Left) then
         Result := Right;
      else
         Result := Left;
      end if;

      return Result;

   end GetSymmetricOperatorType;

   --------------------------------------------------------------------------------

   function GetMultiplicationOperatorType (Left, Right : Symbol) return Symbol
   --# global in Dict;
   is
      Result : Symbol;
   begin

      if TypeIsFixedPoint (Left) then
         if TypeIsInteger (Right) then
            Result := Left;
         elsif TypeIsFixedPoint (Right) then
            Result := GetUniversalFixedType;
         else
            Result := Right;
         end if;
      elsif IsUniversalRealType (Left) then
         if IsUniversalIntegerType (Right) then
            Result := Left;
         else
            Result := Right;
         end if;
      elsif TypeIsUnknown (Left) then
         if IsPredefinedIntegerType (Right) or else IsUniversalIntegerType (Right) or else TypeIsFixedPoint (Right) then
            Result := Left;
         else
            Result := Right;
         end if;
      elsif IsPredefinedIntegerType (Left) or else IsUniversalIntegerType (Left) then
         Result := Right;
      else
         Result := Left;
      end if;

      return Result;

   end GetMultiplicationOperatorType;

   --------------------------------------------------------------------------------

   function GetDivisionOperatorType (Left, Right : Symbol) return Symbol
   --# global in Dict;
   is
      Result : Symbol;
   begin

      if TypeIsFixedPoint (Left) then
         if TypeIsInteger (Right) then
            Result := Left;
         elsif TypeIsFixedPoint (Right) then
            Result := GetUniversalFixedType;
         else
            Result := Right;
         end if;
      elsif IsUniversalRealType (Left) then
         if IsUniversalIntegerType (Right) or else TypeIsUnknown (Right) then
            Result := Left;
         else
            Result := Right;
         end if;
      elsif TypeIsUnknown (Left) then
         if IsPredefinedIntegerType (Right) or else IsUniversalIntegerType (Right) then
            Result := Left;
         elsif TypeIsFixedPoint (Right) then
            Result := GetUniversalFixedType;
         else
            Result := Right;
         end if;
      else
         Result := Left;
      end if;

      return Result;

   end GetDivisionOperatorType;

   --------------------------------------------------------------------------------
   -- For the types Time and Time_Span in package Ada.Real_Time, the adding
   -- operator results are defined as follows, with T = Time and TS = Time_Span:
   --
   -- Op "+" Right  T   TS   U           Op "-" Right  T   TS   U
   -- Left                                Left
   --  T            X   T    T            T            TS  T    T
   --  TS           T   TS   TS           TS           X   TS   TS
   --  U            T   TS   X            U            T   TS   X
   --
   -- The multiplying operator results for type Time_Span are defined as follows,
   -- with I = Integer:
   --
   -- Op "*" Right  I   TS   U           Op "/" Right  I   TS   U
   -- Left                                Left
   --  I            X   TS   X            I            X   X    X
   --  TS           TS  X    TS           TS           TS  I    TS
   --  U            X   TS   X            U            X   TS   X
   --

   function GetTimeTypeOperatorType (Left, Right : Symbol) return Symbol
   --# global in Dict;
   --#        in Name;
   is
      Result : Symbol;
   begin

      if TypeIsUnknown (Left) then
         Result := Right;
      elsif TypeIsUnknown (Right) then
         Result := Left;
      else
         case Name is
            when SPSymbols.plus | SPSymbols.minus =>
               if Left = Right then
                  Result := GetPredefinedTimeSpanType;
               else
                  Result := GetPredefinedTimeType;
               end if;
            when SPSymbols.multiply =>
               Result := GetPredefinedTimeSpanType;
            when SPSymbols.divide =>
               if Left = Right then
                  Result := GetPredefinedIntegerType;
               else
                  Result := GetPredefinedTimeSpanType;
               end if;
            when others => -- Only Relational ops are defined
               Result := GetPredefinedBooleanType;
         end case;
      end if;

      return Result;

   end GetTimeTypeOperatorType;

   --------------------------------------------------------------------------------

begin

   if CommandLineData.Ravenscar_Selected and then (IsPredefinedTimeType (Left) or else IsPredefinedTimeType (Right)) then
      -- The rules for the results of time type operators are very different from
      -- the standard type operators, and so we have specific processing.
      Result := GetTimeTypeOperatorType (Left, Right);
   else
      case Name is
         when SPSymbols.RWand |
           SPSymbols.RWor  |
           SPSymbols.RWxor |
           SPSymbols.plus  |
           SPSymbols.minus |
           SPSymbols.RWmod |
           SPSymbols.RWrem =>
            Result := GetSymmetricOperatorType (Left, Right);
         when SPSymbols.equals           |
           SPSymbols.not_equal        |
           SPSymbols.less_than        |
           SPSymbols.less_or_equal    |
           SPSymbols.greater_than     |
           SPSymbols.greater_or_equal =>
            Result := GetPredefinedBooleanType;
         when SPSymbols.multiply =>
            Result := GetMultiplicationOperatorType (Left, Right);
         when SPSymbols.divide =>
            Result := GetDivisionOperatorType (Left, Right);
         when others =>
            Result := Left;
      end case;
   end if;
   return Result;

end GetBinaryOperatorType;
