-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
function LookupItem
  (Name              : LexTokenManager.Lex_String;
   Scope             : Scopes;
   Context           : Contexts;
   Full_Package_Name : Boolean)
  return    Symbol
is
   Item            : Symbol;
   Is_Visible      : Boolean;
   Current_Scope   : Scopes;
   Current_Region  : Symbol;
   Enclosing_Scope : Scopes;
   In_A_Subprogram : Boolean;
   Stop_At         : LexTokenManager.Lex_String;

   ------------------------------------------------------------------------------

   procedure Lookup_Context_Clauses
     (Name              : in     LexTokenManager.Lex_String;
      Scope             : in     Scopes;
      Start_Pos         : in     Scopes;
      Context           : in     Contexts;
      Full_Package_Name : in     Boolean;
      The_Package       :    out Symbol;
      Is_Visible        :    out Boolean)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Is_Visible  from Context,
   --#                          Dict,
   --#                          Full_Package_Name,
   --#                          LexTokenManager.State,
   --#                          Name,
   --#                          Scope,
   --#                          Start_Pos &
   --#         The_Package from Dict,
   --#                          Full_Package_Name,
   --#                          LexTokenManager.State,
   --#                          Name,
   --#                          Scope;
   is

      Region          : Symbol;
      Inherit_Clause  : Symbol;
      Current_Package : Symbol;

      --------------------------------------------------------------------------------

      function Has_Been_Withed (The_Package : Symbol;
                                Scope       : Scopes) return Boolean
      --# global in Dict;
      is
         Current  : Scopes;
         Last1    : Scopes;
         Ancestor : Symbol;
         Found    : Boolean;
      begin

         Current := Scope;
         Last1   := Current;

         loop
            exit when GetRegion (Current) = GetPredefinedPackageStandard;
            exit when IsWithed (The_Package, Current);
            Last1   := Current;
            Current := GetEnclosingScope (Current);
         end loop;

         Found := GetRegion (Current) /= GetPredefinedPackageStandard;

         if not Found and then Last1 /= Current and then IsPackage (GetRegion (Last1)) then -- search through ancestors
            Ancestor := RawDict.GetPackageParent (GetRegion (Last1));
            loop
               exit when Ancestor = NullSymbol;
               exit when IsWithed (The_Package, VisibleScope (Ancestor));
               Ancestor := RawDict.GetPackageParent (Ancestor);
            end loop;
            Found := Ancestor /= NullSymbol;
         end if;

         return Found;

      end Has_Been_Withed;

      --------------------------------------------------------------------------------

      function Is_Directly_Visible (The_Package_Or_Generic : Symbol;
                                    Scope                  : Scopes) return Boolean
      --# global in Dict;
      is
         The_Parent  : Symbol;
         Lib_Package : Symbol;
         Result      : Boolean;
      begin
         if Is_Generic_Subprogram (The_Symbol => The_Package_Or_Generic) then
            Result := True;
         else
            The_Parent := RawDict.GetPackageParent (The_Package_Or_Generic);
            if The_Parent = NullSymbol then
               Result := True;
            elsif not IsPackage (GetRegion (Scope)) then
               Result := False;
            else  --  The_Package is a child and Scope is in another package
                  -- OK if Scope is (possibly embedded within) The_Package's parent
                  -- or a descendent of the parent
               Lib_Package := GetLibraryPackage (Scope);
               Result      := Lib_Package = The_Parent or else IsProperDescendent (Lib_Package, The_Parent);
            end if;
         end if;

         return Result;

      end Is_Directly_Visible;

   begin -- Lookup_Context_Clauses
      Trace_Lex_Str (Msg => "   In Lookup_Context_Clauses, seeking ",
                     L   => Name);

      Region := GetRegion (Scope);

      if IsPackage (Region) then
         Inherit_Clause := RawDict.GetPackageInheritClauses (Region);
      elsif IsMainProgram (Region) then
         Inherit_Clause := Dict.Main.InheritClauses;
      elsif IsType (Region) and then TypeIsProtected (Region) then
         Region         := GetEnclosingPackage (Scope);
         Inherit_Clause := RawDict.GetPackageInheritClauses (Region);
      else
         Inherit_Clause := NullSymbol;
      end if;

      loop
         if Inherit_Clause = NullSymbol then
            The_Package := NullSymbol;
            Is_Visible := False;
            exit;
         end if;
         Current_Package := RawDict.GetContextClausePackage (Inherit_Clause);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (Current_Package),
                                                                 Lex_Str2 => Name) =
           LexTokenManager.Str_Eq
           and then Is_Directly_Visible (The_Package_Or_Generic => Current_Package,
                                         Scope                  => Scope)
           and then (not Full_Package_Name
                       or else (GetPackageParent (ThePackage => Current_Package) = NullSymbol)) then
            The_Package := Current_Package;
            Is_Visible :=
              Context = ProofContext
              or else (IsPackage (Current_Package) and then IsEmbeddedPackage (Current_Package))
              or else IsLocal (Scope, LocalScope (Current_Package))
              or else (IsPackage (Region)
                         and then IsPackage (Current_Package)
                         and then IsProperDescendent (GetLibraryPackage (Scope), Current_Package))
              or else Has_Been_Withed (The_Package => Current_Package,
                                       Scope       => Start_Pos);
            exit;
         end if;
         Inherit_Clause := RawDict.GetNextContextClause (Inherit_Clause);
      end loop;
      Trace_Sym (Msg   => "   found in  Lookup_Context_Clauses ",
                 Sym   => The_Package,
                 Scope => Scope);
   end Lookup_Context_Clauses;

   ------------------------------------------------------------------------------

   procedure Lookup_Private_Children
     (Name        : in     LexTokenManager.Lex_String;
      Region      : in     Symbol;
      Scope       : in     Scopes;
      Context     : in     Contexts;
      The_Package :    out Symbol;
      Is_Visible  :    out Boolean)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Is_Visible,
   --#         The_Package from Context,
   --#                          Dict,
   --#                          LexTokenManager.State,
   --#                          Name,
   --#                          Region,
   --#                          Scope;
   is
      Current_Package : Symbol;

      -----------------------------------------

      function Check_Is_Withed (Pack_Sym : Symbol;
                                Scope    : Scopes;
                                Context  : Contexts) return Symbol
      --# global in Dict;
      is
         Current : Scopes;
         Result  : Symbol;
      begin
         if Context = ProofContext then
            Result := Pack_Sym;
         else
            Current := Scope;
            loop
               exit when GetRegion (Current) = GetPredefinedPackageStandard;
               exit when IsWithed (Pack_Sym, Current);
               Current := GetEnclosingScope (Current);
            end loop;

            if GetRegion (Current) = GetPredefinedPackageStandard then
               Result := NullSymbol;
            else
               Result := Pack_Sym;
            end if;
         end if;
         return Result;
      end Check_Is_Withed;

   begin -- Lookup_Private_Children
      The_Package := NullSymbol;
      if IsPackage (Region) then
         Current_Package := RawDict.GetPackageFirstChild (Region, PrivateChild);
         loop
            exit when Current_Package = NullSymbol;

            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (Current_Package),
                                                                    Lex_Str2 => Name) =
              LexTokenManager.Str_Eq then
               The_Package := Current_Package;
               exit;
            end if;

            Current_Package := RawDict.GetPackageSibling (Current_Package);
         end loop;

         if The_Package /= NullSymbol then
            The_Package := Check_Is_Withed (Pack_Sym => The_Package,
                                            Scope    => Scope,
                                            Context  => Context);
         end if;
      end if;

      Is_Visible  := The_Package /= NullSymbol;
   end Lookup_Private_Children;

begin
   Current_Scope  := Scope;
   Current_Region := GetRegion (Current_Scope);

   TraceMsg ("--------------------------------------------------------------------------");
   Trace_Lex_Str (Msg => "In LookupItem, seeking ",
                  L   => Name);
   Trace_Sym (Msg   => "   in ",
              Sym   => Current_Region,
              Scope => Scope);

   loop
      LookupScope (Name, LexTokenManager.Null_String, Current_Scope, Current_Scope, Context, Item, Is_Visible);
      exit when Item /= NullSymbol;
      exit when IsCompilationUnit (Current_Region);
      exit when IsProtectedType (Current_Region);
      exit when IsTaskType (Current_Region);
      Current_Scope  := GetEnclosingScope (Current_Scope);
      Current_Region := GetRegion (Current_Scope);
   end loop;

   if Item = NullSymbol then

      In_A_Subprogram := IsSubprogram (Current_Region) or else IsTaskType (Current_Region);

      loop
         Lookup_Private_Children (Name        => Name,
                                  Region      => Current_Region,
                                  Scope       => Scope,
                                  Context     => Context,
                                  The_Package => Item,
                                  Is_Visible  => Is_Visible);
         exit when Item /= NullSymbol;

         Lookup_Context_Clauses (Name              => Name,
                                 Scope             => Current_Scope,
                                 Start_Pos         => Scope,
                                 Context           => Context,
                                 Full_Package_Name => Full_Package_Name,
                                 The_Package       => Item,
                                 Is_Visible        => Is_Visible);
         exit when Item /= NullSymbol;
         exit when IsPackage (Current_Region);
         exit when IsMainProgram (Current_Region);
         Enclosing_Scope := GetEnclosingScope (Current_Scope);
         if IsSubprogram (Current_Region) and then IsLocalScope (Enclosing_Scope) then
            Stop_At := GetSimpleName (Current_Region);
         elsif IsProtectedType (Current_Region) and then IsLocalScope (Enclosing_Scope) then
            Stop_At := GetSimpleName (Current_Region);
         elsif IsTaskType (Current_Region) and then IsLocalScope (Enclosing_Scope) then
            Stop_At := GetSimpleName (Current_Region);
         else
            Stop_At := LexTokenManager.Null_String;
         end if;
         LookupScope (Name, Stop_At, Enclosing_Scope, Enclosing_Scope, Context, Item, Is_Visible);
         if In_A_Subprogram and then IsVariable (Item) then
            Is_Visible := False;
         end if;
         exit when Item /= NullSymbol;
         Current_Scope  := Enclosing_Scope;
         Current_Region := GetRegion (Current_Scope);
      end loop;

      if Item = NullSymbol and then Current_Region /= GetPredefinedPackageStandard then
         LookupScope (Name, LexTokenManager.Null_String, PredefinedScope, PredefinedScope, Context, Item, Is_Visible);
      end if;

   end if;

   if not Is_Visible then
      Item := NullSymbol;
   end if;

   Trace_Sym (Msg   => "Found in LookUpItem ",
              Sym   => Item,
              Scope => Scope);
   TraceMsg ("--------------------------------------------------------------------------");
   return Item;

end LookupItem;
