-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler.Conversions.ToString)
procedure DepSemanticErr (Err_Num          : in     Error_Types.NumericError;
                          With_Explanation : in     Boolean;
                          E_Str            : in out E_Strings.T) is

   procedure DepSemanticErrExpl (E_Str : in out E_Strings.T)
   --# global in Err_Num;
   --# derives E_Str from *,
   --#                    Err_Num;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure Append_Explanation
   --# global in     Err_Num;
   --#        in     With_Explanation;
   --#        in out E_Str;
   --# derives E_Str from *,
   --#                    Err_Num,
   --#                    With_Explanation;
   is
      Explanation_String : E_Strings.T := E_Strings.Empty_String;
   begin
      if With_Explanation then
         -- we need to at least look for an explanation
         DepSemanticErrExpl (Explanation_String);
         if E_Strings.Get_Length (E_Str => Explanation_String) > 0 then
            -- there actually is one
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Prefix);
            E_Strings.Append_Examiner_String (E_Str1 => E_Str,
                                              E_Str2 => Explanation_String);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Postfix);
         end if;
      end if;
   end Append_Explanation;

begin
   case Err_Num.ErrorNum is

      -- HTML Directives
      --! <NameFormat> <"flow-"><Name>
      --! <ErrorFormat> <"!!! Flow Error : "><Name><" : "><Error>

      when 1 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The previously stated updating of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has been omitted");
         --! XXX occurred as an export in the earlier dependency relation but
         --! neither XXX nor any refinement constituent of it occurs in the
         --! refined dependency relation.

      when 2 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The updating of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has not been previously stated");
         --! A refinement constituent of XXX occurs as an export in the
         --! refined dependency relation but XXX does not occur as an export in
         --! the earlier dependency relation.

      when 3 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The previously stated dependency of the exported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " on the imported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has been omitted");
         --! The dependency of the exported value of XXX on the imported value
         --! of YYY occurs in the earlier dependency relation but in the refined
         --! dependency relation, no constituents of XXX
         --! depend on any constituents of YYY.

      when 4 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The dependency of the exported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " on the imported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has not been previously stated");
         --! A refined dependency relation states a dependency of XXX or a
         --! constituent of XXX on YYY or a constituent of YYY, but in the
         --! earlier relation, no dependency of XXX on YYY is stated.

      when 5 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The (possibly implicit) dependency of the exported value of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " on its imported value");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has not been previously stated");
         --! Either a dependency of a constituent of XXX on at least one
         --! constituent of XXX occurs in the refined dependency relation, or
         --! not all the constituents of XXX occur as exports in
         --! the refined dependency relation. However, the dependency of XXX on
         --! itself does not occur in the earlier dependency relation.

      when others =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "UNKNOWN ERROR NUMBER PASSED TO DepSemanticErr");
   end case;

   Append_Explanation;
   E_Strings.Append_String (E_Str => E_Str,
                            Str   => ".");
end DepSemanticErr;
