-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_aggregate_choice
  (Node   : in     STree.SyntaxNode;
   Scope  : in     Dictionary.Scopes;
   EStack : in out ExpStack.ExpStackType) is
   type CaseChoiceSorts is (SingleExpression, ExplicitRange, RangeConstraint);

   CaseChoiceSort                     : CaseChoiceSorts;
   First_Node, Second_Node            : STree.SyntaxNode;
   NameExp, FirstResult, SecondResult : Exp_Record;

   SemanticErrorsFound                : Boolean        := False;
   ChoiceLowerMathsValue              : Maths.Value;
   ChoiceUpperMathsValue              : Maths.Value    := Maths.NoValue;  -- actually ineffectual, but stops spurious flowerrs
   IsARange                           : Boolean;
   IndexTypeSymbol                    : Dictionary.Symbol;
   IndexTypeLowerBound                : Typ_Type_Bound;
   IndexTypeUpperBound                : Typ_Type_Bound;
   AggregateFlags                     : Typ_Agg_Flags;
   EntryCounter                       : Natural;
   CompleteRec                        : CompleteCheck.T;
   ChoiceLowerBound, ChoiceUpperBound : Typ_Type_Bound := Unknown_Type_Bound;  -- actually ineffectual, but stops spurious flowerrs

   LowerBoundUnknown         : Boolean;
   UpperBoundUnknown         : Boolean := True;  -- actually ineffectual, but stops spurious flowerrs
   LowerBoundOutOfRange      : Boolean;
   UpperBoundOutOfRange      : Boolean := True;  -- actually ineffectual, but stops spurious flowerrs
   OutOfRangeSeen            : Boolean;
   OverlapSeen               : CompleteCheck.TypOverlapState;
   BothChoiceBoundsKnown     : Boolean := False;  -- actually ineffectual, but stops spurious flowerrs
   RangeConstraintLowerBound : Typ_Type_Bound;
   RangeConstraintUpperBound : Typ_Type_Bound;

   ------------------------------------------------------------------------

   procedure ConvertChoiceBound
     (MathsValue      : in     Maths.Value;
      Bound           :    out Typ_Type_Bound;
      UnknownBound    :    out Boolean;
      BoundOutOfRange :    out Boolean)
   --# derives Bound,
   --#         BoundOutOfRange,
   --#         UnknownBound    from MathsValue;
   is
      Int        : Integer;
      MathsError : Maths.ErrorCode;
   begin
      if Maths.HasNoValue (MathsValue) then
         Bound           := Typ_Type_Bound'(Value      => 0,
                                            Is_Defined => False);
         UnknownBound    := True;
         BoundOutOfRange := False;
      else
         Maths.ValueToInteger (MathsValue, Int, MathsError);
         if MathsError = Maths.NoError then
            Bound           := Typ_Type_Bound'(Value      => Int,
                                               Is_Defined => True);
            UnknownBound    := False;
            BoundOutOfRange := False;
         else
            Bound           := Typ_Type_Bound'(Value      => 0,
                                               Is_Defined => False);
            UnknownBound    := False;
            BoundOutOfRange := True;
         end if;
      end if;
   end ConvertChoiceBound;

   ------------------------------------------------------------------------
   -- note: returns True if any of the bounds is undefined, unless the
   -- choice is not a range, in which case, ChoiceUpper is unused
   function IsChoiceInRange
     (ChoiceLower   : Typ_Type_Bound;
      ChoiceUpper   : Typ_Type_Bound;
      ChoiceIsRange : Boolean;
      RangeLower    : Typ_Type_Bound;
      RangeUpper    : Typ_Type_Bound)
     return          Boolean
   is
      Result : Boolean;
   begin
      if (ChoiceLower.Is_Defined and RangeLower.Is_Defined and ChoiceLower.Value < RangeLower.Value) or
        (ChoiceLower.Is_Defined and RangeUpper.Is_Defined and ChoiceLower.Value > RangeUpper.Value) or
        (ChoiceIsRange and ChoiceUpper.Is_Defined and RangeUpper.Is_Defined and ChoiceUpper.Value > RangeUpper.Value) then
         Result := False;
      else
         Result := True;
      end if;

      return Result;
   end IsChoiceInRange;

   ------------------------------------------------------------------------

   function IsRangeEmpty (RangeLower : Typ_Type_Bound;
                          RangeUpper : Typ_Type_Bound) return Boolean is
      Result : Boolean;
   begin
      if RangeLower.Is_Defined and RangeUpper.Is_Defined and RangeLower.Value > RangeUpper.Value then
         Result := True;
      else
         Result := False;
      end if;

      return Result;
   end IsRangeEmpty;

   -----------------------------------------------------------------------
   procedure ConvertBooleanMathsValue (Value : in out Maths.Value)
   --# derives Value from *;
   is
   begin
      if Value = Maths.FalseValue then
         Value := Maths.ZeroInteger;
      elsif Value = Maths.TrueValue then
         Value := Maths.OneInteger;
      end if;
   end ConvertBooleanMathsValue;

begin  -- up_wf_aggregate_choice

   -- ASSUME Node = aggregate_choice OR annotation_aggregate_choice
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.aggregate_choice
        or else Syntax_Node_Type (Node => Node) = SPSymbols.annotation_aggregate_choice,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = aggregate_choice OR annotation_aggregate_choice in Up_Wf_Aggregate_Choice");
   -- Assume aggregate is array aggregate with named association
   AggregateStack.Pop (IndexTypeSymbol, IndexTypeLowerBound, IndexTypeUpperBound, AggregateFlags, EntryCounter, CompleteRec);

   First_Node  := Child_Node (Node);
   Second_Node := Next_Sibling (First_Node);
   -- ASSUME First_Node = simple_expression OR annotation_simple_expression
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => First_Node) = SPSymbols.simple_expression
        or else Syntax_Node_Type (Node => First_Node) = SPSymbols.annotation_simple_expression,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect First_Node = simple_expression OR annotation_simple_expression in Up_Wf_Aggregate_Choice");
   -- ASSUME Second_Node = range_constraint OR simple_expression OR
   --                      annotation_range_constraint OR annotation_simple_expression OR NULL
   if Second_Node = STree.NullNode then
      -- ASSUME Second_Node = NULL
      CaseChoiceSort := SingleExpression;
   elsif Syntax_Node_Type (Node => Second_Node) = SPSymbols.simple_expression
     or else Syntax_Node_Type (Node => Second_Node) = SPSymbols.annotation_simple_expression then
      -- ASSUME Second_Node = simple_expression OR annotation_simple_expression
      CaseChoiceSort := ExplicitRange;
   elsif Syntax_Node_Type (Node => Second_Node) = SPSymbols.range_constraint
     or else Syntax_Node_Type (Node => Second_Node) = SPSymbols.annotation_range_constraint then
      -- ASSUME Second_Node = range_constraint OR annotation_range_constraint
      CaseChoiceSort := RangeConstraint;
   else
      CaseChoiceSort := SingleExpression;
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Second_Node = range_constraint OR simple_expression OR annotation_range_constraint OR" &
           " annotation_simple_expression OR NULL in Up_Wf_Aggregate_Choice");
   end if;

   --# assert True; -- for RTC generation

   case CaseChoiceSort is
      when SingleExpression =>
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if Dictionary.IsUnknownTypeMark (FirstResult.Type_Symbol) then
            null;
         elsif Dictionary.CompatibleTypes
           (Scope,
            FirstResult.Type_Symbol,
            Dictionary.GetArrayIndex (NameExp.Type_Symbol, NameExp.Param_Count)) then
            if not FirstResult.Is_Static then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 36,
                  Reference => 1,
                  Position  => Node_Position (Node => First_Node),
                  Id_Str    => LexTokenManager.Null_String);
               SemanticErrorsFound := True;
            end if;
         else
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to work out whether we have a single choice or a
         -- range and to collect the appropriate values
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         if FirstResult.Is_ARange then
            IsARange              := True;
            ChoiceUpperMathsValue := FirstResult.Range_RHS;
         else
            IsARange := False;
         end if;
         NameExp.Errors_In_Expression := SemanticErrorsFound or NameExp.Errors_In_Expression or FirstResult.Errors_In_Expression;

      when ExplicitRange =>
         ExpStack.Pop (SecondResult, EStack);
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, SecondResult.Type_Symbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 42,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes
           (Scope,
            FirstResult.Type_Symbol,
            Dictionary.GetArrayIndex (NameExp.Type_Symbol, NameExp.Param_Count)) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 106,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.Is_Static and SecondResult.Is_Static) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 45,
               Reference => 1,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found#
         ChoiceLowerMathsValue        := FirstResult.Value;
         ChoiceUpperMathsValue        := SecondResult.Value;
         IsARange                     := True;
         NameExp.Errors_In_Expression := SemanticErrorsFound or
           NameExp.Errors_In_Expression or
           FirstResult.Errors_In_Expression or
           SecondResult.Errors_In_Expression;

      when RangeConstraint =>
         ExpStack.Pop (SecondResult, EStack);
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, SecondResult.Type_Symbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 106,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes
           (Scope,
            FirstResult.Type_Symbol,
            Dictionary.GetArrayIndex (NameExp.Type_Symbol, NameExp.Param_Count)) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.Is_Constant and FirstResult.Is_ARange) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 95,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not SecondResult.Is_Static then
            ErrorHandler.Semantic_Error
              (Err_Num   => 45,
               Reference => 1,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue        := SecondResult.Value;
         ChoiceUpperMathsValue        := SecondResult.Range_RHS;
         IsARange                     := True;
         NameExp.Errors_In_Expression := SemanticErrorsFound or
           NameExp.Errors_In_Expression or
           FirstResult.Errors_In_Expression or
           SecondResult.Errors_In_Expression;
         -- somewhere need to check that SecondResult range is within the type
         -- given by FirstResult
   end case;

   ExpStack.Push (NameExp, EStack);

   --# assert True;

   if not SemanticErrorsFound then
      ConvertBooleanMathsValue (ChoiceLowerMathsValue);
      ConvertChoiceBound (ChoiceLowerMathsValue, ChoiceLowerBound, LowerBoundUnknown, LowerBoundOutOfRange);
      if IsARange then
         ConvertBooleanMathsValue (ChoiceUpperMathsValue);  -- CUMV always defined here
         ConvertChoiceBound (ChoiceUpperMathsValue, ChoiceUpperBound, UpperBoundUnknown, UpperBoundOutOfRange);
      else
         ChoiceUpperBound := Unknown_Type_Bound;
      end if;

      --# assert True;

      if LowerBoundOutOfRange or
        (IsARange
           and then -- UBOOR always defined here
           UpperBoundOutOfRange) then
         BothChoiceBoundsKnown := False;
         ErrorHandler.Semantic_Warning
           (Err_Num  => 305,
            Position => Node_Position (Node => First_Node),
            Id_Str   => LexTokenManager.Null_String);
      elsif LowerBoundUnknown or  -- UBOOR always defined here
        (IsARange and then UpperBoundUnknown) then
         BothChoiceBoundsKnown      := False;
         CompleteRec.Undeterminable := True;
         ErrorHandler.Semantic_Warning
           (Err_Num  => 200,
            Position => Node_Position (Node => First_Node),
            Id_Str   => LexTokenManager.Null_String);
      else
         BothChoiceBoundsKnown := True;
      end if;

      --# assert True;

      if BothChoiceBoundsKnown then
         -- check the case choice lies within controlling type
         if not IsChoiceInRange (ChoiceLowerBound, ChoiceUpperBound, IsARange, IndexTypeLowerBound, IndexTypeUpperBound) then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 410,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 410,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => First_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         elsif IsARange and IsRangeEmpty (ChoiceLowerBound, ChoiceUpperBound) then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 409,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 409,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => First_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         end if;

         -- check the case choice lies within RangeConstraint type
         if CaseChoiceSort = RangeConstraint then
            Get_Type_Bounds
              (Type_Symbol => FirstResult.Type_Symbol,
               Lower_Bound => RangeConstraintLowerBound,
               Upper_Bound => RangeConstraintUpperBound);

            if not IsChoiceInRange
              (ChoiceLowerBound,
               ChoiceUpperBound,
               IsARange,
               RangeConstraintLowerBound,
               RangeConstraintUpperBound) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 413,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
               SemanticErrorsFound := True;
            end if;
         end if;
      end if;
   end if;

   --# assert True;

   if (not SemanticErrorsFound) -- BCBK always defined if we reach it
     and then BothChoiceBoundsKnown
     and then (AggregateFlags.Check_Completeness or AggregateFlags.Check_Overlap) then
      if IsARange then
         CompleteCheck.SeenRange (CompleteRec, ChoiceLowerBound.Value, ChoiceUpperBound.Value, OutOfRangeSeen, OverlapSeen);
      else
         CompleteCheck.SeenElement (CompleteRec, -- flow error expected
                                    ChoiceLowerBound.Value, OutOfRangeSeen, OverlapSeen);
      end if;
      if OutOfRangeSeen then
         AggregateFlags.Out_Of_Range_Seen := True;
      end if;
      if AggregateFlags.Check_Overlap and OverlapSeen = CompleteCheck.Overlap then
         ErrorHandler.Semantic_Error
           (Err_Num   => 407,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => First_Node),
            Id_Str    => LexTokenManager.Null_String);
         SemanticErrorsFound := True;
      end if;
   end if;

   --# assert True;

   if SemanticErrorsFound then
      AggregateFlags.Check_Completeness := False;
   end if;

   AggregateStack.Push (IndexTypeSymbol, IndexTypeLowerBound, IndexTypeUpperBound, AggregateFlags, EntryCounter, CompleteRec);

end up_wf_aggregate_choice;
