-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_case (Node  : in STree.SyntaxNode;
                   Scope : in Dictionary.Scopes) is
   CaseExp     : Exp_Record;
   RefVar      : SeqAlgebra.Seq;
   CaseFlags   : Typ_Case_Flags;
   UpperBound  : Typ_Type_Bound;
   LowerBound  : Typ_Type_Bound;
   CompleteADT : CompleteCheck.T;

   CompleteCheckRangeFrom  : Integer;
   CompleteCheckRangeTo    : Integer;
   CompleteCheckRangeState : CompleteCheck.TypRangeState;

begin
   -- ASSUME Node = case_statement
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.case_statement,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = case_statement in Wf_Case");

   SeqAlgebra.CreateSeq (TheHeap, RefVar);
   WalkExpression
     (Exp_Node                => Child_Node (Current_Node => Node),
      Scope                   => Scope,
      Type_Context            => Dictionary.GetUnknownTypeMark,
      Context_Requires_Static => False,
      Result                  => CaseExp,
      Ref_Var                 => RefVar,
      Component_Data          => GlobalComponentData);
   -- distinguish between the different possible situations, and
   -- set up the case checking accordingly
   if Dictionary.IsUnknownTypeMark (CaseExp.Type_Symbol) or not Dictionary.IsDiscreteTypeMark (CaseExp.Type_Symbol, Scope) then
      UpperBound := Unknown_Type_Bound;
      LowerBound := Unknown_Type_Bound;

      -- for unknown or non-discrete types
      -- for unknown types still attempt overlap checking
      ErrorHandler.Semantic_Error
        (Err_Num   => 46,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Node),
         Id_Str    => LexTokenManager.Null_String);
      CaseFlags :=
        Typ_Case_Flags'
        (Check_Completeness  => False,
         Signal_Out_Of_Range => False,
         Out_Of_Range_Seen   => False,
         Check_Overlap       => Dictionary.IsUnknownTypeMark (CaseExp.Type_Symbol),
         Warn_No_Others      => False,
         Others_Mandatory    => False);
      -- the completeness checker object will not be used if the type mark
      -- is not discrete
      CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
      CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
      --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
      --   so the value is ignored, giving a flow error
      --# accept Flow, 10, CompleteCheckRangeState, "Expected ineffective assignment";
      CompleteCheck.Init (CompleteADT, -- expect flow error
                          CompleteCheckRangeFrom, CompleteCheckRangeTo, CompleteCheckRangeState);

   elsif Dictionary.IsUniversalIntegerType (CaseExp.Type_Symbol) then
      UpperBound := Unknown_Type_Bound;
      LowerBound := Unknown_Type_Bound;
      -- for universal Integer: others is mandatory
      CaseFlags              :=
        Typ_Case_Flags'
        (Check_Completeness  => False,
         Signal_Out_Of_Range => True,
         Out_Of_Range_Seen   => False,
         Check_Overlap       => True,
         Warn_No_Others      => False,
         Others_Mandatory    => True);
      CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
      CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
      --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
      --   so the value is ignored, giving a flow error
      CompleteCheck.Init (CompleteADT, -- expect flow error
                          CompleteCheckRangeFrom, CompleteCheckRangeTo, CompleteCheckRangeState);

   else
      -- get bounds from dictionary
      Get_Type_Bounds (Type_Symbol => CaseExp.Type_Symbol,
                       Lower_Bound => LowerBound,
                       Upper_Bound => UpperBound);

      if not (LowerBound.Is_Defined and UpperBound.Is_Defined) then
         -- one or other bound is unknown to the dictionary
         CaseFlags :=
           Typ_Case_Flags'
           (Check_Completeness  => False,
            Signal_Out_Of_Range => True,
            Out_Of_Range_Seen   => False,
            Check_Overlap       => True,
            Warn_No_Others      => True,
            Others_Mandatory    => False);
         -- if both bounds unknown use symmetric range
         if (not LowerBound.Is_Defined) and (not UpperBound.Is_Defined) then
            CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
            CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
            -- otherwise use range extending from known bound
         elsif LowerBound.Is_Defined then
            CompleteCheckRangeFrom := LowerBound.Value;
            CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
         else  -- UpperBound.IsDefined
            CompleteCheckRangeTo   := UpperBound.Value;
            CompleteCheckRangeFrom := (CompleteCheckRangeTo - ExaminerConstants.CompleteCheckSize) + 1;
         end if;
         --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
         --   so the value is ignored, giving a flow error
         CompleteCheck.Init (CompleteADT, -- expect flow error
                             CompleteCheckRangeFrom, CompleteCheckRangeTo, CompleteCheckRangeState);
         --# end accept;

      else -- both bounds known to dictionary: set up completeness checker
         CompleteCheck.Init (CompleteADT, LowerBound.Value, UpperBound.Value, CompleteCheckRangeState);
         if CompleteCheckRangeState = CompleteCheck.RangeDoesFit then
            -- range fits in completeness checker
            CaseFlags :=
              Typ_Case_Flags'
              (Check_Completeness  => True,
               Signal_Out_Of_Range => False,
               Out_Of_Range_Seen   => False,
               Check_Overlap       => True,
               Warn_No_Others      => False,
               Others_Mandatory    => False);
         else -- range does not fit in completeness checker
            CaseFlags :=
              Typ_Case_Flags'
              (Check_Completeness  => False,
               Signal_Out_Of_Range => True,
               Out_Of_Range_Seen   => False,
               Check_Overlap       => True,
               Warn_No_Others      => True,
               Others_Mandatory    => False);
         end if;
      end if;
   end if;

   Case_Stack.Push (CaseFlags, CompleteADT, CaseExp.Type_Symbol, LowerBound, UpperBound);

   -- add reference variable list to RefList hash table
   RefList.AddRelation (Table, TheHeap, Child_Node (Node), Dictionary.NullSymbol, RefVar);
end wf_case;
