-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_delay_until (Node  : in STree.SyntaxNode;
                          Scope : in Dictionary.Scopes) is
   EncUnitSym : Dictionary.Symbol;
   ResultType : Exp_Record;
   RefVar     : SeqAlgebra.Seq;
begin
   -- ASSUME Node = delay_statement
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.delay_statement,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = delay_statement in Wf_Delay_Until");

   SeqAlgebra.CreateSeq (TheHeap, RefVar);

   -- Process the expression of the delay_statement. Must be of type Ada.Real_Time.
   WalkExpression
     (Exp_Node                => Next_Sibling (Current_Node => Child_Node (Current_Node => Node)),
      Scope                   => Scope,
      Type_Context            => Dictionary.GetPredefinedTimeType,
      Context_Requires_Static => False,
      Result                  => ResultType,
      Ref_Var                 => RefVar,
      Component_Data          => GlobalComponentData);

   -- Verify that ResultType is compatible with Ada.Real_Time
   if not Dictionary.CompatibleTypes
     (Scope => Scope,
      Left  => ResultType.Type_Symbol,
      Right => Dictionary.GetPredefinedTimeType) then
      ErrorHandler.Semantic_Error
        (Err_Num   => 866,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Next_Sibling (Child_Node (Node))),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   -- Verify that if the enclosing unit is a procedure then
   -- it has a delay annotation. Other enclosing units are invalid.
   -- Also mark the enclosing unit as having seen a delay statement.
   EncUnitSym := Dictionary.GetEnclosingCompilationUnit (Scope);

   if Dictionary.IsProcedure (EncUnitSym) and not Dictionary.IsOrIsInProtectedScope (Scope) then

      if Dictionary.HasDelayProperty (EncUnitSym) then
         Dictionary.MarkAccountsForDelay (EncUnitSym);

      else
         ErrorHandler.Semantic_Error
           (Err_Num   => 867,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Dictionary.GetSimpleName (EncUnitSym));
      end if;

   elsif not Dictionary.IsTaskType (EncUnitSym) then
      -- Cannot call delay from the enclosing construct.
      ErrorHandler.Semantic_Error
        (Err_Num   => 907,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Node),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   -- NullVariable is an out stream and so need to be considered an import
   AddStreamEffects (Table, TheHeap, Node, Dictionary.GetNullVariable, RefVar);
   -- add reference variable list to RefList hash table for the statement node
   RefList.AddRelation (Table, TheHeap, Node, Dictionary.GetNullVariable, -- ref vars dumped in data sink
                        RefVar);
end wf_delay_until;
