-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_return (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes) is
   SubProgSym, ExpectedType : Dictionary.Symbol;
   ResultType               : Exp_Record;
   LocalNode                : STree.SyntaxNode;
   RefVar                   : SeqAlgebra.Seq;

begin
   -- ASSUME Node = return_statement
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.return_statement,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = return_statement in Wf_Return");

   -- conditions of Section 5.8 of the SPARK definition apply to this statement.

   SeqAlgebra.CreateSeq (TheHeap, RefVar);
   SubProgSym := Dictionary.GetEnclosingCompilationUnit (Scope);

   if Dictionary.IsProcedure (SubProgSym) then
      ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Return_In_Proc,
                                       Position => Node_Position (Node => Node));

   else --subprogram is a function, check location of return within it;
      LocalNode := ParentOfSequence (Node);
      case Syntax_Node_Type (Node => LocalNode) is
         when SPSymbols.if_statement               |
           SPSymbols.elsif_part                 |
           SPSymbols.else_part                  |
           SPSymbols.loop_statement             |
           SPSymbols.case_statement_alternative |
           SPSymbols.others_part                =>
            ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Misplaced_Return,
                                             Position => Node_Position (Node => Node));
         when others =>
            if not IsLastInSequence (Node) then
               ErrorHandler.Control_Flow_Error
                 (Err_Type => ErrorHandler.Misplaced_Return,
                  Position => Node_Position (Node => Node));
            end if;
      end case;
      -- seed syntax tree with expected type for run-time check
      ExpectedType := Dictionary.GetType (SubProgSym);
      STree.AddNodeSymbol (Node, ExpectedType);

      -- now check return type is correct
      WalkExpression
        (Exp_Node                => Child_Node (Current_Node => Node),
         Scope                   => Scope,
         Type_Context            => ExpectedType,
         Context_Requires_Static => False,
         Result                  => ResultType,
         Ref_Var                 => RefVar,
         Component_Data          => GlobalComponentData);
      AssignmentCheck (Node_Position (Node => Node), Scope, ExpectedType, ResultType);

      -- if expression represents an IN stream variable then issue warning about
      -- possible validity problems if it is not marked valid
      if ResultType.Is_AVariable
        and then Dictionary.GetOwnVariableOrConstituentMode (ResultType.Variable_Symbol) = Dictionary.InMode then
         -- Check to see if the variable has been marked valid. Note that the
         -- OtherSymbol is checked,not the variableSymbol, since this will be
         -- the Subcomponent symbol if we are referring to a record component
         if Dictionary.VariableOrSubcomponentIsMarkedValid (ResultType.Other_Symbol) then
            --MCA: do we also need to add a use of 'Valid to the summary?
            -- Debug.PrintSym ("Return Access is Valid =", ResultType.OtherSymbol);
            null;
         else
            -- The warning is stronger when the external variable is a type that doesn't
            -- generate run-time checks
            if Dictionary.TypeIsScalar (ResultType.Type_Symbol)
              and then not Dictionary.TypeIsBoolean (ResultType.Type_Symbol) then
               --weaker warning
               ErrorHandler.Semantic_Warning_Sym
                 (Err_Num  => 392,
                  Position => Node_Position (Node => Child_Node (Node)),
                  Sym      => ResultType.Other_Symbol,
                  Scope    => Scope);
            else
               --stronger warning
               ErrorHandler.Semantic_Warning_Sym
                 (Err_Num  => 393,
                  Position => Node_Position (Node => Child_Node (Node)),
                  Sym      => ResultType.Other_Symbol,
                  Scope    => Scope);
            end if;
         end if;
      end if;
   end if;
   -- add reference variable list to RefList hash table
   RefList.AddRelation (Table, TheHeap, Node, Dictionary.NullSymbol, RefVar);
end wf_return;
