-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_entire_variable
  (Node         : in     STree.SyntaxNode;
   CurrentScope : in     Dictionary.Scopes;
   ErrorHint    : in     Visibility_Error_Hint;
   VarSym       :    out Dictionary.Symbol;
   Dotted       :    out Boolean) is
   IdNode, NextIdNode : STree.SyntaxNode;
   It                 : STree.Iterator;
   PIdStr, IdStr      : LexTokenManager.Lex_String;
   Sym, SymSoFar      : Dictionary.Symbol;
   PrefixOk           : Boolean;

begin
   -- Node is the root of any variable (dotted or otherwise).
   -- ASSUME Node = entire_variable OR annotation_primary
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.entire_variable
        or else Syntax_Node_Type (Node => Node) = SPSymbols.annotation_primary,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = entire_variable OR annotation_primary in Wf_Entire_Variable");
   -- This procedure traverses (depth first) the tree under the root looking for
   -- identifiers. VarSym is made up of all the identifiers found.

   It := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                          From_Root    => Node,
                          In_Direction => STree.Down);

   IdNode := Get_Node (It => It);
   IdStr  := Node_Lex_String (Node => IdNode);
   PIdStr := LexTokenManager.Null_String;
   Sym    := Dictionary.LookupItem (Name              => IdStr,
                                    Scope             => CurrentScope,
                                    Context           => Dictionary.ProofContext,
                                    Full_Package_Name => False);
   loop
      if Sym = Dictionary.NullSymbol then

         case ErrorHint is
            when No_Hint =>
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 1,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str1   => IdStr,
                  Id_Str2   => PIdStr);
            when In_Global_List =>
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 144,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str1   => IdStr,
                  Id_Str2   => PIdStr);
            when In_Derives_Import_List =>
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 752,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str1   => IdStr,
                  Id_Str2   => PIdStr);
            when In_Derives_Export_List =>
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 753,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str1   => IdStr,
                  Id_Str2   => PIdStr);
            when In_Suspends_List =>
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 755,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str1   => IdStr,
                  Id_Str2   => PIdStr);
         end case;

         exit;
      end if;

      It         := STree.NextNode (It);
      NextIdNode := Get_Node (It => It);

      if Dictionary.IsVariable (Sym) then
         -- at this point Sym is a variable, final check that there is no dotted
         -- part to the right of it as there would be if a record field was there
         if NextIdNode /= STree.NullNode then
            ErrorHandler.Semantic_Error
              (Err_Num   => 156,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         else
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdNode);
         end if;
         exit;
      end if;

      if Dictionary.IsConstant (Sym) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 150,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      if not Dictionary.IsPackage (Sym) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 156,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      if NextIdNode = STree.NullNode then
         -- package without a selected component
         ErrorHandler.Semantic_Error
           (Err_Num   => 156,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      Check_Package_Prefix (Node     => IdNode,
                            Pack_Sym => Sym,
                            Scope    => CurrentScope,
                            OK       => PrefixOk);
      if not PrefixOk then
         Sym := Dictionary.NullSymbol;
         exit;
      end if;
      STree.Set_Node_Lex_String (Sym  => Sym,
                                 Node => IdNode);
      PIdStr   := IdStr;
      IdNode   := NextIdNode;
      IdStr    := Node_Lex_String (Node => IdNode);
      SymSoFar := Sym;
      Sym      :=
        Dictionary.LookupSelectedItem
        (Prefix   => Sym,
         Selector => IdStr,
         Scope    => CurrentScope,
         Context  => Dictionary.ProofContext);
      -- check to see if we are getting the same symbol over and again
      if Sym = SymSoFar then            -- P.P.P.P.X case
         Sym := Dictionary.NullSymbol;  -- to cause "Not visible" error at top of loop
      end if;

   end loop;

   Dotted :=
     LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => PIdStr,
                                                          Lex_Str2 => LexTokenManager.Null_String) /=
     LexTokenManager.Str_Eq
     and then Sym /= Dictionary.NullSymbol;
   VarSym := Sym;
end wf_entire_variable;
