-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit.Wf_Full_Type_Declaration)
procedure wf_real
  (Node       : in STree.SyntaxNode;
   Scope      : in Dictionary.Scopes;
   Ident_Node : in STree.SyntaxNode;
   DecLoc     : in LexTokenManager.Token_Position) is
   ChildNode : STree.SyntaxNode;
   NodeType  : SPSymbols.SPSymbol;

   ------------------------------------------------------------------------

   procedure CheckAccuracy
     (Node       : in     STree.SyntaxNode;
      Scope      : in     Dictionary.Scopes;
      IsFloating : in     Boolean;
      HasRange   : in     Boolean;
      Accuracy   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     DecLoc;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives Accuracy                   from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsFloating,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         HasRange,
   --#                                         IsFloating,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      AccuracyType        : Exp_Record;
      UnusedComponentData : ComponentManager.ComponentData;
      UnwantedSeq         : SeqAlgebra.Seq;

      -------------------------------

      function TypeCorrect
        (TypeSym    : Dictionary.Symbol;
         Scope      : Dictionary.Scopes;
         IsFloating : Boolean)
        return       Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean;

      begin
         if IsFloating then
            Result := Dictionary.IsIntegerTypeMark (TypeSym, Scope);
         else
            Result := Dictionary.IsRealTypeMark (TypeSym, Scope);
         end if;
         return Result or Dictionary.IsUnknownTypeMark (TypeSym);
      end TypeCorrect;

      function ErrorToRaise (IsFloating : Boolean) return Natural is
         E : Natural;
      begin
         if IsFloating then
            E := 108;
         else
            E := 109;
         end if;
         return E;
      end ErrorToRaise;

      -------------------------------

      procedure CheckAgainstSystemConstants
      --# global in     AccuracyType;
      --#        in     CommandLineData.Content;
      --#        in     DecLoc;
      --#        in     Dictionary.Dict;
      --#        in     HasRange;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from AccuracyType,
      --#                                         CommandLineData.Content,
      --#                                         DecLoc,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         HasRange,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys;
      is
         SystemSym        : Dictionary.Symbol;
         MaxDigitsSym     : Dictionary.Symbol;
         MaxBaseDigitsSym : Dictionary.Symbol;
         UnwantedME       : Maths.ErrorCode;
         MaxVal           : LexTokenManager.Lex_String;
         Result           : Maths.Value;
      begin

         case CommandLineData.Content.Language_Profile is
            when CommandLineData.SPARK83 =>
               -- No System in SPARK83, so nothing can be done here
               null;

            when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

               -- see if package System exists...
               SystemSym :=
                 Dictionary.LookupItem
                 (Name              => LexTokenManager.System_Token,
                  Scope             => Dictionary.GlobalScope,
                  Context           => Dictionary.ProgramContext,
                  Full_Package_Name => False);
               if SystemSym /= Dictionary.NullSymbol then
                  if not HasRange then
                     -- if so, and we're an unranged fp type then see if
                     -- System.Max_Digits exists ...
                     MaxDigitsSym :=
                       Dictionary.LookupSelectedItem
                       (Prefix   => SystemSym,
                        Selector => LexTokenManager.Max_Digits_Token,
                        Scope    => Dictionary.GetScope (SystemSym),
                        Context  => Dictionary.ProgramContext);
                     if MaxDigitsSym /= Dictionary.NullSymbol then
                        -- if it does...
                        MaxVal := Dictionary.GetValue (MaxDigitsSym);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => MaxVal,
                           Lex_Str2 => LexTokenManager.Null_String) /=
                          LexTokenManager.Str_Eq then
                           -- and if it has a sensible value, then check
                           -- that the requested number of digits does not
                           -- exceed it.
                           --# accept Flow, 10, UnwantedME, "Expected ineffective assignment";
                           Maths.Greater (AccuracyType.Value, Maths.ValueRep (MaxVal), Result, UnwantedME);
                           if Result = Maths.BoolToValue (True) then
                              ErrorHandler.Semantic_Error
                                (Err_Num   => 785,
                                 Reference => ErrorHandler.No_Reference,
                                 Position  => DecLoc,
                                 Id_Str    => LexTokenManager.Null_String);
                           end if;
                        end if;
                     end if;
                  else
                     -- if so, and we're a ranged fp type then see if
                     -- System.Max_Base_Digits exists ...
                     MaxBaseDigitsSym :=
                       Dictionary.LookupSelectedItem
                       (Prefix   => SystemSym,
                        Selector => LexTokenManager.Max_Base_Digits_Token,
                        Scope    => Dictionary.GetScope (SystemSym),
                        Context  => Dictionary.ProgramContext);
                     if MaxBaseDigitsSym /= Dictionary.NullSymbol then
                        -- if it does...
                        MaxVal := Dictionary.GetValue (MaxBaseDigitsSym);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => MaxVal,
                           Lex_Str2 => LexTokenManager.Null_String) /=
                          LexTokenManager.Str_Eq then
                           -- and if it has a sensible value, then check
                           -- that the requested number of digits does not
                           -- exceed it.
                           Maths.Greater (AccuracyType.Value, Maths.ValueRep (MaxVal), Result, UnwantedME);
                           if Result = Maths.BoolToValue (True) then
                              ErrorHandler.Semantic_Error
                                (Err_Num   => 786,
                                 Reference => ErrorHandler.No_Reference,
                                 Position  => DecLoc,
                                 Id_Str    => LexTokenManager.Null_String);
                           end if;
                        end if;
                     end if;
                  end if;
               end if;
         end case;
         --# accept Flow, 33, UnwantedME, "Expected to be neither referenced nor exported";
      end CheckAgainstSystemConstants;

      procedure CheckAccuracyIsPositive
      --# global in     AccuracyType;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     IsFloating;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from AccuracyType,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         IsFloating,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      is
         T   : Maths.Value;
         Err : Natural;
      begin
         T := AccuracyType.Value;
         Maths.Absolute (T);
         if (AccuracyType.Value = Maths.ZeroReal)
           or else (AccuracyType.Value = Maths.ZeroInteger)
           or else (AccuracyType.Value /= T) then

            if IsFloating then
               Err := 787;
            else
               Err := 788;
            end if;
            ErrorHandler.Semantic_Error
              (Err_Num   => Err,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end CheckAccuracyIsPositive;

   begin -- CheckAccuracy
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression
        (Exp_Node                => Node,
         Scope                   => Scope,
         Type_Context            => Dictionary.GetUnknownTypeMark,
         Context_Requires_Static => True,
         Result                  => AccuracyType,
         Ref_Var                 => UnwantedSeq,
         Component_Data          => UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Maths.StorageRep (AccuracyType.Value, Accuracy);
      if not TypeCorrect (AccuracyType.Type_Symbol, Scope, IsFloating) then
         Accuracy := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => ErrorToRaise (IsFloating),
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      if AccuracyType.Is_Static then
         if AccuracyType.Is_ARange then
            ErrorHandler.Semantic_Error
              (Err_Num   => 114,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
         else
            CheckAccuracyIsPositive;
         end if;
      else
         ErrorHandler.Semantic_Error
           (Err_Num   => 36,
            Reference => 1,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
      if IsFloating then
         -- Check against the target configuration file;
         -- System.Max_Digits or Max_Base_Digits may be defined.
         CheckAgainstSystemConstants;
      end if;
   end CheckAccuracy;

   ------------------------------------------------------------------------

   procedure CheckRangeItem
     (Node       : in     STree.SyntaxNode;
      Scope      : in     Dictionary.Scopes;
      IsFloating : in     Boolean;
      Static     :    out Boolean;
      BoundVal   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Static,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         BoundVal                   from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsFloating,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsFloating,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      RangeType           : Exp_Record;
      UnusedComponentData : ComponentManager.ComponentData;
      UnwantedSeq         : SeqAlgebra.Seq;

      -------------------------------

      function TypeCorrect
        (TypeSym    : Dictionary.Symbol;
         Scope      : Dictionary.Scopes;
         IsFloating : Boolean)
        return       Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean;

      begin
         if IsFloating then
            Result := Dictionary.IsRealTypeMark (TypeSym, Scope);
         else
            Result := (Dictionary.IsUniversalRealType (TypeSym) or Dictionary.IsFixedPointTypeMark (TypeSym, Scope));
         end if;
         return Result or Dictionary.IsUnknownTypeMark (TypeSym);
      end TypeCorrect;

      -------------------------------

   begin -- CheckRangeItem
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression
        (Exp_Node                => Node,
         Scope                   => Scope,
         Type_Context            => Dictionary.GetUnknownTypeMark,
         Context_Requires_Static => True,
         Result                  => RangeType,
         Ref_Var                 => UnwantedSeq,
         Component_Data          => UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Maths.StorageRep (RangeType.Value, BoundVal);
      if not TypeCorrect (RangeType.Type_Symbol, Scope, IsFloating) then
         BoundVal := LexTokenManager.Null_String;
         ErrorHandler.Semantic_Error
           (Err_Num   => 38,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
      Static := RangeType.Is_Static;
   end CheckRangeItem;

   ------------------------------------------------------------------------

   function AttributeFound (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
   begin
      return Syntax_Node_Type (Node => Child_Node (Child_Node (Node))) = SPSymbols.attribute;
   end AttributeFound;

   ------------------------------------------------------------------------

   procedure wf_floating_point_constraint
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     DecLoc;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      AccuracyNode, RangeNode     : STree.SyntaxNode;
      LeftIsStatic, RightIsStatic : Boolean;
      Accuracy                    : LexTokenManager.Lex_String;
      Upper                       : LexTokenManager.Lex_String := LexTokenManager.Null_String;
      Lower                       : LexTokenManager.Lex_String := LexTokenManager.Null_String;
      Type_Symbol                 : Dictionary.Symbol;
   begin  --wf_floating_point_constraint
      AccuracyNode := Child_Node (Node);
      RangeNode    := Next_Sibling (AccuracyNode);

      CheckAccuracy (Child_Node (AccuracyNode), Scope, True, RangeNode /= STree.NullNode, Accuracy);

      if RangeNode /= STree.NullNode then
         if AttributeFound (RangeNode) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 98,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => RangeNode),
               Id_Str    => LexTokenManager.Null_String);
         else
            --# accept Flow, 10, AggregateStack.State, "Expected ineffective assignment";
            CheckRangeItem ( -- Ineffective AS.S, ES.S, LH OK
                             Child_Node (Child_Node (RangeNode)), Scope, True,
                             --to get
                             LeftIsStatic, Lower);
            --# end accept;

            CheckRangeItem (Next_Sibling (Child_Node (Child_Node (RangeNode))), Scope, True,
                            --to get
                            RightIsStatic, Upper);

            if not (LeftIsStatic and RightIsStatic) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 45,
                  Reference => 1,
                  Position  => Node_Position (Node => RangeNode),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;
      end if;

      EmptyTypeCheck (DecLoc, Lower, Upper);
      Dictionary.AddFloatingPointType
        (Name        => Node_Lex_String (Node => Ident_Node),
         Comp_Unit   => ContextManager.Ops.Current_Unit,
         Declaration => Dictionary.Location'(Start_Position => DecLoc,
                                             End_Position   => DecLoc),
         Lower       => Lower,
         Upper       => Upper,
         ErrorBound  => Accuracy,
         Scope       => Scope,
         Context     => Dictionary.ProgramContext,
         Type_Symbol => Type_Symbol);
      if ErrorHandler.Generate_SLI then
         SLI.Generate_Xref_Symbol
           (Comp_Unit      => ContextManager.Ops.Current_Unit,
            Parse_Tree     => Ident_Node,
            Symbol         => Type_Symbol,
            Is_Declaration => True);
      end if;
   end wf_floating_point_constraint;

   ------------------------------------------------------------------------

   procedure wf_fixed_point_constraint
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     DecLoc;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      AccuracyNode, RangeNode     : STree.SyntaxNode;
      LeftIsStatic, RightIsStatic : Boolean;
      Accuracy                    : LexTokenManager.Lex_String;
      Upper                       : LexTokenManager.Lex_String := LexTokenManager.Null_String;
      Lower                       : LexTokenManager.Lex_String := LexTokenManager.Null_String;
      Type_Symbol                 : Dictionary.Symbol;
   begin
      AccuracyNode := Child_Node (Node);
      RangeNode    := Next_Sibling (AccuracyNode);
      if RangeNode = STree.NullNode then
         ErrorHandler.Semantic_Error
           (Err_Num   => 57,
            Reference => ErrorHandler.No_Reference,
            Position  => DecLoc,
            Id_Str    => LexTokenManager.Null_String);
      end if;

      CheckAccuracy (Child_Node (AccuracyNode), Scope, False, False, -- ignored
                     Accuracy);

      if RangeNode /= STree.NullNode then
         if AttributeFound (RangeNode) then
            Lower := LexTokenManager.Null_String;
            Upper := LexTokenManager.Null_String;
            ErrorHandler.Semantic_Error
              (Err_Num   => 98,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => RangeNode),
               Id_Str    => LexTokenManager.Null_String);
         else
            --# accept Flow, 10, AggregateStack.State, "Expected ineffective assignment";
            CheckRangeItem (  -- Ineffective AS.S, ES.S, LH OK
                              Child_Node (Child_Node (RangeNode)), Scope, True,
                              --to get
                              LeftIsStatic, Lower);
            --# end accept;

            CheckRangeItem (Next_Sibling (Child_Node (Child_Node (RangeNode))), Scope, True,
                            --to get
                            RightIsStatic, Upper);

            if not (LeftIsStatic and RightIsStatic) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 45,
                  Reference => 1,
                  Position  => Node_Position (Node => RangeNode),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;
      end if;

      EmptyTypeCheck (DecLoc, Lower, Upper);
      Dictionary.AddFixedPointType
        (Name        => Node_Lex_String (Node => Ident_Node),
         Comp_Unit   => ContextManager.Ops.Current_Unit,
         Declaration => Dictionary.Location'(Start_Position => DecLoc,
                                             End_Position   => DecLoc),
         Lower       => Lower,
         Upper       => Upper,
         ErrorBound  => Accuracy,
         Scope       => Scope,
         Context     => Dictionary.ProgramContext,
         Type_Symbol => Type_Symbol);
      if ErrorHandler.Generate_SLI then
         SLI.Generate_Xref_Symbol
           (Comp_Unit      => ContextManager.Ops.Current_Unit,
            Parse_Tree     => Ident_Node,
            Symbol         => Type_Symbol,
            Is_Declaration => True);
      end if;
   end wf_fixed_point_constraint;

   ------------------------------------------------------------------------

begin   --wf_real_type_definition
   ChildNode := Child_Node (Node);
   NodeType  := Syntax_Node_Type (Node => ChildNode);

   if NodeType = SPSymbols.fixed_point_constraint then
      wf_fixed_point_constraint (Node       => ChildNode,
                                 Scope      => Scope,
                                 Ident_Node => Ident_Node);
   else
      wf_floating_point_constraint (Node       => ChildNode,
                                    Scope      => Scope,
                                    Ident_Node => Ident_Node);
   end if;
   Heap.ReportUsage (TheHeap);

end wf_real;
