-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit.Wf_Full_Type_Declaration)
procedure Wf_Task_Type_Declaration (Node  : in STree.SyntaxNode;
                                    Scope : in Dictionary.Scopes) is
   Task_Type_Sym      : Dictionary.Symbol;
   Task_Scope         : Dictionary.Scopes;
   Sym                : Dictionary.Symbol;
   Ident_Node         : STree.SyntaxNode;
   Anno_Node          : STree.SyntaxNode;
   Closing_Ident_Node : STree.SyntaxNode;
   Discriminant_Node  : STree.SyntaxNode;
   Ident_Str          : LexTokenManager.Lex_String;
   Pragma_Node        : STree.SyntaxNode;
   Global_Error       : Boolean;
   Derives_Error      : Boolean := False;

   -----------------------------------------------------------------------------------
   --
   -- Grammar:
   -- task_type_declaration
   --         |
   --     identifier -- [known_discriminant_part] -- task_type_annotation -- task_definition
   --                                                         |                    |
   --               +-----------------------------------------+                    |
   --               |                                                              |
   --  moded_global_definition -- [dependency_relation] -- [declare_annotation]    |
   --                                                              |               |
   --                      +---------------------------------------+               |
   --                      |                                                       |
   --                property_list                                                 |
   --                      |...                                                    |
   --                property_rep                                                  |
   --                      |                                                       |
   --                  property                                                    |
   --                      |                                                   task_pragma -- identifier
   --              name_value_property  or  delay_property                           |...
   --                      |                      |                            task_pragma -- apragma
   --  identifier -- global_variable_list or     RWDelay
   --  identifier -- integer_number                                            priority_pragma
   --
   --
   --

   function Get_Discriminant_Node (Task_Type_Declaration_Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      -- ASSUME Task_Type_Declaration_Node = task_type_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Task_Type_Declaration_Node) = SPSymbols.task_type_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Task_Type_Declaration_Node = task_type_declaration in Get_Discriminant_Node");
      return Next_Sibling (Current_Node => Child_Node (Current_Node => Task_Type_Declaration_Node));
   end Get_Discriminant_Node;

   ----------

   function Get_Anno_Node (Task_Type_Declaration_Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   is
      Result : STree.SyntaxNode;
   begin
      -- ASSUME Task_Type_Declaration_Node = task_type_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Task_Type_Declaration_Node) = SPSymbols.task_type_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Task_Type_Declaration_Node = task_type_declaration in Get_Anno_Node");
      Result := Get_Discriminant_Node (Task_Type_Declaration_Node => Task_Type_Declaration_Node);
      -- ASSUME Result = known_discriminant_part OR task_type_annotation
      if Syntax_Node_Type (Node => Result) = SPSymbols.known_discriminant_part then
         Result := Next_Sibling (Current_Node => Result);
      elsif Syntax_Node_Type (Node => Result) /= SPSymbols.task_type_annotation then
         Result := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Result = known_discriminant_part OR task_type_annotation in Get_Anno_Node");
      end if;
      -- ASSUME Result = task_type_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Result) = SPSymbols.task_type_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Result = task_type_annotation in Get_Anno_Node");
      return Child_Node (Current_Node => Result);
   end Get_Anno_Node;

   ----------

   function Get_Task_Definition_Node (Task_Type_Declaration_Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      -- ASSUME Task_Type_Declaration_Node = task_type_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Task_Type_Declaration_Node) = SPSymbols.task_type_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Task_Type_Declaration_Node = task_type_declaration in Get_Task_Definition_Node");
      return Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Task_Type_Declaration_Node));
   end Get_Task_Definition_Node;

   ----------

   function Get_Closing_Ident_Node (Task_Type_Declaration_Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      -- ASSUME Task_Type_Declaration_Node = task_type_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Task_Type_Declaration_Node) = SPSymbols.task_type_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Task_Type_Declaration_Node = task_type_declaration in Get_Closing_Ident_Node");
      return Next_Sibling
        (Current_Node => Child_Node
           (Current_Node => Get_Task_Definition_Node (Task_Type_Declaration_Node => Task_Type_Declaration_Node)));
   end Get_Closing_Ident_Node;

   ----------

   function Get_Priority_Pragma_Node (Task_Type_Declaration_Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      -- ASSUME Task_Type_Declaration_Node = task_type_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Task_Type_Declaration_Node) = SPSymbols.task_type_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Task_Type_Declaration_Node = task_type_declaration in Get_Priority_Pragma_Node");
      return Parent_Node
        (Current_Node => Last_Child_Of
           (Start_Node => Get_Task_Definition_Node (Task_Type_Declaration_Node => Task_Type_Declaration_Node)));
   end Get_Priority_Pragma_Node;

   ----------

   procedure Check_Pragma_Validity (End_Node_Position : in LexTokenManager.Token_Position;
                                    Task_Type_Sym     : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         End_Node_Position,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Task_Type_Sym;
   is
      Priority_Found, Interrupt_Priority_Found : Boolean;
   begin
      Priority_Found           := Dictionary.GetTypeHasPragma (Task_Type_Sym, Dictionary.Priority);
      Interrupt_Priority_Found := Dictionary.GetTypeHasPragma (Task_Type_Sym, Dictionary.InterruptPriority);

      -- There must be either Priority or Interrupt_Priority
      if not (Priority_Found or else Interrupt_Priority_Found) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 876,
            Reference => ErrorHandler.No_Reference,
            Position  => End_Node_Position,
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end Check_Pragma_Validity;

begin -- Wf_Task_Type_Declaration

   -- ASSUME Node = task_type_declaration
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.task_type_declaration,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = task_type_declaration in Wf_Task_Type_Declaration");

   Ident_Node        := Child_Node (Node);
   Ident_Str         := Node_Lex_String (Node => Ident_Node);
   Discriminant_Node := Get_Discriminant_Node (Task_Type_Declaration_Node => Node);
   Sym               := Dictionary.LookupItem
     (Name              => Ident_Str,
      Scope             => Scope,
      Context           => Dictionary.ProofContext,
      Full_Package_Name => False);

   if Sym = Dictionary.NullSymbol
     or else (Dictionary.IsTypeMark (Sym) and then Dictionary.TypeIsAnnounced (Sym) and then not Dictionary.IsDeclared (Sym)) then
      if Sym /= Dictionary.NullSymbol then
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => Ident_Node);
      end if;
      Dictionary.AddTaskType
        (Name        => Node_Lex_String (Node => Ident_Node),
         Comp_Unit   => ContextManager.Ops.Current_Unit,
         Declaration => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                             End_Position   => Node_Position (Node => Node)),
         Scope       => Scope,
         Context     => Dictionary.ProgramContext,
         Constrained => (Syntax_Node_Type (Node => Discriminant_Node) /= SPSymbols.known_discriminant_part),
         TaskType    => Task_Type_Sym);
      if ErrorHandler.Generate_SLI then
         SLI.Generate_Xref_Symbol
           (Comp_Unit      => ContextManager.Ops.Current_Unit,
            Parse_Tree     => Ident_Node,
            Symbol         => Task_Type_Sym,
            Is_Declaration => True);
      end if;
      Task_Scope := Dictionary.VisibleScope (Task_Type_Sym);

      -- wff discriminants here
      if Syntax_Node_Type (Node => Discriminant_Node) = SPSymbols.known_discriminant_part then
         -- ASSERTS Discriminant_Node = known_discriminant_part
         Wf_Known_Discriminant_Part (Node               => Discriminant_Node,
                                     Protected_Type_Sym => Task_Type_Sym,
                                     Scope              => Scope);
      end if;

      -- handle annotation
      -- global
      Anno_Node := Get_Anno_Node (Task_Type_Declaration_Node => Node);
      wf_global_definition (Anno_Node, Scope, -- was Task_Scope, but for global look up we should start in enclosing scope
                            Task_Type_Sym, True, -- FirstSeen
                            -- to get
                            Global_Error);

      if not CommandLineData.Content.Do_Information_Flow then
         CreateFullSubProgDependency (Node, Task_Type_Sym, Dictionary.IsAbstract);
      end if;

      -- derives
      Anno_Node := Next_Sibling (Current_Node => Anno_Node);
      if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.dependency_relation then
         wf_dependency_relation (Anno_Node, Task_Scope, Task_Type_Sym, True, -- FirstSeen
                                 Global_Error);
      else
         -- no derives so check if its required
         if CommandLineData.Content.Do_Information_Flow then
            Derives_Error := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 501,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Get_Anno_Node (Task_Type_Declaration_Node => Node)),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end if;

      -- asserts
      if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.dependency_relation then
         Anno_Node := Next_Sibling (Current_Node => Anno_Node);
      end if;
      if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.declare_annotation then
         -- ASSUME Anno_Node = declare_annotation
         Wf_Declare_Annotation (Node         => Anno_Node,
                                Scope        => Task_Scope,
                                Task_Or_Proc => Task_Type_Sym,
                                First_Seen   => True);
      end if;

      -- if there are errors in the task type signature then mark it as malformed
      if Global_Error or else Derives_Error then
         Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Task_Type_Sym);
      end if;

      -- deal with priority pragma which should be first
      Wf_Priority_Pragma (Node  => Get_Priority_Pragma_Node (Task_Type_Declaration_Node => Node),
                          Scope => Task_Scope);

      -- check any other pragmas
      Pragma_Node := Get_Priority_Pragma_Node (Task_Type_Declaration_Node => Node);
      loop
         Pragma_Node := Next_Sibling (Parent_Node (Current_Node => Pragma_Node));
         exit when Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.identifier;
         wf_pragma (Pragma_Node, Task_Scope);
      end loop;

      -- closing identifier must match initial
      Closing_Ident_Node := Get_Closing_Ident_Node (Task_Type_Declaration_Node => Node);
      if LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Ident_Str,
         Lex_Str2 => Node_Lex_String (Node => Closing_Ident_Node)) /=
        LexTokenManager.Str_Eq then
         ErrorHandler.Semantic_Error
           (Err_Num   => 58,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Closing_Ident_Node),
            Id_Str    => Ident_Str);
      end if;

      Check_Pragma_Validity (End_Node_Position => Node_Position (Node => Closing_Ident_Node),
                             Task_Type_Sym     => Task_Type_Sym);

   else -- illegal redeclaration
      ErrorHandler.Semantic_Error
        (Err_Num   => 10,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Ident_Node),
         Id_Str    => Ident_Str);
   end if;
end Wf_Task_Type_Declaration;
