-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Grammar (with punctuation tokens removed)
--
--  generic_subprogram_instantiation :
--         generic_procedure_instantiation
--       | generic_function_instantiation
--       | generic_child_function_instantiation ;

--  generic_procedure_instantiation :
--         procedure_specification procedure_annotation identifier
--            generic_actual_part semicolon
--       | procedure_specification procedure_annotation identifier semicolon ;

--  generic_function_instantiation :
--         identifier [is new] identifier
--            generic_actual_part semicolon
--       | identifier [is new] identifier semicolon ;

--  generic_child_function_instantiation :
--         identifier [is new] identifier point identifier
--            generic_actual_part semicolon
--       | identifier [is new] identifier point identifier semicolon ;

--  generic_actual_part :
--         left_paren name_argument_list right_paren ;

separate (Sem.CompUnit)
procedure wf_generic_subprogram_instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes) is
   type ExpectedGenericKind is (GenericProcedure, GenericFunction);

   procedure CheckValidInstantiationIdent (IdentNode : in     STree.SyntaxNode;
                                           Scope     : in     Dictionary.Scopes;
                                           Ok        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IdentNode,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Ok                         from Dictionary.Dict,
   --#                                         IdentNode,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   is
      Sym      : Dictionary.Symbol;
      IdentStr : LexTokenManager.Lex_String;
   begin
      -- check that name is not already in use
      IdentStr := Node_Lex_String (Node => IdentNode);
      Sym      := Dictionary.LookupItem (Name              => IdentStr,
                                         Scope             => Scope,
                                         Context           => Dictionary.ProofContext,
                                         Full_Package_Name => False);

      if Sym = Dictionary.NullSymbol then
         Ok := True;
      else
         Ok := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);
      end if;

   end CheckValidInstantiationIdent;

   ----------------------------------------------------------

   procedure CheckGeneric
     (IdentNode  : in     STree.SyntaxNode;
      Scope      : in     Dictionary.Scopes;
      Kind       : in     ExpectedGenericKind;
      GenericSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IdentNode,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         GenericSym,
   --#         STree.Table                from Dictionary.Dict,
   --#                                         IdentNode,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   is
      -- Checks that the identifier after "is new" represents a visible generic unit of the
      -- appropriate kind.  Returns symbol of this generic unit if legal or a null symbol
      -- otherwise.
      Sym : Dictionary.Symbol;
   begin
      Sym :=
        Dictionary.LookupItem (Name              => Node_Lex_String (Node => IdentNode),
                               Scope             => Scope,
                               Context           => Dictionary.ProgramContext,
                               Full_Package_Name => False);
      -- Kludge Alert
      -- If I am honest, I don't really understand why this is needed.  We have looked
      -- up the generic in program context so I would expect to get the Ada function
      -- symbol.  A test case showed otherwise, hence this is in for now.  PNA 13/1/4
      if Dictionary.IsProofFunction (Sym) then
         Sym := Dictionary.GetAdaFunction (Sym);
      end if;

      -- Validate generic unit symbol
      if Sym = Dictionary.NullSymbol then
         ErrorHandler.Semantic_Error
           (Err_Num   => 1,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => Node_Lex_String (Node => IdentNode));

      elsif Dictionary.Is_Generic_Subprogram (The_Symbol => Sym) then
         if Kind = GenericProcedure and Dictionary.IsFunction (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.Semantic_Error
              (Err_Num   => 631,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdentNode),
               Id_Str    => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         elsif Kind = GenericFunction and Dictionary.IsProcedure (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.Semantic_Error
              (Err_Num   => 632,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdentNode),
               Id_Str    => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;

         elsif not Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Sym) then
            -- right sort of subprogram, but generic declaration had errors
            ErrorHandler.Semantic_Warning
              (Err_Num  => 390,
               Position => Node_Position (Node => IdentNode),
               Id_Str   => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         else
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdentNode);
         end if;

      else -- not a generic subprgoram at all
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 630,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Sym       => Sym,
            Scope     => Scope);
         Sym := Dictionary.NullSymbol;
      end if;

      GenericSym := Sym;
   end CheckGeneric;

   ----------------------------------------------------------

   procedure ProcessGenericProcedureInstantiation (Node  : in STree.SyntaxNode;
                                                   Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      GenericSym, InstantiationSym : Dictionary.Symbol;
      GenericNode                  : STree.SyntaxNode;
      InstantiationIdentNode       : STree.SyntaxNode;
      ErrorsFound                  : Boolean;
      Ok                           : Boolean;
   begin
      --  generic_procedure_instantiation :
      --         procedure_specification procedure_annotation identifier
      --            generic_actual_part
      --       | procedure_specification procedure_annotation identifier ;
      GenericNode            := Next_Sibling (Next_Sibling (Child_Node (Node)));
      InstantiationIdentNode := Child_Node (Child_Node (Node));
      CheckValidInstantiationIdent (InstantiationIdentNode, Scope,
                                    -- to get
                                    Ok);
      if Ok then
         CheckGeneric (GenericNode, Scope, GenericProcedure,
                       -- to get
                       GenericSym);

         if GenericSym /= Dictionary.NullSymbol then
            -- check parameters etc.
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation
              (Name          => Node_Lex_String (Node => Child_Node (Child_Node (Node))),
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                     End_Position   => Node_Position (Node => InstantiationIdentNode)),
               TheGeneric    => GenericSym,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Scope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => InstantiationSym);

            -- check parameters etc.
            wf_generic_actual_part (GenericNode, GenericSym, InstantiationSym, Scope,
                                    -- to get
                                    ErrorsFound);
            if ErrorsFound then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, InstantiationSym);
            else
               -- do formal/actual substitutions
               Dictionary.InstantiateSubprogramParameters
                 (GenericSubprogramSym => GenericSym,
                  ActualSubprogramSym  => InstantiationSym,
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  Declaration          => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                               End_Position   => Node_Position (Node => InstantiationIdentNode)));
            end if;

         end if;
      end if;
   end ProcessGenericProcedureInstantiation;

   ----------------------------------------------------

   procedure ProcessGenericFunctionInstantiation (Node  : in STree.SyntaxNode;
                                                  Scope : in Dictionary.Scopes)

   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      GenericSym, InstantiationSym : Dictionary.Symbol;
      InstantiationIdentNode       : STree.SyntaxNode;
      GenericNode                  : STree.SyntaxNode;
      ErrorsFound                  : Boolean;
      Ok                           : Boolean;
   begin
      --  generic_function_instantiation :
      --         identifier [is new] identifier
      --            generic_actual_part semicolon
      --       | identifier [is new] identifier semicolon ;
      GenericNode            := Next_Sibling (Child_Node (Node));
      InstantiationIdentNode := Child_Node (Node);
      CheckValidInstantiationIdent (InstantiationIdentNode, Scope,
                                    -- to get
                                    Ok);
      if Ok then
         CheckGeneric (GenericNode, Scope, GenericFunction,
                       -- to get
                       GenericSym);
         if GenericSym /= Dictionary.NullSymbol then
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation
              (Name          => Node_Lex_String (Node => Child_Node (Node)),
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                     End_Position   => Node_Position (Node => InstantiationIdentNode)),
               TheGeneric    => GenericSym,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Scope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => InstantiationSym);

            -- check parameters etc.
            wf_generic_actual_part (GenericNode, GenericSym, InstantiationSym, Scope,
                                    -- to get
                                    ErrorsFound);

            if ErrorsFound then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, InstantiationSym);
            else
               Dictionary.InstantiateSubprogramParameters
                 (GenericSubprogramSym => GenericSym,
                  ActualSubprogramSym  => InstantiationSym,
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  Declaration          => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                               End_Position   => Node_Position (Node => InstantiationIdentNode)));
            end if;

         end if;
      end if;
   end ProcessGenericFunctionInstantiation;

   ----------------------------------------------------

   procedure ProcessGenericChildFunctionInstantiation (Node  : in STree.SyntaxNode;
                                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      PrefixSym, GenericSym, InstantiationSym : Dictionary.Symbol;
      PrefixNode, GenericNode                 : STree.SyntaxNode;
      InstantiationIdentNode                  : STree.SyntaxNode;
      ErrorsFound                             : Boolean;
      Ok                                      : Boolean;
   begin
      --  generic_child_function_instantiation :
      --         identifier [is new] identifier point identifier
      --            generic_actual_part semicolon
      --       | identifier [is new] identifier point identifier semicolon ;
      PrefixNode             := Next_Sibling (Child_Node (Node));
      GenericNode            := Next_Sibling (PrefixNode);
      InstantiationIdentNode := Child_Node (Node);
      CheckValidInstantiationIdent (InstantiationIdentNode, Scope,
                                    -- to get
                                    Ok);
      if Ok then
         -- check prefix, in practice the only thing acceptable here will be package Ada
         PrefixSym :=
           Dictionary.LookupItem
           (Name              => Node_Lex_String (Node => PrefixNode),
            Scope             => Scope,
            Context           => Dictionary.ProgramContext,
            Full_Package_Name => False);
         if PrefixSym = Dictionary.NullSymbol then
            ErrorHandler.Semantic_Error
              (Err_Num   => 1,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => PrefixNode),
               Id_Str    => Node_Lex_String (Node => PrefixNode));

         elsif not Dictionary.IsPackage (PrefixSym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 18,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => PrefixNode),
               Id_Str    => Node_Lex_String (Node => PrefixNode));

         else
            -- potentially valid prefix
            CheckGeneric (GenericNode, Dictionary.VisibleScope (PrefixSym), GenericFunction,
                          -- to get
                          GenericSym);
            if GenericSym /= Dictionary.NullSymbol then
               -- add the instantiation
               Dictionary.AddSubprogramInstantiation
                 (Name          => Node_Lex_String (Node => Child_Node (Node)),
                  Comp_Unit     => ContextManager.Ops.Current_Unit,
                  Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                        End_Position   => Node_Position (Node => InstantiationIdentNode)),
                  TheGeneric    => GenericSym,
                  Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                        End_Position   => Node_Position (Node => Node)),
                  Scope         => Scope,
                  Context       => Dictionary.ProgramContext,
                  Subprogram    => InstantiationSym);
               -- check parameters etc.
               wf_generic_actual_part (GenericNode, GenericSym, InstantiationSym, Scope,
                                       -- to get
                                       ErrorsFound);
               if ErrorsFound then
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, InstantiationSym);
               else
                  STree.Set_Node_Lex_String (Sym  => PrefixSym,
                                             Node => PrefixNode);
                  -- do formal/actual substitutions
                  Dictionary.InstantiateSubprogramParameters
                    (GenericSubprogramSym => GenericSym,
                     ActualSubprogramSym  => InstantiationSym,
                     Comp_Unit            => ContextManager.Ops.Current_Unit,
                     Declaration          => Dictionary.Location'(Start_Position => Node_Position (Node => InstantiationIdentNode),
                                                                  End_Position   => Node_Position (Node => InstantiationIdentNode)));
               end if;
            end if;
         end if;
      end if;
   end ProcessGenericChildFunctionInstantiation;

   ----------------------------------------------------
begin -- wf_generic_subprogram_instantiation
   if Syntax_Node_Type (Node => Child_Node (Node)) = SPSymbols.generic_procedure_instantiation then
      ProcessGenericProcedureInstantiation (Child_Node (Node), Scope);

   elsif Syntax_Node_Type (Node => Child_Node (Node)) = SPSymbols.generic_function_instantiation then
      ProcessGenericFunctionInstantiation (Child_Node (Node), Scope);

   elsif Syntax_Node_Type (Node => Child_Node (Node)) = SPSymbols.generic_child_function_instantiation then
      ProcessGenericChildFunctionInstantiation (Child_Node (Node), Scope);

   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Unknown generic kind in wf_generic_subprogram_instantiation");
   end if;
end wf_generic_subprogram_instantiation;
