-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.Wf_Package_Specification)
procedure CheckTypesCanBeUsed (PackSym : in Dictionary.Symbol;
                               ErrNode : in STree.SyntaxNode) is
   PrivateTypeIt : Dictionary.Iterator;
   PrivateType   : Dictionary.Symbol;

   -- Following function detects a special case and is used by both InitializingProc and
   -- InitializingFuncExists.  The idea is that although we are generally looking for
   -- a subprogram that exports soemthing of the private type without importing it such
   -- an import is acceptable if it takes the form of a global which is an initialized
   -- (or mode IN) own variables
   function ImportIsInitializedOrModeInOwnVar (PackSym, ImportSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsOwnVariable (ImportSym)
        and then Dictionary.GetOwner (ImportSym) = PackSym
        and then (Dictionary.OwnVariableIsInitialized (ImportSym)
                    or else (Dictionary.GetOwnVariableOrConstituentMode (ImportSym) = Dictionary.InMode));
   end ImportIsInitializedOrModeInOwnVar;

   ------------------------------------------------------------------------------------------------------
   -- search for constructor procedures
   ------------------------------------------------------------------------------------------------------

   -- function used by both InitializingProcedureExists and InitializingProtectedProcedureExists
   function IsSuitableProcedure (PackSym, SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      function TypeIsExported (SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result    : Boolean := False;
         ExportIt  : Dictionary.Iterator;
         ExportSym : Dictionary.Symbol;
      begin
         ExportIt := Dictionary.FirstExport (Dictionary.IsAbstract, SubprogSym);
         while not Dictionary.IsNullIterator (ExportIt) loop
            ExportSym := Dictionary.CurrentSymbol (ExportIt);
            Result    := Dictionary.GetType (ExportSym) = TypeSym
              and then not Dictionary.IsImport (Dictionary.IsAbstract, -- check for IN OUT case
                                                SubprogSym, ExportSym);
            exit when Result;

            ExportIt := Dictionary.NextSymbol (ExportIt);
         end loop;
         return Result;
      end TypeIsExported;

      function TypeIsImported (PackSym, SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result    : Boolean := False;
         ImportIt  : Dictionary.Iterator;
         ImportSym : Dictionary.Symbol;
      begin
         ImportIt := Dictionary.FirstImport (Dictionary.IsAbstract, SubprogSym);
         while not Dictionary.IsNullIterator (ImportIt) loop
            ImportSym := Dictionary.CurrentSymbol (ImportIt);
            Result    := Dictionary.GetType (ImportSym) = TypeSym
              and then not ImportIsInitializedOrModeInOwnVar (PackSym, ImportSym);
            exit when Result;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         return Result;
      end TypeIsImported;

   begin --IsSuitableProcedure
      return Dictionary.IsProcedure (SubprogSym)
        and then TypeIsExported (SubprogSym, TypeSym)
        and then not TypeIsImported (PackSym, SubprogSym, TypeSym);
   end IsSuitableProcedure;

   function InitializingProcedureExists (PackSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      function InitializingProcedureExistsLocal
        (SubprogIt        : Dictionary.Iterator;
         PackSym, TypeSym : Dictionary.Symbol)
        return             Boolean
      --# global in Dictionary.Dict;
      is
         It         : Dictionary.Iterator;
         Result     : Boolean := False;
         SubprogSym : Dictionary.Symbol;
      begin --InitializingProcedureExists
         It := SubprogIt;
         while not Dictionary.IsNullIterator (It) loop
            SubprogSym := Dictionary.CurrentSymbol (It);
            Result     := IsSuitableProcedure (PackSym, SubprogSym, TypeSym);
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
         return Result;
      end InitializingProcedureExistsLocal;

   begin -- InitializingProcedureExists
      return InitializingProcedureExistsLocal (Dictionary.FirstVisibleSubprogram (PackSym), PackSym, TypeSym)
        or else InitializingProcedureExistsLocal (Dictionary.FirstPrivateSubprogram (PackSym), PackSym, TypeSym);
   end InitializingProcedureExists;

   function InitializingProtectedProcedureExists (PackSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;

      procedure CheckOpsInProtectedTypes (It : in Dictionary.Iterator)
      --# global in     Dictionary.Dict;
      --#        in     PackSym;
      --#        in     TypeSym;
      --#        in out Result;
      --# derives Result from *,
      --#                     Dictionary.Dict,
      --#                     It,
      --#                     PackSym,
      --#                     TypeSym;
      is
         OpIt, TypeIt : Dictionary.Iterator;
      begin
         TypeIt := It;
         while not Dictionary.IsNullIterator (TypeIt) loop
            OpIt := Dictionary.FirstVisibleSubprogram (Dictionary.CurrentSymbol (TypeIt));
            while not Dictionary.IsNullIterator (OpIt) loop
               Result := IsSuitableProcedure (PackSym, Dictionary.CurrentSymbol (OpIt), TypeSym);
               exit when Result;

               OpIt := Dictionary.NextSymbol (OpIt);
            end loop;
            exit when Result;

            TypeIt := Dictionary.NextSymbol (TypeIt);
         end loop;
      end CheckOpsInProtectedTypes;

   begin --InitializingProtectedProcedureExists
      CheckOpsInProtectedTypes (Dictionary.FirstVisibleProtectedType (PackSym));
      if not Result then
         CheckOpsInProtectedTypes (Dictionary.FirstPrivateProtectedType (PackSym));
      end if;
      return Result;
   end InitializingProtectedProcedureExists;

   ------------------------------------------------------------------------------------------------------
   -- search for constructor functions
   ------------------------------------------------------------------------------------------------------

   -- function used by InitializingFunctionExists and InitializingProtectedFunctionExists
   function IsSuitableFunction (PackSym, SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      function TypeIsParameter (SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result    : Boolean := False;
         ImportIt  : Dictionary.Iterator;
         ImportSym : Dictionary.Symbol;
      begin
         ImportIt := Dictionary.FirstSubprogramParameter (SubprogSym);
         while not Dictionary.IsNullIterator (ImportIt) loop
            ImportSym := Dictionary.CurrentSymbol (ImportIt);
            Result    := Dictionary.GetType (ImportSym) = TypeSym;
            exit when Result;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         return Result;
      end TypeIsParameter;

      function TypeIsGlobal (PackSym, SubprogSym, TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result    : Boolean := False;
         ImportIt  : Dictionary.Iterator;
         ImportSym : Dictionary.Symbol;
      begin
         ImportIt := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, SubprogSym);
         while not Dictionary.IsNullIterator (ImportIt) loop
            ImportSym := Dictionary.CurrentSymbol (ImportIt);
            Result    := Dictionary.GetType (ImportSym) = TypeSym
              and then not ImportIsInitializedOrModeInOwnVar (PackSym, ImportSym);
            exit when Result;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         return Result;
      end TypeIsGlobal;

   begin -- IsSuitableFunction
      return Dictionary.IsFunction (SubprogSym)
        and then Dictionary.GetType (SubprogSym) = TypeSym
        and then not TypeIsParameter (SubprogSym, TypeSym)
        and then not TypeIsGlobal (PackSym, SubprogSym, TypeSym);
   end IsSuitableFunction;

   function InitializingFunctionExists (PackSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      function InitializingFunctionExistsLocal
        (SubprogIt        : Dictionary.Iterator;
         PackSym, TypeSym : Dictionary.Symbol)
        return             Boolean
      --# global in Dictionary.Dict;
      is
         Result     : Boolean := False;
         SubprogSym : Dictionary.Symbol;
         It         : Dictionary.Iterator;

      begin -- InitializingFunctionExistsLocal
         It := SubprogIt;
         while not Dictionary.IsNullIterator (It) loop
            SubprogSym := Dictionary.CurrentSymbol (It);
            Result     := IsSuitableFunction (PackSym, SubprogSym, TypeSym);
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
         return Result;
      end InitializingFunctionExistsLocal;

   begin -- InitializingFunctionExists
      return InitializingFunctionExistsLocal (Dictionary.FirstVisibleSubprogram (PackSym), PackSym, TypeSym)
        or else InitializingFunctionExistsLocal (Dictionary.FirstPrivateSubprogram (PackSym), PackSym, TypeSym);
   end InitializingFunctionExists;

   function InitializingProtectedFunctionExists (PackSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;

      procedure CheckOpsInProtectedTypes (It : in Dictionary.Iterator)
      --# global in     Dictionary.Dict;
      --#        in     PackSym;
      --#        in     TypeSym;
      --#        in out Result;
      --# derives Result from *,
      --#                     Dictionary.Dict,
      --#                     It,
      --#                     PackSym,
      --#                     TypeSym;
      is
         OpIt, TypeIt : Dictionary.Iterator;
      begin
         TypeIt := It;
         while not Dictionary.IsNullIterator (TypeIt) loop
            OpIt := Dictionary.FirstVisibleSubprogram (Dictionary.CurrentSymbol (TypeIt));
            while not Dictionary.IsNullIterator (OpIt) loop
               Result := IsSuitableFunction (PackSym, Dictionary.CurrentSymbol (OpIt), TypeSym);
               exit when Result;

               OpIt := Dictionary.NextSymbol (OpIt);
            end loop;
            exit when Result;

            TypeIt := Dictionary.NextSymbol (TypeIt);
         end loop;
      end CheckOpsInProtectedTypes;

   begin --InitializingProtectedFunctionExists
      CheckOpsInProtectedTypes (Dictionary.FirstVisibleProtectedType (PackSym));
      if not Result then
         CheckOpsInProtectedTypes (Dictionary.FirstPrivateProtectedType (PackSym));
      end if;

      return Result;
   end InitializingProtectedFunctionExists;

   ------------------------------------------------------------------------------------------------------
   -- search for constructor constants
   ------------------------------------------------------------------------------------------------------

   function InitializingConstantExists (PackSym, TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      DefConIt : Dictionary.Iterator;
      Result   : Boolean := False;
   begin
      DefConIt := Dictionary.FirstDeferredConstant (PackSym);
      while not Dictionary.IsNullIterator (DefConIt) loop
         Result := Dictionary.GetType (Dictionary.CurrentSymbol (DefConIt)) = TypeSym;
         exit when Result;

         DefConIt := Dictionary.NextSymbol (DefConIt);
      end loop;
      return Result;
   end InitializingConstantExists;

   ------------------------------------------------------------------------------------------------------
   -- main procedures
   ------------------------------------------------------------------------------------------------------

   procedure CheckOnePrivateType (PackSym, Sym : in Dictionary.Symbol;
                                  ErrNode      : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrNode,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym;
   is
      Ok : Boolean;
   begin
      if Dictionary.IsLimitedPrivateType (Sym) then
         Ok := InitializingProcedureExists (PackSym, Sym) or else InitializingProtectedProcedureExists (PackSym, Sym);
      else -- private, not limited
         Ok := InitializingProcedureExists (PackSym, Sym)
           or else InitializingProtectedProcedureExists (PackSym, Sym)
           or else InitializingFunctionExists (PackSym, Sym)
           or else InitializingProtectedFunctionExists (PackSym, Sym)
           or else InitializingConstantExists (PackSym, Sym);
      end if;

      if not Ok then

         case CommandLineData.Content.Language_Profile is
            when CommandLineData.SPARK83 =>

               ErrorHandler.Semantic_Warning
                 (Err_Num  => 397,
                  Position => Node_Position (Node => ErrNode),
                  Id_Str   => Dictionary.GetSimpleName (Sym));

            when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

               -- SPARK 95 or 2005, weaker warning because of child packages
               ErrorHandler.Semantic_Warning
                 (Err_Num  => 394,
                  Position => Node_Position (Node => ErrNode),
                  Id_Str   => Dictionary.GetSimpleName (Sym));
         end case;
      end if;
   end CheckOnePrivateType;

begin --CheckTypesCanBeUsed
   PrivateTypeIt := Dictionary.FirstPrivateType (PackSym);
   while not Dictionary.IsNullIterator (PrivateTypeIt)  -- exit when no more private types
   loop
      PrivateType := Dictionary.CurrentSymbol (PrivateTypeIt);
      CheckOnePrivateType (PackSym, PrivateType, ErrNode);

      PrivateTypeIt := Dictionary.NextSymbol (PrivateTypeIt);
   end loop;
end CheckTypesCanBeUsed;
