-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with CompleteCheck;
with ComponentManager;
with Dictionary;
with ErrorHandler;
with ExaminerConstants;
with E_Strings;
with FlowAnalyser;
with Heap;
with LexTokenManager;
with Lists;
with Maths;
with RefList;
with RelationAlgebra;
with SeqAlgebra;
with SimpleLists;
with SPARK_IO;
with SPSymbols;
with SystemErrors;

use type CompleteCheck.TypCompleteState;
use type CompleteCheck.TypRangeState;
use type CompleteCheck.TypOverlapState;
use type Dictionary.Abstractions;
use type Dictionary.Contexts;
use type Dictionary.Iterator;
use type Dictionary.KindsOfOp;
use type Dictionary.Modes;
use type Dictionary.PrefixSort;
use type Dictionary.Scopes;
use type Dictionary.Symbol;
use type ErrorHandler.Error_Level;
use type ErrorHandler.Justification_Identifier;
use type ErrorHandler.Justification_Kinds;
use type LexTokenManager.Lex_String;
use type LexTokenManager.Str_Comp_Result;
use type Maths.ErrorCode;
use type Maths.Value;
use type SPSymbols.SPSymbol;

package body Sem is
   -- Operator renames

   -- Long subprogram prefix renames
   function Child_Node (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Child_Node;

   function Next_Sibling (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Next_Sibling;

   function Parent_Node (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Parent_Node;

   function Find_First_Node
     (Node_Kind    : SPSymbols.SPSymbol;
      From_Root    : STree.SyntaxNode;
      In_Direction : STree.TraverseDirection)
     return         STree.Iterator renames STree.Find_First_Node;

   function Find_First_Branch_Node (From_Root    : STree.SyntaxNode;
                                    In_Direction : STree.TraverseDirection) return STree.Iterator
     renames STree.Find_First_Branch_Node;

   function Get_Node (It : STree.Iterator) return STree.SyntaxNode renames STree.Get_Node;

   function Syntax_Node_Type (Node : STree.SyntaxNode) return SPSymbols.SPSymbol renames STree.Syntax_Node_Type;

   function Node_Position (Node : STree.SyntaxNode) return LexTokenManager.Token_Position renames STree.Node_Position;

   function Node_Lex_String (Node : STree.SyntaxNode) return LexTokenManager.Lex_String renames STree.Node_Lex_String;

   -- Function returns the left most leaf node of the tree.
   function Last_Child_Of (Start_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Last_Child_Of;

   function Last_Sibling_Of (Start_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Last_Sibling_Of;

   ----------------------------------------------------------------------------

   type Exp_Record_Sort is (
                            Type_Result, -- should be this anywhere above primary
                            Is_Unknown,
                            Is_Parameter_Name, -- used in named association checks
                            Is_Package,
                            Is_Object,
                            Is_Function,
                            Is_Type_Mark);

   type Exp_Record is record
      Type_Symbol             : Dictionary.Symbol;
      Other_Symbol            : Dictionary.Symbol;
      Stream_Symbol           : Dictionary.Symbol;
      Tagged_Parameter_Symbol : Dictionary.Symbol;
      Variable_Symbol         : Dictionary.Symbol;
      Param_Count             : Natural;
      Param_List              : Lists.List;
      Sort                    : Exp_Record_Sort;
      Arg_List_Found          : Boolean;
      Is_AVariable            : Boolean;
      Is_An_Entire_Variable   : Boolean;
      Errors_In_Expression    : Boolean;
      Has_Operators           : Boolean;

      Is_Static, -- 3 flags meaningless unless Sort=Type_Result
        Is_Constant, Is_ARange : Boolean;

      String_Value : LexTokenManager.Lex_String; -- if a String literal or constant
      Value        : Maths.Value;                -- value of scalar, or value of L if Is_ARange
      Range_RHS    : Maths.Value;                -- if Is_ARange (e.g. L .. R) then this is the value of R
   end record;

   Null_Exp_Record : constant Exp_Record :=
     Exp_Record'
     (Type_Symbol             => Dictionary.NullSymbol,
      Other_Symbol            => Dictionary.NullSymbol,
      Stream_Symbol           => Dictionary.NullSymbol,
      Tagged_Parameter_Symbol => Dictionary.NullSymbol,
      Variable_Symbol         => Dictionary.NullSymbol,
      Param_Count             => 0,
      Param_List              => Lists.Null_List,
      Sort                    => Is_Unknown,
      Arg_List_Found          => False,
      Is_AVariable            => False,
      Is_An_Entire_Variable   => False,
      Errors_In_Expression    => False,
      Has_Operators           => False,
      Is_Static               => False,
      Is_Constant             => False,
      Is_ARange               => False,
      String_Value            => LexTokenManager.Null_String,
      Value                   => Maths.NoValue,
      Range_RHS               => Maths.NoValue);

   type Typ_Case_Flags is record
      Check_Completeness  : Boolean;
      Signal_Out_Of_Range : Boolean;
      Out_Of_Range_Seen   : Boolean;
      Check_Overlap       : Boolean;
      Warn_No_Others      : Boolean;
      Others_Mandatory    : Boolean;
   end record;

   Null_Case_Flags : constant Typ_Case_Flags := Typ_Case_Flags'(False, False, False, False, False, False);

   type Typ_Type_Bound is record
      Is_Defined : Boolean;
      Value      : Integer;
   end record;

   Unknown_Type_Bound : constant Typ_Type_Bound := Typ_Type_Bound'(Is_Defined => False,
                                                                   Value      => 0);

   type Typ_Agg_Association_Type is (Aggregate_Is_Positional, Aggregate_Is_Named, Aggregate_Is_Lone_Others);

   type Typ_Agg_Flags is record
      Check_Completeness        : Boolean;
      Warn_No_Others            : Boolean;
      Check_Overlap             : Boolean;
      Signal_Out_Of_Range       : Boolean;
      Out_Of_Range_Seen         : Boolean;
      More_Entries_Than_Natural : Boolean;
      Has_Others_Part           : Boolean;
      Association_Type          : Typ_Agg_Association_Type;
   end record;

   Null_Typ_Agg_Flags : constant Typ_Agg_Flags :=
     Typ_Agg_Flags'(False, False, False, False, False, False, False, Typ_Agg_Association_Type'First);

   ----------------------- Subprograms ---------------------------

   -- Put_Exp_Record is handy for debugging expression walking, but
   -- is uncalled in production builds.
   procedure Put_Exp_Record (R : in Exp_Record)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                R;
   is
      --# hide Put_Exp_Record;
      F : SPARK_IO.File_Type;
   begin
      F := SPARK_IO.Standard_Output;
      SPARK_IO.Put_String (F, "Sort                 => ", 0);
      SPARK_IO.Put_Line (F, Exp_Record_Sort'Image (R.Sort), 0);

      SPARK_IO.Put_String (F, "Type_Symbol           => ", 0);
      E_Strings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Dictionary.GetSimpleName (R.Type_Symbol)));

      SPARK_IO.Put_String (F, "Other_Symbol          => ", 0);
      E_Strings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Dictionary.GetSimpleName (R.Other_Symbol)));

      SPARK_IO.Put_String (F, "String_Value         => ", 0);
      E_Strings.Put_Line (File  => F,
                          E_Str => LexTokenManager.Lex_String_To_String (R.String_Value));

      SPARK_IO.Put_String (F, "Value                => ", 0);
      E_Strings.Put_Line (File  => F,
                          E_Str => Maths.ValueToString (R.Value));

      SPARK_IO.Put_String (F, "Arg_List_Found         => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Arg_List_Found), 0);

      SPARK_IO.Put_String (F, "Range_RHS             => ", 0);
      E_Strings.Put_Line (File  => F,
                          E_Str => Maths.ValueToString (R.Range_RHS));

      SPARK_IO.Put_String (F, "Param_Count           =>", 0);
      SPARK_IO.Put_Line (F, Natural'Image (R.Param_Count), 0);

      SPARK_IO.Put_Line (F, "Param_List            => (...not printed...)", 0);

      SPARK_IO.Put_String (F, "Is_Static             => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Is_Static), 0);
      SPARK_IO.Put_String (F, "Is_Constant           => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Is_Constant), 0);
      SPARK_IO.Put_String (F, "Is_ARange             => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Is_ARange), 0);

      SPARK_IO.Put_String (F, "Variable_Symbol       => ", 0);
      E_Strings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Dictionary.GetSimpleName (R.Variable_Symbol)));

      SPARK_IO.Put_String (F, "Is_AVariable          => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Is_AVariable), 0);
      SPARK_IO.Put_String (F, "Is_An_Entire_Variable   => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Is_An_Entire_Variable), 0);
      SPARK_IO.Put_String (F, "Errors_In_Expression   => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Errors_In_Expression), 0);
      SPARK_IO.Put_String (F, "Has_Operators         => ", 0);
      SPARK_IO.Put_Line (F, Boolean'Image (R.Has_Operators), 0);

      SPARK_IO.Put_String (F, "Stream_Symbol         => ", 0);
      E_Strings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Dictionary.GetSimpleName (R.Stream_Symbol)));
   end Put_Exp_Record;

   ---------------------------------------------------------------------

   procedure Get_Type_Bounds (Type_Symbol              : in     Dictionary.Symbol;
                              Lower_Bound, Upper_Bound :    out Typ_Type_Bound)
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --# derives Lower_Bound,
   --#         Upper_Bound from Dictionary.Dict,
   --#                          LexTokenManager.State,
   --#                          Type_Symbol;
   is
      Maths_Error : Maths.ErrorCode;
      Bound_Val   : Integer;
   begin
      if Dictionary.IsBooleanTypeMark (Type_Symbol) then
         Lower_Bound := Typ_Type_Bound'(Is_Defined => True,
                                        Value      => 0);
         Upper_Bound := Typ_Type_Bound'(Is_Defined => True,
                                        Value      => 1);
      else
         Maths.ValueToInteger
           (Maths.ValueRep
              (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                   LexTokenManager.First_Token, Type_Symbol)),
            Bound_Val, -- to get
            Maths_Error); -- to get

         Lower_Bound := Typ_Type_Bound'(Is_Defined => (Maths_Error = Maths.NoError),
                                        Value      => Bound_Val);

         Maths.ValueToInteger
           (Maths.ValueRep
              (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                   LexTokenManager.Last_Token, Type_Symbol)),
            Bound_Val, -- to get
            Maths_Error); -- to get

         Upper_Bound := Typ_Type_Bound'(Is_Defined => (Maths_Error = Maths.NoError),
                                        Value      => Bound_Val);
      end if;
   end Get_Type_Bounds;

   ---------------------------------------------------------------------

   procedure Check_Package_Prefix
     (Node     : in     STree.SyntaxNode;
      Pack_Sym : in     Dictionary.Symbol;
      Scope    : in     Dictionary.Scopes;
      OK       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         OK                         from Dictionary.Dict,
   --#                                         Pack_Sym,
   --#                                         Scope;
   is
   begin
      if Dictionary.PrefixAllowed (Pack_Sym, Scope) then
         OK := True;
         if Dictionary.Is_Generic_Package (The_Symbol => Pack_Sym) then
            OK := False;
            ErrorHandler.Semantic_Error
              (655,
               ErrorHandler.No_Reference,
               Node_Position (Node),
               Dictionary.GetSimpleName (Pack_Sym));
         end if;
      else
         OK := False;
         ErrorHandler.Semantic_Error (337, ErrorHandler.No_Reference, Node_Position (Node), Dictionary.GetSimpleName (Pack_Sym));
      end if;
   end Check_Package_Prefix;
   ------------------------------------------------------------------

   function In_Package_Initialization (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsLocalScope (Scope) and then Dictionary.IsPackage (Dictionary.GetEnclosingCompilationUnit (Scope));
   end In_Package_Initialization;

   ----------------------------------------------------------------

   function IndexesMatch (Target, Source : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   -------------------------------------------------------------------

   function IsExternalInterface (Pragma_Node : STree.SyntaxNode) return Boolean
   --# global in CommandLineData.Content;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
      is separate;
   -------------------------------------------------------------------

   -- Exported subprogram
   procedure CompUnit (Top_Node : in STree.SyntaxNode;
                       Do_VCG   : in Boolean) is separate;

end Sem;
