-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (SparkLex.Lex)
procedure NextLex (Token : out SPSymbols.SPTerminal)

-- The Ada Lexis allows the class of a token to be determined by the
-- first character in the text string representing it. Given the first
-- character of the string representing the token is at the current position
-- in the line buffer, CurrLine, NextLex determines the class of the token and
-- directly recognises the token or calls a procedure to recognise that class
-- of token.
-- On exit Token is set to a value representing the token and the line buffer
-- position is updated.

is
   Ch, Ch1 : Character;
begin

   LineManager.Record_Curr_Pos;

   LineManager.Inspect_Char (Ch => Ch);

   case Ch is
      when 'a' .. 'z' | 'A' .. 'Z' | '$' =>      -- letter or '$'

         --  In Ada and SPARK, identifiers may not start with a '$'
         --  character, but these are used by GNATPREP to introduce
         --  symbols that might be replaced, and we do want
         --  SPARKFormat to be able to process these successfully in
         --  annotations.  We therefore allow '$' to introduce an
         --  identififier here.  Rejecting the illegal case (in the
         --  Examiner) is performed in SPARKLex.Lex
         GetIdent (Token);
      when '0' .. '9' =>               -- digit
         GetNumber (Token);
      when '-' =>                      -- minus, comment or annotation_start
         HyphIntro (Token);
      when '"' =>                      -- string
         GetString (Token);
      when ''' =>                      -- apostrope or character literal
         ApostIntro (Token);
      when '<' =>                      -- < << <->
         LTIntro (Token);
      when '=' =>                      -- check for compound delimiters
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '>' =>
               Token := SPSymbols.arrow;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.equals;
         end case;
      when '.' =>                      -- point or double_dot
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '.' =>
               Token := SPSymbols.double_dot;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.point;
         end case;
      when '*' =>
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '*' =>
               Token := SPSymbols.double_star;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.multiply;
         end case;
      when ':' =>
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '=' =>
               Token := SPSymbols.becomes;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.colon;
         end case;
      when '/' =>
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '=' =>
               Token := SPSymbols.not_equal;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.divide;
         end case;
      when '>' =>
         LineManager.Accept_Char;
         LineManager.Inspect_Char (Ch => Ch1);
         case Ch1 is
            when '=' =>
               Token := SPSymbols.greater_or_equal;
               LineManager.Accept_Char;
            when '>' =>
               Token := SPSymbols.right_label_paren;
               LineManager.Accept_Char;
            when others =>
               Token := SPSymbols.greater_than;
         end case;
      when '&' =>                      -- simple delimiters
         Token := SPSymbols.ampersand;
         LineManager.Accept_Char;
      when '(' =>
         Token := SPSymbols.left_paren;
         LineManager.Accept_Char;
      when ')' =>
         Token := SPSymbols.right_paren;
         LineManager.Accept_Char;
      when '+' =>
         Token := SPSymbols.plus;
         LineManager.Accept_Char;
      when ',' =>
         Token := SPSymbols.comma;
         LineManager.Accept_Char;
      when ';' =>
         Token := SPSymbols.semicolon;
         LineManager.Accept_Char;
      when '|' =>
         Token := SPSymbols.vertical_bar;
         LineManager.Accept_Char;
      when '[' =>
         Token := SPSymbols.square_open;
         LineManager.Accept_Char;
      when ']' =>
         Token := SPSymbols.square_close;
         LineManager.Accept_Char;
      when '~' =>
         Token := SPSymbols.tilde;
         LineManager.Accept_Char;
      when '%' =>
         Token := SPSymbols.percent;
         LineManager.Accept_Char;
      when End_Of_Annotation =>
         Token := SPSymbols.annotation_end;
         LineManager.Accept_Char;
      when Line_Too_Long =>
         Token := SPSymbols.LONGLINE;
         LineManager.Accept_Char;
      when End_Of_Text =>
         Token := SPSymbols.SPEND;
      when others =>
         Token := SPSymbols.illegal_token;
         LineManager.Accept_Char;
   end case;

end NextLex;
