#include <math.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <pthread.h>
#include <glib.h>

#include "patch.h"
#include "specimen.h"
#include "driver.h"
#include "sample.h"

#define FT 0.0075                        /* time in seconds for fade out (Fade Time) */
#define FC 0.1                           /* constant factor by which fade reduces volume per second */
#define TC (DRIVER->getrate() * FT)      /* number of frames to fade out over (tail count) */
#define TP (1.0 / DRIVER->getrate())     /* period in seconds of one frame (tail period) */
#define CHANS 2                          /* stereo for now */

/* magic numbers */
enum {
     EVENTMAX = 1024,
     POLYPHONY = 64,
};

/* event codes */
enum {
     EV_OFF=0,
     EV_PLAY,
     EV_STOP
};

/* ADSR state codes */
enum {
     ST_IDLE = 0,
     ST_OFF,
     ST_ATTACK,
     ST_DECAY,
     ST_SUSTAIN,
     ST_RELEASE
};

/* contains information on a per note basis about envelope status */
typedef struct env_state_s {
     int state;
     int tick;
     float val;
     float release_val;  /* used during release to determine what level to decrease from */
     env_t *env;
} env_state_t;

/* type for array of currently playing notes  */
typedef struct playlist_s {
     int         active;         /* whether or not this not is being used */
     tick_t      tick;           /* when this note was supposed to be played */
     int         offset;         /* how many ticks until this note should be played */
     patch_t    *patch;          /* the patch associated with this note */
     int         note;           /* the midi note that activated us */
     float       velocity;       /* how loud to play this note (0 to 1.0) */
     int         positioni;      /* integer position indicator */
     guint32     positionf;      /* fractional position indicator */
     int         stepi;          /* integer rate ratio */
     guint32     stepf;          /* fractional rate ratio */
     char        dir;            /* current direction: 1 for forward, -1 for reverse */ 
     float       fl1;            /* low pass filter buffer */
     float       fb1;            /* band pass filter buffer */
     env_state_t vol;            /* volume envelope state tracker */
     env_state_t ffreq;          /* cutoff frequency envelope state tracker */
     env_state_t freso;          /* resonance envelope state */
     env_state_t pan;            /* pan envelope state */
} playlist_t;

/* type for ringbuffer of incoming events */
typedef struct event_s {
     patch_t *patch;
     tick_t tick;
     int event;
     float velocity;
     int note;
} event_t;

/* special structure for previewing samples */
typedef struct preview_s {
     sample_t sample;
     int active;
     int next_frame;
} preview_t;

/* general variables */
static int init = 0;                   /* whether mixer_init() has been called or not */
static float volume;                   /* master volume */
static preview_t preview;              
static playlist_t playlist[POLYPHONY];
static event_t event[EVENTMAX];
static event_t *event_feeder = event;
static event_t *event_consumer = event;

/* cubic interpolation filter coefficients, ripped off from
 * SoundTracker (do NOT fuck with Michael Krause) */
static float ct0[256];
static float ct1[256];
static float ct2[256];
static float ct3[256];

static void mixer_init()
{
     int i;

     /* build cubic coeff table */
     for(i = 0; i < 256; i++) {
	  float x1 = i / 256.0;
	  float x2 = x1*x1;
	  float x3 = x1*x1*x1;
	  ct0[i] = -0.5*x3 + x2 - 0.5*x1;
	  ct1[i] = 1.5*x3 - 2.5 * x2+1;
	  ct2[i] = -1.5*x3 + 2*x2 + 0.5*x1;
	  ct3[i] = 0.5*x3 - 0.5*x2;
     }
}

/* cubic interpolation routine:
   y0 = sample previous to current sample
   y1 = current sample
   y2 = sample after current sample
   y3 = sample after y2
   x  = fractional between y1 and y2 */
static float cubici(float y0, float y1, float y2, float y3, guint32 d)
{
     float s0;

     s0 =  y0 * ct0[d >> 24];
     s0 += y1 * ct1[d >> 24];
     s0 += y2 * ct2[d >> 24];
     s0 += y3 * ct3[d >> 24];

     return s0;
}

/* old fashioned linear interpolator */
static float lerp(float y0, float y1, float d)
{
     return y0*(1-d)+y1*d;
}

static void advance_feeder()
{
     event_feeder = (event_feeder + 1 >= event + EVENTMAX) ? 
	  event : event_feeder + 1;
}

static void advance_consumer()
{
     event_consumer = (event_consumer + 1 >= event + EVENTMAX) ?
	  event : event_consumer +1;
}

static void get_events(tick_t tick, tick_t last_tick)
{
     register int i;
     int gzr = 0; /* geezer, get it? */
     tick_t oldest = tick;
     float tmp;
     playlist_t *pl;

     /* check for new events */
     while (event_consumer != event_feeder) {
	  if (event_consumer->tick > tick)
	       break;
	  if (event_consumer->event == EV_PLAY) {
	       for (i = 0; i < POLYPHONY; i++) {
		    if (playlist[i].tick <= oldest) {
			 oldest = playlist[i].tick;
			 gzr = i;
		    }
		    if (!playlist[i].active)
			 break;
	       }

	       /* if we couldn't find a free slot, put this note in
		* place of the oldest one*/
	       if (i == POLYPHONY)
		    i = gzr;

	       /* grab this slot */
	       pl = &playlist[i];
	       pl->active = 1;
	       pl->patch = event_consumer->patch;

	       /* activate envelopes */
	       pl->vol.state = ST_ATTACK;
	       pl->vol.tick  = 0;
	       pl->vol.val   = 0.0;

	       memcpy(&pl->ffreq, &pl->vol, sizeof pl->ffreq);
	       memcpy(&pl->freso, &pl->vol, sizeof pl->freso);
	       memcpy(&pl->pan,   &pl->vol, sizeof pl->freso);

	       pl->vol.env   = &pl->patch->env_vol;
	       pl->pan.env   = &pl->patch->env_pan;
	       pl->ffreq.env = &pl->patch->env_ffreq;
	       pl->freso.env = &pl->patch->env_freso;

	       /* set time related params */
	       pl->tick = event_consumer->tick;
	       pl->offset = event_consumer->tick - last_tick;
	       if (pl->offset < 0)
		    pl->offset = 0;
	       
	       /* set other params */
	       pl->note = event_consumer->note;
	       pl->velocity = event_consumer->velocity;
	       pl->fl1 = 0;
	       pl->fb1 = 0;
	       
	       /* setup pitch scaling */
	       tmp = pow(2, (event_consumer->note - pl->patch->note)/12.0);
	       pl->stepi = tmp;
	       pl->stepf = (tmp - pl->stepi) * ~((guint32)0);
	       pl->positionf = 0;

	       /* setup direction */
	       if (pl->patch->play_mode & PM_REVERSE) {
		    pl->positioni = pl->patch->sample_stop;
		    pl->dir = -1;
	       
	       } else {
		    pl->positioni = pl->patch->sample_start;
		    pl->dir = 1;
	       }
	  } else if (event_consumer->event == EV_STOP) {
	       for (i = 0; i < POLYPHONY; i++) {
		    pl = &playlist[i];
		    if (pl->active && 
			pl->patch == event_consumer->patch &&
			(pl->note == event_consumer->note || event_consumer->note == -1)) {

			 /* tell envelopes to release */
			 pl->vol.state = ST_RELEASE;
			 pl->vol.tick  = event_consumer->tick - last_tick;
			 if (pl->vol.tick > 0) /* should almost always be true */
			      pl->vol.tick *= -1;

			 memcpy(&pl->ffreq, &pl->vol, sizeof(int)*2);
			 memcpy(&pl->freso, &pl->vol, sizeof(int)*2);
			 memcpy(&pl->pan,   &pl->vol, sizeof(int)*2);

			 pl->vol.release_val   = pl->vol.val;
			 pl->pan.release_val   = pl->pan.val;
			 pl->ffreq.release_val = pl->ffreq.val;
			 pl->freso.release_val = pl->freso.val;
		    }
	       }
	  }
	  advance_consumer();
     }
}

static void mixdown_preview(float *buf, int frames)
{
     register int i, j;
     
     /* if there is a sample currently being previewed let's throw it into the mix */
     if (preview.active && pthread_mutex_trylock(&preview.sample.mutex) == 0) {

	  if (preview.sample.sp != NULL) {

	       for (i = 0, j = preview.next_frame*CHANS; i < frames*CHANS && j < preview.sample.frames*CHANS; i++, j++)
		    buf[i] += preview.sample.sp[j] * (volume);

	       if ((preview.next_frame = j/CHANS) >= preview.sample.frames) {

		    preview.active = 0;

		    if (preview.sample.sp != NULL)
			 free(preview.sample.sp);

		    preview.sample.sp = NULL;
	       }
	  } else {
	       preview.active = 0;
	  }
	  pthread_mutex_unlock(&preview.sample.mutex);
     }
}

static void update_env(env_state_t *e)
{
     float vol;
     float d;

     /* inactive envelopes always stay at their peak */
     if (!e->env->active) {
	  e->val = e->env->peak;
	  return;
     }
     switch(e->state) {
     case ST_ATTACK:
	  if (e->env->attack == 0) {
	       vol = e->env->peak;
	  
	  } else {
	       d = (e->tick * 1.0) / e->env->attack;
	       vol = lerp(0.0, e->env->peak, d);
	  }

	  if (++e->tick > e->env->attack) {
	       e->state = ST_DECAY;
	       e->tick = 0;
	  }
	  break;
     case ST_DECAY:
	  if (e->env->decay == 0) {
	       vol = e->env->sustain * e->env->peak;
	  } else {
	       d = (e->tick * 1.0) / e->env->decay;
	       vol = lerp(e->env->peak, e->env->sustain * e->env->peak, d);
	  }

	  if (++e->tick > e->env->decay) {
	       e->state = ST_SUSTAIN;
	       e->tick = 0;
	  }
	  break;
     case ST_SUSTAIN:
	  vol = e->env->sustain * e->env->peak;
	  break;
     case ST_RELEASE:
	  if (e->env->release == 0) { /* should never be true */
	       vol = 0.0;
	  } else if (e->tick < 0){
	       vol = e->val;
	  } else {
	       d = (e->tick * 1.0) / e->env->release;
	       vol = lerp(e->release_val, 0.0, d);
	  }

	  if (++e->tick > e->env->release) {
	       e->state = ST_IDLE;
	       e->tick = 0;
	  }
	  break;
     default:
     case ST_IDLE:
	  vol = 0.0;
	  break;
     }

     e->val = vol;
}     

static void update_state(int i)
{
     playlist_t *pl = &playlist[i];
     float d;
     int release;

     /* update envelopes */
     update_env(&pl->ffreq);
     update_env(&pl->freso);
     update_env(&pl->pan);
     update_env(&pl->vol);

     /* if our volume envelope is inactive,
	we have to manually process the
	fade out to prevent clicks */
     if (!pl->vol.env->active && pl->vol.state == ST_RELEASE) {
	  release = DRIVER->getrate() * PATCH_RELEASE_MIN;

	  if (pl->vol.tick >= 0) {
	       d = (pl->vol.tick * 1.0) / release;
	       pl->vol.val = lerp(pl->vol.env->peak, 0.0, d);
	  }
	  if (++pl->vol.tick > release) {
	       pl->vol.state = ST_IDLE;
	       pl->vol.tick = 0;
	  }
     }
	 
     /* we stop the note when volume stops */
     if (pl->vol.state == ST_IDLE)
	  pl->active = 0;
}

int mixer_set_volume(float vol)
{
     if (vol < 0.0 || vol > 1.0)
	  return -1;

     volume = vol;
     return 0;
}

void mixer_play(patch_t *patch, char note, char vel, tick_t tick)
{
     event_feeder->patch = patch;
     event_feeder->note = note;
     event_feeder->tick = tick;
     event_feeder->event = EV_PLAY;
     event_feeder->velocity = vel / 127.0;
     advance_feeder();
}

void mixer_stop(patch_t *patch, char note, tick_t tick)
{
     event_feeder->patch = patch;
     event_feeder->tick = tick;
     event_feeder->note = note;
     event_feeder->event = EV_STOP;
     advance_feeder();
}

void mixer_preview(char *name)
{
     preview.active = 0;
     sample_load_file(&preview.sample, name);
     preview.next_frame = 0;
     preview.active = 1;
}


void mixer_flush()
{
     register int i;

     /* stop any currently playing events */
     for (i = 0; i < POLYPHONY; i++) 
	  playlist[i].active = 0;

     /* skip any queued ringbuffer events */
     event_consumer = event_feeder;
}

#define ACTIVE  (playlist[i].active)
#define DIR     (playlist[i].dir)
#define FB1     (playlist[i].fb1)
#define FFREQ   (playlist[i].ffreq.val)
#define FL1     (playlist[i].fl1)
#define FRAMES  (playlist[i].patch->sample.frames)
#define FRESO   (playlist[i].freso.val)
#define LSTART  (playlist[i].patch->loop_start)
#define LSTOP   (playlist[i].patch->loop_stop)
#define MODE    (playlist[i].patch->play_mode)
#define MUTEX   (&playlist[i].patch->sample.mutex)
#define MVOL    (volume)
#define OFFSET  (playlist[i].offset)
#define PAN     (playlist[i].pan.val)
#define POSF    (playlist[i].positionf)
#define POSI    (playlist[i].positioni)
#define PVOL    (playlist[i].vol.val)
#define SP      (playlist[i].patch->sample.sp)
#define SSTART  (playlist[i].patch->sample_start)
#define SSTOP   (playlist[i].patch->sample_stop)
#define STEPF   (playlist[i].stepf)
#define STEPI   (playlist[i].stepi)
#define VVOL    (playlist[i].velocity)

void mixer_mixdown(float *buf, int frames)
{
     register int i, j, k;                /* standard fare counters */
     int count;                           /* number of frames we will actually write for a given patch */
     float sample;                        /* middle man between what we have and what gets written */
     tick_t current_tick;                 /* what was this invocation called at (ass) */
     static tick_t last_tick = 0;         /* current_tick of last invocation */
     int posf_new;

     current_tick = getticks();

     /* make sure we do any initialization routines at least once */
     if (init == 0) {
	  mixer_init();
	  init = 1;
     }

     /* zero out the buffer */
     for (i = 0; i < (frames*CHANS); i++)
	  buf[i] = 0;

     /* get new events */
     get_events(current_tick, last_tick);

     /* do the actual stereo mixing */
     for (i = 0; i < POLYPHONY; i++) {
	  if (ACTIVE) {
	       if (pthread_mutex_trylock(MUTEX) != 0)
		    continue;

	       if (SP == NULL) {
		    ACTIVE = 0;
		    pthread_mutex_unlock(MUTEX);
		    continue;
	       }


	       /* determine the number of frames to write */
	       if (MODE & (PM_SINGLESHOT | PM_TRIM)) {
		    if ((DIR>0) && (frames + POSI - OFFSET > SSTOP)) {
			 count = SSTOP - POSI + 1;
		
		    } else if ((DIR<0) && (POSI - frames + OFFSET < SSTART)) {
			 count = POSI - SSTART + 1;

		    } else {
			 count = frames;
		    }
	       } else { /* loop modes can always provide the requested
			 * frames */
		    count = frames;
	       }
	       
	       for (j = OFFSET*CHANS; (j < count*CHANS) && (ACTIVE) && /* the last two tests are just insurance */
			 (POSI < FRAMES) && (POSI >= 0); j++)
	       {

		    k = j%CHANS; /* indicates the current channel */

		    /* get our sample, interpolating for pitch scaling
		     * if necessary */
		    if (STEPF == 0 && STEPI == 1) { /* no interpolation necessary */
			 sample = SP[POSI*CHANS+k];

		    }  /* forward, singleshot or trim */
		    else if ((MODE & PM_FORWARD) && (MODE & (PM_SINGLESHOT | PM_TRIM)))
		    {
			 sample = cubici((POSI<1)? 0: SP[POSI*CHANS-CHANS+k],
					      
					 SP[POSI*CHANS+k],
					      
					 (POSI+1>SSTOP)? 0: SP[POSI*CHANS+CHANS+k],
					      
					 (POSI+2>SSTOP)? 0: SP[POSI*CHANS+CHANS*2+k],
					      
					 POSF);

		    } /* forward and loop */
		    else if  ((MODE & PM_FORWARD) && (MODE & PM_LOOP))
		    {
			 sample = cubici((POSI==LSTART)? SP[LSTOP*CHANS+k]: ((POSI<1)? 0: SP[POSI*CHANS-CHANS+k]),

					 SP[POSI*CHANS+k],

					 (POSI+1>LSTOP)? SP[LSTART*CHANS+k]: SP[POSI*CHANS+CHANS+k],

					 (POSI+2>LSTOP)? SP[LSTART*CHANS+k]: SP[POSI*CHANS+CHANS*2+k],

					 POSF);
		    } /* reverse, singleshot or trim */
		    else if ((MODE & PM_REVERSE) && (MODE & (PM_SINGLESHOT | PM_TRIM)))
		    {
			 sample = cubici((POSI>=SSTOP)? 0: SP[POSI*CHANS+CHANS+k],
					      
					 SP[POSI*CHANS+k],
					      
					 (POSI-1<SSTART)? 0: SP[POSI*CHANS-CHANS+k],
					      
					 (POSI-2<SSTART)? 0: SP[POSI*CHANS-CHANS*2+k],
					      
					 POSF);
		    } /* reverse and loop */
		    else if  ((MODE & PM_REVERSE) && (MODE & PM_LOOP))
		    {
			 sample = cubici((POSI==LSTOP)? SP[LSTART*CHANS+k]: ((POSI>SSTOP)? 0: SP[POSI*CHANS+CHANS+k]),
					      
					 SP[POSI*CHANS+k],
					      
					 (POSI-1<LSTART)? SP[LSTOP*CHANS+k]: SP[POSI*CHANS-CHANS+k],
					      
					 (POSI-2<LSTART)? SP[LSTOP*CHANS+k]: SP[POSI*CHANS-CHANS*2+k],
					      
					 POSF);
		    } /* ping pong forwards */
		    else if ((MODE & PM_PINGPONG) && (DIR > 0))
		    {
			 sample = cubici((POSI==LSTART||POSI<1)? SP[POSI*CHANS+CHANS+k]: SP[POSI*CHANS-CHANS+k],

					 SP[POSI*CHANS+k],

					 (POSI+1>LSTOP)? SP[POSI*CHANS-CHANS+k]: SP[POSI*CHANS+CHANS+k],

					 (POSI+2>LSTOP)? SP[POSI*CHANS-CHANS+k]: SP[POSI*CHANS+CHANS*2+k],

					 POSF);
		    } /* ping pong reverse */
		    else if ((MODE & PM_PINGPONG) && (DIR < 0))
		    {
			 sample = cubici((POSI==LSTOP||POSI>SSTOP)? SP[POSI*CHANS-CHANS+k]: SP[POSI*CHANS+CHANS+k],

					 SP[POSI*CHANS+k],

					 (POSI-1<LSTART)? SP[POSI*CHANS+CHANS+k]: SP[POSI*CHANS-CHANS+k],

					 (POSI-2<LSTART)? SP[POSI*CHANS+CHANS+k]: SP[POSI*CHANS-CHANS*2+k],

					 POSF);
		    } else {
			 sample = 0; /* should never get here */
		    }


		    /* update envelopes */
		    update_state(i);

		    /* filter sample */
		    FB1 = FRESO * FB1 + FFREQ * (sample - FL1);
		    FL1 += FFREQ * FB1;
		    sample = FL1;
		    
		    /* gain adjust */
		    sample *= VVOL * PVOL * MVOL;

		    /* this part handles panning */
		    if (k == 0) {
			 buf[j]   += sample * ((PAN<0.0)? 1.0: 1-PAN);
			 buf[j+1] += sample * ((PAN>0.0)? 1.0: 1.0+PAN);

		    } else {
			 buf[j]   += sample * ((PAN>0.0)? 1.0: 1.0+PAN);
			 buf[j-1] += sample * ((PAN<0.0)? 1.0: 1-PAN);
	
			 /* determine the new value for POSI */
			 posf_new = POSF + STEPF;
			 if (posf_new < POSF)
			      POSI += DIR;
			 POSF = posf_new;
			 POSI += STEPI*DIR;

			 /* handle looping */
			 if (MODE & PM_LOOP) {
			      
			      if (MODE & PM_PINGPONG) {
				   
				   if ((DIR > 0) && (POSI > LSTOP)) {
					POSI = LSTOP - ((STEPI<1)? 1: STEPI);
					DIR = -1;

				   } else if ((DIR < 0) && (POSI < LSTART)) {
					POSI = LSTART + ((STEPI<1)? 1: STEPI);
					DIR = 1;
				   }
			      } else {
				   if ((DIR > 0) && (POSI > LSTOP)) {
					POSI = LSTART;

				   } else if ((DIR < 0) && (POSI < LSTART)) {
					POSI = LSTOP;
				   }
			      }
			 }
		    }
	       }
	       OFFSET -= count;
	       if (OFFSET < 0) OFFSET = 0;

	       /* stop if we're out of samples */
	       if ((POSI > SSTOP) && (MODE & (PM_SINGLESHOT | PM_TRIM)) && (DIR > 0)) {
		    ACTIVE = 0;

	       } else if ((POSI < SSTART) && (MODE & (PM_SINGLESHOT | PM_TRIM)) && (DIR < 0)) {
		    ACTIVE = 0;
	       }

	       pthread_mutex_unlock(MUTEX);
	  }
     }

     /* get the preview if any */
     mixdown_preview(buf, frames);

     last_tick = current_tick;
     return;
}
