#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <pthread.h>
#include <jack/types.h>

#include "specimen.h"
#include "patch.h"
#include "driver.h"
#include "mixer.h"
#include "sample.h"

static patch_t patch[PATCH_COUNT];

/* saves some typing */
const char *patch_strerror(int error)
{
     switch (error) {
     case PATCH_PARAM_INVALID:
	  return "patch parameter is invalid";
	  break;
     case PATCH_ID_INVALID:
	  return "patch id is invalid";
	  break;
     case PATCH_ALLOC_FAIL:
	  return "failed to allocate space for patch";
	  break;
     case PATCH_NOTE_INVALID:
	  return "specified note is invalid";
	  break;
     case PATCH_PAN_INVALID:
	  return "specified panning is invalid";
	  break;
     case PATCH_CHANNEL_INVALID:
	  return "specified channel is invalid";
	  break;
     case PATCH_VOL_INVALID:
	  return "specified volume is invalid";
	  break;
     case PATCH_MASTER_VOLUME_INVALID:
	  return "specified master volume is invalid";
	  break;
     case PATCH_PLAY_MODE_INVALID:
	  return "specified patch play mode is invalid";
	  break;
     case PATCH_LIMIT:
	  return "maximum patch count reached, can't create another";
	  break;
     case PATCH_SAMPLE_INDEX_INVALID:
	  return "specified sample is invalid";
	  break;
     default:
	  return "unknown error";
     }
}

/* makes sure a given patch is active and properly configured */
int patch_verify(int id)
{
     if (id < 0 || id >= PATCH_COUNT)
	  return 0;
     if (!patch[id].active)
	  return 0;

     return 1;
}

/* destroy all patches */
void patch_destroy_all()
{
     int id;

     for (id = 0; id < PATCH_COUNT; id++)
	  patch_destroy(id);

     return;
}
     
/* activates all patches matching given criteria */
void patch_activate(int chan, int note, int vel)
{
     int i, j;
     tick_t tick = getticks();

     /* process the cuts */
     for (i = 0; i < PATCH_COUNT; i++) {
	  if (patch[i].active && patch[i].channel == chan &&
	      (patch[i].note == note || (patch[i].range > 0 && note >= patch[i].lower_note &&
		   note <= patch[i].upper_note))) {

	       /* it's important that we send stop events before we do
		  start events, otherwise weird incidences may occur
		  where two samples are played with the same cut
		  values end up nullifying each other before they ever
		  get played */
	       if (patch[i].cut > 0) {
		    for (j = 0; j < PATCH_COUNT; j++) {
			 if (patch[j].active && 
			     patch[j].cut_by == patch[i].cut) {

			      mixer_stop(&patch[j], -1, tick);
			 }
		    }
	       }		    
	  }
     }

     /* process play events */
     for (i = 0; i < PATCH_COUNT; i++) {
	  if (patch[i].active && patch[i].channel == chan &&
	      (patch[i].note == note || (patch[i].range > 0 && note >= patch[i].lower_note &&
		   note <= patch[i].upper_note))) {

	       mixer_play(&patch[i], note, vel, tick);
	  }
     }
	      
     return;
}

/* activate a single patch with given id */
void patch_activate_by_id(int id)
{
     tick_t tick = getticks();

     if (id < 0 || id >= PATCH_COUNT)
	  return;
     if (!patch[id].active)
	  return;

     mixer_play(&patch[id], patch[id].note, 127, tick);
     return;
}
     
/* deactivate all active patches matching given criteria */
void patch_deactivate(int chan, int note)
{
     tick_t tick = getticks();
     int i;

     for (i = 0; i < PATCH_COUNT; i++) {
	  if (patch[i].active && !(patch[i].play_mode & PM_SINGLESHOT) &&
	      patch[i].channel == chan &&
	      (patch[i].note == note || (patch[i].range > 0 && note >= patch[i].lower_note &&
					 note <= patch[i].upper_note))) {
	       
	       mixer_stop(&patch[i], note, tick);
	  }
     }
     
     return;
}

void patch_deactivate_by_id(int id)
{
     tick_t tick = getticks();

     if (id < 0 || id >= PATCH_COUNT)
	  return;
     if (!patch[id].active)
	  return;
     if (patch[id].play_mode & PM_SINGLESHOT)
	  return;

     mixer_stop(&patch[id], patch[id].note, tick);
     return;
}

/* returns assigned patch id on success, negative value on failure */
int patch_create(const char *name)
{
     int id, i;
     env_t env_default;

     /* find an unoccupied patch id */
     for (id = 0; patch[id].active; id++)
	  if (id == PATCH_COUNT)
	       return PATCH_LIMIT;

     patch[id].active = 1;
     
     debug("Creating patch %s (%d).\n", name, id);

     strncpy(patch[id].name, name, PATCH_MAX_NAME);
     patch[id].name[PATCH_MAX_NAME-1] = '\0';
     patch[id].sample.name[0] = '\0';
     patch[id].sample.frames = 0;
     patch[id].channel = 0;
     patch[id].note = 60;
     patch[id].range = 0;
     patch[id].lower_note = 60;
     patch[id].upper_note = 60;
     patch[id].play_mode = PM_FORWARD | PM_SINGLESHOT;
     patch[id].cut = 0;
     patch[id].cut_by = 0;
     patch[id].sample_start = 0;
     patch[id].sample_stop = 0;
     patch[id].loop_start = 0;
     patch[id].loop_stop = 0;

     /* setup our generic envelope */
     env_default.active = 0.0;
     env_default.peak = 1.0;
     env_default.attack = 0.0;
     env_default.decay = 0.0;
     env_default.sustain = 1.0;
     env_default.release = DRIVER->getrate() * PATCH_RELEASE_MIN; /* by default, we want to have a tiny release to prevent clicks */

     /* setup patch envelopes */
     memcpy(&patch[id].env_vol,   &env_default, sizeof env_default);
     memcpy(&patch[id].env_ffreq, &env_default, sizeof env_default);
     memcpy(&patch[id].env_freso, &env_default, sizeof env_default);
     memcpy(&patch[id].env_pan,   &env_default, sizeof env_default);

     /* we want resonance and pan to be at zero by default */
     patch[id].env_freso.peak = 0.0;
     patch[id].env_pan.peak = 0.0;

     pthread_mutex_lock(&patch[id].sample.mutex);
     patch[id].sample.sp = NULL;
     pthread_mutex_unlock(&patch[id].sample.mutex);

     /* set display_index to next unique value */
     patch[id].display_index = 0;
     for (i = 0; i < PATCH_COUNT; i++) {
	  if (i == id)
	       continue;
	  if (patch[i].active && patch[i].display_index >= patch[id].display_index) {
	       patch[id].display_index = patch[i].display_index + 1;
	  }
     }

     return id;
}

int patch_duplicate(int target)
{
     int id, i;
     
     if (target < 0 || target > PATCH_COUNT ||
	 !patch[target].active)
	  return PATCH_ID_INVALID;

     /* find an unoccupied patch id */
     for (id = 0; patch[id].active; id++)
	  if (id == PATCH_COUNT)
	       return PATCH_LIMIT;

     debug("Creating patch (%d) from patch %s (%d).\n", id,
	   patch[target].name, target);

     memcpy(&patch[id], &patch[target], sizeof patch[id]);

     /* we don't want to share memory with the target patch */
     pthread_mutex_lock(&patch[id].sample.mutex);
     patch[id].sample.sp = NULL;
     pthread_mutex_unlock(&patch[id].sample.mutex);

     /* load same soundfile as target patch */
     if (patch[target].sample.sp != NULL) {
	  patch_sample_load(id, patch[target].sample.name);
     }

     /* set display_index to next unique value */
     patch[id].display_index = 0;
     for (i = 0; i < PATCH_COUNT; i++) {
	  if (i == id)
	       continue;
	  if (patch[i].active && patch[i].display_index >= patch[id].display_index) {
	       patch[id].display_index = patch[i].display_index + 1;
	  }
     }

     debug("chosen display: %d\n", patch[id].display_index);
     return id;
}

/* destroy a single patch with given id */
int patch_destroy(int id)
{
     int index;

     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     debug("Removing patch: %d\n", id);

     patch[id].active = 0;
     pthread_mutex_lock(&patch[id].sample.mutex);
     if (patch[id].sample.sp != NULL) {
	  free(patch[id].sample.sp);
	  patch[id].sample.sp = NULL;
     }
     pthread_mutex_unlock(&patch[id].sample.mutex);
     
     /* every active patch with a display_index greater than
	this patch's needs to have it's value decremented so
	that we preservere continuity */
     index = patch[id].display_index;
     for (id = 0; id < PATCH_COUNT; id++) {
	  if (patch[id].active && patch[id].display_index > index)
	       patch[id].display_index--;
     }

     return 0;
}

int patch_envelope_set(int id, adsr_t param, int val)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  patch[id].env_vol.active = val;
	  break;

     case ADSR_CUTOFF:
	  patch[id].env_ffreq.active = val;
	  break;

     case ADSR_RESONANCE:
	  patch[id].env_freso.active = val;
	  break;

     case ADSR_PANNING:
	  patch[id].env_pan.active = val;
	  break;

     default:
	  return PATCH_PARAM_INVALID;
     }

     return 0;
}

int patch_envelope_get(int id, adsr_t param)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     
     switch (param) {
     case ADSR_VOLUME:
	  return patch[id].env_vol.active;
	  break;

     case ADSR_CUTOFF:
	  return patch[id].env_ffreq.active;
	  break;

     case ADSR_RESONANCE:
	  return patch[id].env_freso.active;
	  break;

     case ADSR_PANNING:
	  return patch[id].env_pan.active;
	  break;
     }

     return PATCH_PARAM_INVALID;
}

int patch_attack_set(int id, adsr_t param, float secs)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (secs < 0 || secs > PATCH_ADSR_MAX)
	  return PATCH_PARAM_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  patch[id].env_vol.attack = sec2tick(secs);
	  break;

     case ADSR_CUTOFF:
	  patch[id].env_ffreq.attack = sec2tick(secs);
	  break;

     case ADSR_RESONANCE:
	  patch[id].env_freso.attack = sec2tick(secs);
	  break;

     case ADSR_PANNING:
	  patch[id].env_pan.attack = sec2tick(secs);
	  break;

     default:
	  return PATCH_PARAM_INVALID;
     }

     return 0;
}

float patch_attack_get(int id, adsr_t param)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     
     switch (param) {
     case ADSR_VOLUME:
	  return tick2sec(patch[id].env_vol.attack);
	  break;

     case ADSR_CUTOFF:
	  return tick2sec(patch[id].env_ffreq.attack);
	  break;

     case ADSR_RESONANCE:
	  return tick2sec(patch[id].env_freso.attack);
	  break;

     case ADSR_PANNING:
	  return tick2sec(patch[id].env_pan.attack);
	  break;
     }

     return PATCH_PARAM_INVALID;
}

int patch_decay_set(int id, adsr_t param, float secs)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (secs < 0 || secs > PATCH_ADSR_MAX)
	  return PATCH_PARAM_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  patch[id].env_vol.decay = sec2tick(secs);
	  break;

     case ADSR_CUTOFF:
	  patch[id].env_ffreq.decay = sec2tick(secs);
	  break;

     case ADSR_RESONANCE:
	  patch[id].env_freso.decay = sec2tick(secs);
	  break;

     case ADSR_PANNING:
	  patch[id].env_pan.decay = sec2tick(secs);
	  break;

     default:
	  return PATCH_PARAM_INVALID;
     }

     return 0;
}

float patch_decay_get(int id, adsr_t param)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  return tick2sec(patch[id].env_vol.decay);
	  break;

     case ADSR_CUTOFF:
	  return tick2sec(patch[id].env_ffreq.decay);
	  break;

     case ADSR_RESONANCE:
	  return tick2sec(patch[id].env_freso.decay);
	  break;

     case ADSR_PANNING:
	  return tick2sec(patch[id].env_pan.decay);
	  break;
     }

     return PATCH_PARAM_INVALID;
}

int patch_sustain_set(int id, adsr_t param, float vol)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (vol < 0 || vol > 1.0)
	  return PATCH_PARAM_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  patch[id].env_vol.sustain = vol;
	  break;

     case ADSR_CUTOFF:
	  patch[id].env_ffreq.sustain = vol;
	  break;

     case ADSR_RESONANCE:
	  patch[id].env_freso.sustain = vol;
	  break;

     case ADSR_PANNING:
	  patch[id].env_pan.sustain = vol;
	  break;

     default:
	  return PATCH_PARAM_INVALID;
     }

     return 0;
}

float patch_sustain_get(int id, adsr_t param)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  return patch[id].env_vol.sustain;
	  break;

     case ADSR_CUTOFF:
	  return patch[id].env_ffreq.sustain;
	  break;

     case ADSR_RESONANCE:
	  return patch[id].env_freso.sustain;
	  break;

     case ADSR_PANNING:
	  return patch[id].env_pan.sustain;
	  break;
     }

     return PATCH_PARAM_INVALID;
}

int patch_release_set(int id, adsr_t param, float secs)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (secs < 0 || secs > PATCH_ADSR_MAX)
	  return PATCH_PARAM_INVALID;

     /* make sure we always have at least a tiny release to ensure
	click free operation */
     if (secs < PATCH_RELEASE_MIN)
	  secs = PATCH_RELEASE_MIN;

     switch (param) {
     case ADSR_VOLUME:
	  patch[id].env_vol.release = sec2tick(secs);
	  break;

     case ADSR_CUTOFF:
	  patch[id].env_ffreq.release = sec2tick(secs);
	  break;

     case ADSR_RESONANCE:
	  patch[id].env_freso.release = sec2tick(secs);
	  break;

     case ADSR_PANNING:
	  patch[id].env_pan.release = sec2tick(secs);
	  break;

     default:
	  return PATCH_PARAM_INVALID;
     }

     return 0;
}

float patch_release_get(int id, adsr_t param)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     switch (param) {
     case ADSR_VOLUME:
	  return tick2sec(patch[id].env_vol.release);
	  break;

     case ADSR_CUTOFF:
	  return tick2sec(patch[id].env_ffreq.release);
	  break;

     case ADSR_RESONANCE:
	  return tick2sec(patch[id].env_freso.release);
	  break;

     case ADSR_PANNING:
	  return tick2sec(patch[id].env_pan.release);
	  break;
     }

     return PATCH_PARAM_INVALID;
}

int patch_cutoff_set(int id, float freq)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (freq < 0.0 || freq > 1.0)
	  return PATCH_PARAM_INVALID;

     patch[id].env_ffreq.peak = freq;
     return 0;
}

float patch_cutoff_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].env_ffreq.peak;
}

int patch_resonance_set(int id, float reso)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (reso < 0.0 || reso > 1.0)
	  return PATCH_PARAM_INVALID;

     patch[id].env_freso.peak = reso;
     return 0;
}

float patch_resonance_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].env_freso.peak;
}

int patch_display_index_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].display_index;
}

int patch_name_set(int id, const char *name)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     strncpy(patch[id].name, name, PATCH_MAX_NAME);
     return 0;
}

char *patch_name_get(int id)
{
     char *name;

     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  name = strdup("\0");
     else
	  name = strdup(patch[id].name);

     return name;
}

char *patch_sample_name_get(int id)
{
     char *name;

     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  name = strdup("\0");
     else
	  name = strdup(patch[id].sample.name);
     return name;
}

float const *patch_sample_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return NULL;

     return patch[id].sample.sp;
}

int patch_volume_set(int id, float vol)
{

     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (vol < 0 || vol > 1.0)
	  return PATCH_VOL_INVALID;

     patch[id].env_vol.peak = vol;
     return 0;
}

float patch_volume_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].env_vol.peak;
}

int patch_channel_set(int id, int channel)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (channel < 0 || channel > 15)
	  return PATCH_CHANNEL_INVALID;

     patch[id].channel = channel;
     return 0;
}

int patch_channel_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].channel;
}

int patch_note_set(int id, int note)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (note < 0 || note > 127)
	  return PATCH_NOTE_INVALID;

     patch[id].note = note;
     return 0;
}

int patch_note_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].note;
}

int patch_range_set(int id, int range)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     patch[id].range = range;
     return 0;
}

int patch_range_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].range;
}

int patch_lower_note_set(int id, int note)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (note < 0 || note > 127)
	  return PATCH_NOTE_INVALID;

     patch[id].lower_note = note;
     return 0;
}

int patch_lower_note_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].lower_note;
}

int patch_upper_note_set(int id, int note)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (note < 0 || note > 127)
	  return PATCH_NOTE_INVALID;

     patch[id].upper_note = note;
     return 0;
}

int patch_upper_note_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].upper_note;
}

int patch_pan_set(int id, float pan)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (pan < -1.0 || pan > 1.0)
	  return PATCH_PAN_INVALID;

     patch[id].env_pan.peak = pan;
     return 0;
}

float patch_pan_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].env_pan.peak;
}

int patch_play_mode_set(int id, play_mode_t mode)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     /* verify direction */
     if (mode & PM_FORWARD) {
	  if (mode & PM_REVERSE) {
	       return PATCH_PLAY_MODE_INVALID;
	  }
     } else if (mode & PM_REVERSE) {
	  if (mode & PM_FORWARD) {
	       return PATCH_PLAY_MODE_INVALID;
	  }
     } else {
	  return PATCH_PLAY_MODE_INVALID;
     }

     /* verify duration */
     if (mode & PM_SINGLESHOT) {
	  if ((mode & PM_TRIM) || (mode & PM_LOOP)) {
	       return PATCH_PLAY_MODE_INVALID;
	  }
     } else if (mode & PM_TRIM) {
	  if ((mode & PM_SINGLESHOT) || (mode & PM_LOOP)) {
	       return PATCH_PLAY_MODE_INVALID;
	  }
     } else if (mode & PM_LOOP) {
	  if ((mode & PM_SINGLESHOT) || (mode & PM_TRIM)) {
	       return PATCH_PLAY_MODE_INVALID;
	  }
     }

     /* make sure pingpong isn't frivolously set (just for style
      * points) */
     if ((mode & PM_PINGPONG) && !(mode && PM_LOOP)) {
	  return PATCH_PLAY_MODE_INVALID;
     }

     patch[id].play_mode = mode;
     return 0;
}

play_mode_t patch_play_mode_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].play_mode;
} 

int patch_cut_set(int id, int cut)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     patch[id].cut = cut;
     return 0;
}

int patch_cut_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].cut;
}

int patch_cut_by_set(int id, int cut_by)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     patch[id].cut_by = cut_by;
     return 0;
}

int patch_cut_by_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].cut_by;
}

int patch_sample_start_set(int id, int start)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (patch[id].sample.sp == NULL)
	  return 0;

     if (start < 0)
	  start = 0;

     patch[id].sample_start = start;
     if (start > patch[id].sample_stop)
	  patch[id].sample_stop = start;
     return 0;
}

int patch_sample_start_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].sample_start;
}

int patch_sample_stop_set(int id, int stop)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (patch[id].sample.sp == NULL)
	  return 0;

     if (stop >= patch[id].sample.frames)
	  stop = patch[id].sample.frames-1;

     patch[id].sample_stop = stop;
     if (stop < patch[id].sample_start)
	  patch[id].sample_start = stop;
     return 0;
}

int patch_sample_stop_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].sample_stop;
}

int patch_loop_start_set(int id, int start)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (patch[id].sample.sp == NULL)
	  return 0;

     if (start < patch[id].sample_start)
	  start = patch[id].sample_start;
     else if (start > patch[id].sample_stop)
	  start = patch[id].sample_stop;

     patch[id].loop_start = start;
     if (start > patch[id].loop_stop)
	  patch[id].loop_stop = start;

     return 0;
}

int patch_loop_start_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].loop_start;
}

int patch_loop_stop_set(int id, int stop)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (patch[id].sample.sp == NULL)
	  return 0;

     if (stop > patch[id].sample_stop)
	  stop = patch[id].sample_stop;
     else if (stop < patch[id].sample_start)
	  stop = patch[id].sample_start;

     patch[id].loop_stop = stop;
     if (stop < patch[id].loop_start)
	  patch[id].loop_start = stop;

     return 0;
}

int patch_loop_stop_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     return patch[id].loop_stop;
}

int patch_frames_get(int id)
{
     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;
     if (patch[id].sample.sp == NULL)
	  return 0;

     return patch[id].sample.frames;
}

int patch_count()
{
     int id, count;

     for (id = count = 0; id < PATCH_COUNT; id++)
	  if (patch[id].active) count++;

     return count;
}

/* place all patch ids, sorted in ascending order by display index,
 into array 'id' and return number of patches */
int patch_dump(int **dump)
{
     int i, j, id, count, tmp;

     *dump = NULL;

     /* determine number of patches */
     count = patch_count();

     if (count == 0)
	  return count;

     /* allocate dump */
     *dump = malloc(sizeof(int) * count);
     if (*dump == NULL)
	  return PATCH_ALLOC_FAIL;

     /* place active patches into dump array */
     for (id = i = 0; id < PATCH_COUNT; id++)
	  if (patch[id].active)
	       (*dump)[i++] = id;

     /* sort dump array by display_index in ascending order */
     for (i = 0; i < count; i++) {
	  for (j = i; j < count; j++) {
	       if (patch[(*dump)[j]].display_index < patch[(*dump)[i]].display_index) {
		    tmp = (*dump)[i];
		    (*dump)[i] = (*dump)[j];
		    (*dump)[j] = tmp;
	       }
	  }
     }

     return count;
}

int patch_sample_load(int id, char *name)
{
     int val;

     debug("Loading sample %s for patch %d\n", name, id);

     if (id < 0 || id >= PATCH_COUNT || !patch[id].active)
	  return PATCH_ID_INVALID;

     val = sample_load_file(&patch[id].sample, name);

     /* set the sample/loop start/stop point appropriately */
     patch[id].sample_start = 0;
     patch[id].sample_stop = patch[id].sample.frames-1;
     patch[id].loop_start = 0;
     patch[id].loop_stop = patch[id].sample.frames-1;

     return val;
}

void patch_sample_unload(int id)
{
     debug("Unloading sample for patch %d\n", id);
     pthread_mutex_lock(&patch[id].sample.mutex);
     
     if (patch[id].sample.sp != NULL) {
	  free(patch[id].sample.sp);
	  patch[id].sample.sp = NULL;
     }

     patch[id].sample.frames = 0;
     patch[id].sample_start = 0;
     patch[id].sample_stop = 0;
     patch[id].loop_start = 0;
     patch[id].loop_stop = 0;
     patch[id].sample.name[0] = '\0';

     pthread_mutex_unlock(&patch[id].sample.mutex);
}

void patch_sample_reload_all()
{
     int id;
     char *name;

     for (id = 0; id < PATCH_COUNT; id++) {
	  if (!patch[id].active)
	       continue;
	  name = patch_sample_name_get(id);
	  patch_sample_load(id, name);
	  if (name != NULL)
	       free(name);
     }

     return;
}
