#ifndef PATCH_H
#define PATCH_H

#include <jack/types.h>
#include <glib.h>

#include "sample.h"

/* magic numbers: integer */
enum { 
     PATCH_COUNT = 64,
     PATCH_MAX_NAME = 64,
     PATCH_ADSR_MAX = 3,
};

/* magic numbers: floating point */
#define PATCH_RELEASE_MIN 0.0075

/* error codes */
enum {
     PATCH_PARAM_INVALID = -1,
     PATCH_ID_INVALID = -2,
     PATCH_ALLOC_FAIL = -3,
     PATCH_NOTE_INVALID = -4,
     PATCH_PAN_INVALID = -5,
     PATCH_CHANNEL_INVALID = -6,
     PATCH_VOL_INVALID = -7,
     PATCH_MASTER_VOLUME_INVALID = -8,
     PATCH_PLAY_MODE_INVALID = -9,
     PATCH_LIMIT = -10,
     PATCH_SAMPLE_INDEX_INVALID = -11,
};

/* These are the bitfield constants for the different ways a patch can
   be played.  I've used comments to indicate mutual exclusion among
   groups. */
enum  { 
     /* direction*/
     PM_FORWARD = 1,
     PM_REVERSE = 1 << 1,
     /************/

     /* duration */
     PM_SINGLESHOT = 1 << 2,
     PM_TRIM = 1 << 3,
     PM_LOOP = 1 << 4,
     /***********/

     /* ping pong mode can be set independently of all the other
	params, but it should only be tested for if PM_LOOP is set */
     PM_PINGPONG = 1 << 5,
};

/* type for playmode bitfield */
typedef guint8 play_mode_t;

/* codes for the different parameters which have envelopes */
typedef enum adsr_s {
     ADSR_VOLUME,
     ADSR_CUTOFF,
     ADSR_RESONANCE,
     ADSR_PANNING,
} adsr_t;

/* the structure for parameters with ADSR envelopes */
typedef struct env_s {
     int attack;
     int decay;
     float sustain;
     int release;
     float peak;
     int active; /* set to non zero to turn on this envelope */
} env_t;

/* type for array of instruments (called patches) */
typedef struct patch_s {
     int active;                        /* whether this instance of patch_s is in use or not */
     sample_t sample;                   /* sample data */
     int display_index;                 /* order in which it is to be displayed */
     char name[PATCH_MAX_NAME];         /* string name */
     int channel;                       /* midi channel to listen on */
     int note;                          /* midi note to listen on */
     int range;                         /* whether to listen to a range of notes or not */
     int lower_note;                    /* lowest note in range */
     int upper_note;                    /* highest note in range */
     int cut;                           /* cut signal this patch emits */
     int cut_by;                        /* what cut signals stop this patch */
     int sample_start;                  /* the first frame to play */
     int sample_stop;                   /* the last frame to play */
     int loop_start;                    /* the first frame to loop at */
     int loop_stop;                     /* the last frame to loop at */
     play_mode_t play_mode;             /* how this patch is to be played */
     env_t env_vol;                     /* volume envelope */
     env_t env_pan;                     /* panning envelop */
     env_t env_ffreq;                   /* cutoff frequency envelope */
     env_t env_freso;                   /* resonance envelope */
} patch_t;

/* patch utilities */
void         patch_activate          (int chan, int note, int vel);
void         patch_activate_by_id    (int id);
int          patch_create            (const char *name);
void         patch_deactivate        (int chan, int note);
void         patch_deactivate_by_id  (int id);
int          patch_destroy           (int id);
void         patch_destroy_all       ();
int          patch_display_index_get (int id);
int          patch_dump              (int **dump);
int          patch_duplicate         (int id);
const char  *patch_strerror          (int error);
int          patch_verify            (int id);

/* envelope management */
int   patch_envelope_set (int id, adsr_t param, int val);
int   patch_attack_set   (int id, adsr_t param, float secs);
int   patch_decay_set    (int id, adsr_t param, float secs);
int   patch_sustain_set  (int id, adsr_t param, float vol);
int   patch_release_set  (int id, adsr_t param, float secs);
int   patch_envelope_get (int id, adsr_t param);
float patch_attack_get   (int id, adsr_t param);
float patch_decay_get    (int id, adsr_t param);
float patch_sustain_get  (int id, adsr_t param);
float patch_release_get  (int id, adsr_t param);

/* parameter setters */
int patch_channel_set      (int id, int channel);
int patch_cut_by_set       (int id, int cut_by);
int patch_cut_set          (int id, int cut);
int patch_cutoff_set       (int id, float freq);
int patch_loop_start_set   (int id, int start);
int patch_loop_stop_set    (int id, int stop);
int patch_lower_note_set   (int id, int note);
int patch_name_set         (int id, const char *name);
int patch_note_set         (int id, int note);
int patch_pan_set          (int id, float pan);
int patch_play_mode_set    (int id, play_mode_t mode);
int patch_range_set        (int id, int range);
int patch_resonance_set    (int id, float reso);
int patch_sample_start_set (int id, int start);
int patch_sample_stop_set  (int id, int stop);
int patch_upper_note_set   (int id, int note);
int patch_volume_set       (int id, float vol);

/* paramter getters */
int          patch_channel_get       (int id);
int          patch_cut_by_get        (int id);
int          patch_cut_get           (int id);
float        patch_cutoff_get        (int id);
int          patch_frames_get        (int id);
int          patch_loop_start_get    (int id);
int          patch_loop_stop_get     (int id);
int          patch_lower_note_get    (int id);
char        *patch_name_get          (int id);
int          patch_note_get          (int id);
float        patch_pan_get           (int id);
play_mode_t  patch_play_mode_get     (int id);
int          patch_range_get         (int id);
float        patch_resonance_get     (int id);
float const *patch_sample_get        (int id);
int          patch_sample_load       (int id, char *file);
char        *patch_sample_name_get   (int id);
void         patch_sample_reload_all ();
int          patch_sample_start_get  (int id);
int          patch_sample_stop_get   (int id);
void         patch_sample_unload     (int id);
int          patch_upper_note_get    (int id);
float        patch_volume_get        (int id);

#endif /* PATCH_H */
