#ifndef _STRINGS_H
#define _STRINGS_H

#include "../../lib/compat.h"
#include "cxx_macros.h"
#include <string>
#include <vector>

// Various symbols that we will need.
// For Greek letters, see greek.h

#define DEGREE_LATIN1	  "\xB0"
#define DEGREE_UTF8	  "\xC2\xB0"
#define COPYRIGHT_UTF8	  "\xC2\xA9"
#define WHITESPACE	  " \f\n\r\t\v"
#define DIGITS		  "0123456789"

// A class adding a few utilities to a vector of strings

class StringList : public std::vector<std::string> {
 private:
  mutable char ** c_strings;
  mutable size_t c_strings_size;
  void delstrings() const;
  
 public:
  typedef std::vector<std::string>::iterator       iterator;
  typedef std::vector<std::string>::const_iterator const_iterator;

  StringList();
  StringList(const StringList &s);
  StringList(size_t n, const std::string &s);
  StringList(const std::string &s, char tokenizer = ' ');
  ~StringList();
  StringList & operator = (const StringList &s);
  std::string & operator[] (size_t n);
  const std::string & operator[] (size_t n) const;

  void push_back(const char *);
  void push_back(const std::string &);
  void push_back(const StringList &);
  void stripspace();
  void eraseempty();
  void toupper();
  void tolower();
  void utf8ize();
  std::string flatten(char spacer = ' ') const;
  char ** c_str() const;
};

// Useful string functions

namespace starstrings {

  // Old compilers (egcs-2.91.66, for example) define tolower, toupper as
  // macros, which messes everything up.  So fix this:
  int toupper(int);
  int tolower(int);
	
  // convert numbers to strings
  std::string ltoa(long int l, int width = 0, bool zeropadding = false);
  std::string itoa(int i, int width = 0, bool zeropadding = false);
  std::string ftoa(double d, int precision = 4, int integer_width = 0,
		   bool zeropadding = false);

  // string utilities
  bool isempty(const std::string &);
  std::string uppercase(const std::string &);
  std::string lowercase(const std::string &);
  void stripspace(std::string &);
  void toupper(std::string &);
  void tolower(std::string &);
  void utf8ize(std::string &);
  bool find_and_replace(std::string & haystack, const std::string & needle,
			const std::string & replacement);

  // compare strings
  bool compare(const std::string &, const std::string &);
  bool case_compare(const std::string &, const std::string &);
  bool compare_n(const std::string &, const std::string &, size_t length);
  bool case_compare_n(const std::string &, const std::string &, size_t length);
    
  // Functions to convert between string and radian representations of
  // coordinates.  "symbols" decides whether or not to append h,m,s,deg,',"
  StringList dec_to_strs(double declination, bool symbols = false);  
  StringList ra_to_strs(double right_ascension, bool celestial = true,
			bool symbols = false);
  std::string dec_to_str(double declination, char spacer = ' ',
			 bool symbols = false);
  std::string ra_to_str(double right_ascension, char spacer = ' ',
			bool celestial_coords = true, bool symbols = false);

  double strs_to_dec(const StringList &decstrings);
  double strs_to_ra(const StringList &rastrings, bool celestial_coords = true);
  double strs_to_dec(const std::string &d, const std::string &m,
		     const std::string &s);
  double strs_to_ra(const std::string &h, const std::string &m,
		    const std::string &s, bool celestial_coords = true);
  double str_to_dec(const std::string &decstring, char tokenizer = ' ');
  double str_to_ra(const std::string &rastring, char tokenizer = ' ',
		   bool celestial_coords = true);

}; // end of namespace


// inline function definitions

inline void
StringList::delstrings() const
{
  for (size_t i = 0; i < c_strings_size; i++) delete [] c_strings[i];
  delete [] c_strings; c_strings = 0;
}

inline StringList::StringList() :
std::vector<std::string>(), c_strings(0), c_strings_size(0) { }

inline StringList::StringList(const StringList &s) :
std::vector<std::string>(dynamic_cast<const std::vector<std::string> &>(s)),
c_strings(0), c_strings_size(0) { }

inline StringList::StringList(size_t n, const std::string &s) :
std::vector<std::string>(n, s), c_strings(0), c_strings_size(0) { }

inline StringList::~StringList() { delstrings(); }

inline StringList &
StringList::operator = (const StringList &s)
{
  if (this == &s) return *this;
  std::vector<std::string>::operator =
    (dynamic_cast<const std::vector<std::string> &>(s));
  delstrings();
  return *this;
}

inline std::string &
StringList::operator [] (size_t n)
{
  if (n >= size()) resize(n + 1, "");
  return std::vector<std::string>::operator[] (n);
}

inline const std::string &
StringList::operator [] (size_t n) const
{
  static std::string empty_str("");
  if (n < size()) return std::vector<std::string>::operator[] (n);
  else return empty_str;
}

inline void
StringList::push_back(const char *s)
{ std::vector<std::string>::push_back(std::string(s)); }

inline void
StringList::push_back(const std::string &s)
{ std::vector<std::string>::push_back(s); }

inline void
StringList::push_back(const StringList &s)
{ citerate (StringList, s, i) push_back(*i); }

inline void
StringList::stripspace()
{ iterate (StringList, *this, i) starstrings::stripspace(*i); }

inline void
StringList::eraseempty()
{
  stripspace();
  iterate (StringList, *this, i)
    if (starstrings::isempty(*i))
      erase(i--);
}

inline void
StringList::toupper()
{ iterate (StringList, *this, i) starstrings::toupper(*i); }

inline void
StringList::tolower()
{ iterate (StringList, *this, i) starstrings::tolower(*i); }

inline void
StringList::utf8ize()
{ iterate (StringList, *this, i) starstrings::utf8ize(*i); }

inline int
starstrings::tolower(int c)
{ return (c >= 'A' && c <= 'Z') ? (c - 'A' + 'a') : c; }

inline int
starstrings::toupper(int c)
{ return (c >= 'a' && c <= 'z') ? (c + 'A' - 'a') : c; }
  
inline std::string
starstrings::itoa(int i, int width, bool zeropadding)
{ return starstrings::ltoa(static_cast<long int>(i), width, zeropadding); }
  
inline bool
starstrings::isempty(const std::string &s)
{ return s.find_first_not_of(WHITESPACE) >= s.size(); }
  
inline void
starstrings::stripspace(std::string &s)
{
  s.replace(0, s.find_first_not_of(WHITESPACE), "");
  s.replace(s.find_last_not_of(WHITESPACE) + 1, s.size(), "");
}

inline bool
starstrings::find_and_replace(std::string & haystack,
			      const std::string & needle,
			      const std::string & replacement)
{
  size_t posn = haystack.find(needle);
  if (posn < haystack.size()) {
    haystack.replace(posn, needle.size(), replacement);
    return true;
  }
  else return false;
}
  
inline std::string
starstrings::uppercase(const std::string &s)
{
  std::string result = s;
  iterate (std::string, result, i) *i = starstrings::toupper(*i);
  return result;
}
    
inline std::string
starstrings::lowercase(const std::string &s)
{
  std::string result = s;
  iterate (std::string, result, i) *i = starstrings::tolower(*i);
  return result;
}
 
inline void
starstrings::toupper(std::string &s)
{ iterate (std::string, s, i) *i = starstrings::toupper(*i); }
  
inline void
starstrings::tolower(std::string &s)
{ iterate (std::string, s, i) *i = starstrings::tolower(*i); }
  
inline bool
starstrings::compare(const std::string &s1, const std::string &s2)
{ return s1 == s2; }
  
inline bool
starstrings::case_compare(const std::string &s1, const std::string &s2)
{ return uppercase(s1) == uppercase(s2); }

inline bool
starstrings::compare_n(const std::string &s1, const std::string &s2,
		       size_t n)
{ return std::string(s1, 0, n) == std::string(s2, 0, n); }

inline bool
starstrings::case_compare_n(const std::string &s1, const std::string &s2,
			    size_t n)
{ return compare_n(uppercase(s1), uppercase(s2), n); }
  
inline std::string
starstrings::dec_to_str(double declination, char spacer, bool symbols)
{ return dec_to_strs(declination, symbols).flatten(spacer); }
  
inline std::string
starstrings::ra_to_str(double right_ascension, char spacer,
		       bool celestial_coords, bool symbols)
{ return ra_to_strs(right_ascension,celestial_coords,symbols).flatten(spacer); }

inline double
starstrings::str_to_dec(const std::string &decstring, char tokenizer)
{ return strs_to_dec(StringList(decstring, tokenizer)); }
  
inline double
starstrings::str_to_ra(const std::string &rastring, char tokenizer,
		       bool celestial_coords)
{ return strs_to_ra(StringList(rastring, tokenizer), celestial_coords); }

#endif

