// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/*
  convert.cc
  Contains the main() function for the starconvert utility and all the
  functions allowing it to convert a star data file to StarPlot format.
*/

#include "convert.h"
#include <iostream>

using std::istream;
using std::ostream;
using std::string;
using std::cin;
using std::cout;
using std::cerr;
using std::endl;

// defined in names.cc:
extern void get_names(const string &, namedata, StringList *, StringList *);

Star parse_star_record(const string &, parsedata);
void get_coordinates(const string &, coordinates, StringList *);
void get_mag_distance(const string &, const string &, characteristics,
		      double *, double *);
void check_for_double_stars(const string &, systems, Star *, Star *);


void parse_star_file(istream & infile, ostream & outfile, parsedata format,
		     bool using_stdout)
{
  int lineno = 0;
  bool sun_found = false, moredata = true;
  char record[1000];
  Star tempstar, currentstar = BADSTAR, previousstar;

  // boring legal disclaimer:
  outfile << "This StarPlot data file was automatically generated by "
          << "starconvert v. " << STARPLOT_VERSION << "." << endl
	  << "If this file was derived from a copyrighted source, you "
	  << "may not redistribute" << endl << "it without the permission of "
	  << "the original author." << endl << endl 
	  << "If you obtained this data file by installing one of the data "
	  << "packages available" << endl << "on the StarPlot web page, you "
	  << "should see the file copyright notice in that" << endl
	  << "package for more information." << endl << endl;

  do {
    moredata = infile.getline(record, 999, '\n');
    record[999] = 0;

    // $ ; and , have special meanings to StarPlot, so purge them:
    for (unsigned int i = 0; i < strlen(record); i++)
      if (record[i] == '$' || record[i] == ';' || record[i] == ',')
        record[i] = ' ';
  
    tempstar = parse_star_record(record, format);
    if (tempstar.GetStarNames()[0] != BADSTARNAME || !moredata) {
      previousstar = currentstar;
      currentstar = tempstar;
    }
    else continue;

    if (previousstar.GetStarNames()[0] != BADSTARNAME) {
      Rules r;
      r.CelestialCoords = true;
      check_for_double_stars(record, format.Systems, 
			     &previousstar, &currentstar);
      StringList starinfo = previousstar.GetInfo(r, false, ',');
      
      // write star information to output
      outfile << "$ " << starinfo[1];
      if (starinfo.size() >= 11) {
	for (unsigned int i = 10; i < starinfo.size(); i++)
	  outfile << ", " << starinfo[i];
      }
      outfile << ";" << endl << "  ";
      
      outfile << starinfo[2] << "; " << starinfo[3] << "; " << starinfo[4] 
	      << "; 0;" << endl << "  " << starinfo[5] << "; ";
      outfile << (starstrings::uppercase(starinfo[1]) == "SUN" ?
		  (sun_found = true, "4.85") : starinfo[6]);
      outfile << "; " << starinfo[9] << "; " << starinfo[7] << ";" << endl;
      
      if (starinfo[8].size())
	outfile << "  " << starinfo[8] << endl;
      outfile << endl;
    }
    record[0] = 0;

    lineno++;
    if (!using_stdout && !(lineno % 1000))
      cout << lineno << " records converted." << endl;

  } while (moredata) ;

  // if the Sun is not in the data file, append it at the end!
  if (!sun_found)
    outfile << "$ Sun, Sol;" << endl << "  ; ; 0; 0;" << endl 
    	    << "  G2 V; 4.85; 0; ;" << endl << endl;
  return;
}


// parse_star_record(): converts the current record to a Star

Star parse_star_record(const string &record, parsedata format)
{
  if (starstrings::isempty(record)) return BADSTAR;

  double starmag, stardist;
  string spectrum;
  StringList starnames, starcoord, starcomments, starmembership;
  Star result;

  // find the first non-empty spectral class datum out of those specified
  //  in the specification file
  unsigned int i = 0;
  do {
    if (format.Charact.specclass_start[i] < record.size()) {
      spectrum = record.substr(format.Charact.specclass_start[i],
		    	       format.Charact.specclass_len[i]);
      starstrings::stripspace(spectrum);
    }
    i++;
  } while (i < format.Charact.specclass_len.size() &&
	   starstrings::isempty(spectrum)) ;

  // convert comments (if any) to a StringList
  if (format.Comments.comments_len > 0 &&
      format.Comments.comments_start < record.size()) {
    string commentstring = record.substr(format.Comments.comments_start,
		    			 format.Comments.comments_len);
    starcomments = StringList(commentstring, ' ');
    starcomments.eraseempty();
  }

  get_coordinates(record, format.Coord, &starcoord);
  get_mag_distance(record, spectrum, format.Charact, &stardist, &starmag);
  get_names(record, format.Names, &starnames, &starcomments);

  if (starnames.size() && (starstrings::uppercase(starnames[0]) == "SUN"))
    stardist = 0.0;

  if (stardist >= 0.0) {
    std::ostringstream starstring;

    citerate_until (StringList, starnames, strptr, -1)
      starstring << *strptr << ", ";
    if (starnames.size())
      starstring << starnames[starnames.size() - 1];
    else
      starstring << "<no name>";
    starstring << "; ";

    for (unsigned int i = 0; i < 6; i++)
      starstring << starcoord[i] << (((i + 1) % 3) ? ", " : "; ");
    starstring << stardist << "; 0; " << spectrum << "; "
	       << starmag << "; " << "0; ; ";

    citerate (StringList, starcomments, strptr)
      starstring << *strptr << " ";

    result = Star(starstring.str(),
		  false /* no fast conversion */,
		  false /* no name translation */);
  }

  else {
    cerr << "*** Cannot obtain distance to star " << starnames[0] << "; "
	 << "leaving it out." << endl;
    result = BADSTAR;
  }

  return result;
}


void get_coordinates(const string & record, coordinates c, 
		     StringList * coordstrings)
{
  coordstrings->clear();
  
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++) {
    if (c.coord_len[i] > 0 && c.coord_start[i] < record.size())
      coordstrings->push_back(record.substr(c.coord_start[i], c.coord_len[i]));
    else
      coordstrings->push_back(string(""));
  }
  coordstrings->stripspace();

  // fourth element is the sign, so concat it with fifth element and erase
  (*coordstrings)[3] += (*coordstrings)[4];
  coordstrings->erase(coordstrings->begin() + 4);
  
  if (!c.isCelestial) {
    // in this case we convert from Galactic coordinates so that contents
    // of output file are in celestial coordinates
    double theta, phi;
    SolidAngle omega;

    phi = starstrings::strs_to_ra((*coordstrings)[0], (*coordstrings)[1],
		  		  (*coordstrings)[2], GALACTIC);
    theta = starstrings::strs_to_dec((*coordstrings)[3], (*coordstrings)[4],
				     (*coordstrings)[5]);
    omega = SolidAngle(phi, theta).toCelestial();
    coordstrings->clear();
    coordstrings->push_back(starstrings::ra_to_strs(omega.getPhi()));
    coordstrings->push_back(starstrings::dec_to_strs(omega.getTheta()));
  }
  return;
}


// get_mag_distance(): This function obtains the distance and magnitude of
//  the star.  It goes through the distance/magnitude formatting pairs 
//  specified in the config file, and uses the first pair which gives an 
//  acceptable result.

void get_mag_distance(const string &record, const string &spectrum,
		      characteristics c, double *dist, double *mag)
{
  double tempmag, tempdist;

  for (unsigned int i = 0; i < c.distarray.size(); i++) {
    if (c.magarray[i].len <= 0 || c.magarray[i].start >= record.size())
      continue;
    char *endptr;
    string magstring = record.substr(c.magarray[i].start, c.magarray[i].len);
    starstrings::stripspace(magstring);
    tempmag = std::strtod(magstring.c_str(), &endptr);
    if (starstrings::isempty(magstring) || (*endptr != 0 && *endptr != '.'))
      continue;

    if (c.distarray[i].type == SPECCLASS) {
      if (starstrings::isempty(spectrum) || c.magarray[i].type == ABSOLUTE)
	continue;
      SpecClass sclass = SpecClass(spectrum);
      sclass.initialize();
      double absmag = sclass.absmag();
      if (absmag < -25 || absmag > 25) continue;
      
      // Technically this should also account for the effect of 
      //  interstellar dust on the visual magnitude...
      tempdist = starmath::get_distance(tempmag, absmag);

      // should NOT be using spectral class distance estimate for nearby stars:
      if (tempdist < 20 /* LY */) continue;

      *dist = starmath::sigdigits(tempdist, 3);
      *mag = absmag;
      return;
    }

    else if (c.distarray[i].len > 0 && c.distarray[i].start < record.size()) {
      string diststring = record.substr(c.distarray[i].start, 
		      			c.distarray[i].len);
      starstrings::stripspace(diststring);
      tempdist = std::strtod(diststring.c_str(), &endptr);
      if (starstrings::isempty(diststring) || (*endptr != 0 && *endptr != '.'))
	continue;

      double parallaxerr = 0.0;
      if ((c.distarray[i].type == ARCSEC || c.distarray[i].type == MILLIARCSEC)
	  && c.distarray[i].errorlen > 0
	  && c.distarray[i].errorstart < record.size()) {
	string errstring = record.substr(c.distarray[i].errorstart,
					 c.distarray[i].errorlen);
	starstrings::stripspace(errstring);
	parallaxerr = std::strtod(errstring.c_str(), &endptr);
	if (starstrings::isempty(errstring) || parallaxerr < 0 
	    || (*endptr != 0 && *endptr != '.'))
	  continue;
      }
	
      switch (c.distarray[i].type) {
      case MILLIARCSEC: tempdist /= 1000.0; parallaxerr /= 1000.0;
      case ARCSEC:
	{
	  // if parallax is < min. acceptable parallax, go to next rule
	  if (tempdist < c.distarray[i].minparallax) break;
	  // if relative error is > max. acceptable error, go to next rule
	  else if (c.distarray[i].errorlen > 0 
		   && parallaxerr > c.distarray[i].maxerror * tempdist) break;
	  else tempdist = 1.0 / tempdist;
	}
      case PC:          tempdist *= LY_PER_PC;
      case LY:
	{
	  *dist = starmath::sigdigits(tempdist, 3);
	  if (c.magarray[i].type == ABSOLUTE)
	    *mag = tempmag;
	  else {
	    tempmag = starmath::get_absmag(tempmag, tempdist);
	    *mag = starmath::roundoff(tempmag, 1);
	  }
	  return;
	}
      case SPECCLASS: break; // do nothing
      } // end switch statement
    } 
  } // end for loop
  
  // if we haven't returned yet, something is wrong; flag the values.
  *mag = *dist = -9999;
  return;
}


// Check for double systems.  
// Not yet implemented.

void check_for_double_stars(const string &record, systems s,
			    Star *previous, Star *current)
{
  double distdiff 
    = (previous->GetStarXYZ() - current->GetStarXYZ()).magnitude();
  if (distdiff < 0.2 /* light-years */) {
    if (previous->GetStarMembership().size() == 0) {
	  




    }
    else {
	
    }
  }
  else {
    
  }
}

#define l(arg) arg << endl
#define t(arg) "\t" << arg << endl

void printusage()
{
  cout << l(PROGNAME << " usage:")
       << l(PROGNAME << " specfile infile [outfile]")
       << t("Uses specifications in `specfile' to convert `infile' to StarPlot")
       << t("data format and writes the results to `outfile' if specified, or")
       << t("standard output if not.  The first two arguments are required.")
       << t("Exactly ONE may be replaced with a dash for reading from stdin.")
       << endl;
  cout << l(PROGNAME << ": version " << STARPLOT_VERSION
       <<   " (C) 2000-2002 Kevin B. McCarty")
       << l(PROGNAME << " comes with ABSOLUTELY NO WARRANTY; for details, see")
       << l("the file " << DOCDIR << "/COPYING.");
  return;
}


// main(): This doesn't do much besides error checking, and calling the two
//  top-level parsing functions (one for the specification file, and one for
//  the data file).

int main(int argc, char **argv)
{
  // we don't need to sync with printf() and friends since they aren't used
  std::ios::sync_with_stdio(false);
	
  if (argc != 3 && argc != 4) {
    printusage();
    return 0;
  }

  parsedata specdata;
  std::ifstream infile, specfile;
  std::ofstream outfile;

  if (strcmp(argv[1], "-") == 0) {
    if (strcmp(argv[2], "-") == 0) {
      printusage();
      return 0;
    }
    else {
      if (! cin.good()) {
	cerr << "Can't read specifications from stdin - not a tty?" << endl;
	return EXIT_FAILURE;
      }
      parse_config_file(cin, &specdata);
    }
  }
  else {
    specfile.open(argv[1]);
    if (! specfile.good()) {
      cerr << "Can't open specification file `" << argv[1]
	   << "' for input; exiting." << endl;
      return EXIT_FAILURE;
    }
    parse_config_file(specfile, &specdata);
    specfile.close();
  }

  if (argc == 4) {
    outfile.open(argv[3]);
    if (! outfile.good()) {
      cerr << "Can't open output file `" << argv[3] 
	   << "' for output; exiting." << endl;
      return EXIT_FAILURE;
    }
  }
  
  if (strcmp(argv[2], "-") == 0) {
    if (! cin.good()) {
      cerr << "Can't read star data from stdin - not a tty?" << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(cin, outfile, specdata, false);
    else           parse_star_file(cin, cout, specdata, true);
  }

  else {
    infile.open(argv[2]);
    if (! infile.good()) {
      cerr << "Can't open star data file `" << argv[2]
	   << "' for input; exiting." << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(infile, outfile, specdata, false);
    else           parse_star_file(infile, cout, specdata, true);
    infile.close();
  }

  if (argc == 4) {
    outfile.close();
  }

  return 0;
}

