// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// gtkviewer.h - wrapper class for the GTK GUI.
//

#ifndef _GTK_VIEWER_H_
#define _GTK_VIEWER_H_

#include "../classes/viewer.h"   // base class
#include "../classes/colors.h"   // for colors
#include "../classes/strings.h"  // for itoa
#include "../classes/mathdefs.h" // for RAD_PER_DEGREE, ROUND
#include "../classes/greek.h"

#include <gtk/gtk.h>
#include <string>

class GTKViewer : public StarViewer {

 private:
  GdkPixmap *pm;
  GtkWidget *widget;
  GdkGC     *gc; /* the `graphics context' has fg and bg colors, etc. */
  int	    filled;
  color_t   drawing_color;
  
  // Functions to help drawing text
  unsigned int draw_text_helper(const std::string &text,
				unsigned int x, unsigned int y);
  void draw_text_with_greek(const std::string &text,
			    unsigned int x, unsigned int y);

 public:
  GTKViewer(GdkPixmap *PM, GtkWidget *w);
  ~GTKViewer();

  unsigned int width() const;
  unsigned int height() const;
  void setcolor(unsigned long color);
  void setfill(bool fillmode);
  void setfont(std::string font_description);
  void fill(unsigned long color);
  void drawline(unsigned int x1, unsigned int y1,
		unsigned int x2, unsigned int y2);
  void drawbox(unsigned int left, unsigned int top,
	       unsigned int width, unsigned int height);
  void drawcircle(unsigned int center_x, unsigned int center_y, int radius);
  void drawstar(unsigned int center_x, unsigned int center_y, int radius);
  void drawellipse(unsigned int center_x, unsigned int center_y,
		   int semiwidth, int semiheight);
  void drawarc(unsigned int center_x, unsigned int center_y, // arc center
	       int semiwidth, int semiheight,    
	       double startangle, double endangle); // angles in radians
  void drawtext(const std::string &text, unsigned int x, unsigned int y);
  void drawtext(char text, unsigned int x, unsigned int y);
};

// inline function definitions

inline GTKViewer::GTKViewer(GdkPixmap *PM, GtkWidget *w) :
pm(PM), widget(w), gc(gdk_gc_new(pm)), filled(true) { gdk_rgb_init(); }

inline GTKViewer::~GTKViewer() { gdk_gc_unref(gc); }

inline unsigned int
GTKViewer::width() const
{ gint w, h; gdk_window_get_size(pm, &w, &h); return w; }

inline unsigned int
GTKViewer::height() const
{ gint w, h; gdk_window_get_size(pm, &w, &h); return h; }

inline void
GTKViewer::setfill(bool fillmode) { filled = fillmode; }

inline void
GTKViewer::setcolor(color_t color)
{
  gdk_rgb_gc_set_foreground(gc, color);
  gdk_rgb_gc_set_background(gc, color);
  drawing_color = color;
}

inline void
GTKViewer::setfont(std::string font_description)
{
  gtk_widget_modify_font (GTK_WIDGET (widget),
    pango_font_description_from_string (font_description.c_str()));
}

inline void
GTKViewer::fill(color_t color)
{
  setcolor(color);
  gdk_draw_rectangle(pm, gc, /*filled*/ true, 0, 0, width(), height());
}

inline void
GTKViewer::drawline(unsigned int x1, unsigned int y1,
		    unsigned int x2, unsigned int y2)
{ gdk_draw_line(pm, gc, x1, y1, x2, y2); }

inline void
GTKViewer::drawbox(unsigned int left, unsigned int top,
		   unsigned int width, unsigned int height)
{ gdk_draw_rectangle(pm, gc, filled, left, top, width, height); }

// GDK uses units of (1/64) degree for angles.  God only knows why.

// gdk_draw_arc has the neat design feature that if you pass it negative
// dimensions, it makes a hyperbola instead of an ellipse.  However, I don't
// think the user would expect a "drawellipse" function to produce a
// hyperbola, so I've used abs() to ensure that all of gdk_draw_arc's args
// are positive.

inline void
GTKViewer::drawcircle(unsigned int xc, unsigned int yc, int radius)
{
  radius = abs(radius);
  gdk_draw_arc(pm, gc, filled, xc - radius, yc - radius,
	       radius * 2, radius * 2, /*angle1*/ 0, /*angle2*/ 360 * 64);
}

inline void
GTKViewer::drawellipse(unsigned int xc, unsigned int yc,
		       int semiwidth, int semiheight)
{
  semiwidth = abs(semiwidth); semiheight = abs(semiheight);
  gdk_draw_arc(pm, gc, filled, xc - semiwidth, yc - semiheight,
	       semiwidth * 2, semiheight * 2, 0, 360 * 64);
}

inline void
GTKViewer::drawarc(unsigned int xc, unsigned int yc, // arc center
		   int semiwidth, int semiheight,
		   double startangle, double endangle) // angles in radians
{
  semiwidth = abs(semiwidth); semiheight = abs(semiheight);
  gdk_draw_arc(pm, gc, false, xc - semiwidth, yc - semiheight,
	       semiwidth * 2, semiheight * 2,
	       ROUND(startangle * 64 / RAD_PER_DEGREE),
	       ROUND((endangle - startangle) * 64 / RAD_PER_DEGREE));
}

inline void
GTKViewer::drawtext(const std::string &text, unsigned int x, unsigned int y)
{ draw_text_with_greek(text, x, y); }

inline void
GTKViewer::drawtext(char text, unsigned int x, unsigned int y)
{ draw_text_helper(std::string(1, text), x, y); }

#endif // _GTK_VIEWER_H_
