# -*- coding: utf-8 -*-
# Copyright (c) 2007 Jimmy Rönnholm
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import os
import shutil

from bootconfig import utils

class Usplash:

    """Represent the Usplash configuration."""

    def __init__(self):
        """Set the variables"""
        self.binary = '/sbin/usplash'
        self.config_file = '/etc/usplash.conf'
        self.themes_directory = '/usr/lib/usplash/'
        self.alternatives_file = '/etc/alternatives/usplash-artwork.so'
        self.update_initramfs_command = 'update-initramfs -uk all'
        if not os.path.isfile(self.binary):
            raise SystemExit('Usplash not found.')
        if not os.path.isdir(self.themes_directory):
            os.mkdir(self.themes_directory)
        if not os.path.isfile(self.config_file):
            self.set_resolution('1024x768')
        self.update_initramfs = False

    def get_resolution(self):
        """Return the resolution in the form '1024x768'.

        Return -1 if failed.
        
        """
        line = utils.get_and_trim_line(self.config_file, 'xres=')
        xres = utils.extract_number(line)
        line = utils.get_and_trim_line(self.config_file, 'yres=')
        yres = utils.extract_number(line)
        if xres == -1 or yres == -1:
            return -1
        resolution = xres + 'x' + yres
        return resolution

    def set_resolution(self, resolution):
        """Set the resolution.

        resolution ex. '1024x768'

        """
        resolution = resolution.split('x')
        xres = resolution[0]
        yres = resolution[1]
        lines = []
        lines.append('# Usplash configuration file\n')
        lines.append('xres=' + xres + '\n')
        lines.append('yres=' + yres + '\n')
        utils.write_lines_to_file(self.config_file, lines)
        self.update_initramfs = True

    def get_active_theme(self):
        """Return the name of the currently active Usplash theme.
        
        Return -1 if one cannot be found.
        
        """
        name = self.themes_directory + 'usplash-artwork.so'
        if os.path.islink(name):
            current_usplash_path = os.path.realpath(name)
            slash = current_usplash_path.rfind('/')
            if slash != -1:
                path = current_usplash_path[:slash + 1]
                if path != self.themes_directory:
                    self.add_theme(current_usplash_path)
                active_name = current_usplash_path[slash + 1:]
                return active_name[:-3]
        return -1

    def set_active_theme(self, name):
        """Set the active Usplash theme."""
        filename = name + '.so'
        for usplash_file in os.listdir(self.themes_directory):
            if filename == usplash_file:
                os.system('ln -sf ' + self.alternatives_file + ' ' + 
                          self.themes_directory + 'usplash-artwork.so')
                command = '/usr/sbin/update-alternatives --set ' + \
                          'usplash-artwork.so /usr/lib/usplash/' + filename
                os.system(command)
                self.update_initramfs = True

    def get_all_themes(self):
        """Return a list of the available themes."""
        themes = []
        usplash_files = os.listdir(self.themes_directory)
        for usplash_file in usplash_files:
            end = usplash_file[usplash_file.rfind('.'):]
            if end == '.so' and usplash_file != 'usplash-artwork.so':
                themes.append(usplash_file[:-3])
        return themes

    def add_theme(self, path):
        """Add theme from path."""
        filename = path[path.rfind('/') + 1:]
        filename = utils.fix_filename_spaces(filename)
        if filename[-3:] == '.so' and filename != 'usplash-artwork.so':
            shutil.copy(path, self.themes_directory)
            command = '/usr/sbin/update-alternatives '+ \
                      '--install /usr/lib/usplash/usplash-artwork.so ' + \
                      'usplash-artwork.so ' + \
                      self.themes_directory + filename + ' 10'
            os.system(command)

    def remove_theme(self, name):
        """Remove theme."""
        name = name + '.so'
        if name != 'usplash-artwork.so':
            path = self.themes_directory + name
            os.remove(path)

    def do_shutdown_tasks(self):
        """Do any post-config tasks necessary."""
        if self.update_initramfs:
            os.system(self.update_initramfs_command)

