/* Copyright (C) 2000-2006  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: loader.pike,v 1.1.1.1 2006/03/27 12:40:04 exodusd Exp $
 */

constant cvs_version="$Id: loader.pike,v 1.1.1.1 2006/03/27 12:40:04 exodusd Exp $";


#include <configure.h>

mapping env = ([
    "LD_LIBRARY_PATH":getcwd()+"/server/libraries:/usr/local/lib",
    ]);



string     pidfile;
Stdio.File outfile;
Stdio.File errfile;
int   run_once = 0;
int          t = 0;
string     gdb = 0;
string      logdir;

string server_ip;

mapping    mServices;
mapping mServiceLogs;
mapping mServicePIDs;
int serverPID;

static int restart = 0;

static int keep_services_running = 1;


object start_service(string service, string ticket) 
{
  array loader_args = ({ });
  mapping config = Config.read_config_file( CONFIG_DIR + "/services/" + service + ".cfg" );
  if ( mappingp(config) && stringp(config["loader_args"]) )
    loader_args += (config["loader_args"] / " ");

  array(string) runArr =  
  ({ "./steam", "--include-path=server/include",
     "--module-path=client", "--module-path=server/libraries", 
     "--program-path=server/" });
  runArr += loader_args;
  Stdio.File serviceLog = Stdio.File(logdir + "/"+service + ".log", "wct");
  Stdio.File ipc = Stdio.File();

  if ( search(service, ".jar") > 0 ) {
    string java = "java";
    string java_home = getenv()["JAVA_HOME"];
    if ( stringp(java_home) )
      java = java_home + "/bin/java";
    runArr = ({ java, "-Djava.awt.headless=true" });
    runArr += loader_args;
    runArr += ({ "-jar", "services/"+service });
  }
  else
    runArr += ({ "services/"+service });

  runArr += ({ "--user=service", "--password="+ticket });
  if ( stringp(server_ip) ) runArr += ({ "--host="+server_ip });

  serviceLog->write("Starting service \"%s\" with ticket %s\n", service, ticket);
  if ( sizeof( loader_args ) > 0 )
    serviceLog->write( "Loader arguments: %O\n", loader_args );
  mServiceLogs[service] = serviceLog;
  return Process.create_process( runArr,
				 ([ "env": getenv(),
				    "cwd": getcwd(),
				    "stdout": serviceLog,
				    "stderr": serviceLog,
				 ]));
}

void start_services()
{
  mServices = ([ ]);
  mServiceLogs = ([ ]);
  mServicePIDs = ([ ]);
  
  array dir = get_dir("services");
  write("Starting services: ");
  if ( arrayp(dir) ) {
    foreach(dir, string service) {
      if ( !Stdio.is_file("services/"+service) )
	continue;
      if ( search(service, "~") >= 0 || search(service, "CVS") >= 0 )
	continue;
      write(" " + service + ", ");
      mServices[service] = 1;
    }
  }
  write("\n");
  keep_services_running = 1;
  thread_create(check_services);
}

void stop_service ( string service )
{
    int pid = mServicePIDs[service];

    // quit process:
    if ( objectp(mServices[service]) && mServices[service]->status() != 0 )
	mServices[service]->kill(signum("SIGQUIT"));
    else if ( pid != 0 )
	kill( pid, signum("SIGQUIT") );

    // if service doesn't quit, kill it:
    if ( objectp(mServices[service]) && mServices[service]->status() != 0 )
	mServices[service]->kill(signum("SIGKILL"));
    else if ( pid != 0 )
	kill( pid, signum("SIGKILL") );

    // remove old pid from pid_file:
    mixed err;
    if ( pid != 0 ) err = catch {
	string pid_content = Stdio.read_file( pidfile );
	pid_content = replace( pid_content, "\n", " " );
	array pids = pid_content / " ";
	for ( int i=sizeof(pids)-1; i>=0; i-- ) {
	    if ( (int)(pids[i]) == pid ) pids -= ({ pids[i] });
	    else if ( pids[i] == "" ) pids -= ({ pids[i] });  // cosmetics
	}
	pid_content = pids * " ";
	int index = search( pid_content, " " );
	if ( index >= 0 ) pid_content[index] = '\n';  // newline after loader PID
	Stdio.write_file( pidfile, pid_content );
	mServicePIDs[service] = 0;
    };
    if ( err != 0 )
	werror( "Error while removing service \"%s\" PID from pidfile: %O\n", service, err );
}

void stop_services ()
{
    keep_services_running = 0;
    foreach(indices(mServices), string service)
	stop_service( service );
}

void check_services() 
{
  string ticket;

  while ( !stringp(ticket) ) {
    sleep(20);
    catch(ticket = Stdio.read_file("service.pass"));
  }
  
  catch(rm("service.pass"));
  
  while ( keep_services_running ) {
    foreach(indices(mServices), string service) {
      if ( !objectp(mServices[service]) || mServices[service]->status() > 0 ) {
	stop_service( service );
	if ( ! keep_services_running ) break;
        // start service:
	mixed err = catch(mServices[service] = start_service(service, ticket));
	if ( err ) {
	    mServiceLogs[service]->write("Failed to start service %s\n%O\n",
					 service, err);
	}
	else {
	  if ( !mServices[service] )
	      mServiceLogs[service]->write(
		  "Fatal Error - failed to start service %s.\n", service);
	  else {
	      mServiceLogs[service]->write("PID: %d\n", 
					   mServices[service]->pid());
	      mServicePIDs[service] = mServices[service]->pid();
	      Stdio.append_file(pidfile, " " + mServices[service]->pid());
	  }
	}
      }
    }
    sleep(60);
  }
}

//! run the server
void run(array(string) params)
{
    int ret = 0;

    Stdio.File exitF;
    array(string) runArr =  
	    ({ "./steam", "--include-path=server/include",
               "--module-path=server/libraries", 
               "--program-path=server", "server/server.pike" }) + params[1..];
    foreach (params, string p) {
      if ( p == "--restart" )
	restart = 1;
    }
    if ( stringp(gdb) ) {
        runArr = ({ "gdb", "steam", "-x", "gdb.run" });
        Stdio.File gdb_file = Stdio.File("gdb.run", "wct");
        gdb_file->write(gdb);
        gdb_file->write((params[1..]*" ")+"\n");
        gdb_file->close();
    }
    string cwd = getcwd();
    mapping cenv = getenv();
    
    rm(logdir+"/exit");
    while ( ret != 1 && ret != 10 && (ret>=0 || restart) ) {
	write("\nCWD: " + getcwd() + " - Starting sTeam Server\n");
	write("------------------------------------------------------\n");
	write("Logfile: "+ sprintf("%O", outfile)+"\n");
	write("LogDir:  "+logdir+"\n");
	write("Params:  "+sprintf("%O", params[1..])+"\n");
	start_services();
	ret = Process.create_process( runArr,
	    ([ "env": cenv + env,
	       "cwd": cwd,
               "stdout": outfile,
               "stderr": errfile,
	     ]))->wait();
	write("Returned: "+  ret+"\n");
	stop_services();
        if ( ret > 0 ) {
            exitF = Stdio.File(logdir+"/exit", "wct");
            exitF->write("Server exited with error...\n");
            exitF->close();
        }
	if ( stringp(gdb) )
	    ret = 1;
	outfile->close();
	errfile->close();
	rotate_log("server.log");
	rotate_log("errors.log");
	rotate_log("http.log");
	rotate_log("events.log");

	outfile = Stdio.File(logdir+"/server.log", "wct");
	errfile = Stdio.File(logdir+"/errors.log", "wct");
    }
    werror("sTeam Server Exited !\n");
    rm(pidfile);
    exit(ret);
}

#define LOGROTATE_DEPTH 5

void rotate_log(string logfile) 
{

  for ( int i = LOGROTATE_DEPTH; i > 0; i-- ) {
    string fromlog, tolog;
    if ( i > 1 )
      fromlog = logdir + "/" + logfile + "." + (i-1);
    else
      fromlog = logdir + "/" + logfile;
    tolog = logdir + "/" + logfile + "." + i;
    mv(fromlog, tolog);
  }
}
    
void got_kill(int sig)
{
    restart = 0;
    werror("Server killed !\n");
    object f = Stdio.File(pidfile,"r");
    string pid = f->read();
    f->close();
    array pids = pid / " ";
    foreach ( pids, string p ) {
	if ( !stringp(p) || sizeof(p)<1 ) continue;
	if ( (int)p != getpid() ) {
	    werror("Killing: " + p + "\n");
	    kill((int)p, signum("SIGQUIT"));
	}
    }
}

void got_hangup(int sig)
{
    if ( run_once )
        got_kill(sig);
}

int 
main(int argc, array(string) argv)
{
    int pid = getpid();
    
    signal(signum("QUIT"), got_kill);
    signal(signum("TERM"), got_kill);
    signal(signum("SIGHUP"), got_hangup);
    signal(signum("SIGINT"), got_hangup);

    mapping server_config = Config.read_config_file( CONFIG_DIR + "/steam.cfg" );
    if ( mappingp(server_config) ) {
      if ( stringp(server_config["ip"]) && sizeof(server_config["ip"])>1 )
	server_ip = server_config["ip"];
      else
	server_ip = 0;
    }
    server_config = 0;
    
    pidfile = getcwd() + "/steam.pid";
    logdir = LOG_DIR;
    foreach(argv, string p) {
	string a,b;
	if ( sscanf(p, "--%s=%s", a, b) == 2 ) {
	    switch( a ) {
	    case "pid":
	    case "pidfile":
		pidfile = b;
		break;
            case "stdout":
                outfile = Stdio.File(b, "wct"); 
		logdir="logs";
                break;
	    case "stderr":
                errfile = Stdio.File(b, "wct"); 
		break;
	    case "logdir":
		outfile = Stdio.File(b + "/server.log", "wct");
                errfile = Stdio.File(b + "/errors.log", "wct"); 
		logdir = b;
		break;
	    }
	}
        if ( p == "--once" ) {
            outfile = Stdio.stdout;
	    errfile = Stdio.stderr;
            run_once = 1;
        }
        else if ( p == "--gdb" ) {
            gdb = "run -I server/include -M server/libraries -P server "+
                  "server/server.pike ";
	    outfile = Stdio.stdout;
	}	
    }

    if ( !objectp(errfile) )
      errfile = Stdio.File(logdir+"/errors.log", "wct");

    if ( !objectp(outfile) ) {
        outfile = Stdio.File(logdir+"/server.log", "wct");
	logdir = LOG_DIR;
    }

    Stdio.File f = Stdio.File( pidfile, "cwt" );
    f->write((string)pid);
    f->write("\n");
    f->close();
    thread_create(run, argv);
    return -17;
}
