/* Copyright (C) 2000-2006  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: database.pike,v 1.7 2006/05/04 21:26:14 exodusd Exp $
 */

constant cvs_version="$Id: database.pike,v 1.7 2006/05/04 21:26:14 exodusd Exp $";

inherit "/base/serialize.pike";
inherit Thread.Mutex : muBusy;
inherit Thread.Mutex : muLowSave;

#include <macros.h>
#include <assert.h>
#include <attributes.h>
#include <database.h>
#include <config.h>
#include <classes.h>
#include <access.h>
#include <roles.h>
#include <events.h>
#include <exception.h>
#include <types.h>

#define MODULE_SECURITY _Server->get_module("security")
#define MODULE_USERS    _Server->get_module("users")

#define PROXY "/kernel/proxy.pike"

private static mapping(int:mapping(string:int))     mCurrMaxID;
private static mapping(int:object  )              mProxyLookup;
private static Thread.Mutex         loadMutex = Thread.Mutex();
private static Thread.Mutex       createMutex = Thread.Mutex();

private static int                     iCacheID;
private static object                 oSQLCache;
private static object                oSaveQueue;

private static object               oTlDbHandle;
private static object                 oDbHandle;
private static object                tSaveDemon;
private static object            oDemonDbHandle;
private static object                  oModules;
private static mapping(string:object)   mDbMaps;

private static string                   sDbUser;
private static string               sDbPassword;
private static int             idbMappingNaming;
private static Stdio.File              lostData;
private static array(object)     tReaderThreads;
private static Thread.Queue           readQueue;
private static Thread.Queue         globalQueue;

#define DBM_UNDEF   0
#define DBM_ID      1
#define DBM_NAME    2

#define CHECKISO(s, mime, obj) if (search(s, mime)>0) { obj->restore_attr_data(mime, DOC_ENCODING); werror(" %s",mime); }

private static mapping(string:object) oModuleCache;
private static Calendar.Calendar cal = Calendar.ISO->set_language("german");

private static int CONVERSION = 0;

private static mapping(string:int) mSaveRequests = ([ ]);

string generate_request_name(int oid, mixed ident, mixed index)
{
    return (string)oid+"|"+(stringp(ident)? ident : "0")+
        "|" + (stringp(index)? index : "0");
}

class SqlReadRecord {
  int  iMaxRecNbr; // record number
  int         iID; // doc_it
  int iNextRecNbr; // record number current
  object   dbfile;
  function restore; // function if record needs to be restored
  Thread.Fifo   contFifo;
  Thread.Mutex fullMutex;

  int stopRead = 0;
  int myId = time();

  int check_timeout() {
    if ( objectp(dbfile) ) {
      int t = time() - dbfile->get_last_access();
      if ( !objectp(contFifo) )
	return 0;
      if ( t > 600 )
	return 0;
      return 1;
    }
    return 0;
  }

}

class SqlHandle {
    Sql.Sql oHandle;
    private string db_connect;

    void keep() {
        Sql.sql_result res =
            oHandle->big_query("select ob_class from ob_class "+
                               "where ob_id = 13");
        res->fetch_row();
    }
    
    void create(string connect) {
        db_connect = connect;
        oHandle = Sql.Sql(db_connect);
    }
    
    int|object big_query(object|string q, mixed ... extraargs) {
        Sql.sql_result res;
        mixed err = catch { res=oHandle->big_query(q, @extraargs); };
        if (err)
        {
	  FATAL(cal->Second()->format_nice()+
		" Database Error ("+(string)oHandle->error()+")\n"+
		master()->describe_backtrace(err));
	  throw(err);
        }
        return res;
    }
    
    array(mapping(string:mixed)) query(object|string q, mixed ... extraargs) {
        array(mapping(string:mixed)) res;
        mixed err = catch { res=oHandle->query(q, @extraargs);};
        if (err)
        {
            FATAL(cal->Second()->format_nice()+
                   " Database Error("+(string)oHandle->error()+")");
            destruct(oHandle);
            oHandle = Sql.Sql(db_connect);
            res = oHandle->query(q, @extraargs);
            return res;
        }
        return res;
    }
    
    function `->(string fname) {
        switch(fname) {
          case "query": return query;
          case "big_query" : return big_query;
	  case "keep": return keep;
	  default : 
	    if ( !objectp(oHandle) )
	      return 0;
	    return oHandle[fname];
        }
    }
    string describe() { return "SqlHandle()"; }
}


/**
 * return a thread-local (valid) db-handle
 *
 * @param  none
 * @return the database handle
 */
private static Sql.Sql db()
{
    if (this_thread() == tSaveDemon) // give saveDemon its own handle
	return oDemonDbHandle;

    // everybody else gets the same shared handle
    if (!objectp(oDbHandle))
    {
	oDbHandle = SqlHandle(STEAM_DB_CONNECT);
        if (!validate_db_handle(oDbHandle))
            setup_sTeam_tables(oDbHandle);
    }


    //    FATAL(cal->Second()->format_nice()+": database handle requested.");
    return oDbHandle;
}
    
/**
 * mimick object id for serialization etc. 
 * @return  ID_DATABASE from database.h
 * @see    object.get_object_id
 * @author Ludger Merkens 
 */
final int get_object_id()
{
    return ID_DATABASE;
}

private static void db_execute(string db_query)
{
    db()->big_query(db_query);
}

int get_save_size()
{
    return oSaveQueue->size();
}

/**
 * demon function to store pending object saves to the database.
 * This function is started as a thread and waits for objects entering
 * a queue to save them to the database.
 *
 * @param  nothing
 * @return void
 * @see    save_object
 * @author Ludger Merkens 
 */
void database_save_demon()
{
    MESSAGE("DATABASE SAVE DEMON ENABLED");
    mixed job;
    object lGuard;
    object lBusy;

    while(1)
    {
	job = oSaveQueue->read();

	if (!lBusy)
	    lBusy = muBusy::lock(); 
	mixed cerr = catch {
            if (arrayp(job))
            {
                object proxy;
                string ident;
                string index;
                [proxy, ident, index] = job;
                //MESSAGE("would save %O, %s, %s", proxy, ident, index);
                low_save_object(proxy, ident, index);
            }
            else
                if (stringp(job))
                    db_execute(job);
        };
	if (oSaveQueue->size() == 0)
	    destruct(lBusy);

	if ( cerr ) {
	  FATAL("/**************** database_save_demon *************/\n"+
		PRINT_BT(cerr));
	}
    }
}

/**
 * wait_for_db_lock waits until all pending database writes are done, and
 * afterwards aquires the save_demon lock, thus stopping the demon. Destruct
 * the resulting object to release the save demon again.
 *
 * @param nothing
 * @return the key object 
 * @see Thread.Mutex->lock
 * @author Ludger Merkens
 */
object wait_for_db_lock()
{
    return muBusy::lock();
}

/**
 * constructor for database.pike
 * - starts thread to keep objects persistent
 * - enables commands in database
 * @param   none
 * @return  void
 * @author Ludger Merkens 
 */
void create()
{
    // first check for lost data, etc.
  
    mProxyLookup = ([ ]);
    mCurrMaxID = ([ ]);
    oSaveQueue = Thread.Queue();
    oTlDbHandle = thread_local();
    mDbMaps = ([]);
}

void init()
{
    _Persistence->register(this_object());
}

object enable_modules()
{
    //    oDemonDbHandle = Sql.Sql(STEAM_DB_CONNECT);
    tSaveDemon = thread_create(database_save_demon);
    tReaderThreads = ({ });
    readQueue = Thread.Queue();
    globalQueue = Thread.Queue();
    for ( int i = 0; i < 2; i++ )
      tReaderThreads += ({ thread_create(db_reader) });
    
    oModules = ((program)"/modules/modules.pike")();
    oModuleCache = ([ "modules": oModules ]);

    oDemonDbHandle = SqlHandle(STEAM_DB_CONNECT);
    int x=validate_db_handle(oDemonDbHandle);
    if (x==0)
	setup_sTeam_tables(oDemonDbHandle);
    else if (x==1)
        add_new_tables(oDemonDbHandle);

    check_journaling(oDemonDbHandle);
    return oModules;
}

//#define DBREAD(l, args...) werror("%O"+l+"\n", Thread.this_thread(), args)
#define DBREAD(l, args ...)

static void add_record(object record)
{
  readQueue->write(record);
}

void db_reader()
{
  SqlReadRecord   record;
  Sql.sql_result odbData;
  array       fetch_line;

  while ( 1 ) {
    DBREAD("Waiting for queue...");
    
    mixed err = catch {
      record = readQueue->read();
      DBREAD("Jobs in readQueue = %d", readQueue->size());
      if ( record->check_timeout() && !record->stopRead ) {
	odbData = db()->big_query
	  ( "select rec_data,rec_order from doc_data"+
	    " where doc_id ="+record->iID+
	    " and rec_order >="+record->iNextRecNbr+
	    " and rec_order < "+(record->iNextRecNbr+READ_ONCE)+
	    " order by rec_order" );
	DBREAD("Queueing read result for %d job=%d",record->iID,record->myId);
	while ( fetch_line=odbData->fetch_row() ) {
	  if ( record->contFifo->size() > 100 ) {
	    break;
	  }
	  record->contFifo->write(fetch_line[0]);
	  record->iNextRecNbr= (int)fetch_line[1] +1;
	}
	DBREAD("next=%d, last=%d", record->iNextRecNbr, record->iMaxRecNbr);
	if ( record->iNextRecNbr > 0 &&
	     record->iNextRecNbr <= record->iMaxRecNbr) 
	{
	  DBREAD("Continue reading...\n");
	  object mlock = record->fullMutex->lock();
	  if ( record->contFifo->size() > 100 ) 
	    record->restore = add_record;
	  else
	    readQueue->write(record); // further reading
	  destruct(mlock);
	}
	else {
	  DBREAD("Read finished...\n");
	  record->contFifo->write(0);
	}

      }
      else 
	destruct(record);
    };
    if ( err ) {
      FATAL("Error while reading from database: %O", err);
      catch {
	DBREAD("finished read on %d", record->iID);
	record->contFifo->write(0);
      };
    }
  }

}

object read_from_database(int id, int nextID, int maxID, object dbfile) 
{
  SqlReadRecord record = SqlReadRecord();
  record->iID = id;
  record->dbfile = dbfile;
  record->iNextRecNbr = nextID;
  record->iMaxRecNbr = maxID;
  record->contFifo = Thread.Fifo();
  record->fullMutex = Thread.Mutex();
  readQueue->write(record);
  globalQueue->write(record);
  return record;
}


int check_save_demon()
{
  if ( CALLER != _Server )
    error( "Unauthorized call to check_save_demon() !" );

  int status = tSaveDemon->status();
  //werror(ctime(time())+" Checking Database SAVE DEMON\n");
  if ( status != 0 ) {
    FATAL("----- DATABASE SAVE DEMON restarted ! ---");
    tSaveDemon = thread_create(database_save_demon);
  }
  oDbHandle->keep();
  oDemonDbHandle->keep();

  if ( objectp(globalQueue) ) {
    int sz = globalQueue->size();
    while ( sz > 0 ) {
      sz--;
      object record = globalQueue->read();
      // record has restore function set for 15 minutes (timeout)
      // this means the record is also not in the readQueue
      if ( functionp(record->restore) && !record->check_timeout() ) { 
	object dbfile = record->dbfile;
	destruct(record->contFifo);
	record->contFifo = 0;
	destruct(record);
	destruct(dbfile); // make sure everything is gone and freed
      }
      else
	globalQueue->write(record); // keep
    }
  }
    
  return status;
}

void register_transient(array(object) obs)
{
    ASSERTINFO(CALLER==MODULE_SECURITY || CALLER== this_object(), 
	       "Invalid CALLER at register_transient()");
    object obj;
    foreach (obs, obj) {
	if (objectp(obj))
	    mProxyLookup[obj->get_object_id()] = obj;
    }
}


/**
 * set_variable is used to store database internal values. e.g. the last
 * object ID, the last document ID, as well as object ID of modules etc.
 * @param name - the name of the variable to store
 * @param int value - the value
 * @author Ludger Merkens
 * @see get_variable
 */
void set_variable(string name, int value)
{
  if(sizeof(db()->query("SELECT var FROM variables WHERE var='"+name+"'"))) 
  {
    db()->big_query("UPDATE variables SET value='"+value+
                    "' WHERE var='"+name+"'" );
  }
  else
  {
    db()->big_query("INSERT into variables values('"+name+"','"+value+"')");
  }
}

/**
 * get_variable reads a value stored by set_variable
 * @param name - the name used by set_variable
 * @returns int - value previously stored under given name
 * @author Ludger Merkens
 * @see set_variable
 */
int get_variable(string name)
{
    object res;
    res = db()->big_query("select value from variables where "+
                          "var ='"+name+"'");
    if (objectp(res) && res->num_rows())
        return (int) res->fetch_row()[0];
    
    return 0;
}
    
/**
 * reads the currently used max ID from the database and given table
 * and increments. for performance reasons this ID is cached.
 * 
 * @param  int       db - database to connect to
 * @param  string table - table to choose
 * @return int          - the calculated ID
 * @see    free_last_db_id
 * @author Ludger Merkens 
 */
private static
int create_new_database_id(string table)
{
    if (!mCurrMaxID[table])
    {
	string          query;
	int            result;
	Sql.sql_result    res;

        result = get_variable(table);
        if (!result)
        {
            switch(table)
            {
              case "doc_data" :
                  query = sprintf("select max(doc_id) from %s",table);
                  res = db()->big_query(query);
                  result = (int) res->fetch_row()[0];
                  break;
              case "ob_class":
                  query  = sprintf("select max(ob_id) from %s",table);
                  res = db()->big_query(query);
                  result = max((int) res->fetch_row()[0], 1);
            }
        }
        mCurrMaxID[table] = result;
    }
    mCurrMaxID[table] += 1;
    //    MESSAGE("Created new database ID"+(int) mCurrMaxID[table]);
    set_variable(table, mCurrMaxID[table]);
    return mCurrMaxID[table];
}

/**
 * called in case, a newly created database id is obsolete,
 * usually called to handle an error occuring in further handling
 *
 * @param  int       db - Database to connect to
 * @param  string table - table choosen
 * @return void
 * @see    create_new_databas_id()
 * @author Ludger Merkens 
 */
void free_last_db_id(string table)
{
    mCurrMaxID[table]--;
}

/**
 * creates a new persistent sTeam object.
 *
 * @param  string prog (the class to clone)
 * @return proxy and id for object
 *         note that proxy creation implies creation of associated object.
 * @see    kernel.proxy.create, register_user
 * @author Ludger Merkens 
 */
mixed new_object(object obj, string prog_name)
{
    int         new_db_id;
    string sData, sAccess;
    object p;
    // check for valid object has to be added
    // create database ID

    if ( CALLER != _Persistence )
      error("Only Persistence Module is allowed to get in here !");

    int id = obj->get_object_id();
    if ( id )
    {
	ASSERTINFO((p=mProxyLookup[id])->get_object_id() == id,
		   "Attempt to reregister object in database!");
	return ({ id, p });
    }

    object lock = createMutex->lock(); // make sure creation is save
    mixed err = catch {

      if (CONVERSION)
        new_db_id = create_new_database_id("objects");
      else
        new_db_id = create_new_database_id("ob_class");
      
      p = new(PROXY, new_db_id, obj );
      if (!objectp(p->get_object())) // error occured during creation
      {
	  free_last_db_id("ob_class");
	  destruct(p);
      }
      
      // insert the newly created Object into the database
      if (prog_name!="-") {
	Sql.sql_result res = db()->big_query(
		     sprintf("insert into ob_class values(%d,'%s')",
			     new_db_id, prog_name)
		     );
        mProxyLookup[new_db_id] = p;
        save_object(p, 0);
      }
    };
    if ( err ) {
      FATAL("database.new_object: failed to create object\n %O", err);
    }
    destruct(lock); 
    return ({ new_db_id, p});
}

/**
 * permanently destroys an object from the database.
 * @param  object represented by (proxy) to delete
 * @return (0|1)
 * @see    new_object
 * @author Ludger Merkens 
 */
bool delete_object(object p)
{
  if ( CALLER!=_Persistence &&
       (!MODULE_SECURITY->valid_object(CALLER) || CALLER->this() != p->this()))
  {
    werror("caller of delete_object() is %O\n", CALLER);
    THROW("Illegal call to database.delete_object", E_ACCESS);
  }
  return do_delete(p);
}

private bool do_delete(object p)
{
    object proxy;
    int iOID = p->get_object_id();
    db()->query("delete from ob_data where ob_id = "+iOID);
    db()->query("delete from ob_class where ob_id = "+iOID);
    proxy = mProxyLookup[iOID];
    if ( objectp(proxy) )
      catch(proxy->set_status(PSTAT_DELETED));
    m_delete(mProxyLookup, iOID);

    return 1;
}

private int|object
get_old_instance(int iOID, string sClass, object proxy)
{
    object o;
    
    if (sClass == "-") {
        return 2;
    }
    mixed catched = catch {
        o = new(sClass, proxy);
    };

    if (!objectp(o)) // somehow failed to load file
    {
        if ( catched ) {
            FATAL("/**** while loading:"+ sClass + "****/\n" +
                  PRINT_BT(catched));
        }
        return 1; // class exists but failes to compile
    }
    return o;
}

/**
 * load and restore values of an object with given Object ID
 * @param   int OID
 * @return  0, object deleted
 * @return  1, object failed to compile
 * @return  2, objects class deleted
 * @return  3, object fails to load
 * @return  the object
 * @see
 * @author Ludger Merkens 
 */
int|object load_object(object proxy, int|object iOID)
{
    string      sClass;
    string      sIdent;
    string     sAttrib;
    string       sData;
    int              i;
    array(string) inds;
    object o;
    
    if ( CALLER != _Persistence )
      error("Unable to load objects directly !");
        
    mixed catched;
    
    if (CONVERSION)
    {
        Sql.sql_result res = db()->big_query(
            sprintf("select ob_class, ob_data from objects where ob_id = %d",
                    iOID) );
        //werror("select executed\n");
        mixed line = res->fetch_row();
        //werror("data fetched\n");
        if ( !arrayp(line) || sizeof(line)!=2 ) {
            FATAL(PRINT_BT(({"database.load_object: Failed to load "+
                             "Object("+iOID+")"+
                             (arrayp(line) ? sizeof(line) : "- not found" ),
                             backtrace()})));
            return 0;
        }
        [sClass, sData] = line;
        if (!objectp(iOID)) {
            o = get_old_instance(iOID, sClass, proxy);
            if (objectp(o)) { proxy->set_steam_obj(o); }
            else return 0;
        }
        else
            o = iOID;
        
        if ( objectp(o) && functionp(o->get_object_id) )
	  compatibility_load_object(iOID, o, sData);

        return o;
    }

    Sql.sql_result res = db()->big_query(
        sprintf("select ob_class from ob_class where ob_id = %d", iOID)
        );
    mixed line = res->fetch_row();
    if (!arrayp(line))
    {
        FATAL("Failed to load object - deleted?\n");
        return 0;
    }
    ASSERTINFO(arrayp(line), "Failed to load object - database corrupt");

    if (objectp(iOID))
        o = iOID;
    else
    {
        catched = catch {
	  sClass = line[0];
	  int pos;
	  if ( (pos = search(sClass, "DB:#")) >= 0 ) {
	    sClass = "/"+sClass[pos..];
	  }
	  o = new(sClass, proxy);
        };

        if (!objectp(o)) // somehow failed to load file
        {
            if ( catched ) {
                _Server->add_error(time(), catched);
                if (!master()->master_file_stat(sClass+".pike"))
                {
                    FATAL("You may have stale objects in the database of class:"+
                          sClass);
                    return 2;
		}
                FATAL("Error loading object %d (%s)\n%s", iOID, sClass, 
                      master()->describe_backtrace(catched));
            }
            return 1; // class exists but failes to compile
        }
        
        proxy->set_steam_obj(o); // o is the real thing - no proxy!
    }
    mapping mData;

    res = db()->big_query(
        sprintf("select ob_ident, ob_attr, ob_data from ob_data where "+
                "ob_id = %d", iOID));

    if ( !objectp(o) ) 
	FATAL("Warning: Object not set %O", iOID);

    mapping mStorage = get_storage_handlers(o);
    if ( !mappingp(mStorage) || equal(mStorage, ([ ])) ) {
	proxy->set_status(PSTAT_FAIL_UNSERIALIZE);
	return 3;
    }
    mapping mIndexedStorage = ([]);

    foreach(indices(mStorage), string _ident) // precreate indexed idents
        if (mStorage[_ident][2])
            mIndexedStorage[_ident]=([]);
    
    int mem = 0;
    while (line = res->fetch_row())
    {
        [sIdent, sAttrib, sData] = line;
        mem += strlen(sData);
        catched = catch {
            mData = unserialize(sData); // second arg is "this_object()"
        };
        if ( catched ) {
            FATAL("While loading ("+iOID+","+sClass+"):\n"+ catched[0] +"\n"+
                  master()->print_backtrace(catched));
            proxy->set_status(PSTAT_FAIL_UNSERIALIZE);
            return 0;
        }
        if ( sAttrib != "")
        {
	  if ( !mIndexedStorage[sIdent] ) {
	    FATAL("Missing Storage %s in %d, %s", sIdent, iOID, sClass);
	    proxy->set_status(PSTAT_FAIL_UNSERIALIZE);
	    return 0;
	  }
	  mIndexedStorage[sIdent][sAttrib]=mData;
        }
	else if ( !mStorage[sIdent] ) {
	  FATAL("No storage handler %s defined in %d (%s).\ndata=%O",
		sIdent, iOID, sClass, mData);
	}
        else
        {
	  catched = catch {
	    if ( proxy->is_loaded() ) {
	      error(sprintf("Fatal error: already loaded %O\n", proxy));
	    }
	    mStorage[sIdent][1](mData); // actual function call
	  };
	  if ( catched ) {
	    FATAL("Error while loading (%d, %s)\n"+
		  "Error while calling storage handler %s: %O\n"+
		  "Full Storage:%O\nData: %O", 
		  iOID, sClass,
		  sIdent, catched, 
		  mStorage, mData);
	  }
        }
    } 

    foreach(indices(mIndexedStorage), string _ident)
        mStorage[_ident][1](mIndexedStorage[_ident]); // prepared call

    mixed err = catch { 
      o->this()->set_status(PSTAT_SAVE_OK);
      o->loaded(); 
    };
    return o;
}

static mapping get_storage_handlers(object o)
{
    if ( !objectp(o) )
	FATAL("Getting storage handlers for %O", o);
    
    mapping storage = _Persistence->get_storage_handlers(o);
    return storage;
}

mixed call_storage_handler(function f, mixed ... params)
{
  if ( CALLER != _Persistence )
    error("Database:: call_storage_handler(): Unauthorized call !");
  mixed res = f(@params);
  return res;
}

mapping convert_attribute_mapping(mapping old_fashion)
{
    // temporary for conversion
    mapping m_conversion = ([
        101 : "OBJ_OWNER", 102 : "OBJ_NAME", 104 : "OBJ_DESC",
        105 : "OBJ_ICON", 111 : "OBJ_KEYWORDS", 112 : "OBJ_COMMAND_MAP",
        113 : "OBJ_POSITION_X", 114 : "OBJ_POSITION_Y", 115 : "OBJ_POSITION_Z",
        116 : "OBJ_LAST_CHANGED", 119 : "OBJ_CREATION_TIME", "url" : "OBJ_URL",
        "obj:link_icon" : "OBJ_LINK_ICON", "obj_script" : "OBJ_SCRIPT",
        "obj_annotations_changed" : "OBJ_ANNOTATIONS_CHANGED",
        207 : "DOC_TYPE", 208 : "DOC_MIME_TYPE", 213 : "DOC_USER_MODIFIED",
        214 : "DOC_LAST_MODIFIED", 215 : "DOC_LAST_ACCESSED",
        216 : "DOC_EXTERN_URL", 217 : "DOC_TIMES_READ",
        218 : "DOC_IMAGE_ROTATION", 219 : "DOC_IMAGE_THUMBNAIL",
        220 : "DOC_IMAGE_SIZEX", 221 : "DOC_IMAGE_SIZEY",
        300 : "CONT_SIZE_X", 301 : "CONT_SIZE_Y", 302 : "CONT_SIZE_Z",
        303 : "CONT_EXCHANGE_LINKS", "cont:monitor" : "CONT_MONITOR",
        "cont_last_modified" : "CONT_LAST_MODIFIED",
        500 : "GROUP_MEMBERSHIP_REQS", 501 : "GROUP_EXITS",
        502 : "GROUP_MAXSIZE", 503 : "GROUP_MSG_ACCEPT",
        504 : "GROUP_MAXPENDING", 611 : "USER_ADRESS", 612 : "USER_FULLNAME",
        613 : "USER_MAILBOX", 614 : "USER_WORKROOM", 615 : "USER_LAST_LOGIN",
        616 : "USER_EMAIL", 617 : "USER_UMASK", 618 : "USER_MODE",
        619 : "USER_MODE_MSG", 620 : "USER_LOGOUT_PLACE",
        621 : "USER_TRASHBIN", 622 : "USER_BOOKMARKROOM",
        623 : "USER_FORWARD_MSG", 624 : "USER_IRC_PASSWORD",
        "user_firstname" : "USER_FIRSTNAME", "user_language" : "USER_LANGUAGE",
        "user_selection" : "USER_SELECTION",
        "user_favorites" : "USER_FAVOURITES", 700 : "DRAWING_TYPE",
        701 : "DRAWING_WIDTH", 702 : "DRAWING_HEIGHT", 703 : "DRAWING_COLOR",
        704 : "DRAWING_THICKNESS", 705 : "DRAWING_FILLED",
        800 : "GROUP_WORKROOM", 801 : "GROUP_EXCLUSIVE_SUBGROUPS",
        1000 : "LAB_TUTOR", 1001 : "LAB_SIZE", 1002 : "LAB_ROOM",
        1003 : "LAB_APPTIME", 1100 : "MAIL_MIMEHEADERS",
        1101 : "MAIL_IMAPFLAGS"
    ]);

    foreach(indices(old_fashion), mixed Attr)
    {
        if (string newAttr=m_conversion[Attr])
        {
            if (old_fashion[newAttr]) {
                FATAL("WarningD conversion - targetname "+
                      "alreay in use - overwriting...");
                //THROW("CONVERSION FAILED - TERMINATING", E_ERROR);
		old_fashion[newAttr]=old_fashion[Attr];
		m_delete(old_fashion, Attr);
            }
            else {
                old_fashion[newAttr]=old_fashion[Attr];
                m_delete(old_fashion, Attr);
            }
        } else
        {
            if (intp(Attr))
            {
                FATAL("FAILED conversion - incomplete conversion "+
                      "table [%O]", Attr);
                THROW("CONVERSION FAILED - TERMINATING", E_ERROR);
            }
            else if (!stringp(Attr))
            {
                FATAL("Warning - strange Attribute name found "+
                      "[%O]\n", Attr);
                m_delete(old_fashion, Attr);
            }
        }
    }
    return old_fashion;
}
    
/**
 * load and restore values of an object with given Object ID
 * @param   int OID
 * @return  0, object deleted
 * @return  1, object failed to compile
 * @return  2, objects class deleted thus instance deleted
 * @return  the object
 * @see
 * @author Ludger Merkens 
 */
private static int|object
compatibility_load_object(int iOID, object o, string sData)
{
    int              i;
    array(string) inds;
    mixed catched;
    mapping mData;
    
    catched = catch {
	mData = unserialize(sData); // second arg is "this_object()"
    };
    if ( catched ) {
	FATAL("Deserialisation failed (%d,%O)\n%s",
              iOID, o, master()->describe_backtrace(catched));
	return 0;
    }
    if ( mappingp(mData) ) {
        array lines = indices(mData);
        lines = ({ "restore_content_data", "restore_data"}) +
            (lines - ({"restore_content_data", "restore_data"}));
	foreach(lines, string line) {
            if (line=="restore_data")
            {

                if (mData[line])
                {
                    mapping mAttributes =
                        convert_attribute_mapping(mData[line]["Attributes"]);
                    mapping mAttributeAcquire =
                        convert_attribute_mapping(mData[line]["AttributesAcquire"]);
		    if ( functionp(o->restore_attr_data))
			o->restore_attr_data(mAttributes);
                    m_delete(mData[line], "Attributes");
                    mData[line]["AttributesAcquire"]= mAttributeAcquire;
		    if ( functionp(o->restore_data) )
			o->restore_data(mData[line]);

                    if (mData["restore_content_data"]) // it's a document
                    {
                        string doctype = o->query_attribute(DOC_MIME_TYPE);
                        if (doctype && strlen(doctype) &&
                            search(doctype, "text")>-1)
                        {
			    //werror(" MIME = %s", doctype);
                            string sContent = o->get_content();
                            if (sContent && strlen(sContent))
                            {
                                CHECKISO(sContent, "iso-8859-1", o);
                                CHECKISO(sContent, "utf-8", o);
                            }
                        }
                    }
                }
            }
            else if (line=="restore_icons" && functionp(o->restore_icons) )
            {
                foreach(indices(mData[line]["icons"]), mixed icondx)
                    o->restore_icons(mData[line]["icons"][icondx],
                                     (stringp(icondx) ? "\""+icondx :
                                      "#"+(string)icondx));
            }
            else if (line=="restore_content_data")
            {
                if (mData[line])
                    o->restore_content_data(mData[line]);
            }
            else if (line=="restore_annotations")
            {
                o->restore_annotations(mData[line]);
                if (mData[line]["Annotates"])
                {
                    string sContent = o->get_content();
                    if (!stringp(sContent))
		      FATAL("\n!!!Missing Content in Annotation %d\n",
			    o->get_object_id());
                    else {
		      //werror("cid = %d\n", o->get_content_id());
                        object oModifiedBy = o->retrieve_attr_data(DOC_USER_MODIFIED);
                        o->set_content(string_to_utf8(sContent));
                        o->restore_attr_data("utf-8", DOC_ENCODING);
                        o->restore_attr_data(oModifiedBy, DOC_USER_MODIFIED);
                        //werror("\nannotation content %s -> %s\n",
			//sContent, o->get_content());
                    }
                }
            }
            else
            {
                //werror("\nrestore-data line=%O o[line]=%O\n",line, o[line]);
	      catched = catch(o[line](mData[line]));
                if (arrayp(catched)) {
                    FATAL("Error in deserialisation (%d, %s):\n%s",
                          iOID, line, master()->describe_backtrace(catched));
                }
	    }
	}
    } 
    //werror("Loaded "+ o->get_object_id() + " - " + o->get_identifier()+"\n");
    mixed err = catch { o->loaded(); };
    return o;
}

/**
 * find an object from the global object cache or retreive it from the
 * database.
 *
 * @param  int - iOID ( object ID from object to find ) 
 * @return object (proxy associated with object)
 * @see    load_object
 * @author Ludger Merkens 
 */
final object find_object(int|string iOID)
{
    object p;

    if ( stringp(iOID) ) 
	return _Server->get_module("filepath:tree")->path_to_object(iOID);

    if ( !intp(iOID) )
	THROW("Wrong argument to find_object() - expected integer!",E_ERROR);

    if ( iOID == 0 ) return 0;
    if ( iOID == 1 ) return this_object();
    
    if ( objectp(p = mProxyLookup[iOID]) )
	return p;

    Sql.sql_result res;
    if (CONVERSION)
        res = db()->big_query(sprintf("select ob_class, ob_data from "+
                                  "objects where ob_id = %d", iOID));
    else
        res = db()->big_query(sprintf("select ob_class from ob_class "+
                                  "where ob_id = %d", iOID));
    
    if (!objectp(res) || res->num_rows()==0)
    {
        return 0;
    }

    // cache the query for a following load_object.
    if (CONVERSION)
    {
        iCacheID = iOID;
        if (objectp(oSQLCache))
            destruct(oSQLCache);
        oSQLCache = res;
    }
    
    // create an empty proxy to avoid recursive loading of objects
    p = new(PROXY, iOID);
    mProxyLookup[iOID] = p;
    return p;
}

/**
 * The function is called to set a flag in an object for saving.
 * Additionally the functions triggers the global EVENT_REQ_SAVE event.
 *  
 * @author Thomas Bopp (astra@upb.de) 
 * @see save_object
 */
void require_save(object proxy, void|string ident, void|string index)
{
    if (CONVERSION) {
      return;
    }
    if (proxy && proxy->status()>=PSTAT_SAVE_OK)
    {
        string request =
            generate_request_name( proxy->get_object_id(), ident, index);
        mSaveRequests[request]++;
        save_object(proxy, ident, index);
    }
}


/**
 * callback-function called to indicate that an object has been modified
 * in memory and needs to be saved to the database.
 *
 * @param  object p - (proxy) object to be saved
 * @return void
 * @see    load_object
 * @see    find_object
 * @author Ludger Merkens 
 */
static void
save_object(object proxy, void|string ident, void|string index)
{
    if ( !objectp(proxy) )
	return;

    if (proxy->status() == PSTAT_SAVE_OK)
	proxy->set_status(PSTAT_SAVE_PENDING);

    oSaveQueue->write(({proxy, ident, index}));
}

/**
 * quote and check for maximum length of serialized data.
 * @param string data      - string to handle
 * @param object o         - object saved (for error reporting)
 * @param string ident     - ident block (for error reporting)
 * @author <a href="mailto:balduin@upb.de">Ludger Merkens</a>
 */
private static string
quote_data(string data, object o, string ident, function quoter, int|void utf8)
{
    if (utf8)
        data = serialize(data, "utf-8");
    else
        data = serialize(data);
    
    if (strlen(data)> 16777215)
        FATAL("!!! FATAL - data truncated inserting %d bytes for %s block %s",
              strlen(data),
              (objectp(o) ? "broken" : o->get_identifier()),
              ident);
    return quoter(data);
}

/**
 * generate a "mysql-specific" replace statement for saving data according
 * to needs of require_save()
 * @param object o          - object to save data from
 * @param mapping storage   - storage_data to access
 * @param string|void ident - optional arg to limit to ident
 * @param string|void index - optional arg to limit to index
 * @return the mysql statement
 * @author Ludger Merkens
 */
private static string
prepare_save_statement(object o, mapping storage,
                       string|void ident, string|void index)
{
    int oid = o->get_object_id();
    array statements = ({});
    // in case you change the behavoir below - remember to change
    // behaviour in prepeare_clear_statement also
    array(string) idents =
        arrayp(storage[ident]) ? ({ ident }) : indices(storage);
    string data;
    string sClass = master()->describe_program(object_program(o->get_object()));
    function db_quote_data = db()->quote;
    foreach(idents, string _ident)
    {
        if (!arrayp(storage[_ident]))
        {
            FATAL("missing storage handler for _ident %O\n", _ident);
            FATAL("prepare_save_statement object=%O, storage=%O, "+
		  "ident=%O, index=%O\n", o, storage, ident, index);
        }
        else if (storage[_ident][2]) // an indexed data-storage
        {
            //werror("indexed data-storage (ident=%s, indexed=%O, storage=%O)\n",
            //     _ident, storage[_ident][2], storage[_ident][0]);
            if (zero_type(index))
            {
                mapping mData = storage[_ident][0](); // retrieve all
                foreach(indices(mData), mixed _index)
                {
                    if (!stringp(_index))
                        continue;
                    data = quote_data(mData[_index], o, _ident,
                                      db_quote_data, CONVERSION);
                    statements +=
                        ({ sprintf("(\"%d\",\"%s\",\"%s\",\"%s\")",
                                   oid, _ident, db_quote_data(_index),
                                   data )
                        });
                }
            }
            else
            {
                if (_ident != "user" && index!="UserPassword")
                    data = quote_data(storage[_ident][0](index), o, _ident,
                                      db_quote_data, CONVERSION);
                else
                    data = quote_data(storage[_ident][0](index), o, _ident,
                                      db_quote_data); // never convert user pw
                                                      // to utf8
                statements += ({
                    sprintf("(\"%d\",\"%s\",\"%s\",\"%s\")",
                            oid, _ident, db_quote_data(index), data)
                });
            }
            
        }
        else // the usual unindexed data-storage
        {
            data = quote_data(storage[_ident][0](), o, "all", db_quote_data,
                              CONVERSION);
            statements += ({
                sprintf("(\"%d\",\"%s\",\"%s\",\"%s\")",
                        oid, _ident, "", data)
            });
        }
    }
    return statements*",";
}


/**
 * generate a delete statement that will clear all entries according to
 * the data that will be saved.
 * @author Ludger Merkens
 * @see prepare_save_statement
 */
private static string
prepare_clear_statement(object o, mapping storage,
                        string|void ident, string|void index, string|void tb)
{
    if ( !stringp(tb) )
      tb = "ob_data";

    if (ident=="0" || index=="0")
      FATAL("strange call to prepare_clear_statement \n%s\n",
	    describe_backtrace(backtrace()));
    
    if (!storage[ident]) ident =0; // better save then sorry - wrong ident
                                   // invoces a full save.
    if (ident && index)
        return sprintf("delete from %s where ob_id='%d' and "+
                       "ob_ident='%s' and ob_attr='%s'",
                       tb, o->get_object_id(), ident, index);
    else if (ident)
        return sprintf("delete from %s where ob_id='%d' and "+
                       "ob_ident='%s'", tb, o->get_object_id(), ident);
    else
        return sprintf("delete from %s where ob_id='%d'",
                       tb, o->get_object_id());
        
}

/**
 * low level database function to store a given (proxy) object into the
 * database immediately.
 *
 * @param  object proxy - the object to be saved
 * @return void
 * @see    save_object
 * @author Ludger Merkens 
 */
private static void
low_save_object(object p, string|void ident, string|void index)
{
    string sStatement;
#if 1
    LOG_DB("low_save_object:"+p->get_object_id()+"("+PSTAT(p->status())+") "+
		(p->status()>0 ? p->get_identifier() : ""));
#endif
    int stat = p->status();
    // saved twice while waiting
    if ( stat == PSTAT_DISK || stat == PSTAT_DELETED )
	return;   // low is local so this will unlock also

    //if ( stat == PSTAT_SAVE_OK )
    //werror("proxy %O twice in queue (ident=%O,index=%O)\n", p, ident, index);
    ASSERTINFO(!objectp(MODULE_SECURITY) ||
	       MODULE_SECURITY->valid_object(p),
	       "invalid object in database.save_object");

    if (p->status() < PSTAT_SAVE_OK)
    {
	FATAL("DBSAVEDEMON ->broken instance not saved(%d, %s, status=%s)",
              p->get_object_id(),
              master()->describe_object(p->get_object()),
	      PSTAT(p->status()));
	return;
    }
    if ( !p->is_loaded() ) {
	FATAL("DBSAVEDEMON ->trying to save an object that was not previously loaded !!!!!\n"+
	      "Object ID="+p->get_object_id() + "\n");
	return;
    }

    Thread.MutexKey low=muLowSave::lock(1);    

    string request = generate_request_name(p->get_object_id(), ident, index);
    mSaveRequests[request]--;
    if (mSaveRequests[request]==0)
        m_delete(mSaveRequests, request);

    if (p->status()<PSTAT_SAVE_OK)
	THROW("Invalid proxy status for object:"+
	      p->get_object_id()+"("+p->status()+")", E_MEMORY);

    mapping storage = get_storage_handlers(p);
    ASSERTINFO(mappingp(storage),
	       "Corrupted data_storage in "+master()->stupid_describe(p));

    
    p->set_status(PSTAT_SAVE_OK);
    destruct(low);                    // status set, so unlock
    
    if (master()->describe_program(object_program(p->get_object()))=="-")
        return; // temporary objects like executer
    
    sStatement =
        prepare_save_statement(p, storage, ident, index );

    ASSERTINFO(strlen(sStatement)!=0,
               sprintf("trying to insert empty data into object %d class %s",
                    p->get_object_id(),
                    master()->describe_program(object_program(p->get_object()))));
    
    mixed err;
    string s;


    err = catch {
      // First journaling 
      catch(db()->big_query("insert into ob_journaling values " + sStatement));
      
      // remove from ob_dat
      s = prepare_clear_statement(p, storage, ident, index, "ob_data");
      db()->big_query(s);

      // add new value
      db()->big_query("insert into ob_data values " + sStatement);

      s = prepare_clear_statement(p, storage, ident, index, "ob_journaling");
      catch(db()->big_query(s));
    };
    if ( err )
    {
      FATAL("!!! FATAL - Error in save-demon ------------\n%s\n---------!!!",
	    master()->describe_backtrace(err));
      if ( objectp(lostData) ) {
	catch {
	  lostData->write(sprintf("%d: %s\n\n", p->get_object_id(), sStatement));
	};
      }
    }
}

/**
 * register an module with its name
 * e.g. register_module("users", new("/modules/users"));
 *
 * @param   string - a unique name to register with this module.
 * @param   object module - the module object to register
 * @param   void|string source - a source directory for package installations 
 * @return  (object-id|0)
 * @see     /kernel/db_mapping, /kernel/secure_mapping
 * @author  Ludger Merkens 
 */
int register_module(string oname, object module, void|string source)
{
    object realObject;
    string version = "";

    FATAL(sprintf("register module %s with %O source %O", 
		   oname, module, source));
    if ( CALLER != _Server && 
	 !MODULE_SECURITY->access_register_module(0, 0, CALLER) )
	THROW("Unauthorized call to register_module() !", E_ACCESS);

    object mod;
    int imod = get_variable("#" + oname);
    
    if ( imod > 0 )
    {
	mod = find_object(imod); // get old module
	FATAL(sprintf("attempting to get object for module %s", oname));
	if ( objectp(mod) ) {
	    object e = master()->getErrorContainer();
	    master()->set_inhibit_compile_errors(e);
	    realObject = mod->get_object();
	    master()->set_inhibit_compile_errors(0);
	    if (!realObject)
	    {
		FATAL("failed to compile new instance - throwing");
		THROW("Failed to load module\n"+e->get()+"\n"+
		      e->get_warnings(), backtrace());
	    }
	    FATAL(sprintf("module found is %O", realObject));
	}
    }
    if ( objectp(realObject) ) {
	FATAL("Found previously registered version of module !");
	if ( objectp(module) && module->get_object() != realObject )
	    THROW("Trying to register a previously registered module.",
		  E_ERROR);
	
	version = realObject->get_version();
	
	mixed erg = master()->upgrade(object_program(realObject));
	FATAL(sprintf("upgrade resulted in %O", erg));
	if (!intp(erg) ||  erg<0)
	{
	    if (stringp(erg))
		THROW(erg, backtrace());
	    else
	    {
		FATAL("New version of "+oname+" doesn't implement old "+
		    "versions interface");
		master()->upgrade(object_program(mod->get_object()),1);
	    }
	}
	    FATAL("Upgrading done !");
	    module = mod;
    }
    else if ( !objectp(module) ) 
    {
	// module is in the /modules directory.
	object e = master()->getErrorContainer();
	master()->set_inhibit_compile_errors(e);
	module = new("/modules/"+oname+".pike");
	master()->set_inhibit_compile_errors(0);
	if (!module)
	{
	    FATAL("failed to compile new instance - throwing");
	    THROW("Failed to load module\n"+e->get()+"\n"+
		  e->get_warnings(), backtrace());
	}
    }
    
    FATAL(sprintf("installing module %s", oname));
    if ( !stringp(source) )
	source = "";
    
    if ( module->get_object_class() & CLASS_PACKAGE ) {
	
	if ( module->install(source, version) == 0 )
	    error("Failed to install module !");
    }    
    _Server->register_module(module);

    _Server->run_global_event(EVENT_REGISTER_MODULE, PHASE_NOTIFY, 
			      this_object(), ({ module }) );
    LOG_DB("event is run");
    if ( objectp(module) ) 
    {
	set_variable("#"+oname, module->get_object_id());
	_Server->register_module(module);
	return module->get_object_id();
    }
    return 0;
}

/**
 * Check if a database handle is connected to a properly setup database.
 *
 * @param   Sql.Sql handle - the handle to check
 * @return  1 - old format
 * @return  2 - new format
 * @see     setup_sTeam_tables
 * @author  Ludger Merkens 
 */
int validate_db_handle(SqlHandle handle)
{
    multiset tables = (<>);
    array(string) aTables = handle->list_tables();

    foreach(aTables, string table)
	tables[table] = true;
    if (tables["objects"] && tables["doc_data"])
        return 1;
    if (tables["ob_class"] && tables["ob_data"] && tables["doc_data"])
        return 2;
}

bool check_convert()
{
    multiset tables = (<>);
    Sql.Sql handle = SqlHandle(STEAM_DB_CONNECT);
    array(string) aTables = handle->list_tables();

    foreach(aTables, string table)
	tables[table] = true;

    // make sure the "var" column of table "variables" is type char(100):
    if ( tables["variables"] ) {
      Sql.sql_result result = handle->big_query("show columns from variables");
      if ( objectp(result) ) {
        array row = result->fetch_row();
        while ( arrayp(row) ) {
          if ( (sizeof(row) >= 2) && (row[0] == "var") && stringp(row[1]) ) {
            if ( lower_case(row[1]) != "char(100)" ) {
              werror( "Database: setting column \"var\" of table \"variables\""
                      " to type \"char(100)\" (was: "+row[1]+")...\n" );
              MESSAGE( "Database: setting column \"var\" of table \"variables\""
                      " to type \"char(100)\" (was: "+row[1]+")..." );
              handle->big_query( "alter table variables modify var char(100)" );
            }
            break;
          }
          row = result->fetch_row();
        }
      }
    }

    // make sure the max_rows value of ob_data is large enough:
    if ( tables["ob_data"] ) {
      int DB_MAX_ROWS = 4294967295;
      string db_name;
      sscanf( STEAM_DB_CONNECT, "mysql://%*s:%*s@%*s/%s", db_name );
      if ( ! stringp(db_name) ) db_name = "steam";
      Sql.sql_result result = handle->big_query( "show table status from "
                                                 + db_name + " like 'ob_data'" );
      array fields = result->fetch_fields();
      array row = result->fetch_row();
      if ( arrayp(fields) && arrayp(row) ) {
        for ( int i=0; i<sizeof(fields); i++ ) {
          if ( fields[i]["name"] == "Max_data_length" ) {
            int max_rows = DB_MAX_ROWS;
            if ( stringp(row[i]) ) sscanf( row[i], "%d", max_rows );
            else if ( intp(row[i]) ) max_rows = row[i];
            if ( max_rows < DB_MAX_ROWS ) {
              werror( "\nDatabase: altering ob_data table to MAX_ROWS = "
                      + DB_MAX_ROWS + " (was: " + max_rows + ").\n"
                      + "This could take a while...\n" );
              MESSAGE( "\nDatabase: altering ob_data table to MAX_ROWS = "
                      + DB_MAX_ROWS + " (was: " + max_rows + ").\n"
                      + "This could take a while..." );
              handle->big_query( "alter table ob_data MAX_ROWS=" + DB_MAX_ROWS );
            }
            break;
          }
        }
      }
    }

    return tables["objects"];
}


void set_converting(int i)
{
    CONVERSION =i;
}

int get_converting()
{
    return CONVERSION;
}

void convert_tables()
{
    if ( CALLER != _Server )
	error("Unauthorized call to convert_tables() !");
    werror("creating new table format\n");
    Sql.Sql handle = SqlHandle(STEAM_DB_CONNECT);
    create_tables(handle);
    werror("Starting database conversion ...\n");
    convert_old_tables(handle);
    werror("Database conversion finished sucessfully.\n");
    handle->big_query("alter table objects rename objectsold");
    exit(1);
}

/**
 * Convert an old database of sTeam Version below 1.5 to new format
 *
 *
 */
private static void
convert_old_tables(SqlHandle handle)
{
    string sOid, sOclass, sOdata;
    mixed erg;
    mapping mData;
    object o,p;
    mapping storage;
    string sStatement;
    
 // uncatched ... we want to die in that case.

    convert_index_tables(handle, "i_groups");
    convert_index_tables(handle, "i_users");

    Sql.Sql writehandle = Sql.Sql(STEAM_DB_CONNECT);
    Sql.sql_result instream =
        handle->big_query("select ob_id, ob_class, ob_data from objects");

    while (erg = instream->fetch_row())
    {
        [sOid, sOclass, sOdata] = erg;
        if (sOclass == "-:15") continue;
        if (sOclass == "-") continue;
        if (sOclass == "/factories/AnnotationFactory.pike") continue;
        if (sOclass == "/modules/keyword_index.pike") continue;
        
        werror("CONV %s(%s):", sOid, sOclass);
        p = find_object((int) sOid);
        werror("id-");
        mixed mIdent = p->get_identifier();
        werror("(%O)", mIdent);
        if (p->status()==PSTAT_SAVE_OK)
        {
            o=p->get_object();
            convert_keywords(writehandle, o);
            sOclass = master()->describe_program(object_program(o));
            writehandle->query(
                sprintf("insert into ob_class values(%s,'%s')",
                        sOid, sOclass));
            // compatibility_load_object((int)sOid, o, sOdata);
            werror("-prep-");
            storage = get_storage_handlers(o);
            sStatement = prepare_save_statement(p, storage);
            werror("save(%O)-", p);
            // again we want to die in case of an error --- no catch
            string clear = prepare_clear_statement(p, storage);

            writehandle->query("replace into ob_data values "+sStatement);
            switch ( sOclass ) {
              case "/classes/User" : write(" post conversion ");
                  mixed err;
                  err = catch {
                      object oForwardModule = get_module("forward");
                      if (p->query_attribute(USER_FORWARD_MSG) && stringp(p->query_attribtue(USER_EMAIL))
                          &&!arrayp(oForwardModule->get_forward(p)))
                      {
                          oForwardModule->add_forward(p, "/"+p->get_identifier());
                          oForwardModule->add_forward(p, p->query_attribute(USER_EMAIL));
                      }
                  };
                  if (err)
                      write("failed : %s\n",master()->describe_backtrace(err));
                  else
                      werror("ok\n");
                  break;
              default : werror("ok\n");
            }
        }
        else
            werror("Failed to load %s (%O)\n", sOid, p);
    }
}

private static void convert_keywords(Sql.Sql handle, object o)
{
    if ( !functionp(o->get_object_id) )
        return;
    Sql.sql_result res =
        handle->big_query("select k from mi_keyword_index where v='%"+
                          o->get_object_id()+"'");
    while (mixed key = res->fetch_row())
        if (key[0] != "0")
            o->restore_keywords(1, string_to_utf8(key[0]));
}

private static void convert_index_tables(Sql.Sql handle, string index)
{
    mixed res = handle->query("select k,v from "+ index);
    handle->query("delete from "+ index);
    foreach(res, mapping line)
    {
        handle->query("insert into "+index+" values(%s,%s)",
                      string_to_utf8(line["k"]), line["v"]);
    }
}


static void check_journaling(Sql.Sql handle) 
{
  mixed row, res, err;

  string lost = Stdio.read_file("/tmp/lost_data.steam");
  lostData = Stdio.File("/tmp/lost_data.steam", "wct");
  if ( stringp(lost) && strlen(lost) > 0 ) {
    array lostlines = lost / "\n\n";
    foreach(lostlines, string ll) {
      werror("LOST DATA: Restoring %s\n", ll);
      MESSAGE("LOST DATA: Restore %s", ll);
      int oid;
      string ident, attr, val;
      if ( sscanf(ll, "%d: (%*s,%s,%s,%s)", oid, ident, attr, val) != 4 )
	continue;
      werror("values are %O %O %O %O\n", oid, ident, attr, val);
      err = catch {
	res = handle->query(sprintf("select ob_data from ob_data where ob_id='%d' and ob_ident=%s and ob_attr=%s", oid, ident, attr));
	row = res->fetch_row();
	if ( sizeof(row) > 0 ) {
	  handle->query(sprintf("update ob_data SET ob_data=%s where ob_id='%d' and ob_ident=%s and ob_attr=%s", val, oid, ident, attr));
	  werror("updated!\n");
	}
	else {
	  handle->query(sprintf("insert into ob_data values (%d,%s,%s,%s)",
				oid, ident, attr, val));
	}
      };
      if ( err ) {
	FATAL("Failed to restore data: %O", err);
      }
    }
  }
  
  handle->query("create table if not exists ob_journaling ("+
		" ob_id int not null, "+
		" ob_ident char(15) not null,"+
		" ob_attr char(50), "+
		" ob_data mediumtext,"+
		" unique(ob_id, ob_ident, ob_attr),"+
		" index i_attr (ob_attr)"+
		")");
  res = 
    handle->big_query("select ob_id, ob_ident, ob_attr, ob_data from ob_journaling");

  if ( !objectp(res) )
    return;
  while ( row = res->fetch_row() ) {
    if ( arrayp(row) ) {
      MESSAGE("JOURNAL: Restoring %O", row);
      string q = sprintf("insert into ob_data values (%d, '%s', '%s', '%s')",
			 (int)row[0], row[1], row[2], handle->quote(row[3]));
      string c = sprintf("delete from ob_data where ob_id='%d' and ob_ident='%s' and ob_attr='%s'", (int)row[0], row[1], row[2]);
      err = catch {
	handle->query(c);
	handle->query(q);
      };
      if ( err != 0 ) {
	error("Failed to restore journaling table: " + err[0] +
            sprintf("\n%O\n%O", err[1], row));
      }
    }
  }
  handle->query("delete from ob_journaling where 1");
}

static void add_new_tables(Sql.Sql handle) {
    MESSAGE("adding new format tables\n");
    MESSAGE("adding ob_class ");
    catch {
        handle->query("drop table if exists ob_class");
        handle->query("drop table if exists ob_data");
    };
    handle->query("create table if not exists ob_class ("+
                  " ob_id int primary key, "+
                  " ob_class char(128) "+
                  ")");

    MESSAGE("adding ob_data ");
    handle->query("create table if not exists ob_data ("+
                  " ob_id int not null, "+
                  " ob_ident char(15) not null,"+
                  " ob_attr char(50), "+
                  " ob_data mediumtext,"+
                  " unique(ob_id, ob_ident, ob_attr),"+
                  " index i_attr (ob_attr),"+
                  " index i_attrdata (ob_attr, ob_data(80))"+
                  ")");

    handle->query("create table if not exists ob_journaling ("+
                  " ob_id int not null, "+
                  " ob_ident char(15) not null,"+
                  " ob_attr char(50), "+
                  " ob_data mediumtext,"+
                  " unique(ob_id, ob_ident, ob_attr),"+
                  " index i_attr (ob_attr)"+
                  ")");
}

static void create_tables(Sql.Sql handle) {
    MESSAGE("creating table \"doc_data\" ");
    handle->big_query("create table if not exists doc_data ("+
                      " rec_data text, "+
		      " doc_id int not null, "+
		      " rec_order int not null, "+
                      " primary key (doc_id, rec_order)"+
                      ") AVG_ROW_LENGTH=65535 MAX_ROWS=4294967296");
    //FIXME: postgres does not support (and probably not even need)
    //AVG_ROW_LENGTH and MAX_ROWS in this place
    
    
    MESSAGE("creating table \"ob_class\" ");
    handle->big_query("create table if not exists ob_class ("+
                      " ob_id int primary key, "+
                      " ob_class char(128) "+
                      ")");

    MESSAGE("creating table \"ob_data\" ");
    handle->big_query("create table if not exists ob_data ("+
                      " ob_id int not null, "+
                      " ob_ident char(15) not null,"+
                      " ob_attr char(50), "+
                      " ob_data mediumtext,"+
                      " unique(ob_id, ob_ident, ob_attr)"+
                      ")");
    MESSAGE("creating table \"variables\" ");
    handle->big_query("create table if not exists variables ("+
                      " var char(100) primary key, "+
                      " value int"+
                      ")");
}

/**
 * set up the base sTeam tables to create an empty database.
 *
 * @param  none
 * @return (1|0)
 * @author Ludger Merkens 
 */
int setup_sTeam_tables(SqlHandle handle)
{
    /* make sure no old tables exist and delete them properly */
    MESSAGE("Checking for old tables.\n");

    if (CONVERSION)
    {
        THROW("preliminary sTeam Table setup during CONVERSION!", E_ERROR);
    }
    //    Sql.Sql_result res = handle->big_query("show tables");
    array(string) res = handle->list_tables();
    if (sizeof(res))
    {
        foreach(res, string table)
	{
	    MESSAGE(sprintf("dropping (%s)\n",table));
	    handle->big_query("drop table "+table);
	}
    }
    else
	MESSAGE("no old tables found");

    MESSAGE("CREATING NEW BASE TABLES:");
    create_tables(handle);
    
    res = handle->list_tables();
    if (!sizeof(res)) {
	FATAL("\nFATAL: failed to create base tables");
    }
    else
    {
	MESSAGE("\nPOST CHECK retrieves: ");
        foreach(res, string table)
	    MESSAGE(table+" ");
    }
    return 1;
}

/**
 * create and return a new instance of db_file
 *
 * @param  int iContentID - 0|ID of a given Content
 * @return the db_file-handle
 * @see    db_file
 * @see    file/IO
 * @author Ludger Merkens 
 */
object new_db_file_handle(int iContentID, string mode)
{
    return new("/kernel/db_file.pike", iContentID, mode);
}

/**
 * Check if a given gile handle is valid, eg inherits db_file.pike
 *  
 * @param object m - the db_file to check
 * @return true or false
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
private static bool valid_db_file(object m)
{
    if (Program.inherits(object_program(m), (program)"/kernel/db_file.pike"))
        return true;
    return false;
}

/**
 * connect_db_file, connect a /kernel/db_file instance with the database
 * calculate new content id if none given.
 *
 * @param    id
 * @return   function db()
 */
final mixed connect_db_file(int id)
{
    //    if (!(CALLINGFUNCTION == "get_database_handle" &&
    //	  valid_db_file(CALLER)))
    //	THROW("illegal access to database ", E_ACCESS);
    return ({ db, (id==0 ? create_new_database_id("doc_data") : id)});
}

/**
 * valid_db_mapping - check if an object pretending to be an db_mapping
 * really inherits /kernel/db_mapping and thus is a trusted program
 * @param     m - object inheriting db_mapping
 * @return    (TRUE|FALSE)
 * @see       connect_db_mapping
 * @author Ludger Merkens 
 */
private static bool valid_db_mapping(object m)
{
    if ( Program.inherits(object_program(m),
			  (program)"/kernel/db_mapping.pike") ||
         Program.inherits(object_program(m),
                          (program)"/kernel/db_n_one.pike") ||
         Program.inherits(object_program(m),
                          (program)"/kernel/db_n_n.pike") ||
         Program.inherits(object_program(m),
                          (program)"/kernel/db_searching.pike"))
	return true;
    return false;
}

/**
 * connect_mapping, connect a /kernel/db_mapping instance with the database
 * @param    none
 * @return   a pair ({ function db, string tablename })
 */
final mixed connect_db_mapping()
{
    if (!(CALLINGFUNCTION == "load_db_mapping" &&
	  valid_db_mapping(CALLER)))
    {
        FATAL("illegal access %s from %O\n",CALLINGFUNCTION, CALLER);
	THROW("illegal access to database ", E_ACCESS);
    }
    string sDbTable;
    // hack to allow the modules table to be a member of _Database

    sDbTable = CALLER->get_table_name();


    if (!sDbTable)
        THROW(sprintf("Invalid tablename [%s] in module \"%s\"\n",sDbTable,
                      master()->describe_program(CALLERPROGRAM)), E_ERROR);
    /*    if (search(db()->list_tables("i_"+CALLER->get_object_id()),
               "i_"+CALLER->get_object_id())!=-1)
    {
        werror(sprintf("Detected invalid tablename %s - fix it by running "+
                      "check_database first - CALLER %s\n",
                      "i_"+CALLER->get_object_id(),
                       master()->describe_program(CALLERPROGRAM)));
        return ({ 0,0 });
        }*/
    return ({ db, sDbTable });
}

string get_identifier() { return "database"; }
string _sprintf() { return "database"; }
string describe() { return "database"; }
int get_object_class() { return CLASS_DATABASE; }
object this() { return this_object(); }




/**
 * get_objects_by_class()
 * mainly for maintenance reasons, retreive all objects matching a given
 * class name, or program
 * @param class (string|object|int) - the class to compare with
 * @return array(object) all objects found in the database
 * throws on access violation. (ROLE_READ_ALL required)
 * @autoher Ludger Merkens
 */
final array(object) get_objects_by_class(string|program|int mClass)
{
    Sql.sql_result res;
    int i, sz;
    object security;
    array(object) aObjects;
    string sClass;
    
    if (security=MODULE_SECURITY)
        ASSERTINFO(security->
                   check_access(0, this_user(), 0, ROLE_READ_ALL, false),
                   "Illegal access on database.get_all_objects");

    if ( intp(mClass) ) {
      mixed factory = _Server->get_factory(mClass);
      if ( !objectp(factory) )
        return UNDEFINED;
      if ( !stringp(CLASS_PATH) || !stringp(factory->get_class_name()) )
        return UNDEFINED;
      mClass = CLASS_PATH + factory->get_class_name();
    }

    if (programp(mClass))
        sClass = master()->describe_program(mClass);
    else
        sClass = mClass;
    
    res = db()->big_query("select ob_id from ob_class where ob_class='"+
                          mClass+"'");

    aObjects = allocate((sz=res->num_rows()));

    for (i=0;i<sz;i++)
    {
        aObjects[i]=find_object((int)res->fetch_row()[0]);
    }
    return aObjects;
}


/**
 * get_all_objects()
 * mainly for maintenance reasons
 * @return array(object) all objects found in the database
 * throws on access violation. (ROLE_READ_ALL required)
 * @autoher Ludger Merkens
 */
final array(object) get_all_objects()
{
    Sql.sql_result res;
    int i, sz;
    object security;
    array(object) aObjects;

#if 0
    if ( !_Server->is_a_factory(CALLER) )
        THROW("Illegal attempt to call database.get_all_objects !", E_ACCESS);

    if (security=MODULE_SECURITY)
        ASSERTINFO(security->
                   check_access(0, this_user(), 0, ROLE_READ_ALL, false),
                   "Illegal access on database.get_all_objects");
#endif
    
    res = db()->big_query("select ob_id from ob_class where ob_class !='-'");
    aObjects = allocate((sz=res->num_rows()));

    for (i=0;i<sz;i++)
    {
        aObjects[i]=find_object((int)res->fetch_row()[0]);
    }
    return aObjects;
}

/**
 * visit_all_objects
 * loads all objects from the database, makes sure each object really loads
 * and calls the function given as "visitor" with consecutive with each object.
 * @param function visitor
 * @return nothing
 * @author Ludger Merkens
 * @see get_all_objects
 * @see get_all_objects_like
 * @caveats Because this function makes sure an object is properly loaded
 *          when passing it to function "visitor", you won't
 *          notice the existence of objects currently not loading.
 */
final void visit_all_objects(function visitor, mixed ... args)
{
    FATAL("visit_all_objects not yet converted to new database format");
    return;
    Sql.sql_result res = db()->big_query("select ob_id,ob_class from ob_class");
    int i;
    int oid;
    string oclass;
    object p;
    FATAL("Number of objects found:"+res->num_rows());
    for (i=0;i<res->num_rows();i++)
    {
        mixed erg =  res->fetch_row();
        oid = (int) erg[0];  // wrong casting with 
        oclass = erg[1];     // [oid, oclass] = res->fetch_row()
        
        if (oclass[0]=='/') // some heuristics to avoid nonsene classes
        {
            p = find_object((int)oid);         // get the proxy
            catch{p->get_object();};      // force to load the object
            if (p->status() > PSTAT_DISK) // positive stati mean object loaded
                visitor(p, @args);
        }
    }
}

/**
 * Check for a list of objects, if they really exist in the database
 *
 * @param objects - the list of object to be checked
 * @return a list of those objects, which really exist.
 * @author Ludger Merkens
 * @see get_not_existing
 */
array(int) get_existing(array(int) ids)
{
    Sql.sql_result res;
    int i, sz;
    string query = "select ob_id from ob_class where ob_id in (";
    array(int) result;
    
    if (!ids || !sizeof(ids))
        return ({ });
    for (i=0,sz=sizeof(ids)-1;i<sz;i++)
        query +=ids[i]+",";
    query+=ids[i]+")";
    res = db()->big_query(query);

    result = allocate((sz=res->num_rows()));
    for (i=0;i<sz;i++)
        result[i]=(int) res->fetch_row()[0];

    return result;
}

/**
 * Get a list of the not-existing objects.
 *  
 * @param objects - the list of objects to be checked
 * @return a list of objects that are not existing
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 * @see 
 */
array(int) get_not_existing(array(int) ids)
{
    return ids - get_existing(ids);
}

object get_environment() { return 0; }
object get_acquire() { return 0; }

mapping get_xml_data()
{
    return ([ "configs":({_Server->get_configs, XML_NORMAL}), ]);
}

/**
 * clears lost content records from the doc_data table, used for the
 * db_file emulation. This function is purely for maintainance reasons, and
 * should be obsolete, since we hope no content records will get lost
 * anymore.
 * @param none
 * @returns a debug string containing the number of deleted doc_id's
 */
string clear_lost_content()
{
    Sql.Sql h = db();
    LOG("getting doc_ids");
    Sql.sql_result res = h->big_query("select distinct doc_id from doc_data");
    array(int) doc_ids = allocate(res->num_rows());
    for(int i=0;i<sizeof(doc_ids);i++)
        doc_ids[i]=(int)res->fetch_row()[0];

    FATAL("deleting '-' files");
    h->big_query("delete from objects where ob_class='-'");
    FATAL("getting all objects");
    res = h->big_query("select ob_id from objects");
    int oid; object p; mixed a;
    while (a = res->fetch_row())
    {
        oid = (int)a[0];
        if (p=find_object(oid))
        {
            FATAL("accessing object"+oid);
            object try;
            catch{try=p->get_object();};
            if (objectp(try) &&
                Program.inherits(object_program(try),
                                 (program)"/base/content"))
            {
                FATAL("content "+p->get_content_id()+" is in use");
                doc_ids  -= ({ p->get_content_id() });
            }
        }
    }

    FATAL("number of doc_ids to be deleted is:"+sizeof(doc_ids));

    foreach (doc_ids, int did)
    {
        h->big_query("delete from doc_data where doc_id = "+did);
        FATAL("deleting doc_id"+did);
    }
    FATAL("calling optimize");
    h->big_query("optimize table doc_data");
    return "deleted "+sizeof(doc_ids)+"lost contents";
}

object lookup (string identifier)
{
  return get_module("objects")->lookup(identifier);
}

object lookup_user (string identifier)
{
  return get_module("users")->get_user(identifier);
}

object lookup_group (string identifier)
{
  return get_module("groups")->get_group(identifier);
}

