/* Copyright (C) 2000-2006  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: smtp.pike,v 1.9 2006/05/03 22:03:14 exodusd Exp $
 */

constant cvs_version="$Id: smtp.pike,v 1.9 2006/05/03 22:03:14 exodusd Exp $";

inherit "/kernel/module";

#include <macros.h>
#include <config.h>
#include <database.h>
#include <attributes.h>

//#define SMTP_DEBUG

#ifdef SMTP_DEBUG
#define SMTP_LOG(s, args...) werror(s+"\n", args)
#else
#define SMTP_LOG(s, args...)
#endif

#if constant(Protocols.SMTP.client) 
#define SMTPCLIENT Protocols.SMTP.client
#else
#define SMTPCLIENT Protocols.SMTP.Client
#endif


//! This is the SMTP module for sTeam. It sends mail to some e-mail adress
//! by using a local mailserver or doing MX lookup and sending directly
//! to the targets mail server.

static Thread.Queue MsgQueue  = Thread.Queue();
static Thread.Queue saveQueue = Thread.Queue();

static object oSMTP; // cache smtp object (connection)
static object oDNS = Protocols.DNS.client(); // cache DNS Client

class Message {
    array|string email;
    string subject;
    string body;
    string from;
    string fromobj;
    string fromname;
    string mimetype;
    string date;
    string message_id;
    string in_reply_to;
    string reply_to;
    string mail_followup_to;
    string rawmime;
    int sent = 0;
  
    void create(void|mapping m) {
	if ( mappingp(m) ) {
	    email = m->email;
	    subject = m->subject;
	    body = m->body;
	    from = m->from;
	    fromobj = m->fromobj;
	    fromname = m->fromname;
	    mimetype = m->mimetype;
	    date = m->date;
	    message_id = m->message_id;
	    in_reply_to = m->in_reply_to;
	    reply_to = m->reply_to;
	    mail_followup_to = m->mail_followup_to;
	    rawmime = m->rawmime;
	}
    }
    
    mapping get_data() {
	return ([
	    "email": email,
	    "subject": subject,
	    "body": body,
	    "from": from,
	    "fromobj": fromobj,
	    "fromname": fromname,
	    "mimetype": mimetype,
	    "date": date,
	    "message_id": message_id,
	    "in_reply_to": in_reply_to,
	    "reply_to": reply_to,
	    "mail_followup_to": mail_followup_to,
	    "rawmime": rawmime,
	    ]);
    }
};

void init_module()
{
    add_data_storage(STORE_SMTP, retrieve_mails, restore_mails);
}

void restore_mails(mapping data)
{
    if ( CALLER != _Database )
	steam_error("CALLER is not the database !");
    foreach ( data->mails, mapping mail ) {
	object msg = Message(mail);
	MsgQueue->write(msg);
    }
}

mapping retrieve_mails()
{
    if ( CALLER != _Database )
	steam_error("CALLER is not the database !");
    mapping data = ([ "mails": ({ }), ]);
    array aMessages = ({ });
    object msg;
    while ( saveQueue->size() > 0 ) {
	msg = saveQueue->read();
	if ( msg->sent == 0 )
	    aMessages += ({ msg });
    }
    foreach ( aMessages, object msg ) {
	data->mails += ({ msg->get_data() });
	saveQueue->write(msg);
    }
    get_module("log")->log_debug("smtp", "Saving messages:\n%d messages\n"
                                 + "%O\n----\n", saveQueue->size(),  data);
    return data;
}
	

void runtime_install()
{
    SMTP_LOG("Init module SMTP !");

    // an initial connection needs to be created to load some libraries
    // otherwise creating connections will fail after the sandbox
    // is in place (chroot("server/"))
    string server = _Server->query_config(CFG_MAILSERVER);
    int port = _Server->query_config(CFG_MAILPORT);
    if ( !intp(port) || port <= 0 ) port = 25;
    mixed err;

    if ( stringp(server) && sizeof(server) > 0 )
      err = catch( oSMTP = SMTPCLIENT( server, port ) );
    if ( err )
      FATAL("Failed to connect to " + server+" :\n"+sprintf("%O\n", err));

    start_thread(smtp_thread);
}

void 
send_mail(array|string email, string subject, string body, void|string from, void|string fromobj, void|string mimetype, void|string fromname, void|string date, void|string message_id, void|string in_reply_to, void|string reply_to, void|string mail_followup_to)
{
    Message msg = Message();
    msg->email   = email;
    msg->subject = subject;
    msg->mimetype = (stringp(mimetype) ? mimetype : "text/plain");
    if ( lower_case(msg->mimetype) == "text/html" )
      msg->body = Messaging.fix_html( body );
    else
      msg->body = body;
    msg->date    = date || timelib.smtp_time(time());
    msg->message_id = message_id||("<"+(string)time()+(fromobj||("@"+_Server->get_server_name()))+">");
    msg->rawmime = 0;
    if(reply_to)
      msg->reply_to=reply_to;
    if(mail_followup_to)
      msg->mail_followup_to=mail_followup_to;
    if(in_reply_to)
      msg->in_reply_to=in_reply_to;

    get_module("log")->log_debug( "smtp", "send_mail: to=%O", email );
    
    if ( stringp(from) && sizeof(from) > 0 )
      msg->from    = from;
    else
      msg->from = this_user()->get_identifier() +
        "@" + _Server->get_server_name();
    if ( stringp(fromobj) )
      msg->fromobj = fromobj;
    if ( stringp(fromname) )
      msg->fromname = fromname;

    MsgQueue->write(msg);
    saveQueue->write(msg);
    call(require_save, 0, STORE_SMTP);
}

void send_mail_raw(string|array email, string data, string from)
{

  Message msg = Message();
  msg->email   = email;
  msg->rawmime = data;
  if ( stringp(from) && sizeof(from) > 0 )
    msg->from = from;
  else
    msg->from = this_user()->get_identifier()
      + "@" + _Server->get_server_name();

  MsgQueue->write(msg);
  saveQueue->write(msg);
  call(require_save, 0, STORE_SMTP);
}

void send_mail_mime(string email, object message)
{
    mapping mimes = message->query_attribute(MAIL_MIMEHEADERS);
    string from;
    sscanf(mimes->from, "%*s<%s>", from);
    if ( !stringp(from) || sizeof(from) < 1 )
      from = this_user()->get_identifier() + "@" + _Server->get_server_name();
    send_mail(email, message->get_identifier(), message->get_content(), from);
}

static mixed cb_tag(Parser.HTML p, string tag)
{
    if ( search(tag, "<br") >= 0 || search (tag, "<BR") >= 0 )
	return ({ "\n" });
    return ({ "" });
}

void send_message(Message msg)
{
  string server;
  int      port;
  mixed     err;
  object   smtp;

  server = _Server->query_config(CFG_MAILSERVER);
  port   = (int)_Server->query_config(CFG_MAILPORT);
  if ( !intp(port) || port <= 0 ) port = 25;

  object log = get_module( "log" );

  log->log_debug( "smtp", "send_message: server: %O:%O", server, port );

  if ( !stringp(msg->from) ) {
    msg->from = "admin@"+_Server->get_server_name();
    log->log_debug( "smtp", "send_message: invalid 'from', using %s",
                    msg->from );
  }
  log->log_debug( "smtp", "send_message: mail from %O to %O (server: %O:%O)\n"
      + "  Subject: %O", msg->from, msg->email, server, port, msg->subject );
  
  if ( !arrayp(msg->email) )
    msg->email = ({ msg->email });

  foreach ( msg->email, string email ) {
    string tmp_server = server;
    int tmp_port = port;
    mixed smtp_error;
    if ( stringp(server) && sizeof(server) > 0 ) {
      smtp_error = catch {
        smtp = SMTPCLIENT( server, port );
      };
    }
    else {
      // if no server is configured use the e-mail of the receiver
      string host;
      sscanf( email, "%*s@%s", host );
      if ( !stringp(host) )
	steam_error("MX Lookup failed, host = 0 in %O", email);
      
      tmp_server = oDNS->get_primary_mx(host);
      array dns_data = oDNS->gethostbyname(tmp_server);
      if ( arrayp(dns_data) && sizeof(dns_data) > 1 &&
           arrayp(dns_data[1]) && sizeof(dns_data[1]) > 0 )
        tmp_server = dns_data[1][0];
      tmp_port = 25;
      log->log_debug( "smtp", "send_message: MX lookup: %O:%O",
                      tmp_server, tmp_port );
      smtp_error = catch {
        smtp = SMTPCLIENT( tmp_server, tmp_port );
      };
    }

    if ( !objectp(smtp) || smtp_error ) {
      string msg = sprintf( "Invalid mail server %O:%O (from %O to %O)\n",
                            tmp_server, tmp_port, msg->from, email );
      if ( smtp_error ) msg += sprintf( "%O", smtp_error );
      log->log_error( "smtp", msg );
      continue;
    }

    if ( stringp(msg->rawmime) ) { 
      // send directly
      smtp_error = catch {
        smtp->send_message(msg->from, ({ email }),  msg->rawmime);
      };
      if ( smtp_error ) {
        log->log_error( "smtp",
            "Failed to send mail directly from %O to %O via %O:%O : %O",
            msg->from, email, tmp_server, tmp_port, smtp_error[0] );
      }
      else {
        log->log_info( "smtp", "Mail sent directly from %O to %O via %O:%O\n",
                       msg->from, email, tmp_server, tmp_port );
      }
      continue;
    }
  
    if ( !stringp(msg->mimetype) )
      msg->mimetype = "text/plain";

    MIME.Message mmsg = MIME.Message(
        msg->body||"",
        ([ "Content-Type": (msg->mimetype||"text/plain") + "; charset=utf-8",
           "Mime-Version": "1.0 (generated by open-sTeam)",
           "Subject": msg->subject||"",
           "Date": msg->date || timelib.smtp_time(time()),
           "From": msg->fromname||msg->from||msg->fromobj||"",
           "To": (msg->fromobj ? msg->fromobj : email)||"",
           "Message-Id": msg->message_id||"",
        ]) );
	 
    if(msg->mail_followup_to)
      mmsg->headers["Mail-Followup-To"]=msg->mail_followup_to;
    if(msg->reply_to)
      mmsg->headers["Reply-To"]=msg->reply_to;
    if(msg->in_reply_to)
      mmsg->headers["In-Reply-To"]=msg->in_reply_to;
  
    smtp_error =  catch {
      smtp->send_message(msg->from, ({ email }), (string)mmsg);
    };
    if ( smtp_error ) {
      log->log_error( "smtp", "Failed to send mail from %O to %O via %O:%O"
          + " : %O\n", msg->from, email, tmp_server, tmp_port, smtp_error[0] );
    }
    else {
      log->log_info( "smtp", "Mail sent from %O to %O via %O:%O\n",
                     msg->from, email, tmp_server, tmp_port );
    }
  }

  call( require_save, 0, STORE_SMTP );
}

static void smtp_thread()
{
    Message msg;

    while ( 1 ) {
	SMTP_LOG("smtp-thread running...");
	msg = MsgQueue->read();
	mixed err = catch {
	    send_message(msg);
	    msg->sent = 1;
	    call(require_save, 0, STORE_SMTP);
	};
	if ( err != 0 ) {
	    FATAL("Error while sending message:" + err[0] + 
		sprintf("\n%O\n", err[1]));
	    FATAL("MAILSERVER="+_Server->query_config(CFG_MAILSERVER));
	    if ( objectp(oSMTP) ) {
		destruct(oSMTP);
		oSMTP = 0;
	    }
	    sleep(60); // wait one minute before retrying
	}
    }
}

string get_identifier() { return "smtp"; }







