/* Copyright (C) 2000-2006  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: server.pike,v 1.3 2006/04/20 13:40:31 astra Exp $
 */

constant cvs_version="$Id: server.pike,v 1.3 2006/04/20 13:40:31 astra Exp $";
#pike 7.4

/*!\mainpage sTeam Function Documentation
 *
 * \section Server Developers
 * - Ludger Merkens
 * - Thomas Bopp
 * - Christian Schmidt
 * - Martin Baehr
 * - Robert Hinn
 * 
 * Consult the Server documentation for more information.
 */

//! server is the most central part of sTeam, loads and handles factories
//! and modules. Global Events are also triggered through the server and
//! can be subcribed by modules.


private static object                   nmaster;
private static int                  iLastReboot;
private static object                 oDatabase;
private static object              oPersistence; // the persistence manager
private static object               oBacktraces;
private static mapping       mGlobalBlockEvents;
private static mapping      mGlobalNotifyEvents;
private static mapping                 mConfigs;
private static mapping         mConfigsFromFile;
private static mapping                     mHBS;
private static mapping                 mClasses;
private static mapping                 mModules;
private static mapping                  mErrors;
private static mapping              mReadConfig;
private static int                        iTest;
private static mapping            mPortPrograms; // mapping of port programs
private static object          _stderr, _stdout;

private static mapping userConfigs = ([ "database": 1, "ip": 1, ]);

#include <config.h>
#include <macros.h>
#include <classes.h>
#include <database.h>
#include <attributes.h>
#include <assert.h>
#include <access.h>
#include <roles.h>
#include <events.h>
#include <functions.h>
#include <configure.h>

#define CONFIG_FILE "steam.cfg"

#define MODULE_SECURITY mModules["security"]
#define MODULE_FILEPATH mModules["filepath:tree"]
#define MODULE_GROUPS   mModules["groups"]
#define MODULE_USERS    mModules["users"]
#define MODULE_OBJECTS  mModules["objects"]

string get_identifier() { return "Server Object"; }
string describe() { return "Server Object"; }
string _sprintf() { return "Server Object"; }

private static string sContext;



string get_config_dir () {
  string dir = mConfigs["config-dir"];
  if ( !stringp(dir) ) dir = CONFIG_DIR;
  return dir;
}

private static void update_config_file ()
{
  string data;
  mapping config = ([ ]);
  array obsolete_files = ({ });
  // from old pre 1.6 XML config file:
  catch {
    data = Stdio.read_file( get_config_dir() + "/config.txt" );
    if ( stringp(data) ) {
      config |= Config.get_config( data, "config" );
      MESSAGE( "Found obsolete config file: " + get_config_dir() + "/config.txt" );
      obsolete_files += ({ get_config_dir() + "/config.txt" });
    }
  };
  // from 1.6 - 2.0 text config file:
  catch {
    data = Stdio.read_file( get_config_dir() + "/steam.cnf" );
    if ( stringp(data) ) {
      config |= Config.get_config( data );
      MESSAGE( "Found obsolete config file: " + get_config_dir() + "/steam.cnf" );
      obsolete_files += ({ get_config_dir() + "/steam.cnf" });
    }
  };
  catch {
    if ( Stdio.exist( CONFIG_DIR + "/config.tmp" ) ) {
      MESSAGE( "Found obsolete config file template: "
               + get_config_dir() + "/config.tmp" );
      obsolete_files += ({ get_config_dir() + "/config.tmp" });
    }
  };
  catch {
    if ( Stdio.exist( get_config_dir() + "/config.template" ) ) {
      MESSAGE( "Found obsolete config file template: "
               + get_config_dir() + "/config.template" );
      obsolete_files += ({ get_config_dir() + "/config.template" });
    }
  };

  // remove any hbs() wrappers:
  foreach ( indices(config), string key ) {
    if ( !stringp(config[key]) ) continue;
    string v;
    if ( sscanf( config[key], "hbs(%s)", v ) > 0 ) {
      config[key] = Config.string_to_value(v);
    }
  }

  // write to new config file:
  if ( sizeof(config) > 0 ) {
    data = Stdio.read_file( get_config_dir() + "/" + CONFIG_FILE );
    mixed err = catch {
      Stdio.write_file( get_config_dir() + "/" + CONFIG_FILE,
                        Config.make_config_text_from_template( data, config ) );
    };
    if ( err != 0 ) {
      werror( "Could not write config file (updated from old configs): "
              + get_config_dir() + "/" + CONFIG_FILE + "\n" );
      return;
    }
    // rename obsolete files:
    foreach ( obsolete_files, string filename ) {
      mixed err2 = catch {
        if ( mv( filename, filename + ".old" ) )
        MESSAGE( "Renamed obsolete file " + filename + " to " + filename + ".old" );
      };
      if ( err2 != 0 ) {
        werror( "Could not rename obsolete file " + filename + " to " + filename + ".old\n" );
      }
    }
  }
}


/**
 * read configurations from server config file
 */
private static void read_config_from_file ()
{
  string data = Stdio.read_file( get_config_dir() + "/" + CONFIG_FILE );
  if ( !stringp(data) )
    error("Missing config file. Check for "+get_config_dir()+"/"+CONFIG_FILE+"\n"+
	  "You can either repeat installation or call ./setup manually !");
  
  m_delete(mConfigs, "database");
  
  mConfigsFromFile = Config.get_config( data );
  foreach ( indices(mConfigsFromFile), string key ) {
    if ( !stringp(mConfigsFromFile[key]) ) continue;
    string v;
    if ( sscanf( mConfigsFromFile[key], "hbs(%s)", v ) > 0 ) {
      mConfigsFromFile[key] = Config.string_to_value(v);
    }
    mConfigs[key] = mConfigsFromFile[key];
  }
}


/**
 * load configurations from admin group (attribute 'configs')
 */
private static void read_config_from_admin ()
{
  mapping confs = ([ ]);
  object  groups, admin;
  
  groups = mModules["groups"];
  if ( objectp(groups) ) {
    admin = groups->lookup("admin");
    if ( objectp(admin) ) {
      confs = admin->query_attribute("configs");
      if ( !mappingp(confs) )
        confs = ([ ]);
    }
  }

  // configs from config file cannot be overwritten:
  confs |= mConfigsFromFile;
  
  // some default configurations to keep compatibility:
  if ( !confs->web_port_http )
    confs->web_port_http = confs->http_port;
  if ( !confs->web_port_ftp )
    confs->web_port_http = confs->ftp_port;
  if ( !confs->web_port )
    confs->web_port = confs->http_port;
  
  string name, domain;
  string hname = gethostname();
  if ( sscanf(hname, "%s.%s", name, domain) != 2 ) 
    name = hname;
  
  if ( !confs->machine || 
       confs->machine == "<autodetect>" || 
       confs->machine == "(autodetect)" )
    confs->machine = name;
  if ( !confs->domain || 
       confs->domain == "<autodetect>" ||
       confs->domain == "(autodetect)" ) 
    confs->domain = domain;
  
  if ( !confs->web_server || 
       confs->web_server == "<autodetect>" ||
       confs->web_server == "(autodetect)" ) 
    {
      if ( confs->domain )
        confs->web_server = sprintf("%s.%s", confs->machine, confs->domain);
      else
        confs->web_server = sprintf("%s", confs->machine);
    }
  
  if ( !confs->web_mount )
    confs->web_mount = "/";
  
  mConfigs = confs | mConfigs;
  write_config_to_admin();
  //write_config_file();
}


private static void write_config_to_admin()
{
    object groups = mModules["groups"];
    if ( objectp(groups) ) {
	object admin = groups->lookup("Admin");
	if ( objectp(admin) ) {
	  admin->set_attribute("configs", mConfigs);
	}
    }
}

/**
 * Save the modules (additional ones perhaps).
 *  
 */
private static void save_modules()
{
    object groups = mModules["groups"];
    if ( objectp(groups) ) {
	object admin = groups->lookup("Admin");
	if ( objectp(admin) ) {
	    admin->set_attribute("modules", mModules);
	    MESSAGE("Modules saved.");
	}
    }
}

int verify_crypt_md5(string password, string hash)
{
#if constant(Crypto.verify_crypt_md5)
  return Crypto.verify_crypt_md5(password, hash);
#else
  return Crypto.crypt_md5(password, hash) == hash;
#endif
}

string sha_hash(string pw)
{
#if constant(Crypto.sha)
    return Crypto.sha()->update(pw)->digest();
#else
    return Crypto.SHA1()->hash(pw);
#endif
}

static string prepare_sandbox()
{
  string sandbox = mConfigs->sandbox;
  if ( !stringp(sandbox) )
    sandbox = getcwd()+"/tmp";
  if ( sandbox[-1] == '/' )
    sandbox = sandbox[0..strlen(sandbox)-2];
  Stdio.recursive_rm(sandbox);
  Stdio.mkdirhier(sandbox);
  string config_dir = get_config_dir();
  if ( config_dir[-1] == '/' )
    config_dir = config_dir[0..strlen(config_dir)-2];
  MESSAGE("Preparing Sandbox in %s (could take a while)", sandbox);
  Process.create_process( ({ "bin/jail", getcwd()+"/server", sandbox, config_dir }),
				 ([ "env": getenv(),
				    "cwd": getcwd(),
				    "stdout": Stdio.stdout,
				    "stderr": Stdio.stderr,
				 ]))->wait();
  
  return sandbox;
}

static int start_server()
{
    string sandbox = 0;
    float boottime = gauge {

        mGlobalBlockEvents  = ([ ]);
        mGlobalNotifyEvents = ([ ]);
        mClasses            = ([ ]);
        mModules            = ([ ]);
        
        iLastReboot = time();
	MESSAGE("Server startup on " + ctime(time()) );

        update_config_file();
        read_config_from_file();
	sandbox = prepare_sandbox();
    
        nmaster = ((program)"kernel/master.pike")();
        replace_master(nmaster);
        
	// default path
	nmaster->mount("/usr", "/usr");
	nmaster->mount("/sw", "/sw");
	nmaster->mount("/var", "/var");

	nmaster->mount("/", sandbox);
	nmaster->mount("/classes", sandbox+"/classes");
	nmaster->mount("/net", sandbox+"/net");
	nmaster->mount("/modules", sandbox+"/modules");
	nmaster->mount("/libraries", sandbox+"/libraries");
	nmaster->mount("/net/base", sandbox+"/net/base");


        nmaster->add_module_path("/libraries");
        nmaster->add_module_path(sandbox+"/libraries");
	nmaster->add_include_path("/include");
	
        add_constant("_Server", this_object());
	add_constant("query_config", query_config);
        add_constant("vartype", nmaster->get_type);
        add_constant("new", nmaster->new);
        add_constant("this_user", nmaster->this_user);
        add_constant("this_socket", nmaster->this_socket);
	add_constant("geteuid", nmaster->geteuid);
	add_constant("seteuid", nmaster->seteuid);
        add_constant("get_type", nmaster->get_type);
        add_constant("get_functions", nmaster->get_functions);
        add_constant("get_dir", nmaster->get_dir);
        add_constant("rm", nmaster->rm);
        add_constant("file_stat", nmaster->file_stat);
        add_constant("get_local_functions", nmaster->get_local_functions);
        add_constant("_exit", shutdown);
        add_constant("call", nmaster->f_call_out);
        add_constant("call_out_info", nmaster->f_call_out_info);
        add_constant("start_thread", nmaster->start_thread);
        add_constant("call_mod", call_module);
        add_constant("get_module", get_module);
        add_constant("get_factory", get_factory);
        add_constant("steam_error", steam_error);
        add_constant("steam_user_error", steam_user_error);
        add_constant("describe_backtrace", nmaster->describe_backtrace);
        add_constant("set_this_user", nmaster->set_this_user);
	add_constant("run_process", Process.create_process);

	// crypto changes in 7.6
	add_constant("verify_crypt_md5", verify_crypt_md5);
#if constant(Crypto.make_crypt_md5)
	add_constant("make_crypt_md5", Crypto.make_crypt_md5);
#else
	add_constant("make_crypt_md5", Crypto.crypt_md5);
#endif
	add_constant("sha_hash", sha_hash);

	MESSAGE("Loading Persistence...");
    
	oPersistence = ((program)"/Persistence.pike")();
        add_constant("_Persistence", oPersistence);

        MESSAGE("Loading Database..."+mConfigs->database);

#if __MINOR__ > 3
        oDatabase = ((program)"/database.pike")();
#else
        oDatabase = new("/database.pike");
#endif
        add_constant("_Database", oDatabase);
	nmaster->register_constants();  // needed for database/persistence registration
	oDatabase->init();
        MESSAGE("Database is "+ master()->describe_object(oDatabase));

        add_constant("find_object", oPersistence->find_object);
        add_constant("serialize", oDatabase->serialize);
        add_constant("unserialize", oDatabase->unserialize);
        
        nmaster->register_server(this_object());
        nmaster->register_constants();
        
	mixed err = catch {
	  if (oDatabase->check_convert())
	    {
	      oDatabase->set_converting(1);
	      MESSAGE("Old Database format detected ... running conversion");
	    }
	};
	if ( err != 0 ) {
	  error("Boot failed: Unable to access database !\n"+
		"1) Is the database running ?\n"+
		"2) Check if database string is set correctly \n    ("+
		mConfigs->database+")\n"+
		"3) The database might not exist - you need to create it.\n"+
		"4) The user might not have access for this database.\n"+
		"5) The Pike version you are using does not support MySQL\n"+
		"     Try pike --features to check this.\n");
	}
        
        oDatabase->enable_modules();
        MESSAGE("Database module support enabled.");
        
	load_modules();
	read_config_from_admin();
	load_factories();
	load_modules_db();
	
	if ( err = catch(load_objects()) ) {
	  FATAL(err[0]+"\n"+
		"Unable to load basic objects of sTeam.\n"+
		"This could mean something is wrong with the database:\n"+
		"If this is a new installation, you have to drop the database and restart.\n");
	  FATAL("-----------------------------------------\n"+PRINT_BT(err));
	  exit(1);
		
	}
	load_programs();
        install_modules();
        
        MESSAGE("Initializing objects... " + (time()-iLastReboot) + "seconds");
        iLastReboot = time();
        MESSAGE("Setting defaults... " + (time()-iLastReboot) + "seconds");
        
        if (oDatabase->get_converting())
        {
            oDatabase->convert_tables();
            MESSAGE("Database converted --- restart server\n");
            return 1;
        }
        
        open_ports();
        iLastReboot = time();
	thread_create(abs);
        // check if root-room is ok...
        ASSERTINFO(objectp(MODULE_OBJECTS->lookup("rootroom")), 
                   "Root-Room is null!!!");
    };
    
    MESSAGE("Server started on " + ctime(time()) + "Startuptime ("+boottime+")");

    start_services();
    check_root();

    nmaster->run_sandbox(sandbox);


    if ( iTest ) 
	test();
    return -17;
}

static void check_root() {
  MESSAGE("Testing root user ...");
  object root = USER("root");
  int repair = 0;

  foreach ( root->get_groups(), object grp) {
    if ( !objectp(grp) )
      MESSAGE("NULL group detected !");
    else {
      MESSAGE("GROUP %s", grp->get_identifier());
    }
  }
  if ( GROUP("admin")->is_member(root) ) 
    MESSAGE("Root is member of ADMIN !");
  else {
    MESSAGE("Root missing in admin group !");
    repair = 1;
  }
  if ( search(root->get_groups(), GROUP("admin")) == -1 )
      repair = 1;

  if ( GROUP("steam")->is_member(root) ) 
    MESSAGE("Root is member of sTeam !");
  else {
    MESSAGE("Root missing in sTeam group !");
    repair = 1;
  }
  if ( !stringp(root->get_user_name()) ) {
    MESSAGE("NULL username of root !");
    repair = 1;
  }
  if ( repair )
    repair_root_user();
}

static void repair_root_user() {
    GROUP("admin")->remove_member(USER("root"));
    GROUP("admin")->add_member(USER("root"));
    GROUP("steam")->remove_member(USER("root"));
    GROUP("steam")->add_member(USER("root"));
    FATAL("Status for user root is " + USER("root")->status());
    catch {
      USER("root")->set_user_name("root");
      USER("root")->set_user_password("steam");
    };

    USER("root")->set_attribute(USER_LANGUAGE, "english");
    USER("root")->set_attribute(USER_FIRSTNAME, "Root");
    USER("root")->set_attribute(USER_LASTNAME, "User");
    USER("root")->set_attribute(OBJ_DESC, "The root user is the first administrator");
    USER("root")->set_attribute("xsl:content", ([ GROUP("steam") : get_module("filepath:tree")->path_to_object("/stylesheets/user_details.xsl"), ]) );
    USER("root")->set_attribute(OBJ_ICON, get_module("filepath:tree")->path_to_object("/images/user_unknown.jpg") );
    object wr = USER("root")->query_attribute(USER_WORKROOM);
    if (!objectp(wr)) {
	wr = oPersistence->find_object(USER("root")->get_object_id() + 1);
	if (objectp(wr) && (wr->get_object_class() & CLASS_ROOM) ) 
	{
	    werror("\nrestoring USER_WORKROOM of root\n");
	    USER("root")->unlock_attribute(USER_WORKROOM);
	    USER("root")->set_attribute(USER_WORKROOM, wr);
	    USER("root")->lock_attribute(USER_WORKROOM);
	}
    }
    object tb = USER("root")->query_attribute(USER_TRASHBIN);
    if (!objectp(tb)) {
      tb = oPersistence->find_object(USER("root")->get_object_id() + 3);
      if (objectp(tb) && (tb->get_object_class() & CLASS_CONTAINER) ) {
        werror("\nrestoring USER_TRASHBIN of root\n");
        USER("root")->unlock_attribute(USER_TRASHBIN);
        USER("root")->set_attribute(USER_TRASHBIN, tb);
        USER("root")->lock_attribute(USER_TRASHBIN);
      }
    }
}

static void start_services()
{
  object u_service = USER("service");

  Stdio.write_file("service.pass", u_service->get_ticket(time()+1200000));
  return;
}

mixed query_config(mixed config)
{
    if ( config == "database" )
	return 0;
    return mConfigs[config];
}


string plusminus(int num)
{
  return ( num > 0 ? "+"+num: (string)num);
}


string get_database()
{
    //MESSAGE("CALLERPRG="+master()->describe_program(CALLERPROGRAM));
    if ( CALLER == oDatabase || 
	 CALLERPROGRAM==(program)"/kernel/steamsocket.pike" )
	return mConfigs["database"];
    MESSAGE("NO ACCESS to database for "+
            master()->describe_program(CALLERPROGRAM)+
            " !!!!!!!!!!!!!!!!!!!!!!!\n\n");
    return "no access";
}

mapping get_configs()
{
    mapping res = copy_value(mConfigs);
    res["database"] = 0;
    return res;
}

mixed get_config(mixed key)
{
    return  mConfigs[key];
}


string get_version()
{
    return STEAM_VERSION;
}

int get_last_reboot()
{
    return iLastReboot;
}

static private void got_kill(int sig)
{
    MESSAGE("Shutting down !\n");
    oDatabase->wait_for_db_lock();
    _exit(1);
}

static private void got_hangup(int sig)
{
    MESSAGE("sTeam: hangup signal received...");
}

mapping get_errors()
{
    return mErrors;
}

void add_error(int t, mixed err)
{
    mErrors[t] = err;
}

int main(int argc, array(string) argv)
{
    
    mErrors = ([ ]);
    mConfigs = ([ ]);
    int i;
    string path;
    int pid = getpid();
    iTest = 0;
    path = getcwd();

    // check the version
    string ver = version();
    string cver =  Stdio.read_file("version");
    if ( stringp(cver) )
    {
	MESSAGE("sTeam was configured with " + cver);
	if ( cver[-1] == '\n' )
	  cver = cver[..strlen(cver)-2];

	if ( cver != ver ) {
	    FATAL("Version mismatch "+
		  "- used Pike version different from configured !");
	    exit(-1);
	}
    }


    string pidfile = path + "/steam.pid";
    
    mConfigs["logdir"] = LOG_DIR;
    if ( mConfigs["logdir"][-1]!='/' ) mConfigs["logdir"] += "/";
    mConfigs["config-dir"] = CONFIG_DIR;
    if ( mConfigs["config-dir"][-1]!='/' ) mConfigs["config-dir"] += "/";

    for ( i = 1; i < sizeof(argv); i++ ) {
	string cfg, val;
	if ( argv[i] == "--test" )
	    iTest = 1;
	else if ( sscanf(argv[i], "--%s=%s", cfg, val) == 2 ) {
	    int v;
	    
	    if ( cfg == "pid" ) {
		pidfile = val;
	    }
	    else if ( sscanf(val, "%d", v) == 1 )
		mConfigs[cfg] = v;
	    else
		mConfigs[cfg] = val;
	}
	else if ( sscanf(argv[i], "-D%s", cfg) == 1 ) {
	  add_constant(cfg, 1);
	}
    }
    mixed err = catch {
	Stdio.File f=Stdio.File (pidfile,"wac");
	f->write(" " + (string)pid);
	f->close;
    };
    if ( err != 0 )
	FATAL("There was an error writting the pidfile...\n");
    
    err = catch(_stderr = Stdio.File(mConfigs["logdir"]+"/errors.log", "r"));
    if(err)
      MESSAGE("Failed to open %s/errors.log", mConfigs["logdir"]);

    signal(signum("SIGQUIT"), got_kill);
    signal(signum("SIGHUP"), got_hangup);
    signal(signum("SIGINT"), got_hangup);
    return start_server();
}

object get_stderr() 
{
  return _stderr;
}

object get_stdout()
{
  return _stdout;
}

mixed get_module(string module_id)
{
    object module;
    module = mModules[module_id];
    if ( objectp(module) && module->status() >= 0 )
      return module->this();
    return 0;
}

mixed call_module(string module, string func, mixed ... args)
{
    object mod = mModules[module];
    if ( !objectp(mod) ) 
	THROW("Failed to call module "+ module + " - not found.", E_ERROR);
    function f = mod->find_function(func);
    if ( !functionp(f) )
	THROW("Function " + func + " not found inside Module " + module +" !", E_ERROR);
    if ( sizeof(args) == 0 )
	return f();
    return f(@args);
}


mapping get_modules()
{
    return copy_value(mModules);
}

array(object) get_module_objs()
{
    return values(mModules);
}

static object f_open_port(string pname)
{
    program prg = (program)("/net/port/"+pname);
    object port = nmaster->new("/net/port/"+pname);
    mPortPrograms[port->get_port_name()] = prg;
    if ( !port->open_port() && port->port_required() ) 
	return 0;
    nmaster->register_port(port);
    return port;
}

/**
 * Open a single port of the server. 
 *  
 * @param string pname - the name of the port to open.
 * @return the port object or zero.
 */
object open_port(string pname)
{
    if ( _ADMIN->is_member(nmaster->this_user()) ) {
	return f_open_port(pname);
    }
    return 0;
}

/**
 * Open all ports of the server.
 * See the /net/port/ Directory for all available ports.
 *  
 */
static void open_ports()
{
    object         port;
    array(string) ports;
    
    mPortPrograms = ([ ]);

    ports = nmaster->get_dir("/net/port");
    MESSAGE("Opening ports ...");
    // check for steam.cer...
    if ( !Stdio.exist(query_config("config-dir")+"steam.cer") ) {
      MESSAGE("Certificate File Missing - creating new one ...\n");
      string cert = cert.create_cert(  ([
	"country": "Germany",
	"organization": "University of Paderborn",
	"unit": "Open sTeam",
	"locality": "Paderborn",
	"province": "NRW",
	"name": get_server_name(),
      ]) );
      Stdio.write_file( query_config("config-dir") + "steam.cer", cert);
    }
    for ( int i = sizeof(ports) - 1; i >= 0; i-- ) {
	program prg;

	if ( ports[i][0] == '#' || ports[i][0] == '.' || ports[i][-1] == '~' )
	    continue;
	if ( sscanf(ports[i], "%s.pike", ports[i]) != 1 ) continue;
	if ( !objectp(f_open_port(ports[i])) ) {
	    MESSAGE("Opening required port " + ports[i] + " failed.");
	    exit(1);
	}
    }
}

/**
 * Get all string identifiers of available ports (open and close).
 *  
 * @return Array of port identifier strings.
 */
array(string) get_ports()
{
    return indices(mPortPrograms);
}

int close_port(object p)
{
    if ( !objectp(p) )
	error("Cannot close NULL port !");
    if ( _ADMIN->is_member(nmaster->this_user()) ) {
	if ( functionp(p->close_port) )
	    p->close_port();
	if ( objectp(p) )
	    destruct(p);
	return 1;
    }
    return 0;
}

int restart_port(object p)
{
    if ( _ADMIN->is_member(nmaster->this_user()) ) {
	program prg = object_program(p);
	if ( functionp(p->close_port) )
	    p->close_port();
	if ( objectp(p) )
	    destruct(p);
	p = prg();
	if ( p->open_port() ) 
	    MESSAGE("Port restarted ....");
	else {
	    MESSAGE("Restarting port failed.");
	    return 0;
	}
	nmaster->register_port(p);
	return 1;
    }
    return 0;
}

mapping debug_memory(void|mapping debug_old)
{
  mapping dmap = Debug.memory_usage();
  if (!mappingp(debug_old) )
    return dmap;
  foreach(indices(dmap), string idx)
    dmap[idx] = (dmap[idx] - debug_old[idx]);
  return dmap;
}


/**
 * Install all modules of the server.
 *  
 */
void install_modules()
{
    mapping modules = get_modules();
    
    foreach ( indices(modules), string module ) {
      if ( !stringp(module) )
	continue;
	mixed err = catch {
	    modules[module]->runtime_install();
	};
        if(err)
          MESSAGE("failed to install %s\n%s", module, PRINT_BT(err));

    }
    foreach ( indices(modules), string module ) {
	if ( !objectp(modules[module]) || 
	     modules[module]->status() == PSTAT_FAIL_DELETED ) 
	{
	    m_delete(mModules, module);
	}
    }
    save_modules();
}

/**
 * register a module - can only be called by database !
 *  
 * @param object mod - the module to register
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
final void register_module(object mod)
{
    if ( CALLER == oDatabase ) {
	mModules[mod->get_identifier()] = mod;
	save_modules();
    }
}

private static mapping load_module_configuration(string name)
{
  mapping config;

  string content = Stdio.read_file(CONFIG_DIR+"/"+name+".xml");
  if ( stringp(content) )
    config = Module.read_config(content, name);
  else
    config = ([ ]);

  // parse module code for "#define DEPENDENCIES" line:
  array file_dependencies;
  if ( catch {
      array tokens = Parser.Pike.split( Stdio.read_file(nmaster->apply_mount_points("/modules")+"/"+name+".pike") );
      string deps;
      foreach ( tokens, string token ) {
	  if ( sscanf( token, "#define%*[ \t]DEPENDENCIES%*[ \t]%s\n", deps ) > 1 ) {
	      file_dependencies = deps / " ";
	      break;
	  }
      }
  } != 0 ) {
      werror( "Could not parse module : %s\n", name );
  }
  if ( arrayp(file_dependencies) ) {
      if ( !arrayp(config->depends) ) config->depends = file_dependencies;
      else foreach ( file_dependencies, string dep )
	       if ( search( config->depends, dep ) < 0 )
		   config->depends += ({ dep });
  }

  config["score"] = 1000;
  return config;
}

/**
 * Load a module
 *  
 * @param string mpath - the filename of the module
 * @return 
 */
private static object load_module(string mpath)
{
    if ( sscanf(mpath, "%s.pike", mpath) != 1 )
	return 0;
    write("LOADING MODULE:" + mpath + " ... ");
    /* dont load the module that keeps the list of all modules
     * Because it is loaded by database 
     */
    if  ( mpath == "modules" ) 
	return 0;
    
    object module = 0;
    int database_id = oDatabase->get_variable("#"+mpath);
    if ( database_id != 0 )
       module = oDatabase->find_object(database_id);

    // we found an already existing one
    if ( objectp(module) && module->status() >= 0 ) 
    {
	if ( objectp(module->get_object()) )
	    mModules[module->get_identifier()] = module;
	else
	    FATAL("Failed to create instance of "+mpath);
    }
    else
    {
	MESSAGE("Creating new instance of "+mpath);
	// first try to read config file for that module (if any)
	
	mixed err = catch {
	    module = nmaster->new("/modules/"+mpath+".pike");
	};
	if ( err != 0 ) {
	    FATAL("Error while creating new instance of " + mpath + "\n" + 
		  PRINT_BT(err));
	}
	err = catch {
	  if (objectp(module)) {
	    if (!functionp(module->this) ) /* check existance of function */
	    {
	      FATAL("unable to register module \""+mpath+
		      "\" it has to inherit /kernel/module or at least "+
		      "/classes/Object");
	      module = 0;
	    }
	    else
	    {
		oDatabase->set_variable("#"+mpath,
					module->get_object_id());
		module = module->this();
		mModules[module->get_identifier()] = module;
		module->set_attribute(OBJ_DESC, "");
		module->loaded();
		module->created();
	    }
	  }
	};
	if ( err != 0 )
	    FATAL("Error registering module \""+mpath+"\":\n"+PRINT_BT(err));
    }

    if (objectp(module)) 
      MESSAGE("alias " + module->get_identifier() +
	      " OID("+module->get_object_id()+")");

    return module;
}

bool is_module(object mod)
{
  if ( !functionp(mod->get_identifier) )
      return false;
  object module = mModules[mod->get_identifier()];
  if ( objectp(module) )
    return module->this() == mod->this();
  return 0;
}

private static int order_modules(string mod1, string mod2, mapping configs)
{
  if ( sscanf(mod1, "%s.pike", mod1) == 0)
    return 0;
  if ( sscanf(mod2, "%s.pike", mod2) == 0 )
    return 0;

  mixed dep1, dep2;

  return configs[mod1]->score < configs[mod2]->score;
}

void update_depend(string depend, mapping conf)
{
  MESSAGE("Updaing: %s", depend);
  if ( conf[depend]->mark == 1 )
    steam_error("Loop in module Dependencies detected !");
  conf[depend]->mark = 1;

  conf[depend]->score++;
  if ( arrayp(conf[depend]->depends) ) {
    foreach(conf[depend]->depends, string depdep) {
      update_depend(depdep, conf);
    }
  }
  conf[depend]->mark = 0;
}

static int filter_system(string fname)
{
  if ( search(fname, "~") >= 0 ) return 0;
  if ( fname[0] == '#' || fname[0] == '.' ) return 0;
  return 1;
}

/**
 * Load all modules.
 *  
 */
void load_modules()
{
    int    i, database_id;
    
    object         module;
    array(string) modules;
    
    mModules = ([]);
    modules = nmaster->get_dir("/modules");
    
    array(string) priority_load = ({ 
	"log.pike", "security.pike", "cache.pike", "groups.pike", "users.pike",
	"objects.pike", "filepath.pike", "message.pike", "mailbox.pike",
	"xml_converter.pike" });
    
    modules -= priority_load;
    modules = priority_load + modules;
    modules = filter(modules, filter_system);

    mapping configurations = ([ ]);
    for ( i = 0; i < sizeof(modules); i++ ) {
        string modname;
        if ( sscanf(modules[i], "%s.pike", modname) == 0 )
            continue;

	mapping conf = load_module_configuration(modname);
	if ( arrayp(conf->depends) ) {
	  foreach(conf->depends, string depend) {
	    if ( !mappingp(configurations[depend]) )
	      configurations[depend] = ([ "score": 1000, "depends": 0, ]);
	    update_depend(depend, configurations);
	  }
	}

	if ( !mappingp(configurations[modname]) )
	  configurations[modname] = conf;
	else
	  configurations[modname]->depends = conf->depends;
    }
    // now we need to sort our modules according to the graph in configurations
    // sortierung durch vergleich 2er element ist ok
    modules = Array.sort_array(modules, order_modules, configurations);
    
    // finally load the modules
    for ( i = 0; i < sizeof(modules); i++ ) {
	if ( search(modules[i], "~") >= 0 ) continue;
	if ( modules[i][0] == '#' || modules[i][0] == '.' ) continue;
	// only load pike programms !
	load_module(modules[i]);
    }

    MESSAGE("Loading modules finished...");
}

void load_programs()
{
    string cl;
    array(string) classfiles = nmaster->get_dir("/classes");
    foreach(classfiles, cl) {
	if ( cl[0] == '.' || cl[0] == '#' || search(cl, "~") >= 0 || 
	     search(cl, "CVS") >= 0 ) continue;

	MESSAGE("Preparing class: " + cl);
	program prg = (program) ("/classes/"+cl);
    }
    classfiles = nmaster->get_dir("/kernel");
    foreach(classfiles, cl) {
	if ( cl[0] == '.' || cl[0] == '#' || search(cl, "~") >= 0 ||
	     search(cl, "CVS") >= 0 ) 
	     continue;

	MESSAGE("Preparing class: " + cl);
	program prg = (program) ("/kernel/"+cl);
    }
}

/**
 * Load all modules from the database ( stored in the admin group )
 *  
 */
static void load_modules_db()
{
    MESSAGE("Loading registered modules from database...");
    mixed err = catch {
	object groups = mModules["groups"];
	if ( objectp(groups) ) {
	    object admin = groups->lookup("Admin");
	    if ( !objectp(admin) )
		return;
	    mapping modules = admin->query_attribute("modules");
	    if ( !mappingp(modules) ) {
		MESSAGE("No additional modules registered yet!");
		return;
	    }
	    // sync modules saved in admin group with already loaded
	    foreach ( indices(modules), string m ) {
		if ( !mModules[m] )
		    mModules[m] = modules[m];
	    }
	}
	MESSAGE("Loading modules from database finished.");
    };
    if ( err != 0 ) 
	FATAL("Loading Modules from Database failed.\n"+PRINT_BT(err));
}

/**
 *
 *  
 * @param 
 * @return 
 * @author Thomas Bopp (astra@upb.de) 
 * @see 
 */
void load_factories()
{
    int      i, database_id;
    string     factory_name;
    object          factory;
    object            proxy;
    mixed               err;
    array(string) factories;
    array(object) loaded = ({});
    
    factories = nmaster->get_dir("/factories");
    factories -= ({ "DateFactory.pike" });
    factories -= ({ "CalendarFactory.pike" });
    factories -= ({ "AnnotationFactory.pike" });
    factories = ({ "DateFactory.pike", "CalendarFactory.pike" }) + factories;
    for ( i = sizeof(factories) - 1; i >= 0; i-- ) {
	if ( sscanf(factories[i], "%s.pike", factory_name) == 0 )
	    continue;

	if ( search(factory_name, "~") >= 0 || search(factory_name, "~")>=0 ||
	     search(factory_name, ".") == 0 || search(factory_name,"#")>=0 )
	    continue;
        MESSAGE("LOADING FACTORY:%s ...", factory_name);
	proxy = MODULE_OBJECTS->lookup(factory_name);
	if ( !objectp(proxy) ) {
	    MESSAGE("Creating new instance...");
	    err = catch {
		factory = nmaster->new("/factories/"+factory_name+".pike", 
				       factory_name);
	    };
	    if ( err != 0 ) {
		MESSAGE("Error while loading factory " + factory_name + "\n"+
			PRINT_BT(err));
		continue;
	    }
	    
	    proxy = factory->this();
            proxy->created();
            if (proxy->status()>=PSTAT_SAVE_OK)
	    {
	      MESSAGE("New Factory registered !");
	      MODULE_OBJECTS->register(factory_name, proxy);
            }
	}
	else {
            int iProxyStatus;
	    err = catch {
                int iProxyStatus = proxy->force_load();
            };
            if (err!=0) {
                MESSAGE("Error while loading factory %s status(%d)\n%s\n",
                        factory_name, iProxyStatus, master()->describe_backtrace(err));
            }
	}

        if (proxy->status() >= PSTAT_SAVE_OK)
        {
            mClasses[proxy->get_class_id()] = proxy;
            loaded += ({ proxy });
            err = catch {
                proxy->unlock_attribute(OBJ_NAME);
                proxy->set_attribute(OBJ_NAME, proxy->get_identifier());
                proxy->lock_attribute(OBJ_NAME);
            };
            if ( err != 0 ) {
                FATAL("There was an error loading a factory...\n"+
                      PRINT_BT(err));
            }
        }
        else
            MESSAGE("factory is %s with status %d", 
		    factory_name, proxy->status());
    }
    MESSAGE("Loading factories finished.\n");
}

/**
 *
 *  
 * @param 
 * @return 
 * @author Thomas Bopp (astra@upb.de) 
 * @see 
 */
void load_objects()
{
    object factory, root, room, admin, world, steam, guest, postman;
    int               i;
    string factory_name;
    mapping vars = ([ ]);
    
    
    MESSAGE("Loading Groups: sTeam");
    steam = MODULE_GROUPS->lookup("sTeam");
    if ( !objectp(steam) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "sTeam";
	steam = factory->execute(vars);
	ASSERTINFO(objectp(steam), "Failed to create sTeam group!");
	steam->set_attribute(OBJ_DESC, "The group of all sTeam users.");
    }
    add_constant("_GroupAll", steam);
    MESSAGE(", Everyone");
    world = MODULE_GROUPS->lookup("Everyone");
    if ( !objectp(world) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "Everyone";
	world = factory->execute(vars);
	ASSERTINFO(objectp(world), "Failed to create world user group!");
	world->set_attribute(
	    OBJ_DESC, "This is the virtual group of all internet users.");
    }
    MESSAGE(", Help");
    object hilfe = MODULE_GROUPS->lookup("help");
    if ( !objectp(hilfe) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "help";
	hilfe = factory->execute(vars);
	ASSERTINFO(objectp(hilfe), "Failed to create hilfe group!");
	hilfe->set_attribute(
	    OBJ_DESC, "This is the help group of steam.");
    }
    mixed err = catch {
        hilfe->sanction_object(steam, SANCTION_READ|SANCTION_ANNOTATE);
    };
    if(err)
      MESSAGE("Failed sanction on hilfe group\n"+PRINT_BT(err));

    MESSAGE(" - done.\n");
    bool rootnew = false;

    MESSAGE("Loading users: root");
    root = MODULE_USERS->lookup("root");
    if ( !objectp(root) ) {
	rootnew = true;
	factory = get_factory(CLASS_USER);
	vars["name"] = "root";
	vars["pw"] = "steam";
	vars["email"] = "";
	vars["fullname"] = "Root User";
	root = factory->execute(vars);
	root->activate_user(factory->get_activation());
	ASSERTINFO(objectp(root), "Failed to create root user !");
	root->set_attribute(
	    OBJ_DESC, "The root user is the first administrator of sTeam.");
    }
    if ( mConfigs->password ) {
      root->set_user_password(mConfigs->password);
      m_delete(mConfigs, "password");
      write_config_to_admin();
    }
    MESSAGE(", guest");
    guest = MODULE_USERS->lookup("guest");
    if ( !objectp(guest) ) {
	factory = get_factory(CLASS_USER);
	vars["name"] = "guest";
	vars["pw"] = "guest";
	vars["email"] = "none";
	vars["fullname"] = "Guest";
	guest = factory->execute(vars);
	
	ASSERTINFO(objectp(guest), "Failed to create guest user !");
	guest->activate_user(factory->get_activation());
	guest->sanction_object(world, SANCTION_MOVE); // move around guest
	object guest_wr = guest->query_attribute(USER_WORKROOM);
	guest_wr->sanction_object(guest, SANCTION_READ|SANCTION_INSERT);
	guest->set_attribute(
	    OBJ_DESC, "Guest is the guest user.");
    }
    get_factory(CLASS_USER)->reset_guest();
    ASSERTINFO(guest->get_user_name() == "guest", "False name of guest !");
    GROUP("everyone")->add_member(guest);

    object service = MODULE_USERS->lookup("service");
    if ( !objectp(service) ) {
	factory = get_factory(CLASS_USER);
	vars["name"] = "service";
	vars["pw"] = "";
	vars["email"] = "none";
	vars["fullname"] = "Service";
	service = factory->execute(vars);
	
	ASSERTINFO(objectp(service), "Failed to create service user !");
	service->activate_user(factory->get_activation());
	service->set_user_password("0", 1);
	service->sanction_object(world, SANCTION_MOVE); // move around service
	object service_wr = service->query_attribute(USER_WORKROOM);
	service_wr->sanction_object(service, SANCTION_READ|SANCTION_INSERT);
	service->set_attribute(
	    OBJ_DESC, "Service is the service user.");
    }

    MESSAGE(", postman");
    postman = MODULE_USERS->lookup("postman");
    if ( !objectp(postman) ) 
    {
        factory = get_factory(CLASS_USER);
        vars["name"] = "postman";
        vars["pw"] = Crypto.randomness.pike_random()->read(10); //disable passwd
        vars["email"] = "";
        vars["fullname"] = "Postman";
        postman = factory->execute(vars);

        ASSERTINFO(objectp(postman), "Failed to create postman user !");
        postman->activate_user(factory->get_activation());
        postman->sanction_object(world, SANCTION_MOVE); // move postman around
        object postman_wr = postman->query_attribute(USER_WORKROOM);
        postman_wr->sanction_object(postman, SANCTION_READ|SANCTION_INSERT);
        postman->set_attribute(OBJ_DESC, 
               "The postman delivers emails sent to sTeam from the outside.");
    }
    ASSERTINFO(postman->get_user_name() == "postman", "False name of postman !");
    err = catch {
	 room = MODULE_OBJECTS->lookup("rootroom");
	 if ( !objectp(room) ) {
	     factory = get_factory(CLASS_ROOM);
	     vars["name"] = "root-room";
	     room = factory->execute(vars);
	     ASSERTINFO(objectp(room), "Failed to create root room !");
	     room->sanction_object(steam, SANCTION_READ);
	     ASSERTINFO(MODULE_OBJECTS->register("rootroom", room),
			"Failed to register room !");
	     root->move(room);
	     room->set_attribute(
		 OBJ_DESC, "The root room contains system documents.");
	 }
    };
    if(err)
      MESSAGE("Failed to create root room\n%s",PRINT_BT(err));

    guest->move(room);
    postman->move(room);
    root->move(room);
    if ( rootnew ) {
	// only create the exit in roots workroom if the user has
	// been just created
        MESSAGE("New roots workroom");
	object workroom = root->query_attribute(USER_WORKROOM);
	if ( objectp(workroom) ) {
	    object exittoroot;
	    factory = get_factory(CLASS_EXIT);
	    exittoroot = factory->execute((["name":"root-room",
					   "exit_to":room,]));
	    exittoroot->move(workroom);
	}
        MESSAGE(" - created exits and root image\n");
    }

    admin = MODULE_GROUPS->lookup("Admin");
    if ( !objectp(admin) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "Admin";
	admin = factory->execute(vars);
	ASSERTINFO(objectp(admin), "Failed to create Admin user group!");
	admin->set_permission(ROLE_ALL_ROLES);
	admin->add_member(root);
	admin->sanction_object(root, SANCTION_ALL);
	admin->set_attribute(
	    OBJ_DESC, "The admin group is the group of administrators.");
    }
    if ( admin->get_permission() != ROLE_ALL_ROLES )
	admin->set_permission(ROLE_ALL_ROLES);
    admin->add_member(root);
    admin->remove_member(service);

    ASSERTINFO(admin->get_permission() == ROLE_ALL_ROLES, 
	       "Wrong permissions for admin group !");

    object groups = MODULE_GROUPS->lookup("PrivGroups");
    if ( !objectp(groups) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "PrivGroups";
	groups = factory->execute(vars);
	ASSERTINFO(objectp(groups), "Failed to create PrivGroups user group!");
	groups->set_attribute(OBJ_DESC, 
			      "The group to create private groups in.");
	groups->sanction_object(_STEAMUSER, SANCTION_INSERT|SANCTION_READ);
	// everyone can add users and groups to that group!
    }

    object wikigroups = MODULE_GROUPS->lookup("WikiGroups");
    if ( !objectp(wikigroups) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "WikiGroups";
	wikigroups = factory->execute(vars);
	ASSERTINFO(objectp(wikigroups),"Failed to create WikiGroups user group!");
	wikigroups->set_attribute(OBJ_DESC, 
			      "The group to create wiki groups in.");
	wikigroups->sanction_object(_STEAMUSER, SANCTION_INSERT|SANCTION_READ);
	// everyone can add users and groups to that group!
    }
    
    // as soon as the coder group has members, the security is enabled!
    object coders = MODULE_GROUPS->lookup("coder");
    if ( !objectp(coders) ) {
	factory = get_factory(CLASS_GROUP);
	vars["name"] = "coder";
	coders = factory->execute(vars);
	ASSERTINFO(objectp(coders), "Failed to create coder user group!");
	coders->set_attribute(OBJ_DESC, 
			      "The group of people allowed to write scripts.");
	//coders->add_member(root);
    }
    mapping roles = ([ ]);

    roles["admin"] = Roles.Role("Administrator", ROLE_ALL_ROLES, 0);
    roles["steam"] = Roles.Role("sTeam User", ROLE_READ_ALL, _STEAMUSER);
    admin->add_role(roles->admin);
    steam->add_role(roles->steam);

    object cont = null;
    err = catch { cont = MODULE_FILEPATH->path_to_object("/factories"); };
    if (err) 
      MESSAGE(PRINT_BT(err));
    
    if ( !objectp(cont) )
    {
	factory = get_factory(CLASS_CONTAINER);
	vars["name"] = "factories";
	cont = factory->execute(vars);
	ASSERTINFO(objectp(cont),"Failed to create the factories container!");
	cont->set_attribute(OBJ_DESC, "This container is for the factories.");
    }
    ASSERTINFO(objectp(cont), "/factories/ not found");
    cont->move(room);
    
    // backtrace container
    
    oBacktraces = MODULE_OBJECTS->lookup("backtraces");
    if ( !objectp(oBacktraces) ) {
      oBacktraces = get_factory(CLASS_CONTAINER)->execute( 
                                 (["name":"backtraces", ]) );
      ASSERTINFO(MODULE_OBJECTS->register("backtraces", oBacktraces),
		 "Failed to register backtraces container !");

    }
    err = catch(oBacktraces->set_attribute(OBJ_URL, "/backtraces"));
    if(err)
      MESSAGE("failed to set attribute ob backtraces\n"+PRINT_BT(err));
    oBacktraces->move(room);

    
    factory = get_factory(CLASS_USER);
    factory->sanction_object(world, SANCTION_READ|SANCTION_EXECUTE);
    
    for ( i = 31; i >= 0; i-- ) {
	factory = get_factory((1<<i));
	if ( objectp(factory) ) {
	    factory->sanction_object(admin, SANCTION_EXECUTE);
	    // give execute permissions to all factories for all steam users
	    factory->sanction_object(_STEAMUSER, SANCTION_EXECUTE);
	    if ( objectp(cont) )
	      factory->move(cont);
	 }
    }

    object steamroom = steam->query_attribute(GROUP_WORKROOM);
    MESSAGE("Placing home module");
    object home = get_module("home");
    if ( objectp(home) ) {
	home->set_attribute(OBJ_NAME, "home");
	home->move(room);
	catch(home->set_attribute(OBJ_URL, "/home"));
    }
    MESSAGE("Placing WIKI module");
    object wiki = get_module("wiki");
    if ( objectp(wiki) ) {
        catch(wiki->set_attribute(OBJ_NAME, "wiki"));
        catch(wiki->set_attribute(OBJ_URL, "/wiki"));
	wiki->move(room);
    }
    MESSAGE("Placing Calendar module");
    object calendar = get_module("calendar");
    object cal = get_module("filepath:tree")->path_to_object("/calendar");
    if ( objectp(cal) )
      cal->move(steamroom);

    if ( objectp(calendar) ) {
      calendar->set_attribute(OBJ_NAME, "calendar");
      catch(calendar->set_attribute(OBJ_URL, "/calendar"));
      calendar->move(room);
    }
    
    MESSAGE(" - done.\n");
}

object insert_backtrace(string btname, string btcontent)
{
  object bt = get_factory(CLASS_DOCUMENT)->execute( ([ "name": btname, ]) );
  bt->set_content(btcontent);
  bt->set_attribute(DOC_MIME_TYPE, "text/html");
  bt->move(oBacktraces);
  bt->sanction_object(GROUP("Everyone"), SANCTION_READ);
  object temp = get_module("temp_objects");
  int bt_time = get_config("keep_backtraces");
  if ( bt_time <= 0 )
    bt_time = 60*60*24*7; // one week!
  if ( objectp(temp) )
    temp->add_temp_object(bt, time() + bt_time); // a week !
  return bt;
}

/**
 *
 *  
 * @param 
 * @return 
 * @author Thomas Bopp (astra@upb.de) 
 * @see 
 */
static void f_run_global_event(int event, int phase, object obj, mixed args)
{
    int i, sz;
    mapping m;
    
    object logs = get_module("log");
    if ( objectp(logs) ) {
	mixed err;
	if ( phase == PHASE_NOTIFY ) {
	    err = catch(logs->log("events", LOG_LEVEL_INFO, Events.event_to_description(
				      event, ({ obj }) + args)+"\n") );
  	    if ( err ) {
		FATAL("While logging event: %O\n\n%s\n%O", args,err[0],err[1]);
	    }
	}
	else {
	    err = catch(logs->log("events", LOG_LEVEL_DEBUG, "TRY " +
				  Events.event_to_description(
				      event, ({ obj }) + args)+"\n"));
  	    if ( err ) {
		FATAL("While logging event: %s\n%O", err[0], err[1]);
	    }
	}
    }

    if ( phase == PHASE_NOTIFY ) 
	m = mGlobalNotifyEvents;
    else 
	m = mGlobalBlockEvents;
    
    if ( !arrayp(m[event]) ) 
	return;
    foreach(m[event], array cb_data) {
	if ( !arrayp(cb_data) ) continue;
	string fname = cb_data[0];
	object o = cb_data[1];
	if (!objectp(o)) continue;

        function f = cb_data[2];
	if ( !functionp(f) ) {
	    if (o["find_function"])
		f = o->find_function(fname);
	    else
		f = o[fname];
	}
	
	if ( functionp(f) && objectp(function_object(f)) )
	  f(event, obj, @args);
    }
}

/**
  *
  *  
  * @param 
  * @return 
  * @author Thomas Bopp (astra@upb.de) 
  * @see 
  */
void run_global_event(int event, int phase, object obj, mixed args)
{
    if ( CALLER->this() != obj ) 
	return;
    f_run_global_event(event, phase, obj, args);
}


/**
  *
  *  
  * @param 
  * @return 
  * @author Thomas Bopp (astra@upb.de) 
  * @see 
  */
void 
add_global_event(int event, function callback, int phase)
{
    // FIXME! This should maybe be secured
    object   obj;
    string fname;

    fname = function_name(callback);
    obj   = function_object(callback);
    if ( !objectp(obj) ) 
	THROW("Fatal Error on add_global_event(), no object !", E_ERROR);
    if ( !functionp(obj->this) )
	THROW("Fatal Error on add_global_event(), invalid object !", E_ACCESS);
    obj   = obj->this();
    if ( !objectp(obj) ) 
	THROW("Fatal Error on add_global_event(), no proxy !", E_ERROR);

    if ( phase == PHASE_NOTIFY ) {
	if ( !arrayp(mGlobalNotifyEvents[event]) ) 
	    mGlobalNotifyEvents[event] = ({ });
	mGlobalNotifyEvents[event] += ({ ({ fname, obj, callback }) });
    }
    else {
	if ( !arrayp(mGlobalBlockEvents[event]) ) 
	    mGlobalBlockEvents[event] = ({ });
	mGlobalBlockEvents[event] += ({ ({ fname, obj, callback }) });
    }
}

void
remove_global_events()
{
    array(int)         events;
    int                 event;
    function               cb;
    array(function) notifiers;

    events = indices(mGlobalNotifyEvents);
    foreach ( events, event ) {
	notifiers = ({ });
	foreach ( mGlobalNotifyEvents[event], array cb_data ) {
	    if ( cb_data[1] != CALLER->this() )
		notifiers += ({ cb_data });
	}
	mGlobalNotifyEvents[event] = notifiers;
    }
    events = indices(mGlobalBlockEvents);
    foreach ( events, event ) {
	notifiers = ({ });
	foreach ( mGlobalBlockEvents[event], array cb_data ) {
	    if ( cb_data[1] != CALLER )
		notifiers += ({ cb_data });
	}
	mGlobalBlockEvents[event] = notifiers;
    }
}

/**
  *
  *  
  * @param 
  * @return 
  * @author Thomas Bopp (astra@upb.de) 
  * @see 
  */
void
shutdown(void|int reboot)
{
    write_config_to_admin();
    object user = nmaster->this_user();
    if ( !_ADMIN->is_member(user) )
	THROW("Illegal try to shutdown server by "+
	      (objectp(user)?user->get_identifier():"none")+"!",E_ACCESS);
    MESSAGE("Shutting down !\n");
    oDatabase->wait_for_db_lock();
    if ( !reboot )
	_exit(1);
    _exit(0);
}

/**
 * Check whether a configuration value is changeable from within the server.
 * Configs that are set through the config file cannot be changed from within
 * the server, only through the file (and, thus, a server restart).
 *
 * @param type the config to check
 * @return 1 if the value can be changed, 0 if it cannot be changed
 * @see set_config, delete_config
 */
int is_config_changeable ( mixed type )
{
  if ( zero_type(mConfigsFromFile[type]) )
    return 1;
  else
    return 0;
}

/**
  * Set a configuration value. Configs that are set through the config
  * file cannot be changed from within the server.
  *  
  * @param type the config to be changed
  * @param val the new value
  * @return 1 if the value was changed, 0 if it could not be changed
  * @author Thomas Bopp (astra@upb.de)
  * @see query_config, get_config, delete_config
  */
int set_config(mixed type, mixed val)
{
    if ( ! is_config_changeable( type ) ) return 0;
    mixed err = catch {
        MODULE_SECURITY->check_access(0, this_object(), 0, 
                                      ROLE_WRITE_ALL, false);
    };
    if ( err != 0 ) {
        MESSAGE("Failed to set configuration !");
        return 0;
    }
    mConfigs[type] = val;
    write_config_to_admin();
    return 1;
}


/**
  * Remove a configuration value. Configs that are set through the config
  * file cannot be changed from within the server.
  *  
  * @param type the config to be removed
  * @return 1 if the value was removed, 0 if it could not be changed
  * @author Thomas Bopp (astra@upb.de)
  * @see query_config, get_config, set_config
  */
int delete_config(mixed type)
{
    if ( ! is_config_changeable( type ) ) return 0;
    if ( ! zero_type(mConfigsFromFile[type]) )
        return 0;
    mixed err = catch {
	MODULE_SECURITY->check_access(0, this_object(), 0, 
				      ROLE_WRITE_ALL, false);
    };
    if ( err != 0 ) {
	MESSAGE("Failed to set configuration !");
	return 0;
    }
    m_delete(mConfigs, type);
    write_config_to_admin();
    return 1;
}

/**
  *
  *  
  * @param 
  * @return 
  * @author Thomas Bopp (astra@upb.de) 
  * @see 
  */
void register_class(int class_id, object factory)
{
    ASSERTINFO(MODULE_SECURITY->check_access(factory, CALLER, 0, 
					     ROLE_REGISTER_CLASSES, false), 
	       "CALLER must be able to register classes !");
    mClasses[class_id] = factory;
}

/**
  *
  *  
  * @param 
  * @return 
  * @author Thomas Bopp (astra@upb.de) 
  * @see 
  */
final object get_factory(int|object|string class_id)
{
    int i, bits;

    if ( stringp(class_id) ) {
	foreach(values(mClasses), object factory) {
	    if ( factory->get_class_name() == class_id )
		return factory;
	}
	return 0;
    }
    if ( objectp(class_id) ) {
	string class_name = 
	    master()->describe_program(object_program(class_id));
	//	MESSAGE("getting factory for "+ class_name);
	if ( sscanf(class_name, "/DB:#%d.%*s", class_id) >= 1 )
	    return oDatabase->find_object(class_id)->get_object();
	class_id = class_id->get_object_class();
    }

    for ( i = 31; i >= 0; i-- ) {
	bits = (1<<i);
	if ( bits <= class_id && bits & class_id ) {
	    if ( objectp(mClasses[bits]) ) {
		return mClasses[bits]->get_object();
	    }
	}    
    }
    return null;
}

/**
  * Check if a given object is the factory of the object class of CALLER.
  *  
  * @param obj - the object to check
  * @return true or false
  * @author Thomas Bopp (astra@upb.de) 
  * @see get_factory
  * @see is_a_factory
  */
bool is_factory(object obj)
{
    object factory;

    factory = get_factory(CALLER->get_object_class());
    if ( objectp(factory) && factory == obj )
	return true;
    return false;
}

/**
  * Check if a given object is a factory. Factories are trusted objects.
  *  
  * @param obj - the object that might be a factory
  * @return true or false
  * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
  * @see is_factory
  * @see get_factory
  */
bool is_a_factory(object obj)
{
    if ( !functionp(obj->this) )
	return false;
    return (search(values(mClasses), obj->this()) >= 0);

}

/**
  * get all classes and their factories.
  *  
  * @return the mapping of all classes
  * @author Thomas Bopp (astra@upb.de) 
  * @see is_factory
  * @see get_factory
  */
final mapping get_classes()
{
    return copy_value(mClasses);
}

array(object) get_factories()
{
    return copy_value(values(mClasses));
}

object get_caller(object obj, mixed bt)
{
    int sz = sizeof(bt);
    object       caller;

    sz -= 3;
    for ( ; sz >= 0; sz-- ) {
	if ( functionp(bt[sz][2]) ) {
	    function f = bt[sz][2];
	    caller = function_object(f);
	    if ( caller != obj ) {
		return caller;
	    }
	}
    }
    return 0;
	
}


void mail_password(object user)
{
    string pw = user->get_ticket(time() + 3600); // one hour
    int https = query_config(CFG_WEBPORT_HTTP);
    get_module("smtp")->send_mail(
	      user->query_attribute(USER_EMAIL),
	      "You Account Data for sTeam",
	      "Use the following link to login to "+
	      "the server\r\n and change your password "+
	      "within an hour:\r\n"+
	      "https://"+user->get_user_name()+":"+
	      pw+"@"+
	      query_config(CFG_WEBSERVER)+
	      (https!=443?":"+query_config(CFG_WEBPORT_HTTP):"")+
	      query_config(CFG_WEBMOUNT)+
	      "register/forgot_change.html");
}

mixed steam_error(string msg, mixed ... args)
{
    if ( sizeof(args) > 0 )
	msg = sprintf(msg, @args);

    throw(errors.SteamError(msg, backtrace()[1..]));
}

mixed steam_user_error(string msg, mixed ... args)
{
    if ( sizeof(args) > 0 )
	msg = sprintf(msg, @args);
    throw(errors.SteamUserError(msg, backtrace()[1..]));
}

string get_server_name()
{
    string domain=query_config("domain");
    if(stringp(domain) && sizeof(domain))
      return query_config("machine") + "." + domain;
    else
      return query_config("machine");
      
}

string ssl_redirect(string url) 
{
    string sname = get_server_name();
    int https = query_config("https_port");
    if ( https == 443 ) 
	return "https://" + sname + url;
    return "https://" + sname + ":" + https + url;
}

string get_server_ip() 
{
    if ( query_config("ip") )
	return query_config("ip");
    array result = System.gethostbyname(get_server_name());
    if ( arrayp(result) ) {
	if ( sizeof(result) >= 2 ) {
	    if ( arrayp(result[1]) && sizeof(result[1]) > 0 )
		return result[1][0];
	}
    }
    return "127.0.0.1";
}

string get_server_url_presentation()
{
    int port = query_config(CFG_WEBPORT_PRESENTATION);
    
    return "http://"+get_server_name()+(port==80?"":":"+port)+"/";
}

string get_server_url_administration()
{
    int port = query_config(CFG_WEBPORT_ADMINISTRATION);
    
    return "https://"+get_server_name()+(port==443?"":":"+port)+"/";
}

static object test_save;

int check_shutdown_condition()
{
  // check if shutdown is possible right now - check for active connections.
  
  return 1;
}

static void abs()
{
  while ( 1 ) {
    mixed err;
    if ( err=catch(oDatabase->check_save_demon()) ) {
      FATAL("FATAL Error, rebooting !\n"+PRINT_BT(err));
      MESSAGE("ABS: Shutting down on fatal error !");
      oDatabase->wait_for_db_lock();
      _exit(2);
    }

    int reboot_hour = (int)get_config("reboot_hour");
    int reboot_day = (int)get_config("reboot_day");
    int reboot_memory = (int)get_config("max_memory");
    int reboot_connections = (int)get_config("max_connections");

    // find out the hour and only reboot hourly
    if ( time() - iLastReboot > 60*60 ) {
	mapping t = localtime(time());
	if ( !reboot_day || reboot_day == t->mday ) {
	    if ( reboot_hour && reboot_hour == t->hour ) {
		MESSAGE("ABS: Shutting down on reboot time !");
		oDatabase->wait_for_db_lock();
		_exit(0);
	    }
	}
    }
    int num_connections = sizeof(nmaster->get_users());
    mapping memory = debug_memory();
    int mem = 0;
    foreach(indices(memory), string idx) 
      if ( search(idx, "_bytes") > 0 )
	mem += memory[idx];
    
#if 0
    GROUP("admin")->mail("Server " + get_server_name() + " Status: <br>"+
			 "Memory: "+ mem/(1024*1024)+ "M<br>"+
			 "Connections: " + num_connections, "Status of " + 
			 get_server_name());
#endif
    if ( reboot_connections > 0 && num_connections > reboot_connections &&
	 check_shutdown_condition() ) 
    {
      MESSAGE("ABS: Shutting down due to number of connections !");
      oDatabase->wait_for_db_lock();
      _exit(0);
    }
    if ( reboot_memory > 0 && reboot_memory < mem && check_shutdown_condition()) 
    {
      MESSAGE("ABS: Shutting down due to memory usage !");
      oDatabase->wait_for_db_lock();
      _exit(0);
    }
    sleep(300); // 10 minutes
  }
}

void test()
{
    object factory;

    MESSAGE("Testing sTeam Server !");
    
    foreach(values(mClasses), factory) 
	factory->test();

    factory = get_factory(CLASS_OBJECT);
    object obj = factory->execute( ([ "name":"object", ]) );
    MESSAGE("Object test------------------------------------------");
    obj->test();
    MESSAGE("Test succeeded.");
    exit(1);
}

int get_object_class() { return 0; }

int get_object_id() { return 0; }

object this() { return this_object(); }
int status() { return PSTAT_SAVE_OK; }

function find_function(string fname) { return this_object()[fname]; }
