/*
 * p r o c e s s . c 		-- Access to processes from STklos
 *
 * Copyright  1994-2001 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 *
 * Permission to use, copy, modify, distribute,and license this
 * software and its documentation for any purpose is hereby granted,
 * provided that existing copyright notices are retained in all
 * copies and that this notice is included verbatim in any
 * distributions.  No written agreement, license, or royalty fee is
 * required for any of the authorized uses.
 * This software is provided ``AS IS'' without express or implied
 * warranty.
 *
 *            Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: ??-???-1994 ??:??
 * Last file update: 16-Jan-2001 23:17 (eg)
 *
 * Code for Win32 conributed by (Paul Anderson <paul@grammatech.com> and 
 * Sarah Calvo <sarah@grammatech.com>) has been deleted for now. It should be
 * reintroduced for a Win32 port. Look at file proces.c in Win32 for that. 
 *
 */

/******************************************************************************
 *
 * Process extended type definition
 *
 ******************************************************************************/

#include "stklos.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <signal.h>


/*
 * Data 
 */
static char *stdStreams[3] = {		/* Used for messages */
  "input",	
  "output",	
  "error",
};

 
#define MAX_PROC_NUM      40 		/* (simultaneous processes) enough? */
 
struct process_obj {
  stk_header header;
  int pid;                      /* Process id */
  SCM streams[3];		/* standard ports for the process */
  int exited;			/* Process is terminated */
  int exit_status;		/* Exit status of the processus */
  int waited_on;		/* non zero if the process is being
   				   waited on by a waitpid(..,..,0) */
};
 

#define PROCESS_PID(p)		(((struct process_obj *) (p))->pid)
#define PROCESS_STREAMS(p)	(((struct process_obj *) (p))->streams)
#define PROCESS_EXITED(p)	(((struct process_obj *) (p))->exited)
#define PROCESS_STATUS(p)	(((struct process_obj *) (p))->exit_status)
#define PROCESS_WAITED(p)	(((struct process_obj *) (p))->waited_on)
#define PROCESSP(p)		(BOXED_TYPE_EQ((p), tc_process))

static SCM all_processes = STk_nil;

#if defined(SIGCHLD) && !defined(HPUX)
#  define USE_SIGCHLD 1 /* What's the problem with HP? */
#endif

#ifdef USE_SIGCHLD
#  define PURGE_PROCESS_TABLE()				    /* Nothing to do */
#else
#  define PURGE_PROCESS_TABLE() process_terminate_handler(0)/* Simulate a SIGCHLD */
#endif
 
/******************************************************************************/

static void error_bad_process(SCM proc)
{
  STk_error("bad process ~S", proc);
}


static int process_alivep(SCM process)
{
   if (PROCESS_EXITED(process)) 
     return FALSE;
   else if (PROCESS_WAITED(process))
     return TRUE;
   else {
     int info, res;
     
     /* Use waitpid to gain the info. */
     res = waitpid(PROCESS_PID(process), &info, WNOHANG);
     if (res == 0)
       /* process is still running */
       return TRUE;
     else 
       if (res == PROCESS_PID(process)) {
	 /* process has terminated and we must save this information */
	 PROCESS_EXITED(process) = TRUE;
	 PROCESS_STATUS(process) = info;
	 return FALSE;
       } else {
	 /* might not have found process because we've already waited for it */
	 /* if so, then status has already been updated */
	 return FALSE;
       }
   }
}
 
static void process_terminate_handler(int sig) /* called when a child dies */
{
  SCM prev, l;
 
  /* Delete the processes which are not alive from the global list
   * This loop may delete nobody if this the process has been deleted
   * before (a previous call to this function may have deleted more than
   * one process. 
   * Note: No assumption is made on the process which has terminated;
   * we act blindly here since it does not seem that there is a POSIX way 
   * to find the id of the process which died.
   */
  for (prev=l=all_processes; !NULLP(l); prev=l, l=CDR(l)) {
    if (!process_alivep(CAR(l))) {
      /* Process died. delete it from the list */
      if (l == all_processes)
	all_processes = CDR(l);
      else
	CDR(prev) = CDR(l);
    }
  }
}


static SCM make_process(void)
{
  SCM z;

  PURGE_PROCESS_TABLE();
  
  NEWCELL(z, process);
  PROCESS_STREAMS(z)[0] = STk_false;
  PROCESS_STREAMS(z)[1] = STk_false;
  PROCESS_STREAMS(z)[2] = STk_false;
  PROCESS_EXITED(z) = 0;
  PROCESS_STATUS(z) = 0;
  return z;
}

static void close_all_files(int pipes[3][2])
{
  int i; 

  for (i = 0; i < 3; i++) {
    if (pipes[i][0] != -1) close(pipes[i][0]);
    if (pipes[i][1] != -1) close(pipes[i][1]);
  }
}

static int same_files(char* f1, char* f2)
{
  struct stat s1, s2;

  if (stat(f1, &s1) < 0) return FALSE;
  if (stat(f2, &s2) < 0) return FALSE;
  
  return (s1.st_dev==s2.st_dev && s1.st_ino==s2.st_ino);
}

static int is_pipe_p(SCM key)
{
  return KEYWORDP(key) && (strcmp(KEYWORD_PNAME(key), "pipe") == 0);
}


static char *maybe_redirect_input(int index, SCM *redir, int pipes[3][2])
{
  char *name = "";
  SCM in     = redir[index];
  int fd;
  
  if (STRINGP(in)) {
    /* redirection in a file */
    name = STRING_CHARS(in);
    fd = open(name, O_RDONLY);
    if (fd < 0) {
      close_all_files(pipes);
      STk_error("cannot redirect input to ~S", in);
    }
    pipes[0][0] = fd;
  } else if (is_pipe_p(in)) {
    /* Redirection in a pipe*/
    if (pipe(pipes[0]) < 0) {
      close_all_files(pipes);
      STk_error("cannot create pipe for input");
    }
  }
  return name; 
}

static char *maybe_redirect_output(int index, SCM *redir, int pipes[3][2], 
				   char *input, char *output)
{
  char *name = "";
  SCM out    = redir[index];
  int fd;

  if (STRINGP(out)) {
    /* redirection in a file */
    name = STRING_CHARS(out);
    if (same_files(name, input))
      STk_error("file ~S used for input and output", out);

    if (same_files(name, output))
      fd = dup(pipes[1][0]);       /* :output "file" :error "file" */
    else
      fd = open(name, O_WRONLY|O_TRUNC|O_CREAT, 0666); /* out != err */

    if (fd < 0) {
      close_all_files(pipes);
      STk_error("cannot redirect input to ~S", out);
    }
    pipes[index][0] = fd;
  } else if (is_pipe_p(out)) {
    /* Redirection in a pipe*/
    if (pipe(pipes[index]) < 0) {
      close_all_files(pipes);
      STk_error("cannot create pipe for output");
    }
  }
  return name; 
}

/*===========================================================================*\
 *
 * 		Implementation of run-process for Unix
 *
\*==========================================================================*/

DEFINE_PRIMITIVE("%run-process", run_process, subr4,
		 (SCM redirections, SCM do_wait, SCM do_fork, SCM args))
{
  SCM z, l;
  char **argv;
  char *in_name, *out_name, *err_name;
  int i, len;
  int pipes[3][2];
  SCM *redir;
  pid_t pid;

  ENTER_PRIMITIVE(run_process);

  /* Initialisations */
  for (i = 0; i < 3; i++)
    pipes[i][0] =  pipes[i][1] = -1;

  redir = VECTOR_DATA(redirections);

  /* Build an argv array for exec system call */
  len = STk_int_length(args);		  /* //FIXME:  Pas trait le rsh */
  if (len < 0)
    STk_error("bad argument list ~S", args);
  argv = STk_must_malloc((len + 3) * sizeof(char *));
  
  for (i=0, l=args; i < len; i++, l=CDR(l)) {
    if (!STRINGP(CAR(l))) STk_error("bad string ~S", CAR(l));
    argv[i] = STRING_CHARS(CAR(l));
  }
  argv[len] = NULL;
  
  /* Do (eventually) redirections */
  in_name  = maybe_redirect_input (0, redir, pipes);
  out_name = maybe_redirect_output(1, redir, pipes, in_name, "");
  err_name = maybe_redirect_output(2, redir, pipes, in_name, out_name);

  /* Build a process object */
  z   = make_process();
  pid = (do_fork == STk_false) ? 0 : fork();
  
  /* Fork another process */
  switch (pid) {
    case -1:  close_all_files(pipes);
	      STk_error("cannot create a new process");	
	      break;
    case 0:  /* CHILD */
      	     for(i = 0; i < 3; i++) {
 	       if (STRINGP(redir[i])) { 
		 /* Redirection in a file */
 		 dup2(pipes[i][0], i);
 		 close(pipes[i][0]);
 	       } else if (is_pipe_p(redir[i])) {
		   /* Redirection in a pipe */
 		   dup2(pipes[i][i==0? 0 : 1], i);
 		   close(pipes[i][0]);
 		   close(pipes[i][1]);
 		 }
 	     }

	     /* close all remaining files */
	     for(i = 3; i < NOFILE; i++) close(i);
 
 	     /*  And then, EXEC'ing...  */
  	     execvp(*argv, argv);
 	     
 	     /* Cannot exec if we are here */
 	     STk_fprintf(STk_curr_eport, "**** Cannot  exec %s!\n", *argv);
 	     exit(1);
    default: /* PARENT */
      	     PROCESS_PID(z) = pid;
       	     for(i = 0; i < 3; i++) {
 	       if (STRINGP(redir[i])) 
 		 /* Redirection in a file */
 		 close(pipes[i][0]);
 	       else if (is_pipe_p(redir[i])) {
		 /* Redirection in a pipe */
		 SCM port;
		 char buffer[100];
		 
		 close(pipes[i][i == 0 ? 0 : 1]);
		 /* Make a new file descriptor to access the pipe */
		 sprintf(buffer, "pipe-%s-%d", stdStreams[i], pid);
		 port = (i == 0) ?
		   		STk_fd2scheme_port(pipes[i][1], "w", buffer) :
		   		STk_fd2scheme_port(pipes[i][0], "r", buffer);
		 if (!port) {
		   close_all_files(pipes);
		   STk_error("cannot reopen pipe %d for process %d", i, pid);
		 }
		 PROCESS_STREAMS(z)[i] = port;
	       }
	       if (do_wait != STk_false) {
		 PROCESS_WAITED(z) = 1;
		 waitpid(pid, &(PROCESS_STATUS(z)), 0);
		 PROCESS_WAITED(z) = 0;
		 PROCESS_EXITED(z) = TRUE;
	       }
	     }
  }
  /* Chain new process in the list of all process */
  all_processes = STk_cons(z, all_processes);
  return z;
} 


/*
<doc ext process?
 * (process? obj)
 *
 * Returns |t| if |obj| is a process , otherwise returns |f|.
doc>
*/
DEFINE_PRIMITIVE("process?", processp, subr1, (SCM obj))
{
  return MAKE_BOOLEAN(PROCESSP(obj));
}

/*
<doc ext process-alive?
 * (process-alive? proc)
 *
 * Returns |t| if process |proc| is currently running, otherwise returns |f|.
doc>
*/
DEFINE_PRIMITIVE("process-alive?", proc_alivep, subr1, (SCM proc))
{
  ENTER_PRIMITIVE(proc_alivep);
  if (!PROCESSP(proc)) error_bad_process(proc);
  return  MAKE_BOOLEAN(process_alivep(proc));
}


/*
<doc ext process-pid
 * (process-pid proc)
 *
 * Returns an integer which represents the Unix identification (PID) of the 
 * processus.
doc>
*/
DEFINE_PRIMITIVE("process-pid", proc_pid, subr1, (SCM proc))
{
  ENTER_PRIMITIVE(proc_pid);
  if (!PROCESSP(proc)) error_bad_process(proc);
  return STk_long2integer((long) PROCESS_PID(proc));
}


/*
<doc ext process-list
 * (process-list)
 *
 * Returns the list of processes which are currently running (i.e. alive).
doc>
*/
DEFINE_PRIMITIVE("process-list", proc_list, subr0, (void))
{
  PURGE_PROCESS_TABLE();
  return STk_copy_tree(all_processes);
}


/*
<doc ext process-input process-output process-error
 * (process-input proc)
 * (process-input proc)
 * (process-input proc)
 *
 * Returns the file port associated to the standard input, output or error
 * of |proc|, if it is redirected in (or to) a pipe; otherwise
 * returns |f|. Note that the returned port is opened for reading
 * when calling |process-output| or |process-error|; it is opened
 * for writing when calling |process-input|.
doc>
*/

DEFINE_PRIMITIVE("process-input", proc_input, subr1, (SCM proc))
{
  ENTER_PRIMITIVE(proc_input);
  if (!PROCESSP(proc)) error_bad_process(proc);
  return PROCESS_STREAMS(proc)[0];
}

DEFINE_PRIMITIVE("process-output", proc_output, subr1, (SCM proc))
{
  ENTER_PRIMITIVE(proc_output);
  if (!PROCESSP(proc)) error_bad_process(proc);
  return PROCESS_STREAMS(proc)[1];
}

DEFINE_PRIMITIVE("process-error", proc_error, subr1, (SCM proc))
{
  ENTER_PRIMITIVE(proc_error);
  if (!PROCESSP(proc)) error_bad_process(proc);
  return PROCESS_STREAMS(proc)[2];
}
 

/*
<doc ext process-wait
 * (process-wait proc)
 *
 * Stops the current process (the Scheme process) until |proc| completion. 
 * |Process-wait| returns |f| when |proc| is already terminated; it returns
 * |t| otherwise. 
doc>
*/
DEFINE_PRIMITIVE("process-wait", proc_wait, subr1, (SCM proc))
{
  PURGE_PROCESS_TABLE();

  ENTER_PRIMITIVE(proc_wait);
  if (!PROCESSP(proc)) error_bad_process(proc);
  
  if (PROCESS_EXITED(proc)) return STk_false;
  else {
    int res, info;
    SCM ret_val = STk_false;
    
    PROCESS_WAITED(proc) = 1;
    res = waitpid(PROCESS_PID(proc), &info, 0);
    if (res == PROCESS_PID(proc)) {
      PROCESS_STATUS(proc) = info;
      ret_val =  STk_true;;
    }

    PROCESS_WAITED(proc) = 0;
    PROCESS_EXITED(proc) = TRUE;
    return ret_val;
  }
}
 
/*
<doc ext process-exit-status
 * (process-exit-status proc)
 *
 * Returns the exit status of |proc| if it has finished its execution; 
 * returns |f| otherwise.
doc>
*/ 
DEFINE_PRIMITIVE("process-exit-status", proc_xstatus, subr1, (SCM proc))
{
  int info, n, res;

  PURGE_PROCESS_TABLE();

  ENTER_PRIMITIVE(proc_xstatus);
  if (!PROCESSP(proc)) error_bad_process(proc);
  

  if (PROCESS_EXITED(proc)) {
#ifndef __CYGWIN32__
    if (WIFSIGNALED(PROCESS_STATUS(proc)))
      n = WCOREDUMP(PROCESS_STATUS(proc));
    else
#endif
      n = WEXITSTATUS(PROCESS_STATUS(proc));
  } else {
    res = waitpid(PROCESS_PID(proc), &info, WNOHANG);
    if (res == 0) {
      /* Process is still running */
      return STk_false;
    }
    else if (res == PROCESS_PID(proc)) {
      /* Process is now terminated */
      PROCESS_EXITED(proc) = TRUE;
      PROCESS_STATUS(proc) = info;
      n = WEXITSTATUS(info);
    } else
      return STk_false;
  }
  return STk_long2integer((long) n);
}

/*
<doc ext process-send-signal
 * (process-send-signal proc sig)
 *
 * Sends the integer signal |sig| to |proc|. Since value of |sig| is system
 * dependant, use the symbolic defined signal constants to make your program 
 * independant of the running system (see @pxref{signals}). 
 * The result of |process-send-signal| is @emph{void}.
doc>
*/ 
DEFINE_PRIMITIVE("process-signal", proc_signal, subr2, (SCM proc, SCM sig))
{
  int S;
  PURGE_PROCESS_TABLE();

  ENTER_PRIMITIVE(proc_signal);
  if (!PROCESSP(proc)) error_bad_process(proc);
  S = STk_get_signal_value(sig);

  kill(PROCESS_PID(proc), S);
  return STk_void;
}


/******************************************************************************/
static void print_process(SCM p, SCM port, int mode)
{
  char buffer[100];

  sprintf(buffer, "#<process PID=%d>", PROCESS_PID(p));
  STk_puts(buffer, port);
}
 


struct extended_type_descr xtype_process = {
  "process",
  print_process
};


 
int STk_init_process(void)
{
  /* 
   * On systems which support SIGCHLD, the processes table is cleaned up
   * as soon as a process terminate. On other systems this is done from time
   * to time to avoid a too long list of porcesses
   */
  struct sigaction sigact;

  /* Define information for process type */
  DEFINE_XTYPE(process, &xtype_process);

  /* Define the handler for process termination. */
  sigemptyset(&(sigact.sa_mask));
  sigact.sa_handler = process_terminate_handler;
  sigact.sa_flags   = SA_NOCLDSTOP;     /* Ignore SIGCHLD generated by SIGSTOP */
#ifdef SA_RESTART
  /* Thanks to Harvey J. Stein <hjstein@MATH.HUJI.AC.IL> for the fix */
  sigact.sa_flags  |= SA_RESTART;
#endif
  sigaction(SIGCHLD, &sigact, NULL);

  ADD_PRIMITIVE(run_process);
  ADD_PRIMITIVE(processp);
  ADD_PRIMITIVE(proc_alivep);
  ADD_PRIMITIVE(proc_pid);
  ADD_PRIMITIVE(proc_list);
  ADD_PRIMITIVE(proc_input);
  ADD_PRIMITIVE(proc_output);
  ADD_PRIMITIVE(proc_error);
  ADD_PRIMITIVE(proc_wait);
  ADD_PRIMITIVE(proc_signal);
  return TRUE;
}
