use core:geometry;
use ui;

/**
 * An animation for an element inside a slide.
 *
 * Similar to, but not identical to the animations used for slide intros. These animations are
 * configurable to a higher degree, and multiple of these animations may play simultaneously.
 *
 * Note: Many aspects of the animation are configured from a custom syntax, so that new animations
 * do not have to overload a large number of constructors for commonly used scenarios.
 */
class Animation on Render {
	// Play at step number.
	Nat step;

	// Duration of this animation.
	Duration duration;

	// Offset from the start of this animation step.
	Duration offset;

	// Create, default duration.
	init(Nat step) {
		init() {
			step = step;
			duration = Duration();
			offset = Duration();
		}
	}

	// Setup the animation. Called when added to a slide so that we may set defaults.
	void setup(Presentation p, Element element) {
		if (duration == 0 ms)
			duration = p.animationTime;
	}

	// Called before drawing the element. May return 'false' to prevent painting the element.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : abstract;

	// Called after drawing the element to restore any graphics state set up by 'before'.
	void after(Element element, Graphics g, Nat cStep, Duration time) : abstract;

	// Convert a time into a float in the range 0..1.
	Float val(Duration d) {
		clamp((d - offset) / duration, 0.0, 1.0);
	}

	// Convert to a smooth value in the range 0..1.
	Float smoothVal(Duration d) {
		smooth(val(d));
	}

	// Convert a value in the range 0..1 to a "smooth" value in the same range.
	Float smooth(Float x) : static {
		Angle v = rad(x * pi);
		0.5 - v.cos()*0.5;
	}
}

/**
 * An animation that acts like a trigger, i.e. it triggers once at the desired time and does
 * something at that moment only. This class is intended to provide functionality to other embedded
 * things, and has no real use in isolation as-is.
 */
class Trigger extends Animation {
	// Last time triggered.
	private Duration lastTime;

	// Init.
	init(Nat step) {
		init(step) {}
	}

	// Called when the event is supposed to trigger.
	void trigger(Element element) : abstract;

	// Setup. Initialize 'lastTime'.
	void setup(Presentation p, Element element) : override {
		super:setup(p, element);
		// Won't work properly with a zero animation time.
		if (duration == 0 ms)
			duration = 10 ms;
		lastTime = duration;
	}

	// Look at timestamps to determine when to trigger.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			Bool fire = false;
			if ((lastTime < offset) & (time > offset)) {
				fire = true;
			} else if ((lastTime > time) & (time >= offset)) {
				fire = true;
			}
			lastTime = time;

			if (fire)
				trigger(element);
		}
		true;
	}

	// Need to implement since base is abstract.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}


/**
 * Interface to simplify animations that animate the entrance of an object.
 *
 * Adapts the interface of Animation into an interface that is simpler to use, but less flexible.
 */
class AnimationIntro extends Animation {
	// Init.
	init(Nat step) {
		init(step) {}
	}

	// Called before drawing the element.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override, final {
		if (cStep == step) {
			if (time < offset)
				return false;
			if (time < duration + offset)
				before(element, g, val(time));
		}

		cStep >= step;
	}

	// Called after drawing the element.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override, final {
		if (cStep == step) {
			if (time < offset)
				return;
			if (time < duration + offset)
				after(element, g);
		}
	}

	// Override to update the graphics context. Expected to call "push" in some form. "pos" is a
	// float in range 0..1 indicating the current position of the animation.
	void before(Element element, Graphics g, Float pos) : abstract;

	// Override to restore the graphics context. Default implementation simply pops it once.
	void after(Element element, Graphics g) {
		g.pop();
	}
}

/**
 * Interface to simplify animations that animate the disappearence of an object.
 *
 * Adapts the interface of Animation into an interface that is simpler to use, but less flexible.
 */
class AnimationOutro extends Animation {
	// Init.
	init(Nat step) {
		init(step) {}
	}

	// Called before drawing the element.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override, final {
		if (cStep == step) {
			if (time < offset)
				return true;
			if (time < duration + offset) {
				before(element, g, val(time));
				return true;
			}
			return false;
		}

		cStep < step;
	}

	// Called after drawing the element.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override, final {
		if (cStep == step) {
			if (time < offset)
				return;
			if (time < duration + offset)
				after(element, g);
		}
	}

	// Override to update the graphics context. Expected to call "push" in some form. "pos" is a
	// float in range 0..1 indicating the current position of the animation.
	void before(Element element, Graphics g, Float pos) : abstract;

	// Override to restore the graphics context. Default implementation simply pops it once.
	void after(Element element, Graphics g) {
		g.pop();
	}
}


/**
 * Simple "show" animation. Simply displays the item at the specified step.
 */
class ShowAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
		// To indicate that we're not being animated.
		duration = 1 ms;
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			time >= offset;
		} else {
			cStep >= step;
		}
	}

	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}

/**
 * Simple "hide" animation. Hides the item at the specified step.
 */
class HideAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
		duration = 1 ms;
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			time < offset;
		} else {
			cStep < step;
		}
	}

	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}


/**
 * Fade in animation.
 */
class FadeInAnimation extends AnimationIntro {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		g.push(time);
	}
}

/**
 * Animation that raises the element slightly while fading in.
 */
class RaiseAnimation extends AnimationIntro {
	// Create.
	init(Nat step) {
		init(step) {
			height = 20;
		}
	}

	// Create.
	init(Nat step, Float height) {
		init(step) {
			height = height;
		}
	}

	// Height to raise.
	Float height;

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		Float t = time;
		g.push(t);
		Float offset = 1 - rad((1 - t)*pi/2).cos;
		g.transform = translate(Point(0, offset * height));
	}
}


/**
 * Fade out animation.
 */
class FadeOutAnimation extends AnimationOutro {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		g.push(1 - time);
	}
}


/**
 * Grow vertically.
 */
class GrowAnimation extends AnimationIntro {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		Float scale = smooth(time);
		g.push(scale);

		// The font drawing does not seem to like zero scaling, at least in the Y direction
		// neither on Windows nor on Linux.
		scale = max(scale, 0.05);

		Point center = element.pos.center;
		g.transform(scale(Point(1, scale), center));
	}
}


/**
 * Shrink vertically.
 */
class ShrinkAnimation extends AnimationOutro {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		Float scale = 1 - smooth(time);
		g.push(scale);

		// The font drawing does not seem to like zero scaling, at least in the Y direction
		// neither on Windows nor on Linux.
		scale = max(scale, 0.05);

		Point center = element.pos.center;
		g.transform(scale(Point(1, scale), center));
	}
}


/**
 * Zoom in.
 */
class ZoomInAnimation extends AnimationIntro {
	// Create.
	init(Nat step) {
		init(step) {}
	}
	init(Nat step, Point center) {
		init(step) {
			center = center;
		}
	}

	// Center point, if specified.
	Point? center;

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		Float scale = smooth(time);
		g.push(scale);

		// The font drawing does not seem to like zero scaling, at least in the Y direction
		// neither on Windows nor on Linux.
		scale = max(scale, 0.05);

		Point center = if (c = this.center) { c; } else { element.pos.center; };
		g.transform(scale(Point(scale, scale), center));
	}
}


/**
 * Zoom out vertically.
 */
class ZoomOutAnimation extends AnimationOutro {
	// Create.
	init(Nat step) {
		init(step) {}
	}
	init(Nat step, Point center) {
		init(step) {
			center = center;
		}
	}

	// Center point, if specified.
	Point? center;

	// Set up the animation.
	void before(Element element, Graphics g, Float time) : override {
		Float scale = 1 - smooth(time);
		g.push(scale);

		// The font drawing does not seem to like zero scaling, at least in the Y direction
		// neither on Windows nor on Linux.
		scale = max(scale, 0.05);

		Point center = if (c = this.center) { c; } else { element.pos.center; };
		g.transform(scale(Point(scale, scale), center));
	}
}
