use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * A column declaration.
 */
class Column {
	// Name of the column.
    Str name;

	// Datatype.
	SQLType datatype;

	// Is this column a primary key?
	Bool primaryKey;

	// Is this column allowed to contain NULL?
	Bool allowNull;

	// Unique?
	Bool unique;

	// Auto-increment?
	Bool autoIncrement;

	// Default value?
	SQLLiteral? default;


	init(SStr name, SQLType type) {
		init {
			name = name.v;
			datatype = type;
		}
	}

	// Called from the syntax.
	void setPrimary(Str x) { primaryKey = true; }
	void setAllowNull(Str x) { allowNull = true; }
	void setUnique(Str x) { unique = true; }
	void setAutoIncrement(Str x) { autoIncrement = true; }
	void setDefault(SQLLiteral x) { default = x; }

	// Called from the syntax, gives a decent error message on how we're different from SQL.
	void setNotNull(SStr x) {
		throw SyntaxError(x.pos, "Columns are NOT NULL by default. Use ALLOW NULL to allow nullable columns.");
	}

	// Create an SQL part for this column.
	Str toSQL() {
		StrBuf buffer;
		buffer << "\"" << name << "\" " << datatype.sql;
		modifiers(buffer);

		buffer.toS;
	}

	void toS(StrBuf to) : override {
		to << name << " " << datatype.sql;
		modifiers(to);
		if (primaryKey)
			to << " (primary key)";
	}

	Bool ==(Column o) {
		Bool same = name == o.name;
		same &= datatype.sql == o.datatype.sql;
		same &= primaryKey == o.primaryKey;
		same &= allowNull == o.allowNull;
		same &= unique == o.unique;
		same &= autoIncrement == o.autoIncrement;
		if (default) {
			if (odefault = o.default) {
				same &= default.same(odefault);
			} else {
				same = false;
			}
		} else {
			if (o.default) {
				same = false;
			}
		}
		same;
	}

	// See if this column has some kind of default value. I.e. it is OK to leave one without a value?
	Bool hasDefault(Bool multiPK) {
		if (default)
			return true;
		if (primaryKey & !multiPK)
			return true;
		if (allowNull)
			return true;
		if (autoIncrement)
			return true;
		false;
	}

	private void modifiers(StrBuf to) {
		if (!allowNull)
			to << " NOT NULL";
		if (unique)
			to << " UNIQUE";
		if (autoIncrement)
			to << " AUTOINCREMENT";
		if (default)
			to << " DEFAULT " << default.toSQL();
	}
}

/**
 * Column declaration that acts as standard SQL (we have slightly different defaults).
 */
class ParseColumn extends Column {
	init(SStr name, SQLType type) {
		init(name, type) {}
		allowNull = true;
	}

	void setNotNull(SStr x) : override {
		allowNull = false;
	}

	void toS(StrBuf to) : override {
		super:toS(to);
		to << "!";
	}
}


/**
 * Index for a table.
 */
class Index {
	// Name of the index (might be auto-generated).
	Str name;

	// Columns.
	Str[] columns;

	// Create.
	init(Str name, Str[] columns) {
		init {
			name = name;
			columns = columns;
		}
	}

	// To SQL statement.
	Str toSQL(Str table) {
		StrBuf to;
		to << "CREATE INDEX \"" << name << "\" ON \"" << table << "\"(";
		for (i, c in columns) {
			if (i > 0)
				to << ", ";
			to << "\"" << c << "\"";
		}
		to << ");";

		to.toS;
	}

	// To string. No SQL escaping.
	void toS(StrBuf to) : override {
		to << "INDEX " << name << " ON ?(" << join(columns, ", ") << ")";
	}
}

/**
 * Declaration of an entire table.
 */
class Table {
	// Name of the table.
    Str name;

	// Columns in the table.
    Column[] columns;

	// Indices for this table.
	Index[] indices;

	init(SStr name) {
		init {
			name = name.v;
		}
	}

	// Find a column.
	Column? find(Str name) {
		for (c in columns)
			if (c.name == name)
				return c;
		null;
	}

	// Add column (called from syntax).
	void add(Column col) {
		columns << col;
	}

	// Add primary key(s) (called from syntax).
	void add(Array<SStr> cols) {
		for (c in columns)
			if (c.primaryKey)
				throw SyntaxError(cols[0].pos, "Only one instance of the PRIMARY KEY keyword may be present for each table.");

		for (c in cols) {
			unless (col = find(c.v))
				throw SyntaxError(c.pos, "No column named ${c.v} was declared in this table.");
			col.primaryKey = true;
		}
	}

	// Add index.
	void add(SrcPos pos, Index index) {
		for (i in indices)
			if (i.name == index.name)
				throw SyntaxError(pos, "The index ${index.name} already exists.");

		indices << index;
	}

	// Check if there are multiple primary keys.
	Bool multiplePK() {
		Nat count = 0;
		for (c in columns)
			if (c.primaryKey)
				count++;
		count > 1;
	}

	// Create an SQL statement for this table declaration.
	Str toSQL(Bool ifNotExists, Str[] options) {
		StrBuf buffer;
		buffer << "CREATE TABLE ";
		if (ifNotExists)
			buffer << "IF NOT EXISTS ";
		buffer << "\"" << name << "\" (";

		Str[] pk;
		for (i, col in columns) {
			if (i > 0)
				buffer << ", ";
			buffer << col.toSQL();
			if (col.primaryKey)
				pk << col.name;
		}

		if (pk.any) {
			buffer << ", PRIMARY KEY(";
			for (i, k in pk) {
				if (i > 0)
					buffer << ", ";
				buffer << "\"" << k << "\"";
			}
			buffer << ")";
		}

		buffer << ")";
		if (options.any) {
			buffer << " " << join(options, ", ");
		}
		buffer << ";";
		buffer.toS;
	}
}

/**
 * Helper type for creating indices.
 */
class IndexDecl {
	SrcPos pos;
	SStr table;
	Index index;

	init(SrcPos pos, SStr name, SStr table, SStr[] cols) {
		Str[] c;
		for (x in cols)
			c << x.v;

		init {
			pos = pos;
			table = table;
			index(name.v, c);
		}
	}

	init(SrcPos pos, SStr table, SStr[] cols) {
		StrBuf name;
		name << table.v;

		Str[] c;
		for (x in cols) {
			c << x.v;
			name << "_" << x.v;
		}

		init {
			pos = pos;
			table = table;
			index(name.toS, c);
		}
	}
}

/**
 * Database description.
 */
class Database {
	// Tables declared (indices are stored inside each table).
	Table[] tables;

	// Add a table.
	void add(Table decl) {
		tables.push(decl);
	}

	// Add an index.
	void add(IndexDecl decl) {
		unless (table = find(decl.table.v))
			throw SyntaxError(decl.table.pos, "The table ${decl.table.v} was not declared (yet).");

		table.add(decl.pos, decl.index);
	}

	// Find a table.
	Table? find(Str name) {
		// TODO: Speedier lookup?
		for (table in tables)
			if (table.name == name)
				return table;
		null;
	}
}

/**
 * Declaration of a database.
 */
class DatabaseDecl extends NamedDecl {
    SStr name;
    Scope scope;
	Database contents;

    init(SStr name, Scope scope, Database contents) {
        init() {
            name = name;
            scope = scope;
			contents = contents;
        }
    }

    Named doCreate() {
        DatabaseType(name, contents);
    }
}

/**
 * Type stored in the name tree.
 */
class DatabaseType extends Type {
    Database contents;

    init(SStr name, Database contents) {
        init(name.v, TypeFlags:typeClass) {
            contents = contents;
        }

		setSuper(named{TypedConnection});
		addCtor();
    }

	private void addCtor() {
		BSTreeCtor ctor([thisParam(this), ValParam(named{DBConnection}, "db")], SrcPos());
		CtorBody body(ctor, Scope());
		if (var = body.variable(SimplePart("db"))) {
			Actuals params;
			params.add(LocalVarAccess(SrcPos(), var));
			params.add(ReferDBType(this));
			body.add(InitBlock(SrcPos(), body, params));
		} else {
			throw InternalError("Unable to find the 'db' parameter.");
		}

		ctor.body = body;
		add(ctor);
	}
}

/**
 * Expression that returns a type.
 */
private class ReferDBType extends Expr {
	private DatabaseType referTo;

	init(DatabaseType referTo) {
		init(SrcPos()) { referTo = referTo; }
	}

	ExprResult result() : override {
		ExprResult(Value(named{DatabaseType}));
	}

	void code(CodeGen state, CodeResult result) : override {
		if (result.needed) {
			var v = result.location(state);
			state.l << core:asm:mov(v, core:asm:objPtr(referTo));
			result.created(state);
		}
	}

	Str toS() : override {
		"<type>";
	}
}

/**
 * Base class inherited from the DBType class.
 */
class TypedConnection {
	// Underlying database connection.
	DBConnection connection;

	// Cached prepared statements. TODO: Copy these on a thread switch?
	private Nat->Statement statements;

	// Create and verify the database structure.
	init(DBConnection conn, DatabaseType t) {
		init { connection = conn; }

		checkStructure(t.contents);
	}

	// Close.
	void close() {
		for (k, v in statements)
			v.finalize();
		connection.close();
	}

	// Get a cached prepared statement based on its ID.
	// TODO: One idea here is to remove statements that are "prepared" and not return them
	// until we are done with them. That allows us to be re-entrant properly.
	Statement prepare(Nat id, Str query) {
		var iter = statements.find(id);
		if (iter == statements.end) {
			Statement stmt = connection.prepare(query);
			statements.put(id, stmt);
			return stmt;
		} else {
			return iter.v;
		}
	}

	// Check the database contents for a match.
	private void checkStructure(Database db) {
		// TODO: This should be rewritten and improved.
		for (table in db.tables) {
			if (schema = connection.schema(table.name)) {
				var queries = checkTable(table, schema);
				for (q in queries) {
					var stmt = connection.prepare(q);
					stmt.execute();
					stmt.finalize();
				}
			} else {
				// Create the table.
				Str[] options;
				// TODO: This should be handled in a more general way...
				if (connection as SQLite)
					options << "STRICT";
				var stmt = connection.prepare(table.toSQL(false, options));
				stmt.execute();
				stmt.finalize();

				for (index in table.indices) {
					var stmt = connection.prepare(index.toSQL(table.name));
					stmt.execute();
					stmt.finalize();
				}
			}
		}
	}
}

// Helper (on the Compiler thread) for checking column declarations. We need the parser, that's why
// we run on the compiler thread.
private Str[] checkTable(Table table, Schema schema) on Compiler {
	if (table.columns.count < schema.count)
		throw SchemaError("Too many columns in the database.", table, schema);

	Str[] queries;

	// We need to parse the modifier string...
	Parser<SDatatype> typeParser;
	Parser<SModifiers> modifierParser;
	typeParser.addSyntax(named{lang:bs});
	modifierParser.addSyntax(named{lang:bs});

	Set<Str> pk;
	for (k in schema.primaryKeys)
		pk.put(k);

	for (Nat i = 0; i < schema.count; i++) {
		var ref = table.columns[i];
		var curr = schema[i];

		typeParser.parse(curr.datatype, core:io:Url());
		if (typeParser.hasError())
			throw SQLError("Unknown column datatype for ${curr.name}: ${curr.datatype}.");

		ParseColumn c(SStr(ref.name), typeParser.tree.transform());

		modifierParser.parse(curr.attributes, core:io:Url());
		if (modifierParser.hasError()) {
			modifierParser.throwError();
			throw SQLError("Failed to parse the column attributes for column ${curr.name}: ${curr.attributes}");
		}

		modifierParser.tree.transform(c);

		if (pk.has(ref.name))
			c.primaryKey = true;

		Bool modify = ref != c;

		// Note: On databases other than SQLite, we are probably able to change these aspects.
		if (modify)
			throw SchemaError("Incorrect attributes for column ${i}", table, schema);
	}

	for (Nat i = schema.count; i < table.columns.count; i++) {
		// Add a column!
		var col = table.columns[i];
		queries << "ALTER TABLE \"${table.name}\" ADD ${col.toSQL};";
	}

	// Check indices...
	{
		Set<Str> desired;
		for (x in table.indices)
			desired.put(hashIndex(x));
		Set<Str> current;
		for (x in schema.indices)
			current.put(hashIndex(x));

		// See what to remove...
		for (x in schema.indices) {
			if (!desired.has(hashIndex(x)))
				queries << "DROP INDEX \"${x.name}\";";
		}

		// See what to add...
		for (x in table.indices) {
			if (!current.has(hashIndex(x)))
				queries << x.toSQL(table.name);
		}
	}

	queries;
}

// Produce a "hash" of an index.
private Str hashIndex(Index i) {
	StrBuf b;
	b << i.name << "|" << join(i.columns, ",");
	b.toS;
}
private Str hashIndex(Schema:Index i) {
	StrBuf b;
	b << i.name << "|" << join(i.columns, ",");
	b.toS;
}
