# Copyright (c) 2002 Juri Pakaste
# You may use and distribute this software under the terms of the
# GNU General Public License, version 2 or later
#

cvs_id = "$Id: FeedList.py,v 1.2 2004/04/16 20:26:07 jpakaste Exp $"

import straw

class FeedList(list, straw.SignalEmitter):
    def __init__(self, init_seq = []):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedsChangedSignal,
                              straw.FeedDeletedSignal,
                              straw.FeedCreatedSignal)
        self._loading = False

    def load_data(self):
        def _load(feeds, parent):
            for df in feeds:
                if isinstance(df, list):
                    #fc = straw.FeedCategory()
                    #fc.undump(df[0])
                    #self.append(parent, fc)
                    _load(df[1:], parent)
                else:
                    f = straw.create_empty_feed()
                    f.undump(df)
                    self.append(parent, f)
        self._loading = True
        feeds = straw.Config.get_instance().feeds
        if feeds is not None:
            _load(feeds, None)
        self._loading = False
        self.emit_signal(straw.FeedsChangedSignal(self))

    def connect_signals(self, ob):
        ob.signal_connect(straw.SaveFeedsSignal, self.feeds_changed, False)
        ob.signal_connect(straw.FeedsChangedSignal, self.feeds_changed, True)
        ob.signal_connect(straw.FeedErrorStatusChangedSignal, self.feeds_changed, False)

    def __setitem__(self, key, value):
        list.__setitem__(self, key, value)
        self.connect_signals(value)
        self.save_feeds_and_notify(True)

    def append(self, parent, value):
        list.append(self, value)
        value.parent = parent
        self.connect_signals(value)
        self.save_feeds_and_notify(
            True, signal=straw.FeedCreatedSignal(self, value, parent))

    def insert(self, index, parent, value):
        list.insert(self, index, value)
        value.parent = parent
        self.connect_signals(value)
        self.save_feeds_and_notify(
            True, signal=straw.FeedCreatedSignal(self, value, parent, index))

    def reorder(self, move, delta):
        k = self[:]
        move = list(move)
        move.sort()
        if delta > 0:
            move.reverse()
        if move[0] == 0 and delta < 0 or move[-1] == (len(self) - 1) and delta > 0:
            return
        for m in move:
            k[m + delta], k[m] = k[m], k[m + delta]
        for i in range(len(k)):
            list.__setitem__(self, i, k[i])
        self.save_feeds_and_notify(True)

    def __delitem__(self, value):
        feed = self[value]
        list.__delitem__(self, value)
        self.save_feeds_and_notify(
            True, signal=straw.FeedDeletedSignal(self, feed))

    def save_feeds_and_notify(self, notify, signal=None):
        if not self._loading:
            straw.Config.get_instance().feeds = [f.dump() for f in self]
            if notify:
                if signal is None:
                    signal = straw.FeedsChangedSignal(self)
                self.emit_signal(signal)

    def feeds_changed(self, signal, notify):
        self.save_feeds_and_notify(notify)

    def sort(self, indices = None):
        if not indices or len(indices) == 1:
            list.sort(self, lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
        else:
            items = []
            for i in indices:
                items.append(self[i])
                items.sort(lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
            for i in range(len(items)):
                list.__setitem__(self, indices[i], items[i])
        self.save_feeds_and_notify(True)

    def __hash__(self):
        h = 0
        for item in self:
            h ^= hash(item)
        return h

    def get_feed_with_id(self, id):
        for f in self.flatten_list():
            if f.id == id:
                return f
        return None

    def flatten_list(self, ob=None):
        if ob is None:
            ob = self
        l = []
        for o in ob:
            if isinstance(o, list):
                l = l + self.flatten_list(o)
            else:
                l.append(o)
        return l

feedlist_instance = None

def get_instance():
    global feedlist_instance
    if feedlist_instance is None:
        feedlist_instance = FeedList()
    return feedlist_instance

