/*
 * $Id: st-handler.c,v 1.124 2004/03/27 21:25:03 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include <string.h>
#include "gettext.h"
#include "st-category-bag.h"
#include "st-handler.h"
#include "st-handler-field.h"
#include "sg-util.h"
#include "st-stream-bag.h"
#include "st-category-store.h"
#include "st-stream-store.h"

/*** type definitions ********************************************************/

typedef struct
{
  gpointer cb;
  gpointer data;
} STHandlerCallback;

struct _STHandlerPrivate
{
  char			*name;
  char			*label;
  char			*copyright;
  char			*description;
  char			*home;

  GdkPixbuf		*pixbuf;
  
  GSList		*fields;
  GNode			*stock_categories;
  
  unsigned int		flags;

  STHandlerCallback	callbacks[ST_HANDLER_N_EVENTS];

  STCategoryStore	*categories;
  /*
   * We don't need a categories mutex (in the current design, the
   * store is never accessed by multiple threads concurrently).
   */

  GHashTable		*streams;
  GMutex		*streams_mutex;

  GSList		*remarkable_categories;
  GMutex		*remarkable_categories_mutex;

  int			fields_sort_index;
  int			fields_sort_order;

  int			paned_position;
};

typedef struct
{
  const char		*name;
  STCategoryBag		*category_bag;
} GetStockCategoryInfo;

typedef struct
{
  STHandler		*handler;
  GHashTable		*stream_bag_lists;
} StreamListsToBagsInfo;

/*** function declarations ***************************************************/

static void        st_handler_init			(STHandler   *handler);

static STCategory  *st_handler_category_new_default_cb	(gpointer    data);
static void	   st_handler_category_free_default_cb	(STCategory  *category,
							 gpointer    data);
static STStream    *st_handler_stream_new_default_cb	(gpointer    data);
static void	   st_handler_stream_free_default_cb	(STStream    *stream,
							 gpointer    data);

static gboolean    st_handler_set_stock_categories_cb	(GNode       *node,
							 gpointer    data);

static gboolean    st_handler_get_stock_category_cb	(GNode       *node,
							 gpointer    data);

static void        st_handler_refresh_multiple_cb	(gpointer    key,
							 gpointer    value,
							 gpointer    user_data);

static gboolean    st_handler_refresh_single		(STHandler	*handler,
							 STCategoryBag	*category_bag,
							 GNode		**category_bags,
							 GSList		**stream_bags,
							 GError		**err);
static gboolean    st_handler_refresh_multiple		(STHandler	*handler,
							 GNode		**category_bags,
							 GHashTable	**stream_bag_lists,
							 GError		**err);

static GNode       *st_handler_categories_to_bags	(STHandler   *handler,
							 GNode       *categories);
static gboolean    st_handler_categories_to_bags_cb	(GNode       *node,
							 gpointer    data);
static void        st_handler_categories_free		(STHandler   *handler,
							 GNode       *categories);
static gboolean    st_handler_categories_free_cb	(GNode       *node,
							 gpointer    data);

static GSList      *st_handler_streams_to_bags		(STHandler   *handler,
							 GList       *streams);
static void        st_handler_streams_free              (STHandler   *handler,
							 GList       *streams);

static GHashTable  *st_handler_stream_lists_to_bags	(STHandler   *handler,
							 GHashTable  *stream_lists);
static void        st_handler_stream_lists_to_bags_cb	(gpointer    key,
							 gpointer    value,
							 gpointer    user_data);
static void        st_handler_stream_lists_free		(STHandler   *handler,
							 GHashTable  *stream_lists);
static void        st_handler_stream_lists_free_cb	(gpointer    key,
							 gpointer    value,
							 gpointer    user_data);

static void        st_handler_restore_category_flags	(STHandler     *handler,
							 STCategoryBag *category_bag);

/*** API implementation ******************************************************/

GType
st_handler_get_type (void)
{
  static GType handler_type = 0;
  
  if (! handler_type)
    {
      static const GTypeInfo handler_info = {
	sizeof(STHandlerClass),
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	sizeof(STHandler),
	0,
	(GInstanceInitFunc) st_handler_init,
      };
      
      handler_type = g_type_register_static(G_TYPE_OBJECT,
					    "STHandler",
					    &handler_info,
					    0);
    }
  
  return handler_type;
}

STHandler *
st_handler_new (const char *name)
{
  STHandler *handler;

  g_return_val_if_fail(name != NULL, NULL);

  handler = g_object_new(ST_TYPE_HANDLER, NULL);
  handler->priv->name = g_strdup(name);

  return handler;
}

static STCategory *
st_handler_category_new_default_cb (gpointer data)
{
  return g_new0(STCategory, 1);
}

static void
st_handler_category_free_default_cb (STCategory *category, gpointer data)
{
  st_category_free(category);
}

static STStream *
st_handler_stream_new_default_cb (gpointer data)
{
  return g_new0(STStream, 1);
}

static void
st_handler_stream_free_default_cb (STStream *stream, gpointer data)
{
  st_stream_free(stream);
}

void
st_handler_set_label (STHandler *handler, const char *label)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(label != NULL);
  g_return_if_fail(handler->priv->label == NULL);

  handler->priv->label = g_strdup(label);
}

void
st_handler_set_description (STHandler *handler, const char *description)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(description != NULL);
  g_return_if_fail(handler->priv->description == NULL);

  handler->priv->description = g_strdup(description);
}

void
st_handler_set_home (STHandler *handler, const char *home)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(home != NULL);
  g_return_if_fail(handler->priv->home == NULL);

  handler->priv->home = g_strdup(home);
}

void
st_handler_set_copyright (STHandler *handler, const char *copyright)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(copyright != NULL);
  g_return_if_fail(handler->priv->copyright == NULL);

  handler->priv->copyright = g_strdup(copyright);
}

/*
 * Deprecated
 */
void
st_handler_set_info (STHandler *handler,
		     const char *label,
		     const char *copyright)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(label != NULL);
  g_return_if_fail(handler->priv->label == NULL);
  g_return_if_fail(copyright != NULL);
  g_return_if_fail(handler->priv->copyright == NULL);

  st_handler_set_label(handler, label);
  st_handler_set_copyright(handler, copyright);
}

void
st_handler_set_icon_from_inline (STHandler *handler,
				 int size,
				 const guint8 *data)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(size > 0);
  g_return_if_fail(data != NULL);
  g_return_if_fail(handler->priv->pixbuf == NULL);

  handler->priv->pixbuf = gdk_pixbuf_new_from_inline(size, data, FALSE, NULL);
  g_return_if_fail(GDK_IS_PIXBUF(handler->priv->pixbuf));
}

gboolean
st_handler_set_icon_from_file (STHandler *handler,
			       const char *filename,
			       GError **err)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);
  g_return_val_if_fail(handler->priv->pixbuf == NULL, FALSE);

  handler->priv->pixbuf = gdk_pixbuf_new_from_file(filename, err);
  return GDK_IS_PIXBUF(handler->priv->pixbuf);
}

/*
 * Deprecated.
 */
void
st_handler_set_icon (STHandler *handler, int size, const guint8 *data)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(size > 0);
  g_return_if_fail(data != NULL);
  g_return_if_fail(handler->priv->pixbuf == NULL);

  st_handler_set_icon_from_inline(handler, size, data);
}

void
st_handler_set_stock_categories (STHandler *handler, GNode *categories)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(categories != NULL);
  g_return_if_fail(handler->priv->stock_categories == NULL);

  g_node_traverse(categories,
		  G_PRE_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  st_handler_set_stock_categories_cb,
		  handler);
  handler->priv->stock_categories = categories;
}

static gboolean
st_handler_set_stock_categories_cb (GNode *node, gpointer data)
{
  if (node->data)
    {
      STHandler *handler = data;
      STCategoryBag *bag;

      bag = st_category_bag_new_from_category(handler, node->data);
      node->data = bag;
    }

  return FALSE;			/* continue */
}

void
st_handler_set_flags (STHandler *handler, unsigned int flags)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  handler->priv->flags = flags;
}

void
st_handler_add_field (STHandler *handler, STHandlerField *field)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(field != NULL);

  handler->priv->fields = g_slist_append(handler->priv->fields, field);
}

void
st_handler_bind (STHandler *handler,
		 STHandlerEvent event,
		 gpointer cb,
		 gpointer data)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(event >= 0 && event < ST_HANDLER_N_EVENTS);
  g_return_if_fail(cb != NULL);

  handler->priv->callbacks[event].cb = cb;
  handler->priv->callbacks[event].data = data;
}

/*** private implementation **************************************************/

static void
st_handler_init (STHandler *handler)
{
  handler->priv = g_new0(STHandlerPrivate, 1);

  handler->priv->streams = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);
  handler->priv->streams_mutex = g_mutex_new();
  handler->priv->remarkable_categories_mutex = g_mutex_new();

  handler->priv->paned_position = 150;

  /* provide handy defaults for some mandatory callbacks */

  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_NEW, st_handler_category_new_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_FREE, st_handler_category_free_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, st_handler_stream_new_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, st_handler_stream_free_default_cb, NULL);
}

void
st_handler_complete (STHandler *handler)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  if (! handler->priv->stock_categories)
    {
      STCategoryBag *category_bag;

      handler->priv->stock_categories = g_node_new(NULL);

      category_bag = st_category_bag_new(handler);
      ST_CATEGORY(category_bag)->name = g_strdup(ST_CATEGORY_BAG_MAIN);

      g_node_append_data(handler->priv->stock_categories, category_bag);
    }

  handler->priv->categories = st_category_store_new(handler->priv->stock_categories);
}

gboolean
st_handler_validate (STHandler *handler, GError **err)
{
  if (! handler)
    {
      g_set_error(err, 0, 0, _("passed handler is NULL"));
      return FALSE;
    }
  if (! ST_IS_HANDLER(handler))
    {
      g_set_error(err, 0, 0, _("passed handler is not a STHandler object"));
      return FALSE;
    }
  if (! handler->priv->label)
    {
      g_set_error(err, 0, 0, _("handler label is not set"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_NEW))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_NEW is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FIELD_GET is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FIELD_SET is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_MODIFY))
    {
      GSList *l;

      SG_LIST_FOREACH(l, handler->priv->fields)
        {
	  STHandlerField *field = l->data;

	  if (ST_HANDLER_FIELD_IS_EDITABLE(field))
	    {
	      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_MODIFY is not bound, but field \"%s\" is editable"), st_handler_field_get_label(field));
	      return FALSE;
	    }
	}
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FREE))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FREE is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_CATEGORY_NEW))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_CATEGORY_NEW is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_CATEGORY_FREE))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_CATEGORY_FREE is not bound"));
      return FALSE;
    }
  /*
   * If handler->priv->stock_categories is NULL, we'll add the main
   * category in st_handler_complete().
   */
  if (handler->priv->stock_categories && ! st_handler_get_stock_category(handler, ST_CATEGORY_BAG_MAIN))
    {
      g_set_error(err, 0, 0, _("main category is missing"));
      return FALSE;
    }
  
  return TRUE;
}

gboolean
st_handler_refresh (STHandler *handler,
		    STCategoryBag *category_bag,
		    GNode **categories,
		    STStreamStore **stream_store,
		    GError **err)
{
  gboolean status;

  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(category_bag), FALSE);
  g_return_val_if_fail(categories != NULL, FALSE);
  g_return_val_if_fail(stream_store != NULL, FALSE);

  if (st_handler_event_is_bound(handler, ST_HANDLER_EVENT_REFRESH_MULTIPLE))
    {
      GHashTable *stream_lists;

      status = st_handler_refresh_multiple(handler,
					   categories,
					   &stream_lists,
					   err);
      if (status)
	{
	  g_hash_table_foreach(stream_lists, st_handler_refresh_multiple_cb, handler);
	  g_hash_table_destroy(stream_lists);

	  *stream_store = st_handler_get_streams(handler, ST_CATEGORY(category_bag)->name);
	}
    }
  else
    {
      GSList *streams;
      
      status = st_handler_refresh_single(handler,
					 category_bag,
					 categories,
					 &streams,
					 err);
      if (status)
	{
	  *stream_store = st_stream_store_new(handler);
	  st_stream_store_append_list(*stream_store, streams);
	  sg_objects_free(streams);
	  
	  st_handler_set_streams(handler, ST_CATEGORY(category_bag)->name, *stream_store);
	}
    }

  return status;
}

static void
st_handler_refresh_multiple_cb (gpointer key,
				gpointer value,
				gpointer user_data)
{
  const char *category_name = key;
  GSList *streams = value;
  STHandler *handler = user_data;
  STStreamStore *stream_store;

  stream_store = st_stream_store_new(handler);
  st_stream_store_append_list(stream_store, streams);

  st_handler_set_streams(handler, category_name, stream_store);
  g_object_unref(stream_store);
}

static gboolean
st_handler_refresh_single (STHandler *handler,
			   STCategoryBag *category_bag,
			   GNode **category_bags,
			   GSList **stream_bags,
			   GError **err)
{
  gboolean status;
  GNode *categories = NULL;
  GList *streams = NULL;

  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(category_bag), FALSE);
  g_return_val_if_fail(category_bags != NULL, FALSE);
  g_return_val_if_fail(stream_bags != NULL, FALSE);

  status = st_handler_event_refresh(handler,
				    ST_CATEGORY(category_bag),
				    &categories,
				    &streams,
				    err);
  
  if (status)
    {
      *category_bags = st_handler_categories_to_bags(handler, categories);
      *stream_bags = st_handler_streams_to_bags(handler, streams);

      if (categories) g_node_destroy(categories);
      g_list_free(streams);
    }
  else
    {
      st_handler_categories_free(handler, categories);
      st_handler_streams_free(handler, streams);
    }
  
  return status;
}

static gboolean
st_handler_refresh_multiple (STHandler *handler,
			     GNode **category_bags,
			     GHashTable **stream_bag_lists,
			     GError **err)
{
  gboolean status;
  GNode *categories = NULL;
  GHashTable *stream_lists = NULL;

  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);
  g_return_val_if_fail(category_bags != NULL, FALSE);
  g_return_val_if_fail(stream_bag_lists != NULL, FALSE);

  status = st_handler_event_refresh_multiple(handler,
					     &categories,
					     &stream_lists,
					     err);

  if (status)
    {
      *category_bags = st_handler_categories_to_bags(handler, categories);
      *stream_bag_lists = st_handler_stream_lists_to_bags(handler, stream_lists);

      if (categories) g_node_destroy(categories);
      g_hash_table_destroy(stream_lists);
    }
  else
    {
      st_handler_categories_free(handler, categories);
      st_handler_stream_lists_free(handler, stream_lists);
    }

  return status;
}

static GNode *
st_handler_categories_to_bags (STHandler *handler, GNode *categories)
{
  GNode *copy;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  if (categories)
    {
      copy = g_node_copy(categories);
      g_node_traverse(copy,
		      G_PRE_ORDER,
		      G_TRAVERSE_ALL,
		      -1,
		      st_handler_categories_to_bags_cb,
		      handler);
    }
  else
    copy = g_node_new(NULL);

  return copy;
}

static gboolean
st_handler_categories_to_bags_cb (GNode *node, gpointer data)
{
  if (node->data)
    {
      STHandler *handler = data;
      STCategoryBag *bag;

      bag = st_category_bag_new_from_category(handler, node->data);
      st_handler_restore_category_flags(handler, bag);

      node->data = bag;
    }

  return FALSE;			/* continue */
}

static void
st_handler_categories_free (STHandler *handler, GNode *categories)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  if (categories)
    {
      g_node_traverse(categories,
		      G_PRE_ORDER,
		      G_TRAVERSE_ALL,
		      -1,
		      st_handler_categories_free_cb,
		      handler);
      g_node_destroy(categories);
    }
}

static gboolean
st_handler_categories_free_cb (GNode *node, gpointer data)
{
  if (node->data)
    {
      STHandler *handler = data;
      st_handler_event_category_free(handler, node->data);
    }

  return FALSE;			/* continue */
}

static GSList *
st_handler_streams_to_bags (STHandler *handler, GList *streams)
{
  GSList *bags = NULL;
  GList *l;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  SG_LIST_FOREACH(l, streams)
    {
      STStreamBag *bag;

      bag = st_stream_bag_new_from_stream(handler, l->data);
      bags = g_slist_append(bags, bag);
    }

  return bags;
}

static void
st_handler_streams_free (STHandler *handler, GList *streams)
{
  GList *l;

  g_return_if_fail(ST_IS_HANDLER(handler));

  SG_LIST_FOREACH(l, streams)
    st_handler_event_stream_free(handler, l->data);
  g_list_free(streams);
}

static GHashTable *
st_handler_stream_lists_to_bags (STHandler *handler, GHashTable *stream_lists)
{
  StreamListsToBagsInfo info;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  info.handler = handler;
  info.stream_bag_lists = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify) sg_objects_free);

  if (stream_lists)
    g_hash_table_foreach(stream_lists, st_handler_stream_lists_to_bags_cb, &info);
  
  return info.stream_bag_lists;
}

static void
st_handler_stream_lists_to_bags_cb (gpointer key,
				    gpointer value,
				    gpointer user_data)
{
  const char *category_name = key;
  GList *streams = value;
  GSList *stream_bags;
  StreamListsToBagsInfo *info = user_data;

  stream_bags = st_handler_streams_to_bags(info->handler, streams);
  g_list_free(streams);

  g_hash_table_insert(info->stream_bag_lists, g_strdup(category_name), stream_bags);
}

static void
st_handler_stream_lists_free (STHandler *handler, GHashTable *stream_lists)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  
  if (stream_lists)
    {
      g_hash_table_foreach(stream_lists, st_handler_stream_lists_free_cb, handler);
      g_hash_table_destroy(stream_lists);
    }
}

static void
st_handler_stream_lists_free_cb (gpointer key,
				 gpointer value,
				 gpointer user_data)
{
  GList *streams = value;
  STHandler *handler = user_data;

  st_handler_streams_free(handler, streams);
}

STCategoryStore *
st_handler_get_categories (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);
  g_return_val_if_fail(handler->priv->categories != NULL, NULL);

  g_object_ref(handler->priv->categories);
  return handler->priv->categories;
}

void
st_handler_set_streams (STHandler *handler,
			const char *category_name,
			STStreamStore *streams)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(category_name != NULL);
  g_return_if_fail(ST_IS_STREAM_STORE(streams));

  g_object_ref(streams);

  g_mutex_lock(handler->priv->streams_mutex);
  g_hash_table_insert(handler->priv->streams,
		      g_strdup(category_name),
		      streams);
  g_mutex_unlock(handler->priv->streams_mutex);
}

STStreamStore *
st_handler_get_streams (STHandler *handler, const char *category_name)
{
  STStreamStore *store;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);
  g_return_val_if_fail(category_name != NULL, NULL);

  g_mutex_lock(handler->priv->streams_mutex);
  store = g_hash_table_lookup(handler->priv->streams, category_name);
  if (store)
    g_object_ref(store);
  g_mutex_unlock(handler->priv->streams_mutex);

  return store;
}

STCategoryBag *
st_handler_get_stock_category (STHandler *handler, const char *name)
{
  GetStockCategoryInfo info = { name, NULL };

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);
  g_return_val_if_fail(name != NULL, NULL);

  g_node_traverse(handler->priv->stock_categories,
		  G_PRE_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  st_handler_get_stock_category_cb,
		  &info);

  return info.category_bag;
}

static gboolean
st_handler_get_stock_category_cb (GNode *node, gpointer data)
{
  if (node->data)
    {
      GetStockCategoryInfo *info = data;
      STCategoryBag *bag = node->data;
      
      if (ST_CATEGORY_BAG_IS(bag, info->name))
	{
	  info->category_bag = bag;
	  return TRUE;		/* abort */
	}
    }

  return FALSE;			/* continue */
}

/*
 * If no category is selected, select the __main category (if there is
 * no __main category, abort).
 */
void
st_handler_select_category (STHandler *handler)
{
  STCategoryBag *category_bag = NULL;

  g_return_if_fail(ST_IS_HANDLER(handler));

  category_bag = st_handler_get_selected_category(handler);
  if (category_bag)
    g_object_unref(category_bag); /* we're done */
  else
    {
      category_bag = st_handler_get_stock_category(handler, ST_CATEGORY_BAG_MAIN);

      g_assert(category_bag != NULL);
      st_category_bag_set_flags(category_bag, ST_CATEGORY_BAG_SELECTED);
    }
}

void
st_handler_add_remarkable_category (STHandler *handler,
				    STCategoryBag *category_bag)
{
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(ST_IS_CATEGORY_BAG(category_bag));

  g_mutex_lock(handler->priv->remarkable_categories_mutex);
  if (! g_slist_find(handler->priv->remarkable_categories, category_bag))
    {
      g_object_ref(category_bag);
      handler->priv->remarkable_categories = g_slist_append(handler->priv->remarkable_categories, category_bag);
    }
  g_mutex_unlock(handler->priv->remarkable_categories_mutex);
}

void
st_handler_remove_remarkable_category (STHandler *handler,
				       STCategoryBag *category_bag)
{
  GSList *elem;

  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(ST_IS_CATEGORY_BAG(category_bag));

  g_mutex_lock(handler->priv->remarkable_categories_mutex);
  elem = g_slist_find(handler->priv->remarkable_categories, category_bag);
  if (elem)
    {
      g_object_unref(elem->data);
      handler->priv->remarkable_categories = g_slist_delete_link(handler->priv->remarkable_categories, elem);
    }
  g_mutex_unlock(handler->priv->remarkable_categories_mutex);
}

STCategoryBag *
st_handler_get_selected_category (STHandler *handler)
{
  GSList *l;
  STCategoryBag *selected_category = NULL;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  g_mutex_lock(handler->priv->remarkable_categories_mutex);
  SG_LIST_FOREACH(l, handler->priv->remarkable_categories)
    if (st_category_bag_get_flags(l->data) & ST_CATEGORY_BAG_SELECTED)
      {
	g_object_ref(l->data);
	selected_category = l->data;
	break;
      }
  g_mutex_unlock(handler->priv->remarkable_categories_mutex);
  
  return selected_category;
}

static void
st_handler_restore_category_flags (STHandler *handler, STCategoryBag *bag)
{
  GSList *l;
  STCategoryBag *match = NULL;

  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  g_mutex_lock(handler->priv->remarkable_categories_mutex);

  SG_LIST_FOREACH(l, handler->priv->remarkable_categories)
    {
      STCategoryBag *remarkable_bag = l->data;
	
      if (! strcmp(ST_CATEGORY(bag)->name, ST_CATEGORY(remarkable_bag)->name))
	{
	  g_object_ref(remarkable_bag);
	  match = remarkable_bag;
	  break;
	}
    }

  g_mutex_unlock(handler->priv->remarkable_categories_mutex);

  if (match && match != bag)
    {
      st_category_bag_unset_flags(bag, ~0); /* reset flags */
      st_category_bag_set_flags(bag, st_category_bag_get_flags(match));

      st_category_bag_unset_flags(match, ~0); /* reset flags */
      g_object_unref(match);
    }
}

gboolean
st_handler_event_is_bound (STHandler *handler, STHandlerEvent event)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);
  g_return_val_if_fail(event >= 0 && event < ST_HANDLER_N_EVENTS, FALSE);

  return handler->priv->callbacks[event].cb != NULL;
}

const char *
st_handler_get_name (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->name;
}

const char *
st_handler_get_label (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->label;
}

const char *
st_handler_get_copyright (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->copyright;
}

const char *
st_handler_get_description (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->description;
}

const char *
st_handler_get_home (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->home;
}

GdkPixbuf *
st_handler_get_pixbuf (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  if (handler->priv->pixbuf)
    g_object_ref(handler->priv->pixbuf);
  
  return handler->priv->pixbuf;
}

unsigned int
st_handler_get_flags (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), 0);

  return handler->priv->flags;
}

void
st_handler_set_paned_position (STHandler *handler, int paned_position)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  handler->priv->paned_position = paned_position;
}

int
st_handler_get_paned_position (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), -1);

  return handler->priv->paned_position;
}

GSList *
st_handler_get_fields (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return handler->priv->fields;
}

int
st_handler_count_fields (STHandler *handler, unsigned int flags)
{
  int count = 0;
  GSList *l;

  SG_LIST_FOREACH(l, handler->priv->fields)
    if (st_handler_field_get_flags(l->data) & flags)
      count++;

  return count;
}

void
st_handler_set_fields_sort_index (STHandler *handler, int sort_index)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  handler->priv->fields_sort_index = sort_index;
}

int
st_handler_get_fields_sort_index (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), -1);

  return handler->priv->fields_sort_index;
}

void
st_handler_set_fields_sort_order (STHandler *handler, int sort_order)
{
  g_return_if_fail(ST_IS_HANDLER(handler));

  handler->priv->fields_sort_order = sort_order;
}

int
st_handler_get_fields_sort_order (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), -1);

  return handler->priv->fields_sort_order;
}

/*** events ******************************************************************/

gboolean
st_handler_event_refresh (STHandler *handler,
			  STCategory *category,
			  GNode **categories,
			  GList **streams,
			  GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_REFRESH];
  STHandlerRefreshCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(category, categories, streams, callback->data, err);
}

gboolean
st_handler_event_refresh_multiple (STHandler *handler,
				   GNode **categories,
				   GHashTable **streams,
				   GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_REFRESH_MULTIPLE];
  STHandlerRefreshMultipleCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(categories, streams, callback->data, err);
}

STCategory *
st_handler_event_category_new (STHandler *handler)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_CATEGORY_NEW];
  STCategoryNewCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, NULL);
  return cb(callback->data);
}

void
st_handler_event_category_free (STHandler *handler, STCategory *category)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_CATEGORY_FREE];
  STCategoryFreeCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(category, callback->data);
}

STStream *
st_handler_event_stream_new (STHandler *handler)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_NEW];
  STStreamNewCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, NULL);
  return cb(callback->data);
}

void
st_handler_event_stream_field_get (STHandler *handler,
				   STStream *stream,
				   STHandlerField *field,
				   GValue *value)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_FIELD_GET];
  STStreamFieldGetCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(stream, field, value, callback->data);
}

void
st_handler_event_stream_field_set (STHandler *handler,
				   STStream *stream,
				   STHandlerField *field,
				   const GValue *value)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_FIELD_SET];
  STStreamFieldSetCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(stream, field, value, callback->data);
}

void
st_handler_event_stream_stock_field_get (STHandler *handler,
					 STStream *stream,
					 STHandlerStockField stock_field,
					 GValue *value)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET];
  STStreamStockFieldGetCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(stream, stock_field, value, callback->data);
}

gboolean
st_handler_event_stream_modify (STHandler *handler,
				STStream *stream,
				GSList *fields,
				GSList *values,
				GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_MODIFY];
  STStreamModifyCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, fields, values, callback->data, err);
}

gboolean
st_handler_event_stream_delete (STHandler *handler,
				STStream *stream,
				GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_DELETE];
  STStreamDeleteCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, callback->data, err);
}

void
st_handler_event_stream_free (STHandler *handler, STStream *stream)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_FREE];
  STStreamFreeCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(stream, callback->data);
}

gboolean
st_handler_event_stream_resolve (STHandler *handler,
				 STStream *stream,
				 GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_RESOLVE];
  STStreamResolveCallback *cb = callback->cb;
  
  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, callback->data, err);
}

gboolean
st_handler_event_stream_tune_in (STHandler *handler,
				 STStream *stream,
				 GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_TUNE_IN];
  STStreamTuneInCallback *cb = callback->cb;
  
  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, callback->data, err);
}

gboolean
st_handler_event_stream_tune_in_multiple (STHandler *handler,
					  GSList *streams,
					  GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE];
  STStreamTuneInMultipleCallback *cb = callback->cb;
  
  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(streams, callback->data, err);
}

gboolean
st_handler_event_stream_record (STHandler *handler,
				STStream *stream,
				GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_RECORD];
  STStreamRecordCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, callback->data, err);
}

gboolean
st_handler_event_stream_browse (STHandler *handler,
				STStream *stream,
				GError **err)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_STREAM_BROWSE];
  STStreamBrowseCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, FALSE);
  return cb(stream, callback->data, err);
}

gpointer
st_handler_event_thread_begin (STHandler *handler)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_THREAD_BEGIN];
  STThreadBeginCallback *cb = callback->cb;

  g_return_val_if_fail(cb != NULL, NULL);
  return cb(callback->data);
}

void
st_handler_event_thread_end (STHandler *handler, gpointer thread_data)
{
  STHandlerCallback *callback = &handler->priv->callbacks[ST_HANDLER_EVENT_THREAD_END];
  STThreadEndCallback *cb = callback->cb;

  g_return_if_fail(cb != NULL);
  cb(thread_data, callback->data);
}
