/*
 * $Id: st-stream-view.c,v 1.132 2004/03/19 20:00:21 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <gdk/gdkkeysyms.h>
#include "gettext.h"
#include "sgtk-handlers.h"
#include "sg-util.h"
#include "st-handler.h"
#include "st-handler-field.h"
#include "st-stream-api.h"
#include "st-stream-view.h"
#include "st-stream-menu-items.h"
#include "st-stock.h"
#include "st-shell.h"
#include "st-stream-bag.h"
#include "st-stream-store.h"

#define N_VISIBLE_STOCK_COLUMNS		1

/*** type definitions ********************************************************/

enum {
  SELECTION_CHANGED,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_HANDLER
};
    
struct _STStreamViewPrivate
{
  STHandler		*handler;
  STStreamMenuItems	*menu_items;
  unsigned int		selection_changed_hid;
};
  
/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;
static unsigned int stream_view_signals[LAST_SIGNAL] = { 0 };

/*** function declarations ***************************************************/

static void st_stream_view_init		(STStreamView		*view);
static void st_stream_view_class_init	(STStreamViewClass	*class);
static void st_stream_view_finalize	(GObject		*object);
static void st_stream_view_unrealize	(GtkWidget		*widget);

static GObject *st_stream_view_constructor (GType type,
					    unsigned int n_construct_properties,
					    GObjectConstructParam *construct_params);
static void st_stream_view_set_property (GObject	*object,
					 unsigned int	prop_id,
					 const GValue	*value,
					 GParamSpec	*pspec);

static void st_stream_view_construct_columns_menu	(STStreamView *view);
static void st_stream_view_construct_columns		(STStreamView *view);
static void st_stream_view_column_add_field	(GtkTreeViewColumn *column,
						 STHandlerField    *field,
						 int               column_id);
static void st_stream_view_update_columns_menu		(STStreamView *view);

static void st_stream_view_construct_column_menu  (STStreamView       *view,
						   GtkTreeViewColumn  *column);

static void st_stream_view_column_visible_toggled_h (GtkCheckMenuItem *item,
						     gpointer	      data);
static void st_stream_view_column_hide_activate_h	(GtkMenuItem  *item,
							 gpointer     data);
static void st_stream_view_column_show_all_activate_h	(GtkMenuItem  *item,
							 gpointer     data);
static void st_stream_view_selection_changed_h	(GtkTreeSelection     *selection,
						 gpointer	      user_data);

static gboolean st_stream_view_get_first_selection (STStreamView      *view,
						    GtkTreeIter       *iter);
     
/*** implementation **********************************************************/

GType
st_stream_view_get_type (void)
{
  static GType stream_view_type = 0;
  
  if (! stream_view_type)
    {
      static const GTypeInfo stream_view_info = {
	sizeof(STStreamViewClass),
	NULL,
	NULL,
	(GClassInitFunc) st_stream_view_class_init,
	NULL,
	NULL,
	sizeof(STStreamView),
	0,
	(GInstanceInitFunc) st_stream_view_init,
      };
      
      stream_view_type = g_type_register_static(GTK_TYPE_TREE_VIEW,
						"STStreamView",
						&stream_view_info,
						0);
    }

  return stream_view_type;
}

static void
st_stream_view_class_init (STStreamViewClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_stream_view_finalize;
  object_class->constructor = st_stream_view_constructor;
  object_class->set_property = st_stream_view_set_property;

  widget_class->unrealize = st_stream_view_unrealize;

  g_object_class_install_property(object_class,
				  PROP_HANDLER,
				  g_param_spec_pointer("handler",
						       _("Handler"),
						       _("The view's STHandler object"),
						       G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  stream_view_signals[SELECTION_CHANGED] = g_signal_new("selection-changed",
							ST_TYPE_STREAM_VIEW,
							G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET(STStreamViewClass, selection_changed),
							NULL,
							NULL,
							g_cclosure_marshal_VOID__VOID,
							G_TYPE_NONE,
							0);
}

static void
st_stream_view_init (STStreamView *view)
{
  view->priv = g_new0(STStreamViewPrivate, 1);
  view->columns_menu = NULL;
}

static void
st_stream_view_finalize (GObject *object)
{
  STStreamView *view = ST_STREAM_VIEW(object);

  g_object_unref(view->columns_menu);

  if (view->priv->menu_items)
    st_stream_menu_items_free(view->priv->menu_items);
  g_free(view->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_stream_view_unrealize (GtkWidget *widget)
{
  STStreamView *view = ST_STREAM_VIEW(widget);
  GtkTreeViewColumn *column;
  int i = N_VISIBLE_STOCK_COLUMNS;

  /* save the column widths */
  while ((column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i++)))
    {
      STHandlerField *field = g_object_get_data(G_OBJECT(column), "field");
      st_handler_field_set_width(field, gtk_tree_view_column_get_width(column));
    }

  GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}

static GObject *
st_stream_view_constructor (GType type,
			    unsigned int n_construct_properties,
			    GObjectConstructParam *construct_params)
{
  GObject *object; 
  STStreamView *view;
  GtkTreeSelection *selection;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);
  view = ST_STREAM_VIEW(object);

  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(view), TRUE);
  gtk_tree_view_set_search_equal_func(GTK_TREE_VIEW(view),
				      st_stream_store_search_equal_func,
				      NULL,
				      NULL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

  gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
  view->priv->selection_changed_hid = g_signal_connect(G_OBJECT(selection),
						       "changed",
						       G_CALLBACK(st_stream_view_selection_changed_h),
						       view);

  st_stream_view_construct_columns(view);

  /* the following should be called after the columns have been created */
  st_stream_view_construct_columns_menu(view);
  st_stream_view_update_columns_menu(view);

  return object;
}

static void
st_stream_view_set_property (GObject *object,
			     unsigned int prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
  STStreamView *view;

  view = ST_STREAM_VIEW(object);

  switch (prop_id)
    {
    case PROP_HANDLER:
      view->priv->handler = g_value_get_pointer(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_stream_view_construct_columns_menu (STStreamView *view)
{
  GtkTreeViewColumn *column;
  int i = N_VISIBLE_STOCK_COLUMNS;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  
  view->columns_menu = gtk_menu_new();

  g_object_ref(view->columns_menu);
  gtk_object_sink(GTK_OBJECT(view->columns_menu));

  while ((column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i++)))
    {
      STHandlerField *field = g_object_get_data(G_OBJECT(column), "field");
      GtkWidget *item;
      unsigned int hid;

      item = gtk_check_menu_item_new_with_label(st_handler_field_get_label(field));
      gtk_widget_show(item);

      gtk_menu_shell_append(GTK_MENU_SHELL(view->columns_menu), item);

      hid = g_signal_connect(G_OBJECT(item),
			     "toggled",
			     G_CALLBACK(st_stream_view_column_visible_toggled_h),
			     NULL);

      g_object_set_data(G_OBJECT(item), "hid", GINT_TO_POINTER(hid));
      g_object_set_data(G_OBJECT(item), "field", field);
      g_object_set_data(G_OBJECT(item), "column", column);
    }
}

static void
st_stream_view_column_visible_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  STHandlerField *field = g_object_get_data(G_OBJECT(item), "field");
  GtkTreeViewColumn *column = g_object_get_data(G_OBJECT(item), "column");
  gboolean visible;

  visible = gtk_check_menu_item_get_active(item);
  st_handler_field_set_user_visible(field, visible);
  gtk_tree_view_column_set_visible(column, visible);
}

static void
st_stream_view_update_columns_menu (STStreamView *view)
{
  GList *l;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(GTK_IS_MENU_SHELL(view->columns_menu));

  SG_LIST_FOREACH(l, GTK_MENU_SHELL(view->columns_menu)->children)
    {
      GtkWidget *item = l->data;
      STHandlerField *field = g_object_get_data(G_OBJECT(item), "field");
      unsigned int hid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(item), "hid"));

      g_signal_handler_block(item, hid);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), st_handler_field_get_user_visible(field));
      g_signal_handler_unblock(item, hid);
    }
}

static void
st_stream_view_construct_columns (STStreamView *view)
{
  GSList *l;
  int vi = ST_STREAM_STORE_N_STOCK_COLUMNS; /* visible iterator */
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GSList *pending_fields = NULL;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  renderer = gtk_cell_renderer_pixbuf_new();
  column = gtk_tree_view_column_new_with_attributes(NULL, renderer,
						    "stock-id", ST_STREAM_STORE_COLUMN_STOCK_ID,
						    "pixbuf", ST_STREAM_STORE_COLUMN_PIXBUF,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  SG_LIST_FOREACH(l, st_handler_get_fields(view->priv->handler))
    {
      STHandlerField *field = l->data;

      if (ST_HANDLER_FIELD_IS_VISIBLE(field))
	{
	  if (ST_HANDLER_FIELD_HAS_DEDICATED_COLUMN(field))
	    {
	      GSList *m;
	      int pi = vi - g_slist_length(pending_fields); /* pending iterator */
	      int width;
      
	      column = gtk_tree_view_column_new();

	      SG_LIST_FOREACH(m, pending_fields)
		st_stream_view_column_add_field(column, m->data, pi++);
	      g_slist_free(pending_fields);
	      pending_fields = NULL;

	      st_stream_view_column_add_field(column, field, vi);

	      gtk_tree_view_column_set_title(column, st_handler_field_get_label(field));
	      gtk_tree_view_column_set_visible(column, st_handler_field_get_user_visible(field));
	      gtk_tree_view_column_set_sort_column_id(column, vi);
	      gtk_tree_view_column_set_resizable(column, TRUE);
	      gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);

	      g_object_set_data(G_OBJECT(column), "field", field);

	      width = st_handler_field_get_width(field);
	      /* width == 0 only if loaded from an old config file */
	      gtk_tree_view_column_set_fixed_width(column, width > 0 ? width : ST_HANDLER_FIELD_DEFAULT_WIDTH);
      
	      gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
	      st_stream_view_construct_column_menu(view, column);
	    }
	  else
	    pending_fields = g_slist_append(pending_fields, field);

	  vi++;
	}
    }

  g_slist_free(pending_fields);
  pending_fields = NULL;
}

static void
st_stream_view_column_add_field (GtkTreeViewColumn *column,
				 STHandlerField *field,
				 int column_id)
{
  GType type;
  GtkCellRenderer *renderer;
  const char *property;
  gboolean expand;

  g_return_if_fail(GTK_IS_TREE_VIEW_COLUMN(column));
  g_return_if_fail(field != NULL);

  type = st_handler_field_get_type(field);
  if (type == G_TYPE_BOOLEAN)
    {
      renderer = gtk_cell_renderer_toggle_new();
      property = "active";
      expand = FALSE;
    }
  else if (type == GDK_TYPE_PIXBUF)
    {
      renderer = gtk_cell_renderer_pixbuf_new();
      property = "pixbuf";
      expand = FALSE;
    }
  else
    {
      renderer = gtk_cell_renderer_text_new();
      property = "text";
      expand = TRUE;
    }

  gtk_tree_view_column_pack_start(column, renderer, expand);
  gtk_tree_view_column_set_attributes(column, renderer, property, column_id, NULL);
}

static void
st_stream_view_construct_column_menu (STStreamView *view,
				      GtkTreeViewColumn *column)
{
  GtkWidget *menu;

  GtkWidget *hide;
  GtkWidget *image;
  GtkWidget *separator;
  GtkWidget *show_all;
  
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(GTK_IS_TREE_VIEW_COLUMN(column));
  g_return_if_fail(GTK_IS_WIDGET(column->button));

  menu = gtk_menu_new();

  hide = gtk_image_menu_item_new_with_mnemonic(_("_Hide This Column"));
  image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(hide), image);
  
  separator = gtk_separator_menu_item_new();
  show_all = gtk_menu_item_new_with_mnemonic(_("_Show All Columns"));

  gtk_widget_show(hide);
  gtk_widget_show(image);
  gtk_widget_show(separator);
  gtk_widget_show(show_all);
  
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), hide);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), separator);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), show_all);

  g_signal_connect(G_OBJECT(hide),
		   "activate",
		   G_CALLBACK(st_stream_view_column_hide_activate_h),
		   column);
  g_signal_connect(G_OBJECT(show_all),
		   "activate",
		   G_CALLBACK(st_stream_view_column_show_all_activate_h),
		   view);

  g_signal_connect_swapped(G_OBJECT(column->button),
			   "popup-menu",
			   G_CALLBACK(sgtk_popup_menu_h),
			   G_OBJECT(menu));
  g_signal_connect_swapped(G_OBJECT(column->button),
			   "button-press-event",
			   G_CALLBACK(sgtk_popup_menu_mouse_h),
			   G_OBJECT(menu));
}

static void
st_stream_view_column_hide_activate_h (GtkMenuItem *item, gpointer data)
{
  GtkTreeViewColumn *column = data;
  STHandlerField *field = g_object_get_data(G_OBJECT(column), "field");

  st_handler_field_set_user_visible(field, FALSE);
  gtk_tree_view_column_set_visible(column, FALSE);

  st_stream_view_update_columns_menu(ST_STREAM_VIEW(column->tree_view));
}

static void
st_stream_view_column_show_all_activate_h (GtkMenuItem *item, gpointer data)
{
  STStreamView *view = data;
  GtkTreeViewColumn *column;
  int i = N_VISIBLE_STOCK_COLUMNS;

  while ((column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i++)))
    {
      STHandlerField *field = g_object_get_data(G_OBJECT(column), "field");
	  
      st_handler_field_set_user_visible(field, TRUE);
      gtk_tree_view_column_set_visible(column, TRUE);
    }

  st_stream_view_update_columns_menu(view);
}

static void
st_stream_view_selection_changed_h (GtkTreeSelection *selection,
				    gpointer user_data)
{
  STStreamView *view = user_data;

  g_signal_emit(view, stream_view_signals[SELECTION_CHANGED], 0);
}

GtkWidget *
st_stream_view_new (STHandler *handler)
{
  return g_object_new(ST_TYPE_STREAM_VIEW,
		      "handler", handler,
		      NULL);
}

void
st_stream_view_update_sensitivity (STStreamView *view)
{
  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  if (view->priv->menu_items)
    st_stream_menu_items_update_sensitivity(view->priv->menu_items);
}

void
st_stream_view_select_stream (STStreamView *view, STStreamBag *bag)
{
  GtkTreeSelection *selection;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(ST_IS_STREAM_BAG(bag));

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

  g_signal_handler_block(selection, view->priv->selection_changed_hid);
  gtk_tree_selection_unselect_all(selection);
  gtk_tree_selection_select_iter(selection, &bag->iter);
  g_signal_handler_unblock(selection, view->priv->selection_changed_hid);
}

void
st_stream_view_present_stream (STStreamView *view, STStreamBag *bag)
{
  GtkTreeModel *model;
  GtkTreePath *path;
  
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(ST_IS_STREAM_BAG(bag));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  path = gtk_tree_model_get_path(model, &bag->iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(view), path, NULL, FALSE, 0, 0);
  gtk_tree_path_free(path);
}

gboolean
st_stream_view_can_select_previous (STStreamView *view)
{
  GtkTreeIter iter;
  gboolean can = FALSE;

  g_return_val_if_fail(ST_IS_STREAM_VIEW(view), FALSE);

  if (st_stream_view_get_first_selection(view, &iter))
    {
      GtkTreeModel *model;
      GtkTreePath *path;
      
      model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
      path = gtk_tree_model_get_path(model, &iter);
      can = gtk_tree_path_prev(path);
      gtk_tree_path_free(path);
    }

  return can;
}

void
st_stream_view_select_previous (STStreamView *view)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *path;
  gboolean status;
  
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(st_stream_view_can_select_previous(view));
  
  status = st_stream_view_get_first_selection(view, &iter);
  g_return_if_fail(status == TRUE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  path = gtk_tree_model_get_path(model, &iter);
  status = gtk_tree_path_prev(path);
  g_return_if_fail(status == TRUE);

  status = gtk_tree_model_get_iter(model, &iter, path);
  g_return_if_fail(status == TRUE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  gtk_tree_selection_unselect_all(selection);
  gtk_tree_selection_select_iter(selection, &iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(view), path, NULL, FALSE, 0, 0);

  gtk_tree_path_free(path);
}

gboolean
st_stream_view_can_select_next (STStreamView *view)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_return_val_if_fail(ST_IS_STREAM_VIEW(view), FALSE);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

  return st_stream_view_get_first_selection(view, &iter)
    && gtk_tree_model_iter_next(model, &iter);
}

void
st_stream_view_select_next (STStreamView *view)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *path;
  gboolean status;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(st_stream_view_can_select_next(view));
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

  status = st_stream_view_get_first_selection(view, &iter);
  g_return_if_fail(status == TRUE);
  status = gtk_tree_model_iter_next(model, &iter);
  g_return_if_fail(status == TRUE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  gtk_tree_selection_unselect_all(selection);
  gtk_tree_selection_select_iter(selection, &iter);

  path = gtk_tree_model_get_path(model, &iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(view), path, NULL, FALSE, 0, 0);
  gtk_tree_path_free(path);
}

static gboolean
st_stream_view_get_first_selection (STStreamView *view, GtkTreeIter *iter)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;

  g_return_val_if_fail(ST_IS_STREAM_VIEW(view), FALSE);
  g_return_val_if_fail(iter != NULL, FALSE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

  if (model)
    {
      GList *paths;
      
      paths = gtk_tree_selection_get_selected_rows(selection, NULL);
      if (paths)
	{
	  GList *l;
	  GtkTreePath *path = paths->data;
	  gboolean status;
	  
	  status = gtk_tree_model_get_iter(model, iter, path);
	  g_return_val_if_fail(status == TRUE, FALSE);

	  SG_LIST_FOREACH(l, paths)
	    gtk_tree_path_free(l->data);
	  g_list_free(paths);

	  return TRUE;
	}
    }

  return FALSE;
}

/*
 * This owns ITEMS. The caller must not free it.
 */
void
st_stream_view_set_menu_items (STStreamView *view, STStreamMenuItems *items)
{
  GtkWidget *menu;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(items != NULL);

  view->priv->menu_items = items;
  
  menu = gtk_menu_new();
  st_stream_menu_items_append_to_shell(view->priv->menu_items, GTK_MENU_SHELL(menu));

  g_signal_connect_swapped(G_OBJECT(view),
			   "popup-menu",
			   G_CALLBACK(sgtk_popup_menu_h),
			   G_OBJECT(menu));
  g_signal_connect_swapped(G_OBJECT(view),
			   "button_press_event",
			   G_CALLBACK(sgtk_popup_menu_mouse_h),
			   G_OBJECT(menu));
}

void
st_stream_view_set_store (STStreamView *view, STStreamStore *store)
{
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_signal_emit(view, stream_view_signals[SELECTION_CHANGED], 0);
}

GSList *
st_stream_view_get_selected_streams (STStreamView *view)
{
  GtkTreeModel *model;
  GSList *selected_streams = NULL;

  g_return_val_if_fail(ST_IS_STREAM_VIEW(view), NULL);

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  if (model)
    {
      GtkTreeSelection *selection;
      GList *paths;
      
      selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
      paths = gtk_tree_selection_get_selected_rows(selection, NULL);
      if (paths)
	{
	  GList *l;

	  SG_LIST_FOREACH(l, paths)
	    {
	      GtkTreePath *path = l->data;
	      GtkTreeIter iter;
	      gboolean status;
	      STStreamBag *bag;
	  
	      status = gtk_tree_model_get_iter(model, &iter, path);
	      g_return_val_if_fail(status == TRUE, NULL);

	      gtk_tree_model_get(model, &iter, ST_STREAM_STORE_COLUMN_BAG, &bag, -1);
	      selected_streams = g_slist_append(selected_streams, bag);

	      gtk_tree_path_free(path);
	    }

	  g_list_free(paths);
	}
    }

  return selected_streams;
}

gboolean
st_stream_view_has_selected_streams (STStreamView *view)
{
  GSList *selected_streams;
  gboolean has;

  g_return_val_if_fail(ST_IS_STREAM_VIEW(view), NULL);

  selected_streams = st_stream_view_get_selected_streams(view);
  has = selected_streams != NULL;
  sg_objects_free(selected_streams);

  return has;
}
