/*
 * Copyright (C) 2011 Sansar Choinyambuu
 * HSR Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "tcg_pts_attr_req_file_meta.h"

#include <pa_tnc/pa_tnc_msg.h>
#include <bio/bio_writer.h>
#include <bio/bio_reader.h>
#include <debug.h>

typedef struct private_tcg_pts_attr_req_file_meta_t private_tcg_pts_attr_req_file_meta_t;

/**
 * Request File Metadata
 * see section 3.17.1 of PTS Protocol: Binding to TNC IF-M Specification
 * 
 *					   1				   2				   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |	 Flags	 |   Delimiter	|		  Reserved					|
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  ~	   Fully Qualified File Pathname (Variable Length)			~
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

#define PTS_REQ_FILE_META_SIZE			4
#define PTS_REQ_FILE_META_RESERVED		0x00
#define PTS_REQ_FILE_META_NO_FLAGS		0x00

#define DIRECTORY_CONTENTS_FLAG			(1<<7)

/**
 * Private data of an tcg_pts_attr_req_file_meta_t object.
 */
struct private_tcg_pts_attr_req_file_meta_t {

	/**
	 * Public members of tcg_pts_attr_req_file_meta_t
	 */
	tcg_pts_attr_req_file_meta_t public;

	/**
	 * Attribute vendor ID
	 */
	pen_t vendor_id;

	/**
	 * Attribute type
	 */
	u_int32_t type;

	/**
	 * Attribute value
	 */
	chunk_t value;
	
	/**
	 * Noskip flag
	 */
	bool noskip_flag;
	
	/**
	 * Directory Contents flag
	 */
	bool directory_flag;
	
	/**
	 * UTF8 Encoding of Delimiter Character
	 */
	u_int8_t delimiter;
	
	/**
	 * Fully Qualified File Pathname
	 */
	char *pathname;

	/**
	 * Reference count
	 */
	refcount_t ref;
};

METHOD(pa_tnc_attr_t, get_vendor_id, pen_t,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->vendor_id;
}

METHOD(pa_tnc_attr_t, get_type, u_int32_t,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->type;
}

METHOD(pa_tnc_attr_t, get_value, chunk_t,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->value;
}

METHOD(pa_tnc_attr_t, get_noskip_flag, bool,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->noskip_flag;
}

METHOD(pa_tnc_attr_t, set_noskip_flag,void,
	private_tcg_pts_attr_req_file_meta_t *this, bool noskip)
{
	this->noskip_flag = noskip;
}

METHOD(pa_tnc_attr_t, build, void,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	u_int8_t flags = PTS_REQ_FILE_META_NO_FLAGS;
	chunk_t pathname;
	bio_writer_t *writer;
	
	if (this->directory_flag)
	{
		flags |= DIRECTORY_CONTENTS_FLAG;
	}
	pathname = chunk_create(this->pathname, strlen(this->pathname));

	writer = bio_writer_create(PTS_REQ_FILE_META_SIZE);
	writer->write_uint8 (writer, flags);
	writer->write_uint8 (writer, this->delimiter);
	writer->write_uint16(writer, PTS_REQ_FILE_META_RESERVED);
	
	writer->write_data  (writer, pathname);
	this->value = chunk_clone(writer->get_buf(writer));
	writer->destroy(writer);
}

METHOD(pa_tnc_attr_t, process, status_t,
	private_tcg_pts_attr_req_file_meta_t *this, u_int32_t *offset)
{
	bio_reader_t *reader;
	u_int8_t flags;
	u_int16_t reserved;
	chunk_t pathname;
	
	if (this->value.len < PTS_REQ_FILE_META_SIZE)
	{
		DBG1(DBG_TNC, "insufficient data for Request File Metadata");
		*offset = 0;
		return FAILED;
	}

	reader = bio_reader_create(this->value);
	reader->read_uint8 (reader, &flags);
	reader->read_uint8 (reader, &this->delimiter);
	reader->read_uint16(reader, &reserved);
	
	reader->read_data  (reader, reader->remaining(reader), &pathname);

	this->directory_flag = (flags & DIRECTORY_CONTENTS_FLAG) !=
							PTS_REQ_FILE_META_NO_FLAGS;

	this->pathname = malloc(pathname.len + 1);
	memcpy(this->pathname, pathname.ptr, pathname.len);
	this->pathname[pathname.len] = '\0';

	reader->destroy(reader);
	return SUCCESS;
}

METHOD(pa_tnc_attr_t, destroy, void,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	if (ref_put(&this->ref))
	{
		free(this->pathname);
		free(this->value.ptr);
		free(this);
	}
}

METHOD(pa_tnc_attr_t, get_ref, pa_tnc_attr_t*,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	ref_get(&this->ref);
	return &this->public.pa_tnc_attribute;
}

METHOD(tcg_pts_attr_req_file_meta_t, get_directory_flag, bool,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->directory_flag;
}

METHOD(tcg_pts_attr_req_file_meta_t, get_delimiter, u_int8_t,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->delimiter;
}

METHOD(tcg_pts_attr_req_file_meta_t, get_pathname, char*,
	private_tcg_pts_attr_req_file_meta_t *this)
{
	return this->pathname;
}

/**
 * Described in header.
 */
pa_tnc_attr_t *tcg_pts_attr_req_file_meta_create(bool directory_flag,
												 u_int8_t delimiter,
												 char *pathname)
{
	private_tcg_pts_attr_req_file_meta_t *this;

	INIT(this,
		.public = {
			.pa_tnc_attribute = {
				.get_vendor_id = _get_vendor_id,
				.get_type = _get_type,
				.get_value = _get_value,
				.get_noskip_flag = _get_noskip_flag,
				.set_noskip_flag = _set_noskip_flag,
				.build = _build,
				.process = _process,
				.get_ref = _get_ref,
				.destroy = _destroy,
			},
			.get_directory_flag = _get_directory_flag,
			.get_delimiter = _get_delimiter,
			.get_pathname = _get_pathname,
		},
		.vendor_id = PEN_TCG,
		.type = TCG_PTS_REQ_FILE_META,
		.directory_flag = directory_flag,
		.delimiter = delimiter,
		.pathname = strdup(pathname),
		.ref = 1,
	);

	return &this->public.pa_tnc_attribute;
}


/**
 * Described in header.
 */
pa_tnc_attr_t *tcg_pts_attr_req_file_meta_create_from_data(chunk_t data)
{
	private_tcg_pts_attr_req_file_meta_t *this;

	INIT(this,
		.public = {
			.pa_tnc_attribute = {
				.get_vendor_id = _get_vendor_id,
				.get_type = _get_type,
				.get_value = _get_value,
				.get_noskip_flag = _get_noskip_flag,
				.set_noskip_flag = _set_noskip_flag,
				.build = _build,
				.process = _process,
				.get_ref = _get_ref,
				.destroy = _destroy,
			},
			.get_directory_flag = _get_directory_flag,
			.get_delimiter = _get_delimiter,
			.get_pathname = _get_pathname,
		},
		.vendor_id = PEN_TCG,
		.type = TCG_PTS_REQ_FILE_META,
		.value = chunk_clone(data),
		.ref = 1,
	);

	return &this->public.pa_tnc_attribute;
}
