/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

//sc
#include "ConfigManager.h"
#include "Project.h"
#include "Repository.h"
#include "ColorId.h"
#include "config.cpp"
#include "sublib/ColorStorage.h"
#include "sublib/Utility.h"
#include "sublib/config/ConfigFile.h"
#include "sublib/config/ConfigData.h"

// qt
#include <qstring.h>
#include <qregexp.h>
#include <qapplication.h>
#include <qheader.h>
#include <qdir.h>

///////////////////////////////////////////////////////////////////////////////

// project regexp
static QRegExp reGuid( "project\\.\\d+\\.guid$" );
static QRegExp reName( "project\\.\\d+\\.name$" );
static QRegExp reSort( "project\\.\\d+\\.sortpos$" );
static QRegExp reTrunkName( "\\S+\\.trunk\\.name$" );
static QRegExp reTrunkUrl( "\\S+\\.trunk\\.url$" );
static QRegExp reTrunkSort( "\\S+\\.trunk\\.sortpos$" );
static QRegExp reBranchesName( "\\S+\\.branches\\.name$" );
static QRegExp reBranchesUrl( "\\S+\\.branches\\.url$" );
static QRegExp reBranchesSort( "\\S+\\.branches\\.sortpos$" );
static QRegExp reTagsName( "\\S+\\.tags\\.name$" );
static QRegExp reTagsUrl( "\\S+\\.tags\\.url$" );
static QRegExp reTagsSort( "\\S+\\.tags\\.sortpos$" );
static QRegExp reWcCurrent( "\\S+\\.wc\\.current$" );
static QRegExp reItemName( "\\S+\\.name$" );
static QRegExp reItemPath( "\\S+\\.path$" );
static QRegExp reItemSort( "\\S+\\.sortpos$" );
static QRegExp reItemOpts( "\\S+\\.options$" );
static QRegExp reItemUrl( "\\S+\\.url$" );

// option strings
static sc::String OptCmdForceId("option.command.force");
static sc::String OptCmdRecursiveId("option.command.recursive");
static sc::String OptStatusUpdatesId("option.status.updates");
static sc::String OptStatusIgnoredId("option.status.ignored");
static sc::String OptStatusAllId("option.status.all");
static sc::String OptStatusRecursiveId("option.status.recursive");

// subversion strings
static sc::String SvnAspDotNetHack("subversion.asp_dot_net_hack");

// < 0.15, obsolete
static sc::String OptForceId("option.force");
static sc::String OptRecursiveId("option.recursive");
static sc::String OptWcUpdatesId("option.wc.updates");
static sc::String OptWcIgnoredId("option.wc.ignored");
static sc::String OptWcAllId("option.wc.all");

///////////////////////////////////////////////////////////////////////////////

ConfigManager::ConfigManager()
: _dbgDump(false), _dbgLog(false)
{
  _file = new ConfigFile( sc::String("sc.ini"), defaultConf, sizeof(defaultConf) );
  _cfg = new ConfigData( _file );
}

ConfigManager::~ConfigManager()
{
  delete _cfg;
  delete _file;
}

void ConfigManager::load()
{
  _cfg->load();

  // init colors
  Colors colors;
  getColors(colors);
  ColorStorage::setColors(colors);
}

void ConfigManager::save()
{
  _cfg->save();
}

// Settings
void ConfigManager::getProjects( Projects& projects )
{
  const int maxProject = 64;
  const int maxItems   = 64;

  for( int cnt = 0; cnt < maxProject; cnt++ )
  {
    QString key = QString("project.%1.").arg(cnt);

    ConfigValues values;
    _cfg->getValues( sc::String(key), values );

    if( values.size() == 0 )
    {
      return;
    }

    // collect the project values
    sc::String prjName;
    Uuid       prjId    = Uuid::createEmpty();
    long       prjIndex = cnt;
    long       prjSort  = 0;

    sc::String sTrunkName;
    sc::String sTrunkUrl;
    long       trunkSort = 0;
    sc::String sBranchesName;
    sc::String sBranchesUrl;
    long       branchesSort = 0;
    sc::String sTagsName;
    sc::String sTagsUrl;
    long       tagsSort = 0;
    long wcCurId = 0;

    for( ConfigValues::iterator it = values.begin(); it != values.end(); it++ )
    {
      ConfigValue& cval = *it;
      QString      key  = QString::fromUtf8(cval.getKey());
      sc::String   val  = cval.getStringValue();

      if( reGuid.exactMatch(key) )
      {
        prjId = Uuid(val);
      }
      else if( reName.exactMatch(key) )
      {
        prjName = val;
      }
      else if( reSort.exactMatch(key) )
      {
        prjSort = cval.getNumericValue();
      }
      else if( reTrunkName.exactMatch(key) )
      {
        sTrunkName = val;
      }
      else if( reTrunkUrl.exactMatch(key) )
      {
        sTrunkUrl = val;
      }
      else if( reTrunkSort.exactMatch(key) )
      {
        trunkSort = cval.getNumericValue();
      }
      else if( reBranchesName.exactMatch(key) )
      {
        sBranchesName = val;
      }
      else if( reBranchesUrl.exactMatch(key) )
      {
        sBranchesUrl = val;
      }
      else if( reBranchesSort.exactMatch(key) )
      {
        branchesSort = cval.getNumericValue();
      }
      else if( reTagsName.exactMatch(key) )
      {
        sTagsName = val;
      }
      else if( reTagsUrl.exactMatch(key) )
      {
        sTagsUrl = val;
      }
      else if( reTagsSort.exactMatch(key) )
      {
        tagsSort = cval.getNumericValue();
      }
      else if( reWcCurrent.exactMatch(key) )
      {
        wcCurId = cval.getNumericValue();
      }
    }

    Project* project = new Project( prjId, prjName );
    projects.push_back(project);

    project->setIndex(prjIndex);
    project->setSortPos(prjSort);

    Project::Item trunk;
    trunk.setName(sTrunkName);
    trunk.setSource(sTrunkUrl);
    trunk.setSortPos(trunkSort);
    //trunk.setRevision( svn::RevisionPtr(new svn::Revision(svn::RevHead)) );
    project->setTrunk(trunk);

    Project::Item branches;
    branches.setName(sBranchesName);
    branches.setSource(sBranchesUrl);
    branches.setSortPos(branchesSort);
    //branches.setRevision( svn::RevisionPtr(new svn::Revision(svn::RevHead)) );
    project->setBranches(branches);

    Project::Item tags;
    tags.setName(sTagsName);
    tags.setSource(sTagsUrl);
    tags.setSortPos(tagsSort);
    //tags.setRevision( svn::RevisionPtr(new svn::Revision(svn::RevHead)) );
    project->setTags(tags);

    // get working copies
    // note: the wc numbers are not used as the ProjectItem ids.
    for( int cntwc = 0; cntwc < maxItems; cntwc++ )
    {
      QString keywc = QString("project.%1.wc.%2").arg(cnt).arg(cntwc);

      ConfigValues valueswc;
      _cfg->getValues( sc::String(keywc), values, valueswc, true );

      if( valueswc.size() == 0 )
      {
        break;
      }

      Project::Item item = project->createWorkingCopyItem();

      // map config current id to real current id as returned
      // from createWorkingCopyItem() above.
      if( cntwc == wcCurId )
      {
        wcCurId = item.getId();
      }

      for( ConfigValues::iterator it = valueswc.begin(); it != valueswc.end(); it++ )
      {
        ConfigValue& cval = *it;
        QString      key  = QString::fromUtf8(cval.getKey());
        sc::String   val  = cval.getStringValue();

        if( reItemName.exactMatch(key) )
        {
          item.setName(val);
        }
        else if( reItemPath.exactMatch(key) )
        {
          item.setSource(val);
        }
        else if( reItemSort.exactMatch(key) )
        {
          item.setSortPos(cval.getNumericValue());
        }
        else if( reItemOpts.exactMatch(key) )
        {
          item.setOptions(cval.getOptionValue());
        }
      }

      project->setItem(item);
    }

    project->setCurWorkingCopy( wcCurId );

    // get repositories
    // note: the rp numbers are not used as the ProjectItem ids.
    for( int cntrp = 0; cntrp < maxItems; cntrp++ )
    {
      QString keyrp = QString("project.%1.rp.%2").arg(cnt).arg(cntrp);

      ConfigValues valuesrp;
      _cfg->getValues( sc::String(keyrp), values, valuesrp, true );

      if( valuesrp.size() == 0 )
      {
        break;
      }

      Project::Item item = project->createRepositoryItem();

      for( ConfigValues::iterator it = valuesrp.begin(); it != valuesrp.end(); it++ )
      {
        ConfigValue& cval = *it;
        QString      key  = QString::fromUtf8(cval.getKey());
        sc::String   val  = cval.getStringValue();

        if( reItemName.exactMatch(key) )
        {
          item.setName(val);
        }
        else if( reItemUrl.exactMatch(key) )
        {
          item.setSource(val);
        }
        else if( reItemSort.exactMatch(key) )
        {
          item.setSortPos(cval.getNumericValue());
        }
      }

      project->setItem(item);
    }
  }
}

void ConfigManager::setProject( const Project* prj )
{
  ConfigValues values;
  QString      keypref = QString("project.%1").arg( prj->getIndex() );
  QString      key;

  key = keypref + ".name";
  values.push_back( ConfigValue( sc::String(key.utf8()), prj->getName() ) );
  key = keypref + ".guid";
  values.push_back( ConfigValue( sc::String(key.utf8()), sc::String(prj->getId().toString()) ) );
  key = keypref + ".sortpos";
  values.push_back( ConfigValue( sc::String(key.utf8()), prj->getSortPos()) );

  {
    const Project::Item& item = prj->getTrunkItem();
    key = keypref + ".trunk.name";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getName() ) );
    key = keypref + ".trunk.url";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSource() ) );
    key = keypref + ".trunk.sortpos";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSortPos() ) );
  }
  {
    const Project::Item& item = prj->getBranchesItem();
    key = keypref + ".branches.name";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getName() ) );
    key = keypref + ".branches.url";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSource() ) );
    key = keypref + ".branches.sortpos";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSortPos() ) );
  }
  {
    const Project::Item& item = prj->getTagsItem();
    key = keypref + ".tags.name";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getName() ) );
    key = keypref + ".tags.url";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSource() ) );
    key = keypref + ".tags.sortpos";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSortPos() ) );
  }

  Project::Items items;
  prj->getItems(items);

  long wcCnt = 0;
  long wcCur = 0;
  for( Project::Items::iterator it = items.begin(); it != items.end(); it++ )
  {
    const Project::Item& item = *it;

    if( ! item.isWorkingCopy() )
    {
      continue;
    }

    if( item.getId() == prj->getCurWorkingCopyId() )
    {
      wcCur = wcCnt;
    }

    QString wckeypref = keypref + QString(".wc.%1").arg( wcCnt );
    wcCnt++;

    key = wckeypref + ".name";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getName() ) );
    key = wckeypref + ".path";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSource() ) );
    key = wckeypref + ".sortpos";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSortPos() ) );
    key = wckeypref + ".options";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getOptions() ) );
  }

  key = keypref + ".wc.current";
  values.push_back( ConfigValue( sc::String(key.utf8()), wcCur ) );

  long rpCnt = 0;
  for( Project::Items::iterator it = items.begin(); it != items.end(); it++ )
  {
    const Project::Item& item = *it;

    if( ! item.isRepository() )
    {
      continue;
    }

    QString wckeypref = keypref + QString(".rp.%1").arg( rpCnt );
    rpCnt++;

    key = wckeypref + ".name";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getName() ) );
    key = wckeypref + ".url";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSource() ) );
    key = wckeypref + ".sortpos";
    values.push_back( ConfigValue( sc::String(key.utf8()), item.getSortPos() ) );
  }

  _cfg->setValues( sc::String(keypref.utf8()), values ); 
}

void ConfigManager::removeProject( const Project* prj )
{
  QString key = QString("project.%1").arg( prj->getIndex() );
  _cfg->delValues( sc::String(key) );
}


QFont ConfigManager::getGeneralFont()
{
  ConfigValue fontString = _cfg->getValue( sc::String("font.general") );

  if( fontString.isNull() )
  {
    return qApp->font();
  }

  QFont font;
  font.fromString( (const char*)fontString.getStringValue() );
  return font;
}

void ConfigManager::setGeneralFont( const QFont& font )
{
  if( QApplication::font() != font )
  {
    ConfigValues values;
    values.push_back( ConfigValue(sc::String("font.general"),sc::String(font.toString())) );
    _cfg->setValues( sc::String("font.general"), values );

    QApplication::setFont( font, true );
  }
}

QFont ConfigManager::getEditorFont()
{
  ConfigValue fontString = _cfg->getValue( sc::String("font.editor") );

  if( fontString.isNull() )
  {
#if _WIN32
    return QFont( "Courier New" , 9, QFont::Normal, false );
#elif _MACOSX
    return QFont( "Andale Mono" , 12, QFont::Normal, false );
#else // Unix/Linux
    return QFont( "Courier New" , 9, QFont::Normal, false );
#endif
  }

  QFont font;
  font.fromString( (const char*)fontString.getStringValue() );
  return font;
}

void ConfigManager::setEditorFont( const QFont& font )
{
  ConfigValues values;
  values.push_back( ConfigValue(sc::String("font.editor"),sc::String(font.toString())) );
  _cfg->setValues( sc::String("font.editor"), values );

  //doesn't work as expected, this will change the font of any _new_ QTextEdit
  //to the given font but not the currently open ones.
  //QApplication::setFont( font, true, "QTextEdit" );
}

Color ConfigManager::getColor( long id, const sc::String& name, const QColor& def,
  const sc::String& desc )
{
  ConfigValue color = _cfg->getValue( name );
  if( color.isNull() )
  {
    return Color( id, (const char*)name, def, (const char*)desc );
  }

  return Color( id, QString::fromUtf8(color.getKey()),QColor(color.getStringValue()),
    (const char*)desc );
}

void ConfigManager::getColors( Colors& colors )
{
  // misc colors
  colors.push_back( getColor( ColorFaded,       sc::String("color.faded"),      QColor(150,150,150),
    sc::String("-") ) );

  // status colors
  colors.push_back( getColor( ColorNone,        sc::String("color.none"),        QColor(230,230,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorUnversioned, sc::String("color.unversioned"), QColor(240,240,240),
    sc::String("-") ) );

  colors.push_back( getColor( ColorNormal,      sc::String("color.normal"),      QColor(255,255,255),
    sc::String("-") ) );

  colors.push_back( getColor( ColorAdded,       sc::String("color.added"),       QColor(200,200,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorMissing,     sc::String("color.missing"),     QColor(230,230,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorDeleted,     sc::String("color.deleted"),     QColor(230,200,200),
    sc::String("-") ) );

  colors.push_back( getColor( ColorReplaced,    sc::String("color.replaced"),    QColor(230,230,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorModified,    sc::String("color.modified"),    QColor(200,230,200),
    sc::String("-") ) );

  colors.push_back( getColor( ColorMerged,      sc::String("color.merged"),      QColor(230,230,000),
    sc::String("-") ) );

  colors.push_back( getColor( ColorConflicted,  sc::String("color.conflicted"),  QColor(240,200,200),
    sc::String("-") ) );

  colors.push_back( getColor( ColorIgnored,     sc::String("color.ignored"),     QColor(230,230,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorObstructed,  sc::String("color.obstructed"),  QColor(230,230,230),
    sc::String("-") ) );

  colors.push_back( getColor( ColorExternal,    sc::String("color.external"),    QColor(230,230,230),
    sc::String("-") ) );
  
  colors.push_back( getColor( ColorIncomplete,  sc::String("color.incomplete"),  QColor(230,230,230),
    sc::String("-") ) );
  
  colors.push_back( getColor( ColorFallback,    sc::String("color.fallback"),    QColor(230,230,230),
    sc::String("-") ) );
}

void ConfigManager::setColors( const Colors& colors )
{
  ConfigValues values;

  for( Colors::const_iterator it = colors.begin(); it != colors.end(); it++ )
  {
    const Color& c = *it;
    values.push_back( 
      ConfigValue( sc::String(c._name.utf8()),sc::String(c._color.name())) );
  }

  _cfg->setValues( sc::String("color."), values );

  ColorStorage::setColors(colors);

  // refresh the whole app
  qApp->mainWidget()->hide();
  qApp->mainWidget()->show();
}

QRect ConfigManager::getWindowRect( const QString& key, const QRect& def )
{
  QString     rectName  = QString("layout.%1").arg(key);
  ConfigValue rectValue = _cfg->getValue( sc::String(rectName) );
  
  if( rectValue.isNull() )
  {
    return def;
  }
  
  QString rectVal = (const char*)rectValue.getStringValue();
  QRegExp rectExp("(\\d+),(\\d+),(\\d+),(\\d+)");
  rectExp.exactMatch(rectVal);

  QRect rect;
  rect.setLeft( rectExp.cap(1).toInt() );
  rect.setTop( rectExp.cap(2).toInt() );
  rect.setRight( rectExp.cap(3).toInt() );
  rect.setBottom( rectExp.cap(4).toInt() );
  return rect;
}

void ConfigManager::setWindowRect( const QString& key, const QRect& rect )
{
  QString rectName = QString("layout.%1").arg(key);
  QString rectVal  = QString("%1,%2,%3,%4")
    .arg(rect.left()).arg(rect.top()).arg(rect.right()).arg(rect.bottom());
  
  sc::String valkey = sc::String(rectName);
  ConfigValue value( valkey, sc::String(rectVal) );
  _cfg->setValue( value.getKey(), value );
}

void ConfigManager::getHeaderColumns( const QString& key, QHeader* header )
{
  QString     qkey  = QString("layout.%1").arg(key);
  ConfigValue value = _cfg->getValue( sc::String(qkey) );

  if( value.isNull() )
  {
    return;
  }
  
  QString qval = (const char*)value.getStringValue();

  int cnt = 0;
  while( QString section = qval.section( ',', cnt ) )
  {    
    QRegExp exp("(\\d+):(\\d+)");
    exp.exactMatch(section);
    
    header->moveSection( exp.cap(1).toInt(), cnt );
    header->resizeSection( exp.cap(1).toInt(), exp.cap(2).toInt() );
    cnt++;
  }
}

void ConfigManager::setHeaderColumns( const QString& key, const QHeader* header )
{
  QString qkey = QString("layout.%1").arg(key);
  QString sections;
  
  for( int i = 0; i < header->count(); i++ )
  {
    QString v;
    int s = header->mapToSection(i);
    v.setNum(s);
    sections += v;
    sections += ":";    
    v.setNum( header->sectionSize(s) );
    sections += v;
    sections += ",";    
  }
  // strip trailing ","
  sections.truncate( sections.length()-1 );

  sc::String valkey = sc::String(qkey);
  ConfigValue value( valkey, sc::String(sections) );
  _cfg->setValue( value.getKey(), value );
}

int ConfigManager::getPosition( const QString& key )
{
  QString qkey = QString("layout.%1").arg(key);
  ConfigValue value = _cfg->getValue( sc::String(qkey) );
  
  if( value.isNull() )
  {
    return -1;
  }
  
  return value.getNumericValue();
}

void ConfigManager::setPosition( const QString& key, int pos )
{
  QString qkey = QString("layout.%1").arg(key);
  sc::String valkey = sc::String(qkey);
  ConfigValue value(valkey, (long)pos);
  _cfg->setValue( value.getKey(), value );
}

sc::String ConfigManager::getBuiltinDiffCmd()
{
  QString    path = getMergePath();
  sc::String diff = sc::String(path.utf8());

  diff += " diff -L {llabel} -L {rlabel} {left} {right}";

  return diff;
}

sc::String ConfigManager::getDiffCmd()
{
  ConfigValue option = _cfg->getValue( sc::String("commands.diff") );

  if( option.isNull() )
  {
    return getBuiltinDiffCmd();
  }

  sc::String val = option.getStringValue();
  return val;
}

void ConfigManager::setDiffCmd( const sc::String& cmd )
{
  ConfigValue value(sc::String("commands.diff"),cmd);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );
}

sc::String ConfigManager::getBuiltinMergeCmd()
{
  QString    path  = getMergePath();
  sc::String merge = sc::String(path.utf8());

  merge += " merge {base} {mine} {theirs} {merged}";

  return merge;
}

sc::String ConfigManager::getMergeCmd()
{
  ConfigValue option = _cfg->getValue( sc::String("commands.merge") );

  if( option.isNull() )
  {
    return getBuiltinMergeCmd();
  }

  sc::String val = option.getStringValue();
  return val;
}

void ConfigManager::setMergeCmd( const sc::String& cmd )
{
  ConfigValue value(sc::String("commands.merge"),cmd);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );
}

void ConfigManager::getLogs( Logs& logs )
{
  sc::String path = _file->getConfigPath();
  QString qpath = QString::fromUtf8(path);

  QDir commit( qpath + "/CommitLogHistory" );
  if( ! commit.exists() )
  {
    commit.mkdir( commit.absPath() );
  }

  commit.setSorting( QDir::Time | QDir::Reversed );
  commit.setFilter( QDir::Files );

  const QFileInfoList* list = commit.entryInfoList();
  QFileInfoListIterator it( *list );
  QFileInfo *fi;

  while( (fi = it.current()) != 0 )
  {
    QFile file( fi->absFilePath() );
    QTextStream stream( &file );
    stream.setEncoding( QTextStream::UnicodeUTF8 );

    if( file.open(IO_ReadOnly) )
    {
      QString    qlog = stream.read();
      sc::String log( qlog.utf8() );

      logs.push_back(log);

      file.close();
    }

    ++it;
  }
}

void ConfigManager::addLog( const sc::String& log )
{
  sc::String path = _file->getConfigPath();
  QString qpath = QString::fromUtf8(path);

  QDir commit( qpath + "/CommitLogHistory" );
  commit.setSorting( QDir::Time | QDir::Reversed );
  commit.setFilter( QDir::Files );


  // if we already have the same message delete the older one
  const QFileInfoList* list = commit.entryInfoList();
  QFileInfoListIterator it( *list );
  QFileInfo *fi;

  while( (fi = it.current()) != 0 )
  {
    QFile file( fi->absFilePath() );
    QTextStream stream( &file );
    stream.setEncoding( QTextStream::UnicodeUTF8 );

    if( file.open(IO_ReadOnly) )
    {
      QString    qlog = stream.read();
      sc::String olog( qlog.utf8() );
      file.close();

      if( log == olog )
      {
        fi = it.current();
        QFile file( fi->absFilePath() );
        file.remove();

        commit.refresh();
        break;
      }
    }

    ++it;
  }


  QDateTime time = QDateTime::currentDateTime();

  QFile file( commit.absPath() + "/" + time.toString("yyyyMMddhhmmss") );

  if( file.open(IO_WriteOnly) )
  {
    QTextStream stream( &file );
    stream.setEncoding( QTextStream::UnicodeUTF8 );
    stream << QString::fromUtf8(log);
    file.close();
  }

  // if we have more than 10 logs, delete the oldest log.
  const QFileInfoList* listd = commit.entryInfoList();
  QFileInfoListIterator itd( *listd );
  QFileInfo *fid;

  if( listd->count() > 10 )
  {
    fid = itd.current();
    QFile file( fid->absFilePath() );
    file.remove();
  }
}

bool ConfigManager::getAspDotNetHack()
{
  ConfigValue option = _cfg->getValue( SvnAspDotNetHack );

  if( option.isNull() )
  {
    return false;
  }

  return option.getBoolValue();
}

void ConfigManager::setAspDotNetHack( bool enable )
{
  ConfigValue value(SvnAspDotNetHack,enable);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );
}

bool ConfigManager::getDump()
{
  return _dbgDump;
}

void ConfigManager::setDump(bool b)
{
  _dbgDump = b;
  setDumpOnException(b);
}

bool ConfigManager::getLog()
{
  return _dbgLog;
}

void ConfigManager::setLog(bool b)
{
  _dbgLog = b;
}

bool ConfigManager::getL10n()
{
  ConfigValue option = _cfg->getValue(DebugL10n);

  if( option.isNull() )
  {
    return true;
  }

  return option.getBoolValue();
}

void ConfigManager::setL10n(bool b)
{
  ConfigValue value(DebugL10n,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );
}

bool ConfigManager::getOptCommandForce() const
{
  ConfigValue option = _cfg->getValue( OptCmdForceId );

  if( option.isNull() )
  {
    // < 0.15 option available?
    option = _cfg->getValue( OptForceId );
  }

  if( option.isNull() )
  {
    return false;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptCommandForce( bool b )
{
  ConfigValue value(OptCmdForceId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );

  // < remove < 0.15 option
  _cfg->delValues( OptForceId );
}

bool ConfigManager::getOptCommandRecursive() const
{
  ConfigValue option = _cfg->getValue( OptCmdRecursiveId );

  if( option.isNull() )
  {
    return true;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptCommandRecursive( bool b )
{
  ConfigValue value(OptCmdRecursiveId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );
}

bool ConfigManager::getOptStatusUpdates() const
{
  ConfigValue option = _cfg->getValue( OptStatusUpdatesId );

  if( option.isNull() )
  {
    // < 0.15 option available?
    option = _cfg->getValue( OptWcUpdatesId );
  }

  if( option.isNull() )
  {
    return false;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptStatusUpdates( bool b )
{
  ConfigValue value(OptStatusUpdatesId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );

  // < remove < 0.15 option
  _cfg->delValues( OptWcUpdatesId );
}

bool ConfigManager::getOptStatusIgnored() const
{
  ConfigValue option = _cfg->getValue( OptStatusIgnoredId );

  if( option.isNull() )
  {
    // < 0.15 option available?
    option = _cfg->getValue( OptWcIgnoredId );
  }

  if( option.isNull() )
  {
    return false;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptStatusIgnored( bool b )
{
  ConfigValue value(OptStatusIgnoredId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );

  // < remove < 0.15 option
  _cfg->delValues( OptWcIgnoredId );
}

bool ConfigManager::getOptStatusAll() const
{
  ConfigValue option = _cfg->getValue( OptStatusAllId );

  if( option.isNull() )
  {
    // < 0.15 option available?
    option = _cfg->getValue( OptWcAllId );
  }

  if( option.isNull() )
  {
    return false;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptStatusAll( bool b )
{
  ConfigValue value(OptStatusAllId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );

  // < remove < 0.15 option
  _cfg->delValues( OptWcIgnoredId );
}

bool ConfigManager::getOptStatusRecursive() const
{
  ConfigValue option = _cfg->getValue( OptStatusRecursiveId );

  if( option.isNull() )
  {
    // < 0.15 option available?
    option = _cfg->getValue( OptRecursiveId );
  }

  if( option.isNull() )
  {
    return true;
  }

  return option.getBoolValue();
}

void ConfigManager::setOptStatusRecursive( bool b )
{
  ConfigValue value(OptStatusRecursiveId,b);

  ConfigValues values;
  values.push_back( value );
  _cfg->setValues( value.getKey(), values );

  // < remove < 0.15 option
  _cfg->delValues( OptRecursiveId );
}
