/* ====================================================================
 * Copyright (c) 2003-2008, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "BookmarkProperties.h"
#include "Bookmark.h"
#include "ExternProvider.h"
#include "Project.h"
#include "Settings.h"
#include "sublib/Gui.h"
#include "sublib/Utility.h"
#include "sublib/ExternButton.h"
#include "sublib/settings/LayoutSettings.h"
#include "util/String.h"
#include "util/max.h"

// qt
#include <QtGui/QLayout>
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QFileDialog>
#include <QtGui/QToolTip>
#include <QtGui/QComboBox>
#include <QtGui/QCheckBox>
#include <QtGui/QGroupBox>
#include <QtGui/QDialogButtonBox>



BookmarkProperties::BookmarkProperties( Bookmark* bookmark, ExternProvider* p,
  QWidget* parent ) : super(parent, 0, true), _p(p), _bookmark(bookmark)
{
  setName( "BookmarkProperties" );
  setCaption( _q("subcommander:bookmark properties") );

  QVBoxLayout *vbl = new QVBoxLayout(this);
  vbl->setContentsMargins(5,5,5,5);
  {
    QGroupBox* gb = new QGroupBox(this);
    gb->setTitle( _q("properties: ") );
    gb->setFlat(true);
    vbl->addWidget(gb,0);
    {
      QGridLayout* gl = new QGridLayout(gb);
      int left, top, right, bottom;
      gl->getContentsMargins(&left,&top,&right,&bottom);
      gl->setContentsMargins(0,top,0,bottom);
      gl->setColStretch( 0, 0 );
      gl->setColStretch( 1, 4 );
      {
        QLabel*    tl = new QLabel( _q("bookmark type:"), this );
        QComboBox* tc = new QComboBox( false, this );
        tc->insertItem( getPixmap(), getType() );

        gl->addWidget( tl, 0, 0 );
        gl->addWidget( tc, 0, 1, 1, 2 );

        QLabel* ln = new QLabel( _name,   _q("&name"),    this );
        QLabel* ls = new QLabel( _source, _q("&location"), this );
        gl->addWidget( ln, 1, 0, 1, 1 );
        gl->addWidget( ls, 1, 1, 1, 2 );

        _name   = new QLineEdit(this); 
        _source = new QComboBox(this);
        _ex     = new ExternButton(this);
        _source->setEditable(true);
        _name->setFocus();

        gl->addWidget( _name,   2, 0 ); 
        gl->addWidget( _source, 2, 1 ); 
        gl->addWidget( _ex,     2, 2 );

        _update = new QCheckBox(this);
        _update->setText( _q("auto &update") );
        _update->setChecked( _bookmark->getAutoUpdate() );
        gl->addWidget( _update, 3, 0, 1, 3 );
        QToolTip::add( _update, _q("automatically run update on program startup") );

        if( ! _bookmark->isWorkingCopy() )
        {
          _update->hide();
        }

        connect( _name, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _source, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _ex, SIGNAL(clicked()), SLOT(selectExtern()) );

        _ex->setEnabled( _bookmark->isWorkingCopy() );
      }
    }

    QDialogButtonBox* db = new QDialogButtonBox(this);
    vbl->addWidget(db);
    {
      int minWidth = 0;
 
      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("&Ok") );
      _ok->setDefault(true);
      db->addButton( _ok, QDialogButtonBox::AcceptRole );
      minWidth = std::max( minWidth, _ok->sizeHint().width() );

      _ca = new QPushButton(this);
      _ca->setText( _q("&Cancel") );
      db->addButton( _ca, QDialogButtonBox::RejectRole );
      minWidth = std::max( minWidth, _ca->sizeHint().width() );
      
      _ok->setMinimumWidth(minWidth);
      _ca->setMinimumWidth(minWidth);

      connect( _ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( _ca, SIGNAL(clicked()), SLOT(reject()) );
    }
    

    Project::Items items;
    _bookmark->getProject()->getItems(items);

    for( Project::Items::iterator it = items.begin(); it != items.end(); it++ )
    {
      bool    wc   = (*it).isWorkingCopy();
      QString path = QString::fromUtf8((*it).getSource());

      if( ( _bookmark->isWorkingCopy() &&  wc) || (!_bookmark->isWorkingCopy() && !wc) )
      {
        _source->insertItem(path);
      }
    }

    if( ! _bookmark->getName().isEmpty() )
    {
      _name->setText( QString::fromUtf8(_bookmark->getName()) );
    }

    if( ! _bookmark->getSource().isEmpty() )
    {
      _source->setCurrentText( QString::fromUtf8(_bookmark->getSource()) );
    }
  }

  Settings s;
  resize( s.layout().getSize(name(),QSize()) );
  setMaximumHeight( sizeHint().height() );
}

BookmarkProperties::~BookmarkProperties()
{
  Settings s;
  s.layout().setSize( name(), size() );
}

QString BookmarkProperties::getName()
{
  return _name->text();
}

QString BookmarkProperties::getSource()
{
  return _source->currentText();
}

bool BookmarkProperties::getAutoUpdate()
{
  return _update->isChecked();
}

void BookmarkProperties::checkOk( const QString& text )
{
  QString name   = _name->text();
  QString source = _source->currentText();

  bool allowEmptySource = 
       _bookmark->isTrunk()
    || _bookmark->isBranches()
    || _bookmark->isTags();


  if( (! name.isEmpty() && ! source.isEmpty()) || (! name.isEmpty() && allowEmptySource) )
  {
    _ok->setEnabled(true);
  }
  else
  {
    _ok->setEnabled(false);
  }

  if( ! source.isEmpty() || _bookmark->isWorkingCopy() )
  {
    _ex->setEnabled(true);
  }
  else
  {
    _ex->setEnabled(false);
  }
}

void BookmarkProperties::selectExtern()
{
  sc::String res;

  if( _bookmark->isWorkingCopy() )
  {
    if( _p->selectPath( this, sc::String(_source->currentText().utf8()), res, ExternProvider::Dir ) )
    {
      _source->setCurrentText(QString::fromUtf8(res));
    }
  }
  else
  {
    if( _p->selectUrl( this, sc::String(_source->currentText().utf8()), res, ExternProvider::Dir ) )
    {
      _source->setCurrentText(QString::fromUtf8(res));
    }
  }
}

QPixmap BookmarkProperties::getPixmap()
{
  if( _bookmark->isWorkingCopy() )
  {
    return QPixmap(getIconDir() + "BookmarkWorkingCopy.png");
  }
  else // repository
  {
    return QPixmap(getIconDir() + "BookmarkRepository.png");
  }
}

QString BookmarkProperties::getType()
{
  if( _bookmark->isWorkingCopy() )
  {
    return _q("working copy");
  }
  else if( _bookmark->isTrunk() )
  {
    return _q("trunk");
  }
  else if( _bookmark->isBranches() )
  {
    return _q("branches");
  }
  else if( _bookmark->isTags() )
  {
    return _q("tags");
  }
  else if( _bookmark->isRepository() )
  {
    return _q("repository");
  }
  else
  {
    return "BookmarkProperties::getType: unknown bookmark type!";
  }
}
