/* ====================================================================
 * Copyright (c) 2007-2008  Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "WcViewTree.h"
#include "WcViewModel.h"
#include "WcViewTreeItemModel.h"
#include "WcViewTreeProxyModel.h"
#include "WcViewColumnTooltips.h"
#include "ListViewColumnTooltips.h"
#include "WcViewStatus.h"
#include "WcSelection.h"
#include "CursorSupport.h"
#include "Actions.h"
#include "Bookmark.h"
#include "DragInfo.h"
#include "DragDropMimeTypes.h"
#include "Settings.h"
#include "ListViewHeaderHandler.h"
#include "sublib/ActionStorage.h"
#include "sublib/settings/LayoutSettings.h"
#include "svn/WcStatus.h"
#include "svn/Path.h"

// qt
#include <QtCore/QTimer>
#include <QtGui/QAction>
#include <QtGui/QLineEdit>
#include <QtGui/QHeaderView>
#include <QtGui/QItemDelegate>
#include <QtGui/QDragMoveEvent>
#include <Qt3Support/Q3DragObject>
#include <Qt3Support/Q3PopupMenu>


static ListViewHeaderHandler _headerHandler;


/**
 * WcViewTree item delegate.
 */
class WcViewTreeItemDelegate : public QItemDelegate
{
  typedef QItemDelegate super;

public:
  WcViewTreeItemDelegate( WcViewTree* parent, WcViewViewState* state )
    : super(parent), _view(parent), _state(state)
  {
  }

  /** Creates a text editor with frame. */
  QWidget* createEditor( QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index ) const
  {
    QWidget* editor = super::createEditor(parent,option,index);
    editor->setStyleSheet( "border: 1px solid black" );
    return editor;
  }

  void setEditorData( QWidget* editor, const QModelIndex& index ) const
  {
    super::setEditorData(editor,index);
  }

  void setModelData( QWidget* editor, QAbstractItemModel* model, const QModelIndex& index ) const
  {
    QLineEdit* edit = static_cast<QLineEdit*>(editor);

    // not changed?
    if( edit->text() == index.data() )
      return;

    _view->renameItem(edit->text());
  }

  void paint( QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex &index ) const
  {
    super::paint(painter,option,index);
  }

  void drawDecoration( QPainter* painter, const QStyleOptionViewItem& option, const QRect& rect, const QPixmap& pixmap ) const
  {
    super::drawDecoration(painter,option,rect,pixmap);
  }

  QSize sizeHint( const QStyleOptionViewItem& option, const QModelIndex& index ) const
  {
#ifdef Q_WS_MAC
    return super::sizeHint(option,index);
#else 
    QSize result = super::sizeHint(option,index);
    result.setHeight( result.height() + 2 * _view->style()->pixelMetric(QStyle::PM_FocusFrameVMargin) );
    return result;
#endif // Q_WS_MAC
  }

private:
  WcViewTree*      _view;
  WcViewViewState* _state;
};



WcViewTree::WcViewTree( const sc::String& root, WcViewViewState* state, ActionStorage* as, QWidget* parent )
: super(parent), _state(state), _actions(as), _itemData(new WcViewStatusData()), _restoringState(false)
{
  //setVerticalScrollMode(ScrollPerPixel);
  //setHorizontalScrollMode(ScrollPerPixel);
  setTextElideMode(Qt::ElideMiddle);
  setAllColumnsShowFocus(true);
  setUniformRowHeights(true);  // said to improve performance
  setMouseTracking(true);
  setAnimated(false);

  setDragEnabled(true);
  setAcceptDrops(true);
  setDragDropMode(DragDrop);
  setDropIndicatorShown(true);

  setEditTriggers(SelectedClicked);
  setSelectionBehavior(QAbstractItemView::SelectRows);
  setSelectionMode(QAbstractItemView::ExtendedSelection);

  setItemDelegate( new WcViewTreeItemDelegate(this,_state) );

  setSortingEnabled(true);
  header()->setSortIndicator( 0, Qt::AscendingOrder );
  header()->setResizeMode( QHeaderView::ResizeToContents );
  header()->setStretchLastSection(true);

  _itemModel  = new WcViewTreeItemModel(root,_itemData);
  _proxyModel = new WcViewTreeProxyModel();
  _proxyModel->setSourceModel(_itemModel);
  _proxyModel->setCurrentPath( _state->getPath() );
  setModel( _proxyModel );

#if 0
  new ListViewColumnTooltips(this, WcViewColumnTooltips::getTreeTooltips() );
#endif

  connect( this, SIGNAL(expanded(const QModelIndex&)), this, SLOT(expandedItem(const QModelIndex&)) );
  connect( this, SIGNAL(collapsed(const QModelIndex&)), this, SLOT(collapsedItem(const QModelIndex&)) );
  connect( this, SIGNAL(itemExpanded(const QModelIndex&)), this, SLOT(storeState(const QModelIndex&)) );
  connect( this, SIGNAL(itemCollapsed(const QModelIndex&)), this, SLOT(storeState(const QModelIndex&)) );

  // setup menu
  _menu = new Q3PopupMenu(this);
  {
    QAction* action;

    action = _actions->getAction( ActionWcDiffBase );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcCommit );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcLog );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcLogGraph );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcBlame );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcAdd );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcRevert );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcRemove );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcUpdateRev );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcLock );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcUnlock );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcProperties );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcIgnore );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcEditConflict );
    action->addTo(_menu);
    action = _actions->getAction( ActionWcResolved );
    action->addTo(_menu);

    _menu->insertSeparator();

    action = _actions->getAction( ActionWcMkdir );
    action->addTo(_menu);

    action = _actions->getAction( ActionWcBranchTag );
    action->addTo(_menu);

    action = _actions->getAction( ActionWcExport );
    action->addTo(_menu);

    action = _actions->getAction( ActionWcCleanup );
    action->addTo(_menu);
  }

  Settings s;
  //s.layout().getHeaderColumns(name(),header());
  //_headerHandler.add(header());

  refreshType();
}

WcViewTree::~WcViewTree()
{
  //_headerHandler.del(header());
  delete _itemData;
}

void WcViewTree::refreshType()
{
  static int indent = indentation();

  setWaitCursor();

  bool flat = _state->isFlat();

  if(flat)
  {
    setIndentation(2);
    setStyleSheet(
      "QTreeView::branch:open { border-image: none; image: none;} "
      "QTreeView::branch:closed { border-image: none; image: none;}"
    );
  }
  else
  {
    setIndentation(indent);
    setStyleSheet("");
  }

  setItemsExpandable(!flat);
  setRootIsDecorated(!flat);

  _proxyModel->setFilterFlat(flat);
  _proxyModel->invalidate();
  
  if(flat)
    expandAll();
  else
    restoreState(_proxyModel->index(0,0));

  restoreCursor();
}

void WcViewTree::contextMenuEvent( QContextMenuEvent* e )
{
  _menu->exec(e->globalPos());
}

void WcViewTree::showEvent( QShowEvent* )
{
  svn::WcStatuss statuss;
  getSelection(statuss);
  
  WcSelection sel(statuss);
  emit selectionChanged(sel);

  updateMenu(sel);
}

void WcViewTree::mousePressEvent( QMouseEvent* e )
{
  super::mousePressEvent(e);
}

void WcViewTree::mouseMoveEvent( QMouseEvent* e )
{
  super::mouseMoveEvent(e);

  if( !(e->buttons() & Qt::LeftButton) )
    return;

  // start drag?
  if( ! dragStartable() )
    return;

  QMimeData* data = mimeData( selectedIndexes() );
  if( !data )
    return;

  // move mime data item model
  QDrag* drag = new QDrag(this);
  drag->setMimeData(data);

  drag->exec(Qt::MoveAction|Qt::CopyAction);
}

void WcViewTree::dragEnterEvent( QDragEnterEvent* e )
{
  super::dragEnterEvent(e);

  // we want to receive dragMoveEvents
  e->setAccepted(true);

}

void WcViewTree::dragMoveEvent( QDragMoveEvent* e )
{
  super::dragMoveEvent(e);
  e->setAccepted(false);

  //printf( "action: %s\n", e->proposedAction() == Qt::MoveAction ? "move" : "copy" );

  if( e->mimeData()->hasFormat(ScMimeTypeWcFilesItem) )
  {
    QModelIndex dropIndex = indexAt(e->pos());

    // no dir? Then dissallow drop.
    if( ! dropIndex.data(WcViewTreeItemModel::DirRole).asBool() )
      return;

    e->acceptProposedAction();
  }
}

void WcViewTree::dragLeaveEvent( QDragLeaveEvent* e )
{
  super::dragLeaveEvent(e);
}

void WcViewTree::dropEvent( QDropEvent* e )
{
  super::dropEvent(e);

  if( e->mimeData()->hasFormat(ScMimeTypeWcFilesItem) )
  {
    QModelIndex dropIndex = indexAt(e->pos());
    sc::String  target = dropIndex.data(WcViewTreeItemModel::NameRole).value<sc::String>();

    if( e->proposedAction() == Qt::CopyAction )
    {
      e->accept();
      emit copy(target);
    }
    else if( e->proposedAction() == Qt::MoveAction )
    {
      e->accept();
      emit move(target,false);
    }
  }
}

void WcViewTree::timerEvent( QTimerEvent* e )
{
  super::timerEvent(e);

  if( _state->isFlat() )
    return;

  super::autoExpand(e);
}

void WcViewTree::renameItem( const QString& text )
{
  emit move( svn::Path::getBaseName(sc::String(text.toUtf8())), true );
}

void WcViewTree::expandedItem( const QModelIndex& index )
{
  if( _state->isFlat() || _restoringState )
    return;

  emit itemExpanded(index);
}

void WcViewTree::collapsedItem( const QModelIndex& index )
{
  if( _state->isFlat() || _restoringState )
    return;

  emit itemCollapsed(index);
}

void WcViewTree::storeState( const QModelIndex& index )
{
  // if type is flat the item is always open, so remember _opened in tree mode only 

  if( _state->isFlat() )
    return;

  sc::String name = index.data( WcViewTreeItemModel::NameRole ).value<sc::String>();
  _state->setExpanded( name, isExpanded(index) );
}

void WcViewTree::updateOld( const sc::String& path, const svn::WcStatuss& statuss )
{
  _itemModel->remove(path);
}

void WcViewTree::updateNew( const sc::String& path, const svn::WcStatuss& statuss, bool deep )
{
  WcViewItems items;
  for( svn::WcStatuss::const_iterator it = statuss.begin(); it != statuss.end(); it++ )
  {
    items.push_back( WcViewItemPtr(new WcViewStatus(*it)) );
  }
  _itemModel->add(path,items,deep);


  for( svn::WcStatuss::const_iterator it = statuss.begin(); it != statuss.end(); it++ )
  {
    const sc::String& name = (*it)->getName();

    QModelIndex srcIndex = _itemModel->index(name);
    QModelIndex prxIndex = _proxyModel->mapFromSource(srcIndex);
    
    // on first insert, the index will be invalid
    if(!prxIndex.isValid())
      continue;

    // restore selection
    if(_state->isSelected(name))
    {
      selectionModel()->select(
        prxIndex, QItemSelectionModel::Rows | QItemSelectionModel::Select );
      setCurrentIndex(prxIndex);
      
      // we would like to scroll only if the item is not visible.
      // problem is to find out if it is not visible...
      scrollTo(prxIndex);
    }

    if( _state->isFlat() || _state->isExpanded(name) )
      expand(prxIndex);
  }
}

void WcViewTree::setChildsExpanded( const QModelIndex& index )
{
  if(_proxyModel->hasChildren(index))
  {
    int numRows = _proxyModel->rowCount(index);
    
    for( int row = 0; row < numRows; ++row )
    {
      QModelIndex idx = index.child(row,0);
            
      bool dir = idx.data(WcViewTreeItemModel::DirRole).asBool();
      if( dir && !isExpanded(idx) )
        expand(idx);
    }
  }
}

void WcViewTree::updateDeepStatus(const sc::String& path, const DeepStatus& status )
{
  if( _state->isFlat() && status.modified )
  {    
    QModelIndex srcIndex = _itemModel->index(path);
    QModelIndex prxIndex = _proxyModel->mapFromSource(srcIndex);
    
    if(prxIndex.isValid())
    {
      setChildsExpanded(prxIndex);

      if( !isExpanded(prxIndex) )
        expand(prxIndex);     
    }
    else
      ; //printf("updateDeepStatus - %s\n", (const char*)path );
  }
  
  bool changed = _itemModel->setDeepStatus( path, status.modified );

  if( _state->isFlat() && changed )
    _proxyModel->invalidate();
}

void WcViewTree::mouseDoubleClickEvent( QMouseEvent* e )
{
  QModelIndex clickIndex = indexAt(e->pos());

  if( ! clickIndex.isValid() )
    assert(false);
  
  bool dir = clickIndex.data(WcViewTreeItemModel::DirRole).asBool();
  if( !dir )
    return;

  // ignore clicks on ourself
  sc::String target = clickIndex.data(WcViewTreeItemModel::NameRole).value<sc::String>();
  if( target == _state->getPath() )
    return; 

  _state->addPath(target);
  setCurrentDir(target);
}

void WcViewTree::setCurrentDir( const sc::String& current )
{
  setWaitCursor();

  // set new root index.
  QModelIndex modelIndex = _itemModel->index(current);
  QModelIndex proxyIndex = _proxyModel->mapFromSource(modelIndex);
  QModelIndex parentIndex = proxyIndex.parent();

  setRootIndex( parentIndex );
  expand(proxyIndex);

  _proxyModel->setCurrentPath(current);
  _proxyModel->invalidate();

  if( _state->isFlat() )
  {
    expandAll();
  }
  else
  {
    proxyIndex = _proxyModel->mapFromSource(modelIndex);
    restoreState(proxyIndex);  
  }

  emit currentChanged(current);
  restoreCursor();
}

void WcViewTree::restoreState( const QModelIndex& index )
{
  _restoringState = true;
  restoreStates(index);
  _restoringState = false;
}

void WcViewTree::restoreStates( const QModelIndex& index )
{
  if(_proxyModel->hasChildren(index))
  {
    int numRows = _proxyModel->rowCount(index);
    
    for( int row = 0; row < numRows; ++row )
    {
      QModelIndex idx = index.child(row,0);

      sc::String name = idx.data(WcViewTreeItemModel::NameRole).value<sc::String>();
      bool dir = idx.data(WcViewTreeItemModel::DirRole).asBool();

      if( dir && _state->isExpanded(name) )
      {
        if(!isExpanded(idx))
        {
          expand(idx);
          restoreStates(idx);
        }
      }
      else if( dir && !_state->isExpanded(name) )
      {
        if(isExpanded(idx))
          collapse(idx);        
      }
    }
  }
}
  
void WcViewTree::refresh()
{
  _proxyModel->setFilterFlat( _state->isFlat() );
  _proxyModel->setFilterAll( _state->getViewAll() );
  _proxyModel->setFilterIgnored( _state->getViewIgnored() );
  _proxyModel->setFilterOutOfDate( _state->getViewUpdates() );

  _proxyModel->invalidate();

  if( _state->isFlat() )
    expandAll();
  else
    restoreState(_proxyModel->index(0,0));
}

void WcViewTree::selectionChanged( const QItemSelection& selected, const QItemSelection& deselected )
{
  svn::WcStatuss statuss;
  getSelection(statuss);
  
  WcSelection sel(statuss);
  emit selectionChanged(sel);

  _state->clearSelected();
  for( svn::WcStatuss::iterator it = statuss.begin(); it != statuss.end(); it++ )
  {
    const sc::String& name = (*it)->getName();
    _state->setSelected(name,true);
  }

  updateMenu(sel);

  // properly refresh selection changes..
  super::selectionChanged(selected,deselected);
}

void WcViewTree::updateMenu( const WcSelection& sel )
{
  // enable/disable actions based on current selection

  _actions->enableAction( ActionCmdReload, sel.isVersionedDir() );

  _actions->enableAction( ActionWcDiffBase, sel.isDiffable() );
  _actions->enableAction( ActionWcAdd, sel.isAddable() );
  _actions->enableAction( ActionWcLog, sel.isVersioned() );
  _actions->enableAction( ActionWcRevert, sel.isRevertable() );
  _actions->enableAction( ActionWcRemove, sel.isRemoveable() );
  _actions->enableAction( ActionWcEditConflict, sel.isConflicted() );
  _actions->enableAction( ActionWcResolved, sel.isConflicted() || sel.isPropConflicted() );
  _actions->enableAction( ActionWcCommit, sel.isCommitable() );
  _actions->enableAction( ActionWcProperties, sel.isVersioned() );
  _actions->enableAction( ActionWcIgnore, sel.isUnversionedAll() );
  _actions->enableAction( ActionWcBlame, sel.isVersionedFile() );
  _actions->enableAction( ActionWcLock, sel.isVersionedFile() );
  _actions->enableAction( ActionWcUnlock, sel.isVersionedFile() );
  _actions->enableAction( ActionWcCleanup, sel.isVersionedDir() );
  _actions->enableAction( ActionWcExport, sel.isVersionedDir() );
  _actions->enableAction( ActionWcMkdir, sel.isVersionedDir() );
  _actions->enableAction( ActionWcUpdateRev, sel.isVersioned() );
  _actions->enableAction( ActionWcBranchTag, sel.isVersioned() );
  _actions->enableAction( ActionWcLogGraph, sel.isVersioned() );
}

void WcViewTree::getSelection( svn::WcStatuss& statuss )
{
  QModelIndexList indexList = selectedIndexes();

  for( QModelIndexList::Iterator it = indexList.begin(); it != indexList.end(); it++ )
  {
    QModelIndex index = _proxyModel->mapToSource(*it);

    if( index.column() != 0 )
      continue;

    const WcViewItem* item = index.data(WcViewTreeItemModel::WcViewItemRole).value<const WcViewItem*>();
    svn::WcStatusPtr status = static_cast<const WcViewStatus*>(item)->status();

    statuss.push_back(status);
  }
}

QMimeData* WcViewTree::mimeData( const QModelIndexList& list )
{
  QMimeData *mimeData = new QMimeData();
  QByteArray encodedData;

  QDataStream stream(&encodedData, QIODevice::WriteOnly);

  foreach( QModelIndex index, list )
  {
    if( index.column() != 0 )
      continue;

    QString value = index.data(WcViewTreeItemModel::DragRole).toString();
    stream << value;
  }

  mimeData->setData( ScMimeTypeWcFilesItem, encodedData );
  return mimeData;
}
